/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef WORKERTHREADPOOL_H_
#define WORKERTHREADPOOL_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "ConnAcceptor.h"
#include "RefCountableSAP.h"
#include "IntrusivePtr.h"
#include "IntrusiveList.h"
#include "TimeStamp.h"
#include <ace/config-lite.h>
#include <ace/SOCK_Stream.h>
#include <ace/Thread_Manager.h>
#include <ace/Synch.h>

class Reactor;
class ServiceContext;

/**
 * \brief Accept and process client connections.
 *
 * This class is completely thread-safe.  Public methods can be called
 * from different threads.  No external synchronization is necessary.
 */
class WorkerThreadPool
{
	DECLARE_NON_COPYABLE(WorkerThreadPool)
public:
	/**
	 * \brief Constructs an initially inactive object.
	 */
	WorkerThreadPool();
	
	/**
	 * \brief Destroys the object, deactivating it first.
	 */
	~WorkerThreadPool();
	
	/**
	 * \brief Start accepting and processing connections.
	 *
	 * If already active, does nothing.
	 * \throw std::runtime_error is thrown in case of a failure.
	 *        Most likely this means resource starvation.
	 */
	void activate();
	
	/**
	 * \brief Stop accepting and processing connections.
	 *
	 * Also signals workers to finish their tasks and waits until
	 * they do so.\n
	 * If already inactive, does nothing.\n
	 * Registered acceptors are not removed.
	 * 
	 */
	void deactivate();
	
	/**
	 * \brief Start accepting connections on \a acceptor.
	 *
	 * The actual accepting won't happen until activate() is called.
	 * If activate() has already been called, there is no need to
	 * call it again.
	 * \throw std::runtime_error is thrown if an error occured while
	 *        registering the acceptor.  Most likely this means
	 *        resource starvation.
	 */
	void addAcceptor(ConnAcceptor::AcceptorPtr const& acceptor);
	
	/**
	 * \brief Stop accepting connections on \a acceptor.
	 *
	 * Does nothing if \a acceptor was not registered with addAcceptor().
	 */
	void removeAcceptor(ConnAcceptor::AcceptorPtr const& acceptor);
	
	/**
	 * \brief Stop accepting connections on all registered acceptors.
	 *
	 * Unlike deactivate() it doesn't signal existing workers to finish.
	 */
	void removeAllAcceptors();
private:
	enum { WAIT_FOR_NEW_TASK = 10 }; // seconds
	enum { WAITERS_KEEP_ALIVE = 4 }; // not counting the Leader
	enum Status { INACTIVE, ACTIVE, STOPPING };
	enum NewThreadStatus { THR_READY, THR_STOPPING, THR_FAILURE };
	
	class ReactorRegistration;
	
	typedef IntrusivePtr<RefCountableSAP<ACE_SOCK_Stream> > SocketPtr;
	typedef IntrusiveList<ReactorRegistration> ReactorList;
	typedef ACE_Thread_Mutex Mutex;
	typedef ACE_Mutex LeaderMutex;
	
	void prepareNewThreadLocked();
	
	int waitLocked();
	
	void runServiceUnlocked();
	
	bool tryLeadershipUnlocked(ServiceContext& service_context);
	
	bool tryLeadershipLocked(ServiceContext& service_context);
	
	static void processConnectionUnlocked(
		ServiceContext& service_context, SocketPtr const& client_sock);
	
	static ACE_THR_FUNC_RETURN runThread(void *arg);
	
	ACE_Thread_Manager m_threadManager;
	Mutex m_startStopMutex; // protects activate() and deactivate()
	Mutex m_mutex;
	ACE_Condition<Mutex> m_cond;
	ReactorList m_runningReactors; // protected by m_mutex
	TimeStamp m_lastThreadCreateTime; // protected by m_mutex
	int m_numWaiters; // number of idle threads; protected by m_mutex
	Status m_status; // protected by m_mutex
	LeaderMutex m_leaderMutex; // protects m_connector.accept()
	ConnAcceptor m_connAcceptor;
};

#endif
