/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "JsEnvironment.h"
#include "JsRuntime.h"
#include "JsRequestScope.h"
#include "JsRequestSuspender.h"
#include "BString.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include "URI.h"
#include "Debug.h"
#include "jsapi.h"
#include <ace/config-lite.h>
#include <ace/Singleton.h>
#include <ace/Synch.h>
#include <string>
#include <cstring>
#include <cctype>
#include <sstream>
#include <algorithm>

using namespace std;


class JsEnvironment::JsException
{
};

class JsEnvironment::Context
{
public:
	Context(URI const& page_url);
	
	~Context();
	
	JSContext* getContext() { return m_pContext; }
	
	bool executeScript(BString const& script,
		char const* filename, char const* js_version=0);
	
	bool executeScriptAsFunction(
		BString const& script, char const* filename, int lineno);
	
	void setListener(Listener& listener) {
		m_listenerLink.setObserver(&listener);
	}
	
	void removeListener() {
		m_listenerLink.setObserver(0);
	}
	
	Listener* getListener() {
		if (m_listenerSuspenderLink.getObserver()) {
			return 0;
		}
		return m_listenerLink.getObserver();
	}
	
	void setListenerSuspender(ListenerSuspender& suspender) {
		m_listenerSuspenderLink.setObserver(&suspender);
	}
	
	void removeListenerSuspender() {
		m_listenerSuspenderLink.setObserver(0);
	}
	
	ListenerSuspender* getListenerSuspender() {
		return m_listenerSuspenderLink.getObserver();
	}
	
	void setPageOpenListener(PageOpenListener& listener) {
		m_poListenerLink.setObserver(&listener);
	}
	
	void removePageOpenListener() {
		m_poListenerLink.setObserver(0);
	}
	
	PageOpenListener* getPageOpenListener() {
		return m_poListenerLink.getObserver();
	}
private:
	enum { STACK_SIZE = 15*1024 };
	enum { BRANCH_LIMIT = 20000 };
	enum Access { RO, RW };
	
	static Context* getEnvContext(JSContext* cx);
	
	static void errorReporter(
		JSContext *cx, const char *msg, JSErrorReport *report);
	
	static JSBool branchCallback(JSContext *cx, JSScript *script);
	
	static JSBool cookieGetter(
		JSContext *cx, JSObject *obj, jsval id, jsval *vp);
	
	static JSBool cookieSetter(
		JSContext *cx, JSObject *obj, jsval id, jsval *vp);
	
	static JSBool onloadSetter(
		JSContext *cx, JSObject *obj, jsval id, jsval *vp);
	
	static JSBool innerHtmlSetter(
		JSContext *cx, JSObject *obj, jsval id, jsval *vp);
	
	static JSBool locationSetter(
		JSContext *cx, JSObject *obj, jsval id, jsval *vp);
	
	static JSBool locationToString(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool windowOpen(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool javaEnabled(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool alert(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool documentWrite(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool documentWriteln(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool secondaryDocumentWrite(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool secondaryDocumentWriteln(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool processDocumentWrite(
		JSContext* cx, uintN argc, jsval* argv, jsval* rval,
		Origin origin, bool newline);
	
	static JSBool documentGetElementById(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool documentGetElementsByTagName(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool imageConstructor(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool doNothing(
		JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	void setJsVersion(char const* version=0);
	
	JSObject* jsNewObject(JSClass *clasp, JSObject *proto, JSObject *parent);
	
	JSObject* jsNewArrayObject(jsint length, jsval *vector);
	
	JSString* jsNewString(BString const& str) {
		return jsNewStringCopyN(str.begin(), str.size());
	}
	
	JSString* jsNewStringCopyZ(const char *s);
	
	JSString* jsNewStringCopyN(const char *s, size_t n);
	
	void jsDefineProperty(JSObject *obj, char const* name, jsval value,
		JSPropertyOp getter, JSPropertyOp setter, uintN flags);
	
	void jsDefineProperty(JSObject* obj, char const* name, jsval value, Access access);
	
	JSFunction* jsDefineFunction(JSObject *obj, char const* name,
		JSNative call, uintN nargs, uintN flags);
	
	JSFunction* jsDefineFunction(
		JSObject *obj, char const* name, JSNative call, uintN nargs, Access access);
	
	JSObject* jsDefineObject(JSObject *obj, char const *name,
		JSClass *clasp, JSObject *proto, uintN flags);
	
	JSObject* jsDefineObject(
		JSObject *obj, char const *name, JSClass *clasp, Access access);
	
	void jsSetElement(JSObject *obj, jsint index, jsval *vp);
	
	void initLocationObject(JSObject* obj, URI const& url);
	
	static JSClass m_sWindowClass;
	static JSClass m_sDocumentClass;
	static JSClass m_sBodyClass;
	static JSClass m_sNavigatorClass;
	static JSClass m_sLocationClass;
	static JSClass m_sScreenClass;
	static JSClass m_sImageClass;
	static JSClass m_sDivClass;
	static JSClass m_sMimetypeClass;
	static JSClass m_sPluginClass;
	SingleObserverLink<Listener> m_listenerLink;
	SingleObserverLink<ListenerSuspender> m_listenerSuspenderLink;
	SingleObserverLink<PageOpenListener> m_poListenerLink;
	JSContext* m_pContext;
	JSObject* m_pWindowObj;
	JSObject* m_pDocumentObj;
	JSObject* m_pFormArray;
	int m_branchCount;
	string m_documentCookie;
};


struct JsEnvironment::VersionPair
{
	char const* strv;
	JSVersion jsv;
};


struct JsEnvironment::VersionComparator
{
	VersionComparator() {}
	
	bool operator()(VersionPair const& lhs, char const* rhs) const {
		return strcmp(lhs.strv, rhs) < 0;
	}
	
	bool operator()(char const* lhs, VersionPair const& rhs) const {
		return strcmp(lhs, rhs.strv) < 0;
	}
};


/*===================== JsEnvironment::Context =========================*/

JSClass JsEnvironment::Context::m_sWindowClass = {
	"Window", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sDocumentClass = {
	"HTMLDocument", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sBodyClass = {
	"HTMLBodyElement", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sNavigatorClass = {
	"Navigator", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sLocationClass = {
	"Location", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sScreenClass = {
	"Screen", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sImageClass = {
	"HTMLImageElement", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sDivClass = {
	"HTMLDivElement", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sMimetypeClass = {
	"MimeType", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sPluginClass = {
	"Plugin", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};


inline void
JsEnvironment::Context::jsDefineProperty(
	JSObject* obj, char const* name, jsval value, Access access)
{
	uintN flags = JSPROP_ENUMERATE|JSPROP_PERMANENT;
	if (access == RO) {
		flags |= JSPROP_READONLY;
	}
	jsDefineProperty(obj, name, value, 0, 0, flags);
}

inline JSFunction*
JsEnvironment::Context::jsDefineFunction(
	JSObject *obj, char const* name, JSNative call, uintN nargs, Access access)
{
	uintN flags = JSPROP_ENUMERATE|JSPROP_PERMANENT;
	if (access == RO) {
		flags |= JSPROP_READONLY;
	}
	return jsDefineFunction(obj, name, call, nargs, flags);
}

inline JSObject*
JsEnvironment::Context::jsDefineObject(
	JSObject *obj, char const *name, JSClass *clasp, Access access)
{
	uintN flags = JSPROP_ENUMERATE|JSPROP_PERMANENT;
	if (access == RO) {
		flags |= JSPROP_READONLY;
	}
	return jsDefineObject(obj, name, clasp, 0, access);
}


JsEnvironment::Context::Context(URI const& page_url)
:	m_branchCount(0)
{
	JSRuntime* runtime = JsRuntime::rep();
	m_pContext = JS_NewContext(runtime, STACK_SIZE);
	if (!m_pContext) {
		throw JsException();
	}
	JsRequestScope rscope(m_pContext);
	
	JS_SetContextPrivate(m_pContext, this);
	JS_SetErrorReporter(m_pContext, &errorReporter);
	JS_SetBranchCallback(m_pContext, &branchCallback);
	
	m_pWindowObj = jsNewObject(&m_sWindowClass, 0, 0);
	if (!JS_InitStandardClasses(m_pContext, m_pWindowObj)) {
		throw JsException();
	}
	
	jsDefineProperty(m_pWindowObj, "self", OBJECT_TO_JSVAL(m_pWindowObj), RW);
	jsDefineProperty(m_pWindowObj, "window", OBJECT_TO_JSVAL(m_pWindowObj), RO);
	jsDefineProperty(m_pWindowObj, "top", OBJECT_TO_JSVAL(m_pWindowObj), RW);
	jsDefineProperty(m_pWindowObj, "parent", OBJECT_TO_JSVAL(m_pWindowObj), RW);
	jsDefineProperty(m_pWindowObj, "history", OBJECT_TO_JSVAL(jsNewArrayObject(0, 0)), RO);
	jsDefineFunction(m_pWindowObj, "alert", &alert, 1, RW);
	jsDefineFunction(m_pWindowObj, "open", &windowOpen, 1, RW);
	jsDefineFunction(m_pWindowObj, "setTimeout", &doNothing, 2, RW);
	jsDefineFunction(m_pWindowObj, "setInterval", &doNothing, 2, RW);
	jsDefineFunction(m_pWindowObj, "clearTimeout", &doNothing, 1, RW);
	jsDefineFunction(m_pWindowObj, "clearInterval", &doNothing, 1, RW);
	jsDefineFunction(m_pWindowObj, "focus", &doNothing, 0, RW);
	jsDefineFunction(m_pWindowObj, "blur", &doNothing, 0, RW);
	jsDefineFunction(m_pWindowObj, "resizeTo", &doNothing, 2, RW);
	jsDefineFunction(m_pWindowObj, "moveTo", &doNothing, 2, RW);
	jsDefineFunction(m_pWindowObj, "Image", &imageConstructor, 0, RW);
	jsDefineProperty(m_pWindowObj, "onload", JSVAL_VOID,
		0, &onloadSetter, JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	
	JSObject* location_obj = jsNewObject(&m_sLocationClass, 0, 0);
	jsDefineProperty(
		m_pWindowObj, "location", OBJECT_TO_JSVAL(location_obj),
		0, &locationSetter, JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	
	initLocationObject(location_obj, page_url);
	
	JSObject* nav_obj = jsDefineObject(m_pWindowObj, "navigator", &m_sNavigatorClass, RW);
	jsDefineFunction(nav_obj, "javaEnabled", &javaEnabled, 0, RW);
	jsDefineProperty(nav_obj, "cookieEnabled", JSVAL_FALSE, RO);
	jsDefineProperty(nav_obj, "platform", STRING_TO_JSVAL(jsNewStringCopyZ("Win32")), RO);
	jsDefineProperty(nav_obj, "appCodeName", STRING_TO_JSVAL(jsNewStringCopyZ("Mozilla")), RO);
	jsDefineProperty(nav_obj, "appName", STRING_TO_JSVAL(jsNewStringCopyZ("Netscape")), RO);
	jsDefineProperty(nav_obj, "appVersion",
		STRING_TO_JSVAL(jsNewStringCopyZ("5.0 (Windows; en-US)")), RO
	);
	jsDefineProperty(nav_obj, "userAgent", STRING_TO_JSVAL(jsNewStringCopyZ(
		"Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.8) Gecko/20051111 Firefox/1.5"
	)), RO);
	
	JSObject* mimetypes_array = jsNewArrayObject(0, 0);
	jsDefineProperty(nav_obj, "mimeTypes", OBJECT_TO_JSVAL(mimetypes_array), RO);
	JSObject* plugins_array = jsNewArrayObject(0, 0);
	jsDefineProperty(nav_obj, "plugins", OBJECT_TO_JSVAL(plugins_array), RO);
	JSObject* flash_mimetype_obj = jsDefineObject(
		mimetypes_array, "application/x-shockwave-flash", &m_sMimetypeClass, RW
	);
	jsval flash_mimetype_jsval = OBJECT_TO_JSVAL(flash_mimetype_obj);
	jsSetElement(mimetypes_array, 0, &flash_mimetype_jsval);
	jsDefineProperty(flash_mimetype_obj, "description",
		STRING_TO_JSVAL(jsNewStringCopyZ("Shockwave Flash 7.0")), RO
	);
	jsDefineProperty(flash_mimetype_obj, "type",
		STRING_TO_JSVAL(jsNewStringCopyZ("application/x-shockwave-flash")), RO
	);
	jsDefineProperty(flash_mimetype_obj, "suffixes",
		STRING_TO_JSVAL(jsNewStringCopyZ("swf")), RO
	);
	JSObject *flash_plugin_obj = jsDefineObject(
		flash_mimetype_obj, "enabledPlugin", &m_sPluginClass, RO
	);
	jsval flash_plugin_jsval = OBJECT_TO_JSVAL(flash_plugin_obj);
	jsSetElement(plugins_array, 0, &flash_plugin_jsval);
	jsDefineProperty(plugins_array, "Shockwave Flash", flash_plugin_jsval, RW);
	jsDefineProperty(flash_plugin_obj, "name",
		STRING_TO_JSVAL(jsNewStringCopyZ("Shockwave Flash")), RO
	);
	jsDefineProperty(flash_plugin_obj, "description",
		STRING_TO_JSVAL(jsNewStringCopyZ("Shockwave Flash 7.0")), RO
	);
	jsDefineProperty(flash_plugin_obj, "length", INT_TO_JSVAL(1), RO);
	
	JSObject* screen_obj = jsDefineObject(m_pWindowObj, "screen", &m_sScreenClass, RO);
	jsDefineProperty(screen_obj, "width", INT_TO_JSVAL(1024), RO);
	jsDefineProperty(screen_obj, "height", INT_TO_JSVAL(768), RO);
	jsDefineProperty(screen_obj, "availWidth", INT_TO_JSVAL(1024), RO);
	jsDefineProperty(screen_obj, "availHeight", INT_TO_JSVAL(738), RO);
	jsDefineProperty(screen_obj, "colorDepth", INT_TO_JSVAL(16), RO);
	jsDefineProperty(screen_obj, "pixelDepth", INT_TO_JSVAL(16), RO);
	
	JSObject *frames_obj = jsNewArrayObject(0, 0);
	jsDefineProperty(m_pWindowObj, "frames", OBJECT_TO_JSVAL(frames_obj), RW);
	
	m_pDocumentObj = jsDefineObject(m_pWindowObj, "document", &m_sDocumentClass, RW);
	jsDefineProperty(m_pDocumentObj, "cookie", JS_GetEmptyStringValue(m_pContext),
		&cookieGetter, &cookieSetter, JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	jsDefineProperty(m_pDocumentObj, "domain", JS_GetEmptyStringValue(m_pContext), RW);
	jsDefineProperty(m_pDocumentObj, "referrer",
		STRING_TO_JSVAL(jsNewStringCopyZ("http://whatever")), RO
	);
	JSObject* images_array = jsNewArrayObject(0, 0);
	jsDefineProperty(m_pDocumentObj, "images", OBJECT_TO_JSVAL(images_array), RO);
	jsDefineProperty(m_pDocumentObj, "URL", JS_GetEmptyStringValue(m_pContext), RO);
	jsDefineFunction(m_pDocumentObj, "write", &documentWrite, 1, RW);
	jsDefineFunction(m_pDocumentObj, "writeln", &documentWriteln, 1, RW);
	jsDefineFunction(m_pDocumentObj, "getElementById", &documentGetElementById, 1, RW);
	jsDefineFunction(m_pDocumentObj, "getElementsByTagName", &documentGetElementsByTagName, 1, RW);
	jsDefineFunction(m_pDocumentObj, "open", &doNothing, 0, RW);
	jsDefineFunction(m_pDocumentObj, "close", &doNothing, 0, RW);
	jsDefineProperty(
		m_pDocumentObj, "location", OBJECT_TO_JSVAL(location_obj),
		0, &locationSetter, JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	
	m_pFormArray = jsNewArrayObject(0, 0);
	jsDefineProperty(m_pDocumentObj, "forms", OBJECT_TO_JSVAL(m_pFormArray), RO);
	
	JSObject* body_obj = jsDefineObject(m_pDocumentObj, "body", &m_sBodyClass, RW);
	jsDefineProperty(body_obj, "clientWidth", INT_TO_JSVAL(1024), RO);
	jsDefineProperty(body_obj, "clientHeight", INT_TO_JSVAL(602), RO);
	jsDefineProperty(body_obj, "innerText", JS_GetEmptyStringValue(m_pContext), RW);
	jsDefineProperty(body_obj, "innerHTML", JS_GetEmptyStringValue(m_pContext), RW);
	jsDefineFunction(body_obj, "getElementsByTagName", &documentGetElementsByTagName, 1, RW);
}

JsEnvironment::Context::~Context()
{
	JS_DestroyContext(m_pContext);
}

bool
JsEnvironment::Context::executeScript(
	BString const& script,
	char const* filename, char const* version)
{
	JsRequestScope rscope(m_pContext);
	JSVersion version_saved = JS_GetVersion(m_pContext);
	setJsVersion(version);
	jsval rval;
	JSBool res = JS_EvaluateScript(
		m_pContext, m_pWindowObj,
		script.begin(), script.size(),
		filename, 1, &rval
	);
	JS_SetVersion(m_pContext, version_saved);
	return (res == JS_TRUE);
}

bool
JsEnvironment::Context::executeScriptAsFunction(
	BString const& script, char const* filename, int lineno)
{
	JsRequestScope rscope(m_pContext);
	
	JSFunction* fun = JS_CompileFunction(
		m_pContext,
		0, // obj
		0, // name
		0, // nargs,
		0, // arguments
		script.begin(),
		script.size(),
		filename,
		lineno
	);
	
	bool success = false;
	
	if (fun) {
		jsval rval;
		JSBool res = JS_CallFunction(
			m_pContext, m_pWindowObj, fun,
			0, // argc
			0, // argv
			&rval
		);
		if (res == JS_TRUE) {
			success = true;
		}
	}
	
	return success;
}


// We don't use RequestScope anywhere below, because the code below is
// called either by Context constructor or by Context::executeScript,
// and is protected by their own request scopes.

void
JsEnvironment::Context::setJsVersion(char const* version)
{
	static VersionPair const versions[] = {
		{ "1.0", JSVERSION_1_0 },
		{ "1.1", JSVERSION_1_1 },
		{ "1.2", JSVERSION_1_2 },
		{ "1.3", JSVERSION_1_3 },
		{ "1.4", JSVERSION_1_4 },
		{ "1.5", JSVERSION_1_5 }
	};
	
	JSVersion jsv = JSVERSION_DEFAULT;
	if (version) {
		VersionPair const* end = versions + sizeof(versions)/sizeof(versions[0]);
		VersionComparator comp;
		VersionPair const* it = std::lower_bound(versions, end, version, comp);
		if (it != end && !comp(version, *it)) {
			jsv = it->jsv;
		}
	}
	JS_SetVersion(m_pContext, jsv);
}

JsEnvironment::Context*
JsEnvironment::Context::getEnvContext(JSContext* cx)
{
	return static_cast<JsEnvironment::Context*>(
		JS_GetContextPrivate(cx)
	);
}

void
JsEnvironment::Context::errorReporter(
	JSContext* cx, char const* msg, JSErrorReport* report)
{
#if defined(DEBUG) && 1
	DEBUGLOG("========= JS ERROR =========\n" << msg
		<< " [" << report->filename << ':' << report->lineno << ']');
	if (!report->linebuf) {
		return;
	}
	string line(report->linebuf);
	std::replace(line.begin(), line.end(), '\t', ' ');
	DEBUGLOG(line);
	line.resize(0);
	line.append(report->tokenptr - report->linebuf, ' ');
	line += '^';
	DEBUGLOG(line);
#endif
}

JSBool
JsEnvironment::Context::branchCallback(JSContext* cx, JSScript* script)
{
	Context* context = getEnvContext(cx);
	if (++context->m_branchCount < BRANCH_LIMIT) {
		return JS_TRUE;
	} else {
		// infinite loop?
		DEBUGLOG("JS branch limit reached!");
		return JS_FALSE; // terminate the script
	}
}

JSBool
JsEnvironment::Context::cookieGetter(
	JSContext* cx, JSObject* obj, jsval id, jsval* vp)
{
	Context* context = getEnvContext(cx);
	JSString* cookie = JS_NewStringCopyN(
		cx, context->m_documentCookie.c_str(),
		context->m_documentCookie.length()
	);
	if (cookie) {
		*vp = STRING_TO_JSVAL(cookie);
		return JS_TRUE;
	} else {
		return JS_FALSE;
	}
}

JSBool
JsEnvironment::Context::cookieSetter(
	JSContext* cx, JSObject* obj, jsval id, jsval* vp)
{
	Context* context = getEnvContext(cx);
	JSString* str = JS_ValueToString(cx, *vp);
	if (str) {
		char const* data = JS_GetStringBytes(str);
		if (!data) {
			return JS_FALSE;
		}
		string cookie(data);
		if (context->m_documentCookie.empty()) {
			context->m_documentCookie = cookie;
		} else {
			// quick and dirty solution to convince some
			// scripts that getting/setting a cookie works
			context->m_documentCookie += "; ";
			context->m_documentCookie += cookie;
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::onloadSetter(
	JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	if (!JSVAL_IS_NULL(vp)) {
		Listener* listener = getEnvContext(cx)->getListener();
		if (listener) {
			listener->processOnLoadAssignment();
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::innerHtmlSetter(
	JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	if (JSVAL_IS_STRING(*vp)) {
		Listener* listener = getEnvContext(cx)->getListener();
		if (listener) {
			JSString* str = JS_ValueToString(cx, *vp);
			if (str) {
				char const* data = JS_GetStringBytes(str);
				listener->processInnerHtmlAssignment(data);
			}
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::locationSetter(
	JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	if (JSVAL_IS_STRING(*vp)) {
		PageOpenListener* listener = getEnvContext(cx)->getPageOpenListener();
		if (listener) {
			JSString* str = JS_ValueToString(cx, *vp);
			if (str) {
				char const* url = JS_GetStringBytes(str);
				listener->processPageOpening(url, "");
			}
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::locationToString(
	JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	jsval href;
	if (!JS_GetProperty(cx, obj, "href", &href)) {
		return JS_FALSE;
	}
	*rval = href;
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::windowOpen(
	JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	if (argc < 1) {
		return JS_TRUE;
	}
	
	JSString* url = JS_ValueToString(cx, argv[0]);
	if (!url) {
		return JS_TRUE;
	}
	char const* url_str = JS_GetStringBytes(url);
	
	char const* target_str = "";
	if (argc > 1) {
		JSString* target = JS_ValueToString(cx, argv[0]);
		if (target) {
			target_str = JS_GetStringBytes(target);
		}
	}
	
	PageOpenListener* listener = getEnvContext(cx)->getPageOpenListener();
	if (listener) {
		listener->processPageOpening(url_str, target_str);
	}
	
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::javaEnabled(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_FALSE;
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::alert(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_VOID;
	if (argc > 0) {
		JSString* str = JS_ValueToString(cx, argv[0]);
		if (str) {
			char const* data = JS_GetStringBytes(str);
			DEBUGLOG("JS alert: " << data);
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::documentWrite(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	return processDocumentWrite(cx, argc, argv, rval, PRIMARY_DOCUMENT, false);
}

JSBool
JsEnvironment::Context::documentWriteln(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	return processDocumentWrite(cx, argc, argv, rval, PRIMARY_DOCUMENT, true);
}

JSBool
JsEnvironment::Context::secondaryDocumentWrite(
	JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	return processDocumentWrite(cx, argc, argv, rval, SECONDARY_DOCUMENT, false);
}

JSBool
JsEnvironment::Context::secondaryDocumentWriteln(
	JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	return processDocumentWrite(cx, argc, argv, rval, SECONDARY_DOCUMENT, true);
}

JSBool
JsEnvironment::Context::processDocumentWrite(
	JSContext* cx, uintN argc, jsval* argv, jsval* rval,
	Origin origin, bool newline)
{
	*rval = JSVAL_VOID;
	if (argc > 0) {
		JSString* str = JS_ValueToString(cx, argv[0]);
		if (str) {
			char const* data = JS_GetStringBytes(str);
			if (!data) {
				return JS_FALSE;
			}
			Listener* listener = getEnvContext(cx)->getListener();
			if (listener) {
				JsRequestSuspender suspender(cx);
				listener->processJsOutput(data, origin, newline);
			}
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::documentGetElementById(
	JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	JSObject* el = JS_NewObject(cx, &m_sDivClass, 0, 0);
	if (!el) {
		*rval = JSVAL_NULL;
		return JS_FALSE;
	}
	
	*rval = OBJECT_TO_JSVAL(el);
	JSBool ok = true;
	
	ok = JS_DefineProperty(
		cx, el, "innerHTML", JS_GetEmptyStringValue(cx),
		0, &innerHtmlSetter, JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	if (!ok) {
		return JS_FALSE;
	}
	
	JSObject* content_document = JS_DefineObject(
		cx, el,"contentDocument", &m_sDocumentClass, 0,
		JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	if (!content_document) {
		return JS_FALSE;
	}
	
	JSFunction* fun = JS_DefineFunction(
		cx, content_document, "write", &secondaryDocumentWrite, 1,
		JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	if (!fun) {
		return JS_FALSE;
	}
	
	fun = JS_DefineFunction(
		cx, content_document, "writeln", &secondaryDocumentWriteln, 1,
		JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	if (!fun) {
		return JS_FALSE;
	}
	
	ok = JS_DefineProperty(
		cx, el, "style", JS_GetEmptyStringValue(cx),
		0, 0, JSPROP_ENUMERATE|JSPROP_READONLY|JSPROP_PERMANENT
	);
	if (!ok) {
		return JS_FALSE;
	}
	
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::documentGetElementsByTagName(
	JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	JSObject* empty_array = JS_NewArrayObject(cx, 0, 0);
	if (!empty_array) {
		return JS_FALSE;
	}
	*rval = OBJECT_TO_JSVAL(empty_array);
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::imageConstructor(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	JSObject* image_obj = JS_NewObject(cx, &Context::m_sImageClass, 0, 0);
	if (!image_obj) {
		return JS_FALSE;
	}
	JSBool res = JS_DefineProperty(
		cx, image_obj, "complete", JSVAL_TRUE, 0, 0,
		JSPROP_ENUMERATE|JSPROP_READONLY|JSPROP_PERMANENT
	);
	if (res) {
		*rval = OBJECT_TO_JSVAL(image_obj);
		return JS_TRUE;
	} else {
		return JS_FALSE;
	}
}

JSBool
JsEnvironment::Context::doNothing(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_VOID;
	return JS_TRUE;
}

JSObject*
JsEnvironment::Context::jsNewObject(JSClass *clasp, JSObject *proto, JSObject *parent)
{
	JSObject* obj = JS_NewObject(m_pContext, clasp, proto, parent);
	if (!obj) {
		throw JsException();
	}
	return obj;
}

JSObject*
JsEnvironment::Context::jsNewArrayObject(jsint length, jsval *vector)
{
	JSObject* obj = JS_NewArrayObject(m_pContext, length, vector);
	if (!obj) {
		throw JsException();
	}
	return obj;
}

JSString*
JsEnvironment::Context::jsNewStringCopyZ(const char *s)
{
	JSString* str = JS_NewStringCopyZ(m_pContext, s);
	if (!str) {
		throw JsException();
	}
	return str;
}

JSString*
JsEnvironment::Context::jsNewStringCopyN(const char *s, size_t n)
{
	JSString* str = JS_NewStringCopyN(m_pContext, s, n);
	if (!str) {
		throw JsException();
	}
	return str;
}

void
JsEnvironment::Context::jsDefineProperty(
	JSObject *obj, char const* name, jsval value,
	JSPropertyOp getter, JSPropertyOp setter, uintN flags)
{
	JSBool rval = JS_DefineProperty(
		m_pContext, obj, name, value, getter, setter, flags
	);
	if (!rval) {
		throw JsException();
	}
}

JSFunction*
JsEnvironment::Context::jsDefineFunction(
	JSObject *obj, char const* name,
	JSNative call, uintN nargs, uintN flags)
{
	JSFunction* fun = JS_DefineFunction(
		m_pContext, obj, name, call, nargs, flags
	);
	if (!fun) {
		throw JsException();
	}
	return fun;
}

JSObject*
JsEnvironment::Context::jsDefineObject(
	JSObject *obj, char const *name,
	JSClass *clasp, JSObject *proto, uintN flags)
{
	JSObject* o = JS_DefineObject(
		m_pContext, obj, name, clasp, proto, flags
	);
	if (!o) {
		throw JsException();
	}
	return o;
}

void
JsEnvironment::Context::jsSetElement(JSObject *obj, jsint index, jsval *vp)
{
	JSBool rval = JS_SetElement(m_pContext, obj, index, vp);
	if (!rval) {
		throw JsException();
	}
}

void
JsEnvironment::Context::initLocationObject(JSObject* obj, URI const& url)
{
	SBOutStream strm(1000);
	
	strm << url.getScheme() << ':';
	jsDefineProperty(
		obj, "protocol",
		STRING_TO_JSVAL(jsNewString(strm.data().toBString())), RW
	);
	strm.data().clear();
	
	jsDefineProperty(
		obj, "hostname",
		STRING_TO_JSVAL(jsNewString(url.getHost())), RW
	);
	
	strm << url.getHost();
	if (url.getPort() != -1) {
		strm << ':' << url.getPort();
	}
	jsDefineProperty(
		obj, "host",
		STRING_TO_JSVAL(jsNewString(strm.data().toBString())), RW
	);
	strm.data().clear();
	
	strm << url.guessPort();
	jsDefineProperty(
		obj, "port",
		STRING_TO_JSVAL(jsNewString(strm.data().toBString())), RW
	);
	strm.data().clear();
	
	jsDefineProperty(
		obj, "pathname",
		STRING_TO_JSVAL(jsNewString(url.getRawPath())), RW
	);
	
	if (!url.getRawQuery().empty()) {
		strm << '#' << url.getRawFragment();
	}
	jsDefineProperty(
		obj, "hash",
		STRING_TO_JSVAL(jsNewString(strm.data().toBString())), RW
	);
	strm.data().clear();
	
	jsDefineProperty(
		obj, "search",
		STRING_TO_JSVAL(jsNewString(url.getRawQuery())), RW
	);
	
	jsDefineProperty(
		obj, "href", STRING_TO_JSVAL(jsNewString(url.toBString())),
		0, &locationSetter, JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	
	jsDefineFunction(obj, "toString", &locationToString, 0, RW);
}


/*============================ JsEnvironment ==============================*/

JsEnvironment::JsEnvironment(URI const& page_url)
{
	try {
		m_ptrContext.reset(new Context(page_url));
	} catch (JsException const&) {}
}

JsEnvironment::~JsEnvironment()
{
}

bool
JsEnvironment::executeScript(BString const& script,
	char const* filename, char const* version)
{
	if (!m_ptrContext.get()) {
		// because of a previous exception
		return false;
	}
	
	return m_ptrContext->executeScript(script, filename, version);
}

bool
JsEnvironment::executeScriptAsFunction(
	BString const& script, char const* filename, int lineno)
{
	if (!m_ptrContext.get()) {
		// because of a previous exception
		return false;
	}
	
	return m_ptrContext->executeScriptAsFunction(script, filename, lineno);
}

void
JsEnvironment::setListener(Listener& listener)
{
	if (m_ptrContext.get()) {
		m_ptrContext->setListener(listener);
	}
}

void
JsEnvironment::removeListener()
{
	if (m_ptrContext.get()) {
		m_ptrContext->removeListener();
	}
}

void
JsEnvironment::setListenerSuspender(ListenerSuspender& suspender)
{
	if (m_ptrContext.get()) {
		m_ptrContext->setListenerSuspender(suspender);
	}
}

void
JsEnvironment::removeListenerSuspender()
{
	if (m_ptrContext.get()) {
		m_ptrContext->removeListenerSuspender();
	}
}

void
JsEnvironment::setPageOpenListener(PageOpenListener& listener)
{
	if (m_ptrContext.get()) {
		m_ptrContext->setPageOpenListener(listener);
	}
}

void
JsEnvironment::removePageOpenListener()
{
	if (m_ptrContext.get()) {
		m_ptrContext->removePageOpenListener();
	}
}
