/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef REACTORHELPERS_H_
#define REACTORHELPERS_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Reactor.h"
#include "EventHandler.h"
#include "NonCopyable.h"
#include "TimeStamp.h"
#include "TimeDelta.h"
#include <ace/config-lite.h>
#include <boost/multi_index_container.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/ordered_index.hpp>
#include <boost/multi_index/sequenced_index.hpp>

class ACE_Lock;

namespace ReactorHelpers {

using namespace boost::multi_index;
typedef Reactor::IOEvents IOEvents;
typedef Reactor::EventHandlerPtr EventHandlerPtr;


struct Handler
{
	EventHandlerPtr handler;
	ACE_HANDLE handle;
	IOEvents events;
	mutable unsigned flags; // implementation defined
	mutable void* opaque;   // implementation defined
	
	Handler(ACE_HANDLE handle, EventHandlerPtr const& handler,
	        IOEvents events, unsigned flags, void* opaque)
	: handler(handler), handle(handle), events(events),
	flags(flags), opaque(opaque) {}
};


class SequenceTag {};
class HandleTag {};

typedef multi_index_container<
	Handler,
	indexed_by<
		sequenced<tag<SequenceTag> >,
		ordered_unique<
			tag<HandleTag>,
			member<Handler, ACE_HANDLE, &Handler::handle>
		>
	>
> HandlerContainer;


class HandlerRepository : public HandlerContainer
{
	DECLARE_NON_COPYABLE(HandlerRepository)
public:
	typedef index<HandleTag>::type HandleIdx;
	typedef HandleIdx::iterator HandleIter;
	
	HandlerRepository() : first_inactive(end()) {}
	
	ReactorHandlerId findByHandle(ACE_HANDLE handle) const;
	
	ReactorHandlerId add(Handler const& new_handler);
	
	void remove(ReactorHandlerId const& id);
	
	void setEvents(iterator const& it, IOEvents events);
	
	static ReactorHandlerId iterToId(iterator const& it) {
		return ReactorHandlerId(it);
	}
	
	static iterator const& iterFromId(ReactorHandlerId const& id) {
		return id.rep<iterator>();
	}
	
	iterator first_inactive;
};


struct Timer
{
	enum { DISPATCHING = 1, REMOVED = 2 };
	EventHandlerPtr handler;
	TimeStamp abs_time;
	
	Timer(EventHandlerPtr const& handler, TimeStamp const& abs_time)
	: handler(handler), abs_time(abs_time) {}
	
	bool operator<(Timer const& rhs) const { return abs_time < rhs.abs_time; }
};


typedef multi_index_container<
	Timer,
	indexed_by<
		ordered_non_unique<identity<Timer> >
	>
> TimerContainer;


class TimerQueue : public TimerContainer
{
	DECLARE_NON_COPYABLE(TimerQueue)
public:
	TimerQueue();
	
	~TimerQueue();
	
	ReactorTimerId add(EventHandlerPtr const& handler, TimeDelta const* timeout);
	
	void reschedule(ReactorTimerId const& id, TimeDelta const* timeout);
	
	TimeDelta getRemainingTime(ReactorTimerId const& id) const;
	
	TimeStamp getEarliestTime() const;
	
	void updateDispatchThreshold();
	
	void continueDispatching(ACE_Lock& mutex, bool& is_stopped);
	
	static ReactorTimerId iterToId(iterator const& it) {
		return ReactorTimerId(it);
	}
	
	static iterator const& iterFromId(ReactorTimerId const& id) {
		return id.rep<iterator>();
	}
private:
	TimeStamp timeoutToAbsTime(TimeDelta const* timeout) const;
	
	void ensureTimeAfterThreshold(TimeStamp& abs_time) const;
	
	TimeStamp m_dispatchThreshold;
};


class DemultiplexerGuard
{
	DECLARE_NON_COPYABLE(DemultiplexerGuard);
public:
	DemultiplexerGuard(
		ACE_Lock& mutex, ACE_Lock& demux_mutex, bool& is_inside_demux);
	
	~DemultiplexerGuard();
private:
	ACE_Lock& m_rMutex;
	ACE_Lock& m_rDemuxMutex;
	bool& m_rInsideDemux;
};

} // namespace ReactorHelpers

#endif
