/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Timer.h"
#include "IntrusivePtr.h"
#include <cassert>

Timer::Timer(sigc::slot<void> const& callback,
	TimeDelta const& delay, Reactor* reactor)
:	m_pReactor(0),
	m_refCounter(0),
	m_callback(callback),
	m_delay(delay),
	m_isActive(false)
{
	if (reactor) {
		connectToReactor(*reactor);
	}
}

Timer::~Timer()
{
	disconnectFromReactor();
}

void
Timer::connectToReactor(Reactor& reactor)
{
	if (m_pReactor) {
		disconnectFromReactor();
	}
	
	assert(!m_pReactor);
	assert(!m_isActive);
	
	m_pReactor = &reactor;
	m_id = reactor.registerTimer(IntrusivePtr<EventHandlerBase>(this));
}

void
Timer::disconnectFromReactor()
{
	if (m_pReactor) {
		m_pReactor->unregisterTimer(m_id);
		// m_refCounter should reach zero now
		assert(!m_pReactor);
		m_isActive = false;
	}
}

void
Timer::deactivate()
{
	if (!m_pReactor) {
		return;
	}
	
	if (m_isActive) {
		m_pReactor->rescheduleTimer(m_id);
		m_isActive = false;
	}
}

void
Timer::reactivate()
{
	if (!m_pReactor) {
		return;
	}
	
	m_pReactor->rescheduleTimer(m_id, &m_delay);
	m_isActive = true;
}

void
Timer::handleTimeout(ReactorTimerId const&)
{
	m_isActive = false;
	m_callback();
}

void
Timer::ref()
{
	++m_refCounter;
}

bool
Timer::unref()
{
	if (--m_refCounter == 0) {
		m_pReactor = 0;
		m_id = ReactorTimerId();
	}
	return true;
}
