/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef TIMEDELTA_H_
#define TIMEDELTA_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "types.h"
#include <ace/os_include/os_time.h>
#include <limits>

class TimeDelta
{
public:
	TimeDelta() : m_usec(0) {}
	
	static TimeDelta fromSec(int64_t sec);
	
	static TimeDelta fromMsec(int64_t msec);
	
	static TimeDelta fromUsec(int64_t usec);
	
	static TimeDelta fromTimeval(struct timeval const& tv);
	
	static TimeDelta fromTimespec(struct timespec const& ts);
	
	int64_t toSec() const;
	
	int64_t toMsec() const;
	
	int64_t toUsec() const;
	
	struct timeval toTimeval() const;
	
	struct timespec toTimespec() const;
	
	TimeDelta& operator+=(TimeDelta const& delta);
	
	TimeDelta& operator-=(TimeDelta const& delta);
	
	static TimeDelta min();
	
	static TimeDelta max();
	
	static TimeDelta zero();
private:
	TimeDelta(int64_t usec) : m_usec(usec) {}
	
	int64_t m_usec;
};

TimeDelta operator+(TimeDelta const& lhs, TimeDelta const& rhs);

TimeDelta operator-(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator==(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator!=(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator<(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator>(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator<=(TimeDelta const& lhs, TimeDelta const& rhs);

bool operator>=(TimeDelta const& lhs, TimeDelta const& rhs);



inline TimeDelta
TimeDelta::fromSec(int64_t sec)
{
	return TimeDelta(sec * 1000000L);
}

inline TimeDelta
TimeDelta::fromMsec(int64_t msec)
{
	return TimeDelta(msec * 1000L);
}

inline TimeDelta
TimeDelta::fromUsec(int64_t usec)
{
	return TimeDelta(usec);
}

inline TimeDelta
TimeDelta::fromTimeval(struct timeval const& tv)
{
	int64_t usec = int64_t(tv.tv_sec) * 1000000L;
	usec += tv.tv_usec;
	return TimeDelta(usec);
}

inline TimeDelta
TimeDelta::fromTimespec(struct timespec const& ts)
{
	int64_t usec = int64_t(ts.tv_sec) * 1000000L;
	usec += ts.tv_nsec / 1000L;
	return TimeDelta(usec);
}

inline int64_t
TimeDelta::toSec() const
{
	return m_usec / 1000000L;
}

inline int64_t
TimeDelta::toMsec() const
{
	return m_usec / 1000L;
}

inline int64_t
TimeDelta::toUsec() const
{
	return m_usec;
}

inline struct timeval
TimeDelta::toTimeval() const
{
	struct timeval tv;
	tv.tv_sec = m_usec / 1000000L;
	tv.tv_usec = m_usec % 1000000L;
	return tv;
}

inline struct timespec
TimeDelta::toTimespec() const
{
	struct timespec ts;
	ts.tv_sec = m_usec / 1000000L;
	ts.tv_nsec = (m_usec % 1000000L) * 1000L;
	return ts;
}

inline TimeDelta&
TimeDelta::operator+=(TimeDelta const& delta)
{
	m_usec += delta.m_usec;
	return *this;
}

inline TimeDelta&
TimeDelta::operator-=(TimeDelta const& delta)
{
	m_usec -= delta.m_usec;
	return *this;
}

inline TimeDelta
TimeDelta::min()
{
	return TimeDelta(std::numeric_limits<int64_t>::min());
}

inline TimeDelta
TimeDelta::max()
{
	return TimeDelta(std::numeric_limits<int64_t>::max());
}

inline TimeDelta
TimeDelta::zero()
{
	return TimeDelta();
}

inline TimeDelta operator+(TimeDelta const& lhs, TimeDelta const& rhs)
{
	return TimeDelta::fromUsec(lhs.toUsec() + rhs.toUsec());
}

inline TimeDelta operator-(TimeDelta const& lhs, TimeDelta const& rhs)
{
	return TimeDelta::fromUsec(lhs.toUsec() - rhs.toUsec());
}

inline bool operator==(TimeDelta const& lhs, TimeDelta const& rhs)
{
	return lhs.toUsec() == rhs.toUsec();
}

inline bool operator!=(TimeDelta const& lhs, TimeDelta const& rhs)
{
	return lhs.toUsec() != rhs.toUsec();
}

inline bool operator<(TimeDelta const& lhs, TimeDelta const& rhs)
{
	return lhs.toUsec() < rhs.toUsec();
}

inline bool operator>(TimeDelta const& lhs, TimeDelta const& rhs)
{
	return lhs.toUsec() > rhs.toUsec();
}

inline bool operator<=(TimeDelta const& lhs, TimeDelta const& rhs)
{
	return lhs.toUsec() <= rhs.toUsec();
}

inline bool operator>=(TimeDelta const& lhs, TimeDelta const& rhs)
{
	return lhs.toUsec() >= rhs.toUsec();
}

#endif
