# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#
# Authors: Quinn Storm (quinn@beryl-project.org)
# Copyright (C) 2007 Quinn Storm


#libberylsettings header
#ifndef LIB_BERYL_SETTINGS
#define LIB_BERYL_SETTINGS

#include <glib.h>
#include <beryl.h>
# as noted, all values in the structs are in the current locale, not C
# libberylsettings transparently handles translation back to C for
#		the options of type STRING and LIST[STRING] that have a list
#		of allowed values set.

#cdef struct _BerylSettingsContext BerylSettingsContext
#cdef struct _BerylSetting BerylSetting

#cdef gboolean (*BerylSettingsContextReadInitFunc) (BerylSettingsContext * context)
#cdef void (*BerylSettingsContextReadDoneFunc) (BerylSettingsContext * context)
#cdef gboolean (*BerylSettingsContextWriteInitFunc) (BerylSettingsContext * context)
#cdef void (*BerylSettingsContextWriteDoneFunc) (BerylSettingsContext * context)
#cdef void (*BerylSettingsContextReadSettingFunc)
#	(BerylSettingsContext * context, BerylSetting * setting)
#cdef void (*BerylSettingsContextWriteSettingFunc)
#	(BerylSettingsContext * context, BerylSetting * setting)

cdef struct GSList:
	void * data
	GSList * next

#THIS MAY NOT WORK RIGHT ON ALL ARCH
#TODO - test!
ctypedef int gboolean
ctypedef int gint
ctypedef char gchar
ctypedef unsigned int guint
ctypedef double gdouble
ctypedef void * gpointer

cdef enum BerylSettingConflictType:
	BERYL_SETTING_CONFLICT_TYPE_KEY,
	BERYL_SETTING_CONFLICT_TYPE_BUTTON,
	BERYL_SETTING_CONFLICT_TYPE_EDGE,
	BERYL_SETTING_CONFLICT_TYPE_ANY

cdef enum BerylSettingType:
	BERYL_SETTING_TYPE_BOOL,
	BERYL_SETTING_TYPE_INT,
	BERYL_SETTING_TYPE_FLOAT,
	BERYL_SETTING_TYPE_STRING,
	BERYL_SETTING_TYPE_COLOR,
	BERYL_SETTING_TYPE_BINDING,
	BERYL_SETTING_TYPE_LIST,
	BERYL_SETTING_TYPE_COUNT

cdef extern struct BerylSettingsBackend
cdef extern struct BerylSettingsContext
cdef extern struct BerylSettingsPluginCategory
cdef extern struct BerylSettingsSubGroup
cdef extern struct BerylSettingsGroup
cdef extern struct BerylSettingsPlugin
cdef extern struct BerylSettingValue
cdef extern struct BerylSetting

cdef struct BerylSettingConflict:
	BerylSettingConflictType type
	GSList * list


cdef struct BerylSettingColorValueColor:
	gint red
	gint green
	gint blue
	gint alpha

cdef struct BerylSettingColorValueArray:
	gint array[4]

cdef union BerylSettingColorValue:
	BerylSettingColorValueColor color
	BerylSettingColorValueArray array


cdef extern BerylSettingsContext * beryl_settings_context_new()

cdef extern gboolean beryl_settings_context_set_backend(BerylSettingsContext * context,
	gchar * backend)

cdef extern void beryl_settings_context_destroy(BerylSettingsContext * context)

cdef extern GSList * beryl_settings_plugin_get_groups(BerylSettingsPlugin * plugin)

cdef extern BerylSettingsGroup * beryl_settings_plugin_find_group(BerylSettingsPlugin * plugin,
		gchar * group)

cdef extern gchar * beryl_settings_group_get_name(BerylSettingsGroup * group)

cdef extern GSList * beryl_settings_group_get_subgroups(BerylSettingsGroup * group)

cdef extern BerylSettingsSubGroup * beryl_settings_group_find_subgroup(BerylSettingsGroup * group,
		gchar * subgroup)

cdef extern gchar * beryl_settings_subgroup_get_name(BerylSettingsSubGroup * group)

cdef extern GSList * beryl_settings_subgroup_get_settings(BerylSettingsSubGroup * subGroup)

cdef extern BerylSetting * beryl_settings_subgroup_find_setting(BerylSettingsSubGroup * subGroup,
		gchar * setting)

cdef extern gboolean beryl_setting_has_hint(BerylSetting * setting, gchar * hint)

cdef extern BerylSettingsPlugin * beryl_settings_context_find_plugin(
						BerylSettingsContext * context, gchar * name)

cdef extern BerylSetting * beryl_settings_plugin_find_setting(
						BerylSettingsPlugin * plugin, gchar * name, gboolean is_screen)

cdef extern void beryl_settings_context_read(BerylSettingsContext * context)

cdef extern void beryl_settings_context_write(BerylSettingsContext * context)

cdef extern BerylSettingsContext * beryl_settings_context_comp_new()

cdef extern gboolean beryl_settings_send_reload_signal()

cdef extern void beryl_setting_reset_to_default(BerylSetting * setting)

cdef extern void beryl_setting_list_clear(BerylSetting * setting)

cdef extern BerylSettingValue * beryl_setting_list_append(BerylSetting * setting)

cdef extern gint beryl_setting_list_length(BerylSetting * setting)

cdef extern gint beryl_setting_list_value_index(BerylSettingValue * value)

cdef extern void beryl_setting_list_value_swap_with(BerylSettingValue * value, guint n)

cdef extern void beryl_setting_list_value_move_before(BerylSettingValue * value,
				guint n)

cdef extern void beryl_setting_list_value_remove(BerylSettingValue * value)

cdef extern BerylSettingValue * beryl_setting_get_primary_value(BerylSetting * setting)

#these are value get functions
cdef extern gboolean beryl_setting_value_get_int(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_get_float(BerylSettingValue * value, gdouble * data)
cdef extern gboolean beryl_setting_value_get_bool(BerylSettingValue * value, gboolean * data)
cdef extern gboolean beryl_setting_value_get_string(BerylSettingValue * value, gchar ** data)
cdef extern gboolean beryl_setting_value_get_keysym(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_get_keymods(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_get_button(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_get_buttonmods(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_get_bell(BerylSettingValue * value, gboolean * data)
cdef extern gboolean beryl_setting_value_get_edgemask(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_get_key_enabled(BerylSettingValue * value, gboolean * data)
cdef extern gboolean beryl_setting_value_get_button_enabled(BerylSettingValue * value, gboolean * data)
cdef extern gboolean beryl_setting_value_get_color(BerylSettingValue * value, BerylSettingColorValue * data)

#returns list of BerylSettingValue's that is the list value
cdef extern gboolean beryl_setting_value_get_value_list(BerylSettingValue * value, GSList ** data)

#these are value set functions
cdef extern gboolean beryl_setting_value_set_int(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_set_float(BerylSettingValue * value, gdouble * data)
cdef extern gboolean beryl_setting_value_set_bool(BerylSettingValue * value, gboolean * data)
cdef extern gboolean beryl_setting_value_set_string(BerylSettingValue * value, gchar ** data)
cdef extern gboolean beryl_setting_value_set_keysym(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_set_keymods(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_set_button(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_set_buttonmods(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_set_bell(BerylSettingValue * value, gboolean * data)
cdef extern gboolean beryl_setting_value_set_edgemask(BerylSettingValue * value, gint * data)
cdef extern gboolean beryl_setting_value_set_key_enabled(BerylSettingValue * value, gboolean * data)
cdef extern gboolean beryl_setting_value_set_button_enabled(BerylSettingValue * value, gboolean * data)
cdef extern gboolean beryl_setting_value_set_color(BerylSettingValue * value, BerylSettingColorValue * data)

#these are info get functions
cdef extern gboolean beryl_setting_get_can_set_key(BerylSetting * value, gboolean * data)
cdef extern gboolean beryl_setting_get_can_set_button(BerylSetting * value, gboolean * data)
cdef extern gboolean beryl_setting_get_can_set_edgemask(BerylSetting * value, gboolean * data)
cdef extern gboolean beryl_setting_get_can_set_bell(BerylSetting * value, gboolean * data)
cdef extern gboolean beryl_setting_get_allowed_strings(BerylSetting * value, GSList ** data)
cdef extern gboolean beryl_setting_get_raw_allowed_strings(BerylSetting * value, GSList ** data)
cdef extern gboolean beryl_setting_get_int_min(BerylSetting * value, gint * data)
cdef extern gboolean beryl_setting_get_int_max(BerylSetting * value, gint * data)
cdef extern gboolean beryl_setting_get_float_min(BerylSetting * value, gdouble * data)
cdef extern gboolean beryl_setting_get_float_max(BerylSetting * value, gdouble * data)
cdef extern gboolean beryl_setting_get_float_precision(BerylSetting * value, gdouble * data)
cdef extern gboolean beryl_setting_get_list_type(BerylSetting * value, BerylSettingType * data)

#these are info functions
cdef extern BerylSettingType beryl_setting_get_type(BerylSetting * setting)
cdef extern gchar * beryl_setting_get_short_desc(BerylSetting * setting)
cdef extern gchar * beryl_setting_get_long_desc(BerylSetting * setting)
cdef extern gchar * beryl_setting_get_group(BerylSetting * setting)
cdef extern gchar * beryl_setting_get_subgroup(BerylSetting * setting)
cdef extern gchar * beryl_setting_get_name(BerylSetting * setting)
cdef extern gchar * beryl_settings_plugin_get_short_desc(BerylSettingsPlugin * plugin)
cdef extern gchar * beryl_settings_plugin_get_long_desc(BerylSettingsPlugin * plugin)
cdef extern gchar * beryl_settings_plugin_get_name(BerylSettingsPlugin * plugin)

#context functions
cdef extern GSList * beryl_settings_context_get_plugins(BerylSettingsContext * context)
cdef extern GSList * beryl_settings_plugin_get_settings(BerylSettingsPlugin * plugin)

#privates
cdef extern gpointer beryl_settings_context_get_private(BerylSettingsContext * context)
cdef extern gpointer beryl_settings_plugin_get_private(BerylSettingsPlugin * plugin)
cdef extern gpointer beryl_setting_get_private(BerylSetting * setting)
cdef extern void beryl_settings_context_set_private(BerylSettingsContext * context, gpointer private_ptr)
cdef extern void beryl_settings_plugin_set_private(BerylSettingsPlugin * plugin, gpointer private_ptr)
cdef extern void beryl_setting_set_private(BerylSetting * setting, gpointer private_ptr)

#utility functions
cdef extern unsigned int beryl_settings_get_mods_and_endptr(gchar * src, gchar ** ret)
cdef extern gchar * beryl_settings_mods_to_string(unsigned int mods)

cdef extern int beryl_settings_get_plugin_category_count()
cdef extern BerylSettingsPluginCategory * beryl_settings_context_get_nth_plugin_category(BerylSettingsContext * context, int i)
cdef extern gchar * beryl_settings_plugin_category_get_name(BerylSettingsPluginCategory * category)
cdef extern gchar * beryl_settings_plugin_category_get_short_desc(BerylSettingsPluginCategory * category)
cdef extern gchar * beryl_settings_plugin_category_get_long_desc(BerylSettingsPluginCategory * category)
cdef extern GSList * beryl_settings_plugin_category_get_plugins(BerylSettingsPluginCategory * category)

cdef extern GSList * beryl_settings_plugin_get_provides(BerylSettingsPlugin * plugin)
cdef extern GSList * beryl_settings_plugin_get_requires(BerylSettingsPlugin * plugin)
cdef extern GSList * beryl_settings_plugin_get_load_before(BerylSettingsPlugin * plugin)
cdef extern GSList * beryl_settings_plugin_get_load_after(BerylSettingsPlugin * plugin)

cdef extern gchar * beryl_settings_group_get_desc(BerylSettingsGroup * group)
cdef extern gchar * beryl_settings_subgroup_get_desc(BerylSettingsSubGroup * subGroup)

cdef extern void beryl_settings_context_set_profile(BerylSettingsContext * context, gchar * profile)
cdef extern gchar * beryl_settings_context_get_profile(BerylSettingsContext * context)

cdef extern void beryl_settings_context_set_de_integration_enabled(BerylSettingsContext * context,
		gboolean enable_integration)
cdef extern gboolean beryl_settings_context_get_de_integration_enabled(BerylSettingsContext * context)

cdef extern gboolean beryl_settings_context_get_setting_is_read_only(BerylSettingsContext * context,
		BerylSetting * setting)
cdef extern gboolean beryl_settings_context_get_setting_is_de_integrated(BerylSettingsContext * context,
		BerylSetting * setting)

cdef extern gchar * beryl_settings_context_get_backend(BerylSettingsContext * context)

cdef extern gboolean beryl_setting_get_is_advanced(BerylSetting * setting)

cdef extern gboolean beryl_settings_context_import_from_file(BerylSettingsContext * context,
		gchar * filename, gboolean overwrite)
cdef extern gboolean beryl_settings_context_export_to_file(BerylSettingsContext * context,
		gchar * filename)

cdef extern GSList * beryl_settings_context_get_existing_profiles(BerylSettingsContext * context)
cdef extern void beryl_settings_free_profile_list(GSList * list)

cdef extern BerylSettingsPlugin * beryl_setting_get_plugin(BerylSetting * setting)
cdef extern BerylSetting * beryl_settings_context_find_first_edge_owner(BerylSettingsContext * context, gint edgemask)

cdef extern BerylSettingConflictType beryl_setting_conflict_get_type(BerylSettingConflict * conflict)
cdef extern GSList * beryl_setting_conflict_get_settings(BerylSettingConflict * conflict)
cdef extern GSList * beryl_settings_context_find_conflicts(BerylSettingsContext * context, BerylSettingConflictType type)
cdef extern GSList * beryl_settings_context_find_conflicts_for_setting(BerylSettingsContext * context, BerylSetting * setting, BerylSettingConflictType type)
cdef extern void beryl_settings_free_conflict_list(GSList * list)

cdef extern GSList * beryl_settings_context_get_active_plugins(BerylSettingsContext * context)
cdef extern gboolean beryl_settings_plugin_enable(BerylSettingsPlugin * plugin)
cdef extern gboolean beryl_settings_plugin_disable(BerylSettingsPlugin * plugin)
cdef extern gboolean beryl_settings_plugin_get_is_enabled(BerylSettingsPlugin * plugin)

cdef extern GSList * beryl_settings_get_backends()
cdef extern gchar * beryl_settings_backend_get_name(BerylSettingsBackend * backend)
cdef extern gchar * beryl_settings_backend_get_short_desc(BerylSettingsBackend * backend)
cdef extern gboolean beryl_settings_backend_get_supports_integration(BerylSettingsBackend * backend)
cdef extern gboolean beryl_settings_delete_profile(BerylSettingsContext * context, gchar * profile)
#endif

#glib
cdef extern void g_slist_free(GSList * l)

#X11
cdef extern char * XKeysymToString (int keysym)
cdef extern int XStringToKeysym (gchar * string)

cdef plugin_get_name(BerylSettingsPlugin * plugin):
	cdef gchar * s
	s=beryl_settings_plugin_get_name(plugin)
	if s==NULL:
		p = '_'
	else:
		p = s
	return p

def keytostring(keysym):
	cdef char * s
	s=XKeysymToString(keysym)
	if (s==NULL):
		return "None"
	return s

def modstostring(mods):
	cdef char * s
	s=beryl_settings_mods_to_string(mods)
	if (s==NULL):
		return ""
	return s

def buttontostring(button):
	if (button==0):
		return "None"
	return "Button%d"%button

def edgelisttomask(edge):
	edges=(
			(1,"Left"),
			(2,"Right"),
			(4,"Top"),
			(8,"Bottom"),
			(16,"TopLeft"),
			(32,"TopRight"),
			(64,"BottomLeft"),
			(128,"BottomRight"))
	ret=0
	for each in edge:
		for ev in edges:
			if (str(each).lower()==ev[1].lower()):
				ret=ret|ev[0]
	return ret

def edgetolist(edge):
	edges=(
			(1,"Left"),
			(2,"Right"),
			(4,"Top"),
			(8,"Bottom"),
			(16,"TopLeft"),
			(32,"TopRight"),
			(64,"BottomLeft"),
			(128,"BottomRight"))
	ret=[]
	for p in edges:
		if (edge & p[0]):
			ret = ret + [ p[1]]
	return ret

def send_reload():
	beryl_settings_send_reload_signal()

class PluginSetEnabledException(Exception):
	def __init__(self,plugin,type):
		self.plugin=str(plugin)
		self.type=str(type)
	def __str__(self):
		return repr('Could not %s %s, dependencies not met.'%(self.type,self.plugin))

class PluginEnableException(PluginSetEnabledException):
	def __init__(self,plugin):
		PluginSetEnabledException.__init__(self,plugin,"enable")

class PluginDisableException(PluginSetEnabledException):
	def __init__(self,plugin):
		PluginSetEnabledException.__init__(self,plugin,"disable")

class NotFoundException(Exception):
	def __init__(self,type,value,owner):
		self.value=str(value)
		self.type=str(type)
		self.owner=str(owner)
	def __str__(self):
		return repr('Could not find %s named "%s" in "%s"'%(self.type,self.value,self.owner))

class SettingNotFoundException(NotFoundException):
	def __init__(self,value,owner):
		NotFoundException.__init__(self,"setting",value,owner)

class GroupNotFoundException(NotFoundException):
	def __init__(self,value,owner):
		NotFoundException.__init__(self,"group",value,owner)

class PluginNotFoundException(NotFoundException):
	def __init__(self,value,owner):
		NotFoundException.__init__(self,"plugin",value,owner)

class SubGroupNotFoundException(NotFoundException):
	def __init__(self,value,owner):
		NotFoundException.__init__(self,"subgroup",value,owner)

def Backends():
	cdef GSList * l
	cdef BerylSettingsBackend * b
	l = beryl_settings_get_backends()
	ret=[]
	if (l == NULL):
		return ret
	while(l!=NULL):
		b=<BerylSettingsBackend *>l.data
		ret=ret+[Backend(beryl_settings_backend_get_name(b))]
		l=l.next
	return ret

cdef class Backend:
	cdef BerylSettingsBackend * myBackend

	def __new__(self,nam):
		cdef GSList * l
		cdef BerylSettingsBackend * b
		l = beryl_settings_get_backends()
		while(l!=NULL):
			b = <BerylSettingsBackend *>l.data
			if (beryl_settings_backend_get_name(b)==nam):
				self.myBackend=b
				break
			l=l.next

	property Name:
		def __get__(self):
			return beryl_settings_backend_get_name(self.myBackend)

	property ShortDesc:
		def __get__(self):
			return beryl_settings_backend_get_short_desc(self.myBackend)

	property IntegrationSupport:
		def __get__(self):
			val = beryl_settings_backend_get_supports_integration(self.myBackend)
			if (val):
				return True
			return False

	def __str__(self):
		return "%s backend"%self.Name

cdef class Context:
	cdef BerylSettingsContext * myContext
	cdef char * backend

	def __new__(self):
		self.myContext=beryl_settings_context_new()
		self.backend=beryl_settings_context_get_backend(self.myContext)

	def __dealloc__(self):
		beryl_settings_context_destroy(self.myContext)

	property Backend:
		def __get__(self):
			return beryl_settings_context_get_backend(self.myContext)

		def __set__(self,backend):
			beryl_settings_context_set_backend(self.myContext,backend)

	property Categories:
		def __get__(self):
			cdef int i
			cdef gchar * n
			cdef BerylSettingsPluginCategory * pc
			i=beryl_settings_get_plugin_category_count()
			ret=[]
			for p in range(0,i):
				ret=ret+[Category(self,p)]
			return ret

	property Plugins:
		def __get__(self):
			cdef GSList * l
			l = beryl_settings_context_get_plugins(self.myContext)
			ret=[]
			while (l!=NULL):
				snam=plugin_get_name(<BerylSettingsPlugin *>l.data)
				ret = ret + [Plugin(self,snam)]
				l=l.next
			return ret

	property Profile:
		def __get__(self):
			cdef gchar * s
			s=beryl_settings_context_get_profile(self.myContext)
			if (s==NULL):
				return None
			else:
				return s
		def __set__(self,val):
			cdef gchar * s
			if val == None:
				s = NULL
			else:
				s=val
			beryl_settings_context_set_profile(self.myContext,s)

	property Profiles:
		def __get__(self):
			cdef GSList * l
			cdef gchar * s
			l=beryl_settings_context_get_existing_profiles(self.myContext)
			ret=[]
			while(l!=NULL):
				s = <gchar *>l.data
				ret=ret+[s]
				l=l.next
			beryl_settings_free_profile_list(l)
			return ret

	property Integrated:
		def __get__(self):
			if beryl_settings_context_get_de_integration_enabled(self.myContext):
				return True
			else:
				return False

		def __set__(self,isIntegrated):
			cdef gboolean b
			if (isIntegrated):
				b=True
			else:
				b=False
			beryl_settings_context_set_de_integration_enabled(self.myContext,b)

	property ActivePlugins:
		def __get__(self):
			cdef GSList * l
			cdef GSList * m
			ret=[]
			l = beryl_settings_context_get_active_plugins(self.myContext)
			m = l
			while(m!=NULL):
				s=plugin_get_name(<BerylSettingsPlugin *>m.data)
				ret=ret+[s]
				m=m.next
			g_slist_free(l)
			return ret

	def EdgeOwner(self,edge):
		cdef BerylSettingsPlugin * p
		cdef BerylSetting * s
		cdef gint edm
		edm=edge
		s=beryl_settings_context_find_first_edge_owner(self.myContext,edm)
		if (s!=NULL):
			p=beryl_setting_get_plugin(s)
			return Setting(self.Plugin(plugin_get_name(p)),beryl_setting_get_name(s))
		else:
			return None

	def Conflicts(self,t=BERYL_SETTING_CONFLICT_TYPE_ANY):
		cdef GSList * l
		cdef GSList * m
		cdef GSList * n
		cdef BerylSettingConflictType type
		type=t
		cdef BerylSettingConflict * c
		cdef BerylSetting * s
		cdef BerylSettingsPlugin * p
		l=beryl_settings_context_find_conflicts(self.myContext,type)
		m=l
		ret=[]
		while(m!=NULL):
			oret=[]
			c=<BerylSettingConflict *>m.data
			rtype=beryl_setting_conflict_get_type(c)
			n=beryl_setting_conflict_get_settings(c)
			while(n!=NULL):
				s=<BerylSetting *>n.data
				p=beryl_setting_get_plugin(s)
				oret=oret+[Setting(self.Plugin(plugin_get_name(p)),beryl_setting_get_name(s))]
				n=n.next
			ret = ret + [(rtype,oret)]
			m=m.next
		beryl_settings_free_conflict_list(l)
		return ret

	def Read(self):
		beryl_settings_context_read(self.myContext)

	def Write(self):
		beryl_settings_context_write(self.myContext)

	def Import(self,filename,overwrite=False):
		cdef gboolean ow
		cdef gchar * s
		s=filename
		if (overwrite):
			ow=True
		else:
			ow=False
		return beryl_settings_context_import_from_file(self.myContext,s,ow)

	def Export(self,filename):
		cdef gchar * s
		s=filename
		beryl_settings_context_export_to_file(self.myContext,s)

	def DeleteProfile(self,profile):
		cdef gchar * p
		if (profile!=None):
			p=profile
		else:
			p=NULL
		beryl_settings_delete_profile(self.myContext,p)

	def Plugin(self,name):
		return Plugin(self,name)

	def __str__(self):
		return self.backend+" context"

cdef class Category:
	cdef BerylSettingsPluginCategory * myCategory
	cdef Context pContext

	def __new__(self,Context context,index):
		self.myCategory=beryl_settings_context_get_nth_plugin_category(context.myContext,index)
		self.pContext=context

	property Name:
		def __get__(self):
			return beryl_settings_plugin_category_get_name(self.myCategory)

	property ShortDesc:
		def __get__(self):
			return beryl_settings_plugin_category_get_short_desc(self.myCategory)

	property LongDesc:
		def __get__(self):
			return beryl_settings_plugin_category_get_long_desc(self.myCategory)

	property Plugins:
		def __get__(self):
			cdef GSList * l
			l = beryl_settings_plugin_category_get_plugins(self.myCategory)
			ret=[]
			while (l!=NULL):
				snam=plugin_get_name(<BerylSettingsPlugin *>l.data)
				ret = ret + [Plugin(self.pContext,snam)]
				l=l.next
			return ret

	def __str__(self):
		return self.Name + " category"

cdef class Plugin:
	cdef BerylSettingsPlugin * myPlugin
	cdef Context pContext

	cdef int is_general

	def __new__(self,Context context,plugin):
		self.pContext=context
		if (plugin=="_"):
			self.myPlugin=beryl_settings_context_find_plugin(context.myContext,NULL)
			self.is_general=1
		else:
			self.myPlugin=beryl_settings_context_find_plugin(context.myContext,plugin)
			self.is_general=0
		if (self.myPlugin==NULL):
			raise PluginNotFoundException,(plugin,context)

	property Enabled:
		def __get__(self):
			if (self.is_general):
				return True
			else:
				if beryl_settings_plugin_get_is_enabled(self.myPlugin):
					return True
				else:
					return False
		def __set__(self,val):
			if (self.is_general):
				return
			else:
				if val:
					if not beryl_settings_plugin_enable(self.myPlugin):
						beryl_settings_plugin_disable(self.myPlugin)
						raise PluginEnableException,(self)
				else:
					if not beryl_settings_plugin_disable(self.myPlugin):
						beryl_settings_plugin_enable(self.myPlugin)
						raise PluginDisableException,(self)

	property Name:
		def __get__(self):
			return plugin_get_name(self.myPlugin)

	property ShortDesc:
		def __get__(self):
			if (self.is_general==0):
				return beryl_settings_plugin_get_short_desc(self.myPlugin)
			else:
				return "General Options"

	property LongDesc:
		def __get__(self):
			if (self.is_general==0):
				return beryl_settings_plugin_get_long_desc(self.myPlugin)
			else:
				return "Lots of General Options"

	property Settings:
		def __get__(self):
			cdef GSList * l
			l = beryl_settings_plugin_get_settings(self.myPlugin)
			ret=[]
			while (l!=NULL):
				if not str(beryl_setting_get_name(<BerylSetting *>l.data))=='____plugin_enabled':
					ret = ret + [Setting(self,beryl_setting_get_name(<BerylSetting *>l.data))]
				l=l.next
			return ret

	property Groups:
		def __get__(self):
			cdef GSList * l
			l = beryl_settings_plugin_get_groups(self.myPlugin)
			ret=[]
			while (l!=NULL):
				ret = ret + [Group(self,beryl_settings_group_get_name(<BerylSettingsGroup *>l.data))]
				l=l.next
			return ret

	property Provides:
		def __get__(self):
			cdef GSList * l
			l = beryl_settings_plugin_get_provides(self.myPlugin)
			ret=[]
			while (l!=NULL):
				st=<gchar *>l.data
				ret=ret+[st]
				l=l.next
			return ret

	property Requires:
		def __get__(self):
			cdef GSList * l
			l = beryl_settings_plugin_get_requires(self.myPlugin)
			ret=[]
			while (l!=NULL):
				st=<gchar *>l.data
				ret=ret+[st]
				l=l.next
			return ret

	def Group(self,group):
		return Group(self,group)

	def Setting(self,setting):
		return Setting(self,setting)

	def __str__(self):
		return self.Name+" plugin"

cdef class Group:
	cdef BerylSettingsGroup * myGroup

	cdef Plugin pPlugin

	def __new__(self,Plugin plugin,group):
		self.myGroup=beryl_settings_plugin_find_group(plugin.myPlugin,group)
		if (self.myGroup==NULL):
			raise GroupNotFoundException,(group,plugin)
		self.pPlugin=plugin

	property Plugin:
		def __get__(self):
			return self.pPlugin

	property Name:
		def __get__(self):
			return beryl_settings_group_get_name(self.myGroup)

	property Desc:
		def __get__(self):
			cdef gchar * c
			c=beryl_settings_group_get_desc(self.myGroup)
			if (c==NULL):
				return None
			else:
				return c

	property SubGroups:
		def __get__(self):
			cdef GSList * l
			l = beryl_settings_group_get_subgroups(self.myGroup)
			ret=[]
			while (l!=NULL):
				ret = ret + [SubGroup(self,beryl_settings_subgroup_get_name(<BerylSettingsSubGroup *>l.data))]
				l=l.next
			return ret

	def SubGroup(self,subgroup):
		return SubGroup(self,subgroup)

	def __str__(self):
		return self.Name+" group"

cdef class SubGroup:
	cdef BerylSettingsSubGroup * mySubGroup

	cdef Group pGroup
	cdef Plugin pPlugin

	def __new__(self,Group group,subgroup):
		self.mySubGroup=beryl_settings_group_find_subgroup(group.myGroup,subgroup)
		if self.mySubGroup==NULL:
			raise SubGroupNotFoundException,(subgroup,group)
		self.pPlugin=group.pPlugin
		self.pGroup=group

	property Name:
		def __get__(self):
			return beryl_settings_subgroup_get_name(self.mySubGroup)

	property Desc:
		def __get__(self):
			cdef gchar * c
			c=beryl_settings_subgroup_get_desc(self.mySubGroup)
			if (c==NULL):
				return None
			else:
				return c

	property Settings:
		def __get__(self):
			cdef GSList * l
			l = beryl_settings_subgroup_get_settings(self.mySubGroup)
			ret = []
			while (l!=NULL):
				if not str(beryl_setting_get_name(<BerylSetting *>l.data))=='____plugin_enabled':
					ret = ret + [Setting(self.pPlugin,beryl_setting_get_name(<BerylSetting *>l.data))]
				l=l.next
			return ret

	def Setting(self,setting):
		cdef BerylSetting * s
		s = beryl_settings_subgroup_find_setting(self.mySubGroup,setting)
		if (s==NULL):
			raise SettingNotFoundException,(setting,self)
		return Setting(self.pPlugin,beryl_setting_get_name(s))

	def __str__(self):
		return self.Name+" subgroup"

cdef class Setting:
	cdef BerylSetting * mySetting

	cdef Plugin pPlugin

	def __new__(self,Plugin plugin,setting):
		self.mySetting=beryl_settings_plugin_find_setting(plugin.myPlugin,setting,0)
		if (self.mySetting==NULL):
			self.mySetting=beryl_settings_plugin_find_setting(plugin.myPlugin,setting,1)
		if (self.mySetting==NULL):
			raise SettingNotFoundException,(setting,plugin)
		self.pPlugin = plugin

	property Plugin:
		def __get__(self):
			return self.pPlugin

	property Name:
		def __get__(self):
			return beryl_setting_get_name(self.mySetting)

	property ShortDesc:
		def __get__(self):
			return beryl_setting_get_short_desc(self.mySetting)

	property LongDesc:
		def __get__(self):
			return beryl_setting_get_long_desc(self.mySetting)

	property Group:
		def __get__(self):
			return beryl_setting_get_group(self.mySetting)

	property SubGroup:
		def __get__(self):
			return beryl_setting_get_subgroup(self.mySetting)

	property Advanced:
		def __get__(self):
			if beryl_setting_get_is_advanced(self.mySetting):
				return True
			else:
				return False

	property Integrated:
		def __get__(self):
			if beryl_settings_context_get_setting_is_de_integrated(self.pPlugin.pContext.myContext,
					self.mySetting):
				return True
			else:
				return False

	property ReadOnly:
		def __get__(self):
			if beryl_settings_context_get_setting_is_read_only(self.pPlugin.pContext.myContext,
					self.mySetting):
				return True
			else:
				return False

	def ResetToDefault(self):
		beryl_setting_reset_to_default(self.mySetting)
		return self.Value

	def HasHint(self,hint):
		return beryl_setting_has_hint(self.mySetting,hint)

	def Conflicts(self,t=BERYL_SETTING_CONFLICT_TYPE_ANY):
		cdef GSList * l
		cdef GSList * m
		cdef GSList * n
		cdef gchar * pnam
		cdef BerylSettingConflictType type
		type=t
		cdef BerylSettingConflict * c
		cdef BerylSetting * s
		cdef BerylSettingsPlugin * p
		l=beryl_settings_context_find_conflicts_for_setting(self.pPlugin.pContext.myContext,self.mySetting,type)
		m=l
		ret=[]
		while(m!=NULL):
			oret=[]
			c=<BerylSettingConflict *>m.data
			rtype=beryl_setting_conflict_get_type(c)
			n=beryl_setting_conflict_get_settings(c)
			while(n!=NULL):
				s=<BerylSetting *>n.data
				p=beryl_setting_get_plugin(s)
				oret=oret+[Setting(self.pPlugin.pContext.Plugin(plugin_get_name(p)),beryl_setting_get_name(s))]
				n=n.next
			ret = ret + [(rtype,oret)]
			m=m.next
		beryl_settings_free_conflict_list(l)
		return ret

	def __str__(self):
		return self.Name+" setting"

	property Restrictions:
		def __get__(self):
			cdef BerylSettingType t
			cdef GSList * l
			cdef gint i
			cdef gdouble d
			cdef gboolean b
			t=beryl_setting_get_type(self.mySetting)
			if (t==BERYL_SETTING_TYPE_LIST):
				beryl_setting_get_list_type(self.mySetting,&t)
			if (t==BERYL_SETTING_TYPE_STRING):
				beryl_setting_get_allowed_strings(self.mySetting,&l)
				ret=[]
				while(l!=NULL):
					st=<gchar *>l.data
					ret=ret+[st]
					l=l.next
				return ret
			if (t==BERYL_SETTING_TYPE_INT):
				beryl_setting_get_int_min(self.mySetting,&i)
				imin=i
				beryl_setting_get_int_max(self.mySetting,&i)
				imax=i
				return (imin,imax)
			if (t==BERYL_SETTING_TYPE_FLOAT):
				beryl_setting_get_float_min(self.mySetting,&d)
				fmin=d
				beryl_setting_get_float_max(self.mySetting,&d)
				fmax=d
				beryl_setting_get_float_precision(self.mySetting,&d)
				fprec=d
				return(fmin,fmax,fprec)
			if (t==BERYL_SETTING_TYPE_BINDING):
				beryl_setting_get_can_set_key(self.mySetting,&b)
				csk=b
				beryl_setting_get_can_set_button(self.mySetting,&b)
				csm=b
				beryl_setting_get_can_set_edgemask(self.mySetting,&b)
				cse=b
				beryl_setting_get_can_set_bell(self.mySetting,&b)
				csb=b
				return(csk,csm,csb,cse)
			return None

	property Type:
		def __get__(self):
			cdef BerylSettingType t
			t=beryl_setting_get_type(self.mySetting)
			lret=""
			ret=""
			if (t==BERYL_SETTING_TYPE_LIST):
				beryl_setting_get_list_type(self.mySetting,&t)
				lret="List"
			if (t==BERYL_SETTING_TYPE_STRING):
				ret= "String"
			if (t==BERYL_SETTING_TYPE_BINDING):
				ret= "Binding"
			if (t==BERYL_SETTING_TYPE_COLOR):
				ret= "Color"
			if (t==BERYL_SETTING_TYPE_INT):
				ret= "Int"
			if (t==BERYL_SETTING_TYPE_FLOAT):
				ret= "Float"
			if (t==BERYL_SETTING_TYPE_BOOL):
				ret= "Bool"
			if (ret==""):
				return "Unknown"
			if (lret==""):
				return ret
			else:
				return "List of %s"%ret

	property Value:
		def __set__(self,value):
			cdef BerylSettingType t
			cdef BerylSettingValue * v
			cdef char * s
			cdef int i
			cdef double d
			cdef gboolean b
			cdef BerylSettingColorValue c
			cdef GSList * l
			t = beryl_setting_get_type(self.mySetting)
			v = beryl_setting_get_primary_value(self.mySetting)
			is_list=False
			if (t==BERYL_SETTING_TYPE_LIST):
				is_list=True
				beryl_setting_get_list_type(self.mySetting,&t)
				beryl_setting_list_clear(self.mySetting)
				val=value
			else:
				val=[value]
			for value in val:
				if is_list:
					v=beryl_setting_list_append(self.mySetting)
				if (t==BERYL_SETTING_TYPE_STRING):
					s=value
					beryl_setting_value_set_string(v,&s)
				if (t==BERYL_SETTING_TYPE_INT):
					i=value
					beryl_setting_value_set_int(v,&i)
				if (t==BERYL_SETTING_TYPE_FLOAT):
					d=value
					beryl_setting_value_set_float(v,&d)
				if (t==BERYL_SETTING_TYPE_BOOL):
					if (value):
						b=1
					else:
						b=0
					beryl_setting_value_set_bool(v,&b)
				if (t==BERYL_SETTING_TYPE_COLOR):
					c.color.red=value[0]
					c.color.green=value[1]
					c.color.blue=value[2]
					c.color.alpha=value[3]
					beryl_setting_value_set_color(v,&c)
				if (t==BERYL_SETTING_TYPE_BINDING):
					#(keystr,buttonstr,bellbool,edgelist)
					if self.Restrictions[0]:
						keystr=value[0]
						ke=0
						keysym=0
						mods=0
						if (str(keystr).lower()!="disabled" and keystr!=""):
							mods=beryl_settings_get_mods_and_endptr(keystr,&s)
							keysym=XStringToKeysym(s)
							ke=1
						i=keysym
						beryl_setting_value_set_keysym(v,&i)
						i=mods
						beryl_setting_value_set_keymods(v,&i)
						b=ke
						beryl_setting_value_set_key_enabled(v,&b)
					if self.Restrictions[1]:
						buttonstr=value[1]
						bu=0
						button=0
						mods=0
						if (str(buttonstr).lower()!="disabled" and buttonstr!=""):
							mods=beryl_settings_get_mods_and_endptr(buttonstr,&s)
							try:
								button=int(s[6:])
							except:
								button=0
							bu=1
						i=button
						beryl_setting_value_set_button(v,&i)
						i=mods
						beryl_setting_value_set_buttonmods(v,&i)
						b=bu
						beryl_setting_value_set_button_enabled(v,&b)
					if self.Restrictions[2]:
						if value[2]:
							bel=1
						else:
							bel=0
						b=bel
						beryl_setting_value_set_bell(v,&b)
					if self.Restrictions[3]:
						edg=edgelisttomask(value[3])
						i=edg
						beryl_setting_value_set_edgemask(v,&i)

		def __get__(self):
			cdef BerylSettingType t
			cdef BerylSettingValue * v
			cdef char * s
			cdef int i
			cdef double d
			cdef gboolean b
			cdef BerylSettingColorValue c
			cdef GSList * l
			t = beryl_setting_get_type(self.mySetting)
			v = beryl_setting_get_primary_value(self.mySetting)
			is_list=False
			rem=1
			if (t==BERYL_SETTING_TYPE_LIST):
				is_list=True
				beryl_setting_get_list_type(self.mySetting,&t)
				beryl_setting_value_get_value_list(v,&l)
				rem=0
			ret=[]
			while((rem!=0) or (is_list and l!=NULL)):
				if is_list:
					v=<BerylSettingValue *>l.data
				else:
					rem=rem-1
				if (t==BERYL_SETTING_TYPE_STRING):
					if beryl_setting_value_get_string(v,&s):
						ret = ret + [s]
					else:
						if len(self.Restrictions):
							ret = ret + [self.Restrictions[0]]
						else:
							ret = ret + ['']
				if (t==BERYL_SETTING_TYPE_INT):
					if beryl_setting_value_get_int(v,&i):
						ret = ret + [i]
				if (t==BERYL_SETTING_TYPE_FLOAT):
					if beryl_setting_value_get_float(v,&d):
						ret = ret + [d]
				if (t==BERYL_SETTING_TYPE_BOOL):
					if beryl_setting_value_get_bool(v,&b):
						if b:
							ret = ret + [True]
						else:
							ret = ret + [False]
				if (t==BERYL_SETTING_TYPE_COLOR):
					if beryl_setting_value_get_color(v,&c):
						ret = ret + [(c.color.red,c.color.green,c.color.blue,c.color.alpha)]
				if (t==BERYL_SETTING_TYPE_BINDING):
					beryl_setting_value_get_keysym(v,&i)
					kys=keytostring(i)
					beryl_setting_value_get_keymods(v,&i)
					kym=modstostring(i)
					beryl_setting_value_get_button(v,&i)
					bus=buttontostring(i)
					beryl_setting_value_get_buttonmods(v,&i)
					bum=modstostring(i)
					beryl_setting_value_get_bell(v,&b)
					if (b):
						bes=True
					else:
						bes=False
					beryl_setting_value_get_edgemask(v,&i)
					eds=edgetolist(i)
					beryl_setting_value_get_key_enabled(v,&b)
					if (b):
						kss=kym+kys
					else:
						kss="Disabled"
					beryl_setting_value_get_button_enabled(v,&b)
					if (b):
						bss=bum+bus
					else:
						bss="Disabled"
					ret = ret + [(kss,bss,bes,eds)]
				if is_list:
					l=l.next
			if is_list:
				return ret
			else:
				return ret[0]
