// winscarp.cpp : Defines the entry point for the DLL application.
//
#define wxUSE_GUI 0

#ifndef _WIN32
#include <winscard.h>
#endif

#include "winscarp.h"

#ifdef _WIN32
#include <process.h>
#endif

#include "wx/defs.h"

#include <wx/dynlib.h>

#include "Config.h"
#include "ServerMessageReceiver.h"
#include "ServerPoll.h"
#include "TLVBuffer.h"

////////////
#include "Base/StringToObjectCollection.h"
#include "Base/StringContainer.h" 
#include "Base/ULongContainer.h"
#include "Base/ObjectManager.h"

#include "Comm/ConnectionManager.h" 
#include "Comm/Connection_Mono.h" 
#include "Comm/Message.h" 
#include "Comm/GenericMessage.h" 


// Function prototypes
BOOL Initialize();
BOOL Finalize();

class CStartLoader  
{
public:
    CStartLoader() { Initialize();}
    virtual ~CStartLoader() { Finalize(); }
};

///////////

#ifdef _WIN32
 static const SCARD_IO_REQUEST g_rgSCardT0Pci = {1, 8};
 static const SCARD_IO_REQUEST g_rgSCardT1Pci = {2, 8};
 static const SCARD_IO_REQUEST g_rgSCardRawPci = {65536, 8};

#define SCARD_W_INSERTED_CARD           0x8010006A

#else
#define SCARD_AUTOALLOCATE (DWORD)(-1)
    
 typedef struct _GUID {          // size is 16
    DWORD Data1;
    WORD   Data2;
    WORD   Data3;
    BYTE  Data4[8];
} GUID;
typedef const GUID *LPCGUID;
typedef GUID *LPGUID;
typedef char *LPWSTR;
typedef const char *LPCWSTR;
typedef SCARD_READERSTATE_A *LPSCARD_READERSTATE_W;
typedef const char *LPCSTR;
#endif

#ifdef BELPIC_PIN_PAD
 
// PinPad stuff
#define SCR_INIT_ID          100
#define SCR_VERIFY_ID      101
#define SCR_CHANGE_ID    102
#define SCR_CARD_HANDLE          999
#define PINPADNUM 3

typedef enum 
{
	SCR_SUPPORT_OK = 0L,
	SCR_SUPPORT_INCOMPATIBLE_CALLING_VERSION = 1L,
	SCR_SUPPORT_INCOMPATIBLE_FIRMWARE = 2L,
	SCR_SUPPORT_INCOMPATIBLE_FIRMWARE_VERSION = 3L,
} SCR_SupportConstants;

typedef struct 
{
	BYTE *data;
	DWORD length;
} SCR_Bytes;

typedef struct 
{
	SCARDHANDLE hCard;
	char *language;
	SCR_Bytes id;
	void *pinFormat; /* reserved for future use */
} SCR_Card;

typedef struct 
{
	SCR_Bytes id;
	char *shortString;
	char *longString;
} SCR_Application;

typedef struct 
{
	long code;
	char *shortString;
	char *longString;
} SCR_PinUsage;

typedef struct
{
    char szPinPadDll[256];
    wxDynamicLibrary *pDll;
} PinPadObject;

extern "C"
{
typedef long (*SCR_INIT_PROC)(LPCTSTR, DWORD, SCR_SupportConstants*);
typedef long (*SCR_VERIFYPIN_PROC)(const SCR_Card*, BYTE, const SCR_PinUsage*, const SCR_Application*, BYTE *);
typedef long (*SCR_CHANGEPIN_PROC)(const SCR_Card*, BYTE, const SCR_Application*, BYTE *);
}

#endif // BELPIC_PIN_PAD

// Global Variables
static CStartLoader gLoader;
BOOL gbServerMode = FALSE;
CConnectionManager *gpConnClientMan = NULL;
CConnection *gpConnectionClient = NULL;
CServerMessageReceiver *gpServerMessageReceiver = NULL;
CServerPoll *gpServerPoll = NULL;
CConfig *gpConfig = NULL;

#ifdef BELPIC_PIN_PAD
PinPadObject *gpPinPadDlls = NULL;
#endif

wxDynamicLibrary *gpWinScardLoader = NULL;

///////////////////////////////////////////////////////
// Original Winscard functions
#ifndef WINSCARDAPI
#define WINSCARDAPI
#endif

#ifdef _WIN32
#ifndef WINAPI
    #define WINAPI  __stdcall
#endif
#else
    #define WINAPI
#endif

#ifndef IN
#define IN
#endif

#ifndef OUT
#define OUT
#endif

//#ifdef _WIN32
typedef long (WINAPI *SCTransmit)(SCARDHANDLE,LPCSCARD_IO_REQUEST,LPCBYTE,DWORD,LPSCARD_IO_REQUEST,LPBYTE,LPDWORD);
/*
#else
typedef long (WINAPI *SCTransmit)(long,LPCSCARD_IO_REQUEST,const unsigned char *,unsigned long,LPSCARD_IO_REQUEST,unsigned char *,unsigned long *);
#endif
*/
typedef long (WINAPI *SCBeginTransaction)(SCARDHANDLE);
typedef long (WINAPI *SCEndTransaction)(SCARDHANDLE, DWORD);
typedef long (WINAPI *SCConnectA)(SCARDCONTEXT,LPCSTR,DWORD,DWORD,LPSCARDHANDLE,LPDWORD);
//#ifdef _WIN32
typedef long (WINAPI *SCControl)(SCARDHANDLE,DWORD,LPCVOID,DWORD,LPVOID,DWORD,LPDWORD);
/*
#else
typedef long (WINAPI *SCControl)(SCARDHANDLE,const unsigned char *,DWORD,unsigned char *,LPDWORD);
#endif
*/
typedef long (WINAPI *SCDisconnect)(SCARDHANDLE,DWORD);
typedef long (WINAPI *SCEstablishContext)(DWORD,LPCVOID,LPCVOID,LPSCARDCONTEXT);
typedef long (WINAPI *SCReleaseContext)(SCARDCONTEXT);
typedef long (WINAPI *SCGetStatusChangeA)(SCARDCONTEXT,DWORD,LPSCARD_READERSTATE_A,DWORD);
typedef long (WINAPI *SCListReadersA)(SCARDCONTEXT,LPCSTR,LPTSTR,LPDWORD);
typedef long (WINAPI *SCStatusA)(SCARDHANDLE,LPTSTR,LPDWORD,LPDWORD,LPDWORD,LPBYTE,LPDWORD);

#define WINSCARPAPI WINSCARDAPI

static WINSCARDAPI LONG (WINAPI *Original_SCardTransmit)(
  IN SCARDHANDLE hCard,  
  IN LPCSCARD_IO_REQUEST pioSendPci,
  IN LPCBYTE pbSendBuffer,
  IN DWORD cbSendLength,
  IN OUT LPSCARD_IO_REQUEST pioRecvPci,
  OUT LPBYTE pbRecvBuffer,
  IN OUT LPDWORD pcbRecvLength
) = NULL;

static WINSCARDAPI LONG (WINAPI *Original_SCardBeginTransaction)(
    IN SCARDHANDLE hCard
) = NULL;

static WINSCARDAPI LONG (WINAPI *Original_SCardEndTransaction)(
    IN  SCARDHANDLE hCard,
    IN  DWORD dwDisposition
) = NULL;

static WINSCARDAPI LONG (WINAPI *Original_SCardConnectA)(
    IN  SCARDCONTEXT hContext,
    IN  LPCSTR szReader,
    IN  DWORD dwShareMode,
    IN  DWORD dwPreferredProtocols,
    OUT LPSCARDHANDLE phCard,
    OUT LPDWORD pdwActiveProtocol
) = NULL;

//#ifdef _WIN32
static WINSCARDAPI LONG (WINAPI *Original_SCardControl)(
    IN SCARDHANDLE hCard,
    IN DWORD dwControlCode,
    IN LPCVOID lpInBuffer,
    IN DWORD nInBufferSize,
    OUT LPVOID lpOutBuffer,
    IN  DWORD nOutBufferSize,
    OUT LPDWORD lpBytesReturned
) = NULL;
/*
#else
static WINSCARDAPI LONG (WINAPI *Original_SCardControl)(
    IN SCARDHANDLE hCard,
    IN const unsigned char * lpInBuffer,
    IN DWORD nInBufferSize,
    OUT unsigned char * lpOutBuffer,
    OUT LPDWORD lpBytesReturned
) = NULL;
#endif
*/
static WINSCARDAPI LONG (WINAPI *Original_SCardDisconnect)(
    IN SCARDHANDLE hCard,
    IN DWORD dwDisposition
) = NULL;

static WINSCARDAPI LONG (WINAPI *Original_SCardEstablishContext)(
    IN  DWORD dwScope,
    IN  LPCVOID pvReserved1,
    IN  LPCVOID pvReserved2,
    OUT LPSCARDCONTEXT phContext
) = NULL;

static WINSCARDAPI LONG (WINAPI *Original_SCardReleaseContext)(
    IN SCARDCONTEXT hContext
) = NULL;

static WINSCARDAPI LONG (WINAPI *Original_SCardGetStatusChangeA)(
    IN SCARDCONTEXT hContext,
    IN DWORD dwTimeout,
    IN OUT LPSCARD_READERSTATE_A rgReaderStates,
    IN DWORD cReaders
) = NULL;

static WINSCARDAPI LONG (WINAPI *Original_SCardListReadersA)(
    IN SCARDCONTEXT hContext,
    IN LPCSTR mszGroups,
    OUT LPTSTR mszReaders,
    IN OUT LPDWORD pcchReaders
) = NULL;

static WINSCARDAPI LONG (WINAPI *Original_SCardStatusA)(
    IN SCARDHANDLE hCard,
    OUT LPTSTR szReaderName,
    IN OUT LPDWORD pcchReaderLen,
    OUT LPDWORD pdwState,
    OUT LPDWORD pdwProtocol,
    OUT LPBYTE pbAtr,
    OUT LPDWORD pcbAtrLen
) = NULL;

//////////////////////////////////////////////////////////


/////////////////////////////////////////
// Function prototypes
CGenericMessage *SCardCreateMessage(char *pszName);
void FillUserData(CGenericMessage *pMessage);

#ifdef BELPIC_PIN_PAD
LONG HandlePinPad(const void *lpInBuffer, DWORD nInBufferSize, void *lpOutBuffer, DWORD *lpBytesReturned);
#endif

/////////////////////////////////////////

BOOL Initialize()
{
    BOOL bRet = TRUE;
    if (gpWinScardLoader == NULL)
    {
#ifdef _WIN32
        gpWinScardLoader = new wxDynamicLibrary("winscard");
#else
        gpWinScardLoader = new wxDynamicLibrary("libpcsclite");
#endif
        if(gpWinScardLoader->GetLibHandle() > 0)
        {
            Original_SCardTransmit = (SCTransmit)gpWinScardLoader->GetSymbol("SCardTransmit");
            Original_SCardBeginTransaction = (SCBeginTransaction)gpWinScardLoader->GetSymbol("SCardBeginTransaction");
            Original_SCardEndTransaction = (SCEndTransaction)gpWinScardLoader->GetSymbol("SCardEndTransaction");
#ifdef _WIN32
            Original_SCardConnectA = (SCConnectA)gpWinScardLoader->GetSymbol("SCardConnectA");
#else
            Original_SCardConnectA = (SCConnectA)gpWinScardLoader->GetSymbol("SCardConnect");
#endif
            Original_SCardControl = (SCControl)gpWinScardLoader->GetSymbol("SCardControl");
            Original_SCardDisconnect = (SCDisconnect)gpWinScardLoader->GetSymbol("SCardDisconnect");
            Original_SCardEstablishContext = (SCEstablishContext)gpWinScardLoader->GetSymbol("SCardEstablishContext");
            Original_SCardReleaseContext = (SCReleaseContext)gpWinScardLoader->GetSymbol("SCardReleaseContext");
#ifdef _WIN32
            Original_SCardGetStatusChangeA = (SCGetStatusChangeA)gpWinScardLoader->GetSymbol("SCardGetStatusChangeA");
#else
            Original_SCardGetStatusChangeA = (SCGetStatusChangeA)gpWinScardLoader->GetSymbol("SCardGetStatusChange");
#endif

#ifdef _WIN32
            Original_SCardListReadersA = (SCListReadersA)gpWinScardLoader->GetSymbol("SCardListReadersA");
#else
            Original_SCardListReadersA = (SCListReadersA)gpWinScardLoader->GetSymbol("SCardListReaders");
#endif

#ifdef _WIN32
            Original_SCardStatusA = (SCStatusA)gpWinScardLoader->GetSymbol("SCardStatusA");        
#else
            Original_SCardStatusA = (SCStatusA)gpWinScardLoader->GetSymbol("SCardStatus");        
#endif

        }
    }


#ifdef BELPIC_PIN_PAD
    if(gpPinPadDlls == NULL)
    {
        gpPinPadDlls = new PinPadObject[PINPADNUM];
        memset(gpPinPadDlls, 0, sizeof(PinPadObject) * PINPADNUM);
    }
#endif

    // Read config file
    if(gpConfig == NULL)
    {
        gpConfig = new CConfig;
        gpConfig->Load();
    }

    if(gpConfig->GetServiceEnabled())
    {
        const char *pszServerAddress = gpConfig->GetServerAddress();
        unsigned long ulServerPort = gpConfig->GetServerPort();
 
        if(pszServerAddress != NULL && ulServerPort > 0)
        {
                if(gpServerMessageReceiver == NULL)
                {
		            gpServerMessageReceiver = new CServerMessageReceiver;
		            gpServerMessageReceiver->SetTickTime (2);
		            gpServerMessageReceiver->Start();
                }
            // Try to connect to PS/SC Service
            if (gpConnClientMan == NULL)
            {
                gpConnClientMan = new CConnectionManager();
                gpConnClientMan->SetConnectionType ("Mono");
            }

            if(gpConnectionClient == NULL)
            {
                gpConnectionClient = gpConnClientMan->CreateConnection();

	            CStringToObjectCollection connectionParam;
	            CStringContainer serverAddressParam;
	            CLongContainer serverPortParam;

	            serverAddressParam.SetString((char *)pszServerAddress);
	            serverPortParam.SetULong (ulServerPort);
	            connectionParam.Add ("ServerAddress", &serverAddressParam);
	            connectionParam.Add ("ServerPort", &serverPortParam);

	            gpConnectionClient->SetConnectionParam (&connectionParam);
            }

	        if (!gpConnectionClient->Open())
	        {
                gbServerMode = FALSE;
                gpConnectionClient->Close();
	        }
	        else
	        {
                gbServerMode = TRUE;
		        gpServerMessageReceiver->SetConnection (gpConnectionClient);
	        }
        }

        if(gpServerPoll == NULL)
        {
            gpServerPoll = new CServerPoll();
            gpServerPoll->Start();
        }
    }
    return bRet;
}

BOOL Finalize()
{
    BOOL bRet = TRUE;
    ULONG ulCounter = 0;

    if(NULL != gpServerPoll)
    {
        ulCounter = 0;
        if(gpServerPoll->IsRunning())
        {
            gpServerPoll->Stop();
        }
        while(gpServerPoll->IsRunning() && ulCounter < 15)
        {
            PASS_MILLISLEEP (100);
            ulCounter++;
        }
	    if (ulCounter == 15)
	    {
		    // Reached timeout
		    // Kill handler
		    gpServerPoll->Kill();
	    }
        delete gpServerPoll;
        gpServerPoll = NULL;
    }

    if(NULL != gpServerMessageReceiver)
    {
        ulCounter = 0;
        if(gpServerMessageReceiver->IsRunning())
        {
            gpServerMessageReceiver->Stop();
        }
        while(gpServerMessageReceiver->IsRunning() && ulCounter < 15)
        {
            PASS_MILLISLEEP (100);
            ulCounter++;
        }
	    if (ulCounter == 15)
	    {
		    // Reached timeout
		    // Kill handler
		    gpServerMessageReceiver->Kill();
	    }
        delete gpServerMessageReceiver;
        gpServerMessageReceiver = NULL;
    }
	if (NULL != gpConnectionClient)
	{
        ulCounter = 0;
        gpConnectionClient->Close();
        if(gpConnectionClient->IsRunning())
        {
            gpConnectionClient->Stop();
        }
        while(gpConnectionClient->IsRunning() && ulCounter < 10)
        {
            PASS_MILLISLEEP (100);
            ulCounter++;
        }
	    if (ulCounter == 10)
	    {
		    // Reached timeout
		    // Kill handler
		    gpConnectionClient->Kill();
	    }
        gbServerMode = FALSE;
		delete gpConnectionClient;
		gpConnectionClient = NULL;
	}

     if (gpConnClientMan != NULL)
	{
        ulCounter = 0;
        if(gpConnClientMan->IsRunning())
        {
		    gpConnClientMan->Stop();
        }
        while(gpConnClientMan->IsRunning() && ulCounter < 10)
        {
            PASS_MILLISLEEP (100);
            ulCounter++;
        }
	    if (ulCounter == 10)
	    {
		    // Reached timeout
		    // Kill handler
		    gpConnClientMan->Kill();
	    }
		delete gpConnClientMan;
		gpConnClientMan = NULL;
	}

#ifdef BELPIC_PIN_PAD
    if(gpPinPadDlls != NULL)
    {
        for(int i = 0; i < PINPADNUM; i++)
        {
            if(gpPinPadDlls[i].pDll  != NULL)
            {
                delete gpPinPadDlls[i].pDll;
            }
        }
        delete gpPinPadDlls;
        gpPinPadDlls = NULL;
    }
#endif

    if (gpWinScardLoader != NULL)
    {
        delete gpWinScardLoader;
        gpWinScardLoader = NULL;
    }

    if(gpConfig != NULL)
    {
        gpConfig->Unload();
        delete gpConfig;
        gpConfig = NULL;
    }

    return bRet;
}

void CheckConnection()
{
    if(gpConnectionClient != NULL && gpConnectionClient->GetConnectionBreak())
    {
        gbServerMode = FALSE;        
    }
}

WINSCARPAPI LONG WINAPI SCardEstablishContext(
    IN  DWORD dwScope,
    IN  LPCVOID pvReserved1,
    IN  LPCVOID pvReserved2,
    OUT LPSCARDCONTEXT phContext)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;
        *phContext = 0;

        CGenericMessage *pMessage = SCardCreateMessage("SCardEstablishContext");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Scope", (unsigned long)dwScope);
		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                pMessage->GetValueForKey ("Context", (unsigned long *)phContext);  
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardEstablishContext(dwScope, pvReserved1, pvReserved2, phContext);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardReleaseContext(
    IN      SCARDCONTEXT hContext)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardReleaseContext");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Context", hContext);
		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardReleaseContext(hContext);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardFreeMemory(
    IN SCARDCONTEXT hContext,
    IN LPVOID pvMem)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

//
// Database Reader routines
//

WINSCARPAPI LONG WINAPI SCardListReaderGroupsA(
    IN      SCARDCONTEXT hContext,
    OUT     LPTSTR mszGroups,
    IN OUT  LPDWORD pcchGroups)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardListReaderGroupsW(
    IN      SCARDCONTEXT hContext,
    OUT     LPWSTR mszGroups,
    IN OUT  LPDWORD pcchGroups)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

#ifdef _WIN32
WINSCARPAPI LONG WINAPI SCardListReadersA(
#else
WINSCARPAPI LONG WINAPI SCardListReaders(
#endif
    IN      SCARDCONTEXT hContext,
    IN      LPCSTR mszGroups,
    OUT     LPTSTR mszReaders,
    IN OUT  LPDWORD pcchReaders)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardListReaders");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Context", hContext);
            if(mszGroups != NULL)
            {
                pMessage->AddValueForKey("Groups", (char *)mszGroups);
            }
            pMessage->AddValueForKey("ReadersLen", *pcchReaders);
		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                unsigned long lLen = 0;
                pMessage->GetValueForKey ("ReadersLen", &lLen);
                if(NULL != mszReaders)
                {
                    if(*pcchReaders == SCARD_AUTOALLOCATE )
                    {
                        char *pmszReaders = new char[lLen+1];
                        memset(pmszReaders, 0, lLen+1);
                        pMessage->GetValueForKey ("Readers", (unsigned char *)pmszReaders, lLen);
                        memcpy(mszReaders, &pmszReaders, 4);
                    }
                    else
                    {
                        pMessage->GetValueForKey ("Readers", (unsigned char *)mszReaders, lLen);  
                    }
                }
                *pcchReaders = lLen;
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardListReadersA(hContext, mszGroups, mszReaders, pcchReaders);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardListReadersW(
    IN      SCARDCONTEXT hContext,
    IN      LPCWSTR mszGroups,
    OUT     LPWSTR mszReaders,
    IN OUT  LPDWORD pcchReaders)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardListCardsA(
    IN      SCARDCONTEXT hContext,
    IN      LPCBYTE pbAtr,
    IN      LPCGUID rgquidInterfaces,
    IN      DWORD cguidInterfaceCount,
    OUT     LPTSTR mszCards,
    IN OUT  LPDWORD pcchCards)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardListCardsW(
    IN      SCARDCONTEXT hContext,
    IN      LPCBYTE pbAtr,
    IN      LPCGUID rgquidInterfaces,
    IN      DWORD cguidInterfaceCount,
    OUT     LPWSTR mszCards,
    IN OUT  LPDWORD pcchCards)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardListInterfacesA(
    IN      SCARDCONTEXT hContext,
    IN      LPCSTR szCard,
    OUT     LPGUID pguidInterfaces,
    IN OUT  LPDWORD pcguidInterfaces)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardListInterfacesW(
    IN      SCARDCONTEXT hContext,
    IN      LPCWSTR szCard,
    OUT     LPGUID pguidInterfaces,
    IN OUT  LPDWORD pcguidInterfaces)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardGetProviderIdA(
    IN      SCARDCONTEXT hContext,
    IN      LPCSTR szCard,
    OUT     LPGUID pguidProviderId)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardGetProviderIdW(
    IN      SCARDCONTEXT hContext,
    IN      LPCWSTR szCard,
    OUT     LPGUID pguidProviderId)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

//
// Database Writer routines
//

WINSCARPAPI LONG WINAPI SCardIntroduceReaderGroupA(
    IN SCARDCONTEXT hContext,
    IN LPCSTR szGroupName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardIntroduceReaderGroupW(
    IN SCARDCONTEXT hContext,
    IN LPCWSTR szGroupName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardForgetReaderGroupA(
    IN SCARDCONTEXT hContext,
    IN LPCSTR szGroupName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardForgetReaderGroupW(
    IN SCARDCONTEXT hContext,
    IN LPCWSTR szGroupName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardIntroduceReaderA(
    IN SCARDCONTEXT hContext,
    IN LPCSTR szReaderName,
    IN LPCSTR szDeviceName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI
SCardIntroduceReaderW(
    IN SCARDCONTEXT hContext,
    IN LPCWSTR szReaderName,
    IN LPCWSTR szDeviceName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI
SCardForgetReaderA(
    IN SCARDCONTEXT hContext,
    IN LPCSTR szReaderName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI
SCardForgetReaderW(
    IN SCARDCONTEXT hContext,
    IN LPCWSTR szReaderName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardAddReaderToGroupA(
    IN SCARDCONTEXT hContext,
    IN LPCSTR szReaderName,
    IN LPCSTR szGroupName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardAddReaderToGroupW(
    IN SCARDCONTEXT hContext,
    IN LPCWSTR szReaderName,
    IN LPCWSTR szGroupName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardRemoveReaderFromGroupA(
    IN SCARDCONTEXT hContext,
    IN LPCSTR szReaderName,
    IN LPCSTR szGroupName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardRemoveReaderFromGroupW(
    IN SCARDCONTEXT hContext,
    IN LPCWSTR szReaderName,
    IN LPCWSTR szGroupName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardIntroduceCardTypeA(
    IN SCARDCONTEXT hContext,
    IN LPCSTR szCardName,
    IN LPGUID pguidPrimaryProvider,
    IN LPGUID rgguidInterfaces,
    IN DWORD dwInterfaceCount,
    IN LPCBYTE pbAtr,
    IN LPCBYTE pbAtrMask,
    IN DWORD cbAtrLen)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardIntroduceCardTypeW(
    IN SCARDCONTEXT hContext,
    IN LPCWSTR szCardName,
    IN LPGUID pguidPrimaryProvider,
    IN LPGUID rgguidInterfaces,
    IN DWORD dwInterfaceCount,
    IN LPCBYTE pbAtr,
    IN LPCBYTE pbAtrMask,
    IN DWORD cbAtrLen)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardForgetCardTypeA(
    IN SCARDCONTEXT hContext,
    IN LPCSTR szCardName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardForgetCardTypeW(
    IN SCARDCONTEXT hContext,
    IN LPCWSTR szCardName)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

//  Reader Services
//
//      The following services are supplied to simplify the use of the Service
//      Manager API.
//

WINSCARPAPI LONG WINAPI SCardLocateCardsA(
    IN      SCARDCONTEXT hContext,
    IN      LPCSTR mszCards,
    IN OUT  LPSCARD_READERSTATE_A rgReaderStates,
    IN      DWORD cReaders)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardLocateCardsW(
    IN      SCARDCONTEXT hContext,
    IN      LPCWSTR mszCards,
    IN OUT  LPSCARD_READERSTATE_W rgReaderStates,
    IN      DWORD cReaders)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

#ifdef _WIN32
WINSCARPAPI LONG WINAPI SCardGetStatusChangeA(
#else
WINSCARPAPI LONG WINAPI SCardGetStatusChange(
#endif

    IN      SCARDCONTEXT hContext,
    IN      DWORD dwTimeout,
    IN OUT  LPSCARD_READERSTATE_A rgReaderStates,
    IN      DWORD cReaders)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardGetStatusChange");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Context", hContext);
            pMessage->AddValueForKey("Timeout", dwTimeout);
            pMessage->AddValueForKey("ReadersLen", cReaders);
            // Fill Readers
            for(unsigned int i = 0; i < cReaders; ++i)
            {
                char szReader[16] = {0};
                char szState[16] = {0};
                sprintf(szReader, "Reader%d", i); 
                sprintf(szState, "CurrentState%d", i); 
                pMessage->AddValueForKey(szReader, (char *)rgReaderStates[i].szReader);
                pMessage->AddValueForKey(szState, rgReaderStates[i].dwCurrentState);
            }
		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                for(unsigned int i = 0; i < cReaders; ++i)
                {
                    char szEventstate[16] = {0};
                    char szAtrLen[16] = {0};
                    char szAtr[16] = {0};
                    sprintf(szEventstate, "EventState%d", i); 
                    sprintf(szAtrLen, "AtrLen%d", i); 
                    sprintf(szAtr, "Atr%d", i); 
                    pMessage->GetValueForKey (szEventstate, &(rgReaderStates[i].dwEventState));  
                    pMessage->GetValueForKey (szAtrLen, &(rgReaderStates[i].cbAtr));  
                    pMessage->GetValueForKey (szAtr, rgReaderStates[i].rgbAtr, rgReaderStates[i].cbAtr);  
                }
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);
                delete pMessage;                
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardGetStatusChangeA(hContext, dwTimeout, rgReaderStates, cReaders);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardGetStatusChangeW(
    IN      SCARDCONTEXT hContext,
    IN      DWORD dwTimeout,
    IN OUT  LPSCARD_READERSTATE_W rgReaderStates,
    IN      DWORD cReaders)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardCancel(
    IN      SCARDCONTEXT hContext)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

//  Card/Reader Access Services
//
//      The following services provide means for establishing communication with
//      the card.
//

#ifdef _WIN32
WINSCARPAPI LONG WINAPI SCardConnectA(
#else
WINSCARPAPI LONG WINAPI SCardConnect(
#endif
    IN      SCARDCONTEXT hContext,
    IN      LPCSTR szReader,
    IN      DWORD dwShareMode,
    IN      DWORD dwPreferredProtocols,
    OUT     LPSCARDHANDLE phCard,
    OUT     LPDWORD pdwActiveProtocol)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardConnect");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Context", hContext);
            pMessage->AddValueForKey("Reader", (char *)szReader);
            pMessage->AddValueForKey("ShareMode", dwShareMode);
            pMessage->AddValueForKey("Protocol", dwPreferredProtocols);

		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                pMessage->GetValueForKey ("Card", (unsigned long *)phCard);  
                pMessage->GetValueForKey ("ActiveProtocol", pdwActiveProtocol);  
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardConnectA(hContext, szReader, dwShareMode, dwPreferredProtocols, phCard, pdwActiveProtocol);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardConnectW(
    IN      SCARDCONTEXT hContext,
    IN      LPCWSTR szReader,
    IN      DWORD dwShareMode,
    IN      DWORD dwPreferredProtocols,
    OUT     LPSCARDHANDLE phCard,
    OUT     LPDWORD pdwActiveProtocol)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardReconnect(
    IN      SCARDHANDLE hCard,
    IN      DWORD dwShareMode,
    IN      DWORD dwPreferredProtocols,
    IN      DWORD dwInitialization,
    OUT     LPDWORD pdwActiveProtocol)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardDisconnect(
    IN      SCARDHANDLE hCard,
    IN      DWORD dwDisposition)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardDisconnect");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Card", hCard);
            pMessage->AddValueForKey("Disposition", dwDisposition);

		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardDisconnect(hCard, dwDisposition);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardBeginTransaction(
    IN      SCARDHANDLE hCard)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardBeginTransaction");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Card", hCard);

		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardBeginTransaction(hCard);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardEndTransaction(
    IN      SCARDHANDLE hCard,
    IN      DWORD dwDisposition)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardEndTransaction");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Card", hCard);
            pMessage->AddValueForKey("Disposition", dwDisposition);

		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardEndTransaction(hCard, dwDisposition);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardState(
    IN SCARDHANDLE hCard,
    OUT LPDWORD pdwState,
    OUT LPDWORD pdwProtocol,
    OUT LPBYTE pbAtr,
    OUT LPDWORD pcbAtrLen)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

#ifdef _WIN32
WINSCARPAPI LONG WINAPI SCardStatusA(
#else
WINSCARPAPI LONG WINAPI SCardStatus(
#endif
    IN SCARDHANDLE hCard,
    OUT LPTSTR szReaderName,
    IN OUT LPDWORD pcchReaderLen,
    OUT LPDWORD pdwState,
    OUT LPDWORD pdwProtocol,
    OUT LPBYTE pbAtr,
    IN OUT LPDWORD pcbAtrLen)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardStatus");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Card", hCard);

		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                unsigned long lLen = 0;
                pMessage->GetValueForKey ("ReaderLen", &lLen);
                if(NULL != szReaderName && lLen > 0)
                {
                    if(*pcchReaderLen == SCARD_AUTOALLOCATE )
                    {
                        char *pmszReader = new char[lLen+1];
                        memset(pmszReader, 0, lLen+1);
                        pMessage->GetValueForKey ("ReaderName", pmszReader, lLen);
                        memcpy(szReaderName, &pmszReader, 4);
                    }
                    else
                    {
                        pMessage->GetValueForKey ("ReaderName", szReaderName, lLen);  
                    }
                }
                *pcchReaderLen = lLen;
                pMessage->GetValueForKey ("State", pdwState);  
                pMessage->GetValueForKey ("Protocol", pdwProtocol);  

                lLen = 0;
                pMessage->GetValueForKey ("AtrLen", &lLen);
                if(NULL != pbAtr && lLen > 0)
                {
                    if(*pcbAtrLen == SCARD_AUTOALLOCATE )
                    {
                        BYTE *pAtr = new BYTE[32];
                        memset(pAtr, 0, 32);
                        pMessage->GetValueForKey ("Atr", pAtr, lLen);
                        memcpy(pbAtr, &pAtr, 4);
                    }
                    else
                    {
                        pMessage->GetValueForKey ("Atr", pbAtr, lLen);  
                    }
                }
                if(pcbAtrLen != NULL)
                {
                    *pcbAtrLen = lLen;
                }
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;    
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardStatusA(hCard, szReaderName, pcchReaderLen, pdwState, pdwProtocol, pbAtr, pcbAtrLen);
    }
    return SCARD_E_NO_SERVICE;
}

WINSCARPAPI LONG WINAPI SCardStatusW(
    IN SCARDHANDLE hCard,
    OUT LPWSTR szReaderName,
    IN OUT LPDWORD pcchReaderLen,
    OUT LPDWORD pdwState,
    OUT LPDWORD pdwProtocol,
    OUT LPBYTE pbAtr,
    OUT LPDWORD pcbAtrLen)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

//  I/O Services
//
//      The following services provide access to the I/O capabilities of the
//      reader drivers.  Services of the Smart Card are requested by placing the
//      following structure into the protocol buffer:
//

WINSCARPAPI LONG WINAPI SCardTransmit(
    IN SCARDHANDLE hCard,
    IN LPCSCARD_IO_REQUEST pioSendPci,
    IN LPCBYTE pbSendBuffer,
    IN DWORD cbSendLength,
    IN OUT LPSCARD_IO_REQUEST pioRecvPci,
    OUT LPBYTE pbRecvBuffer,
    IN OUT LPDWORD pcbRecvLength)
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = SCardCreateMessage("SCardTransmit");

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Card", hCard);
            pMessage->AddValueForKey("SendProtocol", pioSendPci->dwProtocol);
            pMessage->AddValueForKey("SendProtocolLen", pioSendPci->cbPciLength);
            pMessage->AddValueForKey("SendBuffer", (BYTE *)pbSendBuffer, cbSendLength);
            pMessage->AddValueForKey("SendBufferLen", cbSendLength);
            if(pioRecvPci != NULL)
            {
                pMessage->AddValueForKey("RecvProtocol", pioRecvPci->dwProtocol);
                pMessage->AddValueForKey("RecvProtocolLen", pioRecvPci->cbPciLength);
            }
            if(pcbRecvLength != NULL && *pcbRecvLength > 0)
            {
                pMessage->AddValueForKey("RecvLen", *pcbRecvLength);
            }

		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                if(pioRecvPci != NULL)
                {
                    pMessage->GetValueForKey("RecvProtocol", &pioRecvPci->dwProtocol);
                    pMessage->GetValueForKey("RecvProtocolLen", &pioRecvPci->cbPciLength);
                }
                unsigned long lLen = 0;
                pMessage->GetValueForKey ("RecvLen", &lLen);
                if(pbRecvBuffer != NULL && lLen > 0)
                {
                    if(*pcbRecvLength == SCARD_AUTOALLOCATE )
                    {
                        BYTE *pBuffer = new BYTE[lLen];
                        memset(pBuffer, 0, lLen);
                        pMessage->GetValueForKey ("RecvBuffer", pBuffer, lLen);
                        memcpy(pbRecvBuffer, &pBuffer, 4);
                    }
                    else
                    {
                        pMessage->GetValueForKey ("RecvBuffer", pbRecvBuffer, lLen);  
                    }
                }
                *pcbRecvLength = lLen;
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
        return Original_SCardTransmit(hCard, pioSendPci, pbSendBuffer, cbSendLength, pioRecvPci, pbRecvBuffer, pcbRecvLength);
    }
    return SCARD_E_NO_SERVICE;
}


//  Reader Control Routines
//
//      The following services provide for direct, low-level manipulation of the
//      reader by the calling application allowing it control over the
//      attributes of the communications with the card.
//

//#ifdef _WIN32
// The API of this function changed. 
// In pcsc-lite 1.2.0 and before the API was not Windows PC/SC compatible. This has been corrected
WINSCARPAPI LONG WINAPI SCardControl(
    IN      SCARDHANDLE hCard,
    IN      DWORD dwControlCode,
    IN      LPCVOID lpInBuffer,
    IN      DWORD nInBufferSize,
    OUT     LPVOID lpOutBuffer,
    IN      DWORD nOutBufferSize,
    OUT     LPDWORD lpBytesReturned)
/*
#else
WINSCARPAPI LONG WINAPI SCardControl(
    IN      SCARDHANDLE hCard,
    IN      const unsigned char *lpInBuffer,
    IN      DWORD nInBufferSize,
    OUT     unsigned char *lpOutBuffer,
    OUT     LPDWORD lpBytesReturned)
#endif
*/
{
    CheckConnection();
    if(gbServerMode)
    {
        long lRet = SCARD_F_COMM_ERROR;

        CGenericMessage *pMessage = NULL;
        SCARDHANDLE hLocal = hCard;

#ifdef BELPIC_PIN_PAD
        // Support for native pinpad
        if(hCard == SCR_CARD_HANDLE)
        {
            pMessage = SCardCreateMessage("SCardPinPadControl");
            // Parse TLV
            CTLVBuffer oTLVBuffer;
            oTLVBuffer.ParseTLV((unsigned char *)lpInBuffer, nInBufferSize);
            long lOp = 0;
            oTLVBuffer.FillLongData(0x01, &lOp);
            if(SCR_VERIFY_ID == lOp || SCR_CHANGE_ID == lOp)
            {
                oTLVBuffer.FillLongData(0x03, (long *)&hLocal);
            }
        }
        else
#endif // BELPIC_PIN_PAD
        {
            pMessage = SCardCreateMessage("SCardControl");
        }

	    if (pMessage != NULL)
	    {
            pMessage->AddValueForKey("Card", hLocal);
//#ifdef _WIN32
            pMessage->AddValueForKey("ControlCode", dwControlCode);
//#endif
            if(lpInBuffer != NULL && nInBufferSize > 0)
            {
                pMessage->AddValueForKey("SendBuffer", (BYTE *)lpInBuffer, nInBufferSize);
                pMessage->AddValueForKey("SendBufferLen", nInBufferSize);
            }
//#ifdef _WIN32
            if(nOutBufferSize > 0)
            {
                pMessage->AddValueForKey("RecvLen", nOutBufferSize);
            }
//#endif
		    unsigned long ulMessageID = gpConnectionClient->SendMessage (pMessage);
		    pMessage = (CGenericMessage *)gpConnectionClient->WaitMessageForID (ulMessageID);
            if(pMessage != NULL)
            {
                unsigned long lLen = 0;
                pMessage->GetValueForKey ("RecvLen", &lLen);
                if(lpOutBuffer != NULL && lLen > 0)
                {
//#ifdef _WIN32
                    if(nOutBufferSize == SCARD_AUTOALLOCATE )
                    {
                        BYTE *pBuffer = new BYTE[lLen];
                        memset(pBuffer, 0, lLen);
                        pMessage->GetValueForKey ("RecvBuffer", pBuffer, lLen);
                        memcpy(lpOutBuffer, &pBuffer, 4);
                    }
                    else
                    {
//#endif
                        pMessage->GetValueForKey ("RecvBuffer", (BYTE *)lpOutBuffer, lLen);  
//#ifdef _WIN32
                    }
//#endif
                }
                if(lpBytesReturned != NULL)
                {
                    *lpBytesReturned = lLen;
                }
                pMessage->GetValueForKey ("Return", (unsigned long *)&lRet);  
                delete pMessage;
            }
        }
        return lRet;
    }
    else if(gpWinScardLoader->GetLibHandle() > 0)
    {
#ifdef BELPIC_PIN_PAD
        // Support for native pinpad
        if(hCard == SCR_CARD_HANDLE)
        {
            return HandlePinPad(lpInBuffer, nInBufferSize, lpOutBuffer, lpBytesReturned);
        }
#endif
//#ifdef _WIN32
        return Original_SCardControl(hCard, dwControlCode, lpInBuffer, nInBufferSize, lpOutBuffer, nOutBufferSize, lpBytesReturned);
/*
#else
        return Original_SCardControl(hCard, lpInBuffer, nInBufferSize, lpOutBuffer, lpBytesReturned);
#endif
*/
    }
    return SCARD_E_NO_SERVICE;
}


WINSCARPAPI LONG WINAPI SCardGetAttrib(
    IN SCARDHANDLE hCard,
    IN DWORD dwAttrId,
    OUT LPBYTE pbAttr,
    IN OUT LPDWORD pcbAttrLen)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

WINSCARPAPI LONG WINAPI SCardSetAttrib(
    IN SCARDHANDLE hCard,
    IN DWORD dwAttrId,
    IN LPCBYTE pbAttr,
    IN DWORD cbAttrLen)
{
    return SCARD_E_CARD_UNSUPPORTED;
}

///////////////////////////////////////////////////////////
CGenericMessage *SCardCreateMessage(char *pszName)
{
	CGenericMessage *pMessage = NULL;
    if(pszName != NULL)
    {
        CObjectManager *pObjectManager = PASS_GET_OBJECTMANAGER;
	    pMessage = (CGenericMessage *)pObjectManager->CreateByNameAndType ("GenericMessage", "Message");
	    if (pMessage != NULL)
	    {
		    pMessage->SetMessageName ("SCardHandler");
            pMessage->AddValueForKey("FunctionName", pszName);
            FillUserData(pMessage);
        }
    }
    return pMessage;
}

void FillUserData(CGenericMessage *pMessage)
{
#ifdef _WIN32
    char szTemp[256] = {0};
    unsigned long ulSize = sizeof(szTemp);
    if (GetUserNameA(szTemp, &ulSize))
    {
        pMessage->AddValueForKey("UserName", szTemp);
    }
#endif

    // Lookup ProcessID
    long lProcessID = getpid();
    pMessage->AddValueForKey("ProcessID", lProcessID);
}


#ifdef BELPIC_PIN_PAD

LONG HandlePinPad(const void *lpInBuffer, DWORD nInBufferSize, void *lpOutBuffer, DWORD *lpBytesReturned)
{
    long lRet = SCARD_F_COMM_ERROR;
    *lpBytesReturned = 0;

    // Parse TLV
    CTLVBuffer oTLVBuffer;
    oTLVBuffer.ParseTLV((unsigned char *)lpInBuffer, nInBufferSize);

    long lOp = 0;
    char szPinPad[256] = {0};
    oTLVBuffer.FillLongData(0x01, &lOp); 
    oTLVBuffer.FillASCIIData(0x02, szPinPad);

    BOOL bFound = FALSE;
    int i = 0;
    int iIndex = -1;
    for(i = 0; i < PINPADNUM; i++)
    {
        if(0 == strcmp(gpPinPadDlls[i].szPinPadDll, szPinPad))
        {
            bFound = TRUE;
            break;
        }
        if(gpPinPadDlls[i].pDll == NULL && iIndex == -1)
        {
            iIndex = i;
            break;
        }
    }
    wxDynamicLibrary *pPinPadLoader = NULL;
    if (bFound)
    {
        pPinPadLoader = gpPinPadDlls[i].pDll;
    }
    else
    {
        pPinPadLoader = new wxDynamicLibrary(szPinPad);
        gpPinPadDlls[iIndex].pDll = pPinPadLoader;
        strcpy(gpPinPadDlls[iIndex].szPinPadDll, szPinPad);
    }

    if(pPinPadLoader && pPinPadLoader->GetLibHandle() > 0)
    {
        if(SCR_INIT_ID == lOp)
        {
            char szReader[256] = {0};
            long lVersion = 0;
            oTLVBuffer.FillASCIIData(0x03, szReader);
            oTLVBuffer.FillLongData(0x04, &lVersion);
            SCR_SupportConstants supported;
            SCR_INIT_PROC scr_init = (SCR_INIT_PROC)pPinPadLoader->GetSymbol("SCR_Init");
            if(scr_init != NULL)
            {
                lRet = scr_init(szReader, lVersion, &supported);
                ((BYTE *)lpOutBuffer)[0] = supported;
                *lpBytesReturned = 1;
            }
        }
        else if(SCR_VERIFY_ID == lOp || SCR_CHANGE_ID == lOp)
        {
            unsigned long ulLen = 0;
            SCR_Card card = {0};
            SCR_PinUsage usage = {0};
            SCR_Application application = {0};
            BYTE pinID = 0;
            BYTE *pCardData = NULL;
            char *pszLanguage = NULL;
            char *pszShortUsage = NULL;
            char *pszLongUsage = NULL;
            BYTE *pAppData = NULL;
            char *pszShortApp = NULL;
            char *pszLongApp = NULL;
            BYTE ucRet[2] = {0};

            oTLVBuffer.FillLongData(0x03, (long *)&card.hCard);
            pszLanguage = new char[256];
            memset(pszLanguage, 0, 256);
            oTLVBuffer.FillASCIIData(0x04, pszLanguage);
            card.language = pszLanguage;
            if(oTLVBuffer.GetTagData(0x05))
            {
                pCardData = new BYTE[256];
                memset(pCardData, 0, 256);
                oTLVBuffer.FillBinaryData(0x05, pCardData, &ulLen);  
                card.id.data = pCardData;
                card.id.length = ulLen;
            }
            oTLVBuffer.FillBinaryData(0x06, &pinID, &ulLen);  
            if(oTLVBuffer.GetTagData(0x07))
            {
                oTLVBuffer.FillLongData(0x07, &usage.code);
                if(oTLVBuffer.GetTagData(0x08))
                {
                    pszShortUsage = new char[256];
                    memset(pszShortUsage, 0, 256);
                    oTLVBuffer.FillASCIIData(0x08, pszShortUsage);
                    usage.shortString = pszShortUsage;
                }
                if(oTLVBuffer.GetTagData(0x09))
                {
                    pszLongUsage = new char[256];
                    memset(pszLongUsage, 0, 256);
                    oTLVBuffer.FillASCIIData(0x09, pszLongUsage);
                    usage.longString = pszLongUsage;
                }
            }
            if(oTLVBuffer.GetTagData(0x0A))
            {
                pAppData = new BYTE[256];
                memset(pAppData, 0, 256);
                oTLVBuffer.FillBinaryData(0x0A, pAppData, &ulLen);  
                application.id.data = pAppData;
                application.id.length = ulLen;
            }
            if(oTLVBuffer.GetTagData(0x0B))
            {
                pszShortApp = new char[256];
                memset(pszShortApp, 0, 256);
                oTLVBuffer.FillASCIIData(0x0B, pszShortApp);
                application.shortString = pszShortApp;

            }
            if(oTLVBuffer.GetTagData(0x0C))
            {
                pszLongApp = new char[256];
                memset(pszLongApp, 0, 256);
                oTLVBuffer.FillASCIIData(0x0C, pszLongApp);
                application.longString = pszLongApp;
            }
            if(SCR_VERIFY_ID == lOp)
            {
                SCR_VERIFYPIN_PROC scr_verify = (SCR_VERIFYPIN_PROC)pPinPadLoader->GetSymbol("SCR_VerifyPIN");
                if(scr_verify != NULL)
                {
                    lRet = scr_verify(&card, pinID, &usage, &application, ucRet);
                    memcpy(lpOutBuffer, ucRet, 2);
                    *lpBytesReturned = 2;
                }
            }
            else if (SCR_CHANGE_ID == lOp)
            {
                SCR_CHANGEPIN_PROC scr_change = (SCR_CHANGEPIN_PROC)pPinPadLoader->GetSymbol("SCR_ChangePIN");
                if(scr_change != NULL)
                {
                    lRet = scr_change(&card, pinID, &application, ucRet);
                    memcpy(lpOutBuffer, ucRet, 2);
                    *lpBytesReturned = 2;
                }
            }
            if(pCardData)
                delete [] pCardData;
            if(pszLanguage)
                delete [] pszLanguage;
            if(pszShortUsage)
                delete [] pszShortUsage;
            if(pszLongUsage)
                delete [] pszLongUsage;
            if(pAppData)
                delete [] pAppData;
            if(pszShortApp)
                delete [] pszShortApp;
            if(pszLongApp)
                delete [] pszLongApp;
        }
    }
    return lRet;
}

#endif // BELPIC_PIN_PAD

char *pcsc_stringify_error(long Error)
{
	static char strError[75];

	switch (Error)
	{
	case SCARD_S_SUCCESS:
		strcpy(strError, "Command successful.");
		break;
	case SCARD_E_CANCELLED:
		strcpy(strError, "Command cancelled.");
		break;
	case SCARD_E_CANT_DISPOSE:
		strcpy(strError, "Cannot dispose handle.");
		break;
	case SCARD_E_INSUFFICIENT_BUFFER:
		strcpy(strError, "Insufficient buffer.");
		break;
	case SCARD_E_INVALID_ATR:
		strcpy(strError, "Invalid ATR.");
		break;
	case SCARD_E_INVALID_HANDLE:
		strcpy(strError, "Invalid handle.");
		break;
	case SCARD_E_INVALID_PARAMETER:
		strcpy(strError, "Invalid parameter given.");
		break;
	case SCARD_E_INVALID_TARGET:
		strcpy(strError, "Invalid target given.");
		break;
	case SCARD_E_INVALID_VALUE:
		strcpy(strError, "Invalid value given.");
		break;
	case SCARD_E_NO_MEMORY:
		strcpy(strError, "Not enough memory.");
		break;
	case SCARD_F_COMM_ERROR:
		strcpy(strError, "RPC transport error.");
		break;
	case SCARD_F_INTERNAL_ERROR:
		strcpy(strError, "Unknown internal error.");
		break;
	case SCARD_F_UNKNOWN_ERROR:
		strcpy(strError, "Unknown internal error.");
		break;
	case SCARD_F_WAITED_TOO_LONG:
		strcpy(strError, "Waited too long.");
		break;
	case SCARD_E_UNKNOWN_READER:
		strcpy(strError, "Unknown reader specified.");
		break;
	case SCARD_E_TIMEOUT:
		strcpy(strError, "Command timeout.");
		break;
	case SCARD_E_SHARING_VIOLATION:
		strcpy(strError, "Sharing violation.");
		break;
	case SCARD_E_NO_SMARTCARD:
		strcpy(strError, "No smartcard inserted.");
		break;
	case SCARD_E_UNKNOWN_CARD:
		strcpy(strError, "Unknown card.");
		break;
	case SCARD_E_PROTO_MISMATCH:
		strcpy(strError, "Card protocol mismatch.");
		break;
	case SCARD_E_NOT_READY:
		strcpy(strError, "Subsystem not ready.");
		break;
	case SCARD_E_SYSTEM_CANCELLED:
		strcpy(strError, "System cancelled.");
		break;
	case SCARD_E_NOT_TRANSACTED:
		strcpy(strError, "Transaction failed.");
		break;
	case SCARD_E_READER_UNAVAILABLE:
		strcpy(strError, "Reader/s is unavailable.");
		break;
	case SCARD_W_UNSUPPORTED_CARD:
		strcpy(strError, "Card is not supported.");
		break;
	case SCARD_W_UNRESPONSIVE_CARD:
		strcpy(strError, "Card is unresponsive.");
		break;
	case SCARD_W_UNPOWERED_CARD:
		strcpy(strError, "Card is unpowered.");
		break;
	case SCARD_W_RESET_CARD:
		strcpy(strError, "Card was reset.");
		break;
	case SCARD_W_REMOVED_CARD:
		strcpy(strError, "Card was removed.");
		break;
	case SCARD_W_INSERTED_CARD:
		strcpy(strError, "Card was inserted.");
		break;
	case SCARD_E_UNSUPPORTED_FEATURE:
		strcpy(strError, "Feature not supported.");
		break;
	case SCARD_E_PCI_TOO_SMALL:
		strcpy(strError, "PCI struct too small.");
		break;
	case SCARD_E_READER_UNSUPPORTED:
		strcpy(strError, "Reader is unsupported.");
		break;
	case SCARD_E_DUPLICATE_READER:
		strcpy(strError, "Reader already exists.");
		break;
	case SCARD_E_CARD_UNSUPPORTED:
		strcpy(strError, "Card is unsupported.");
		break;
	case SCARD_E_NO_SERVICE:
		strcpy(strError, "Service not available.");
		break;
	case SCARD_E_SERVICE_STOPPED:
		strcpy(strError, "Service was stopped.");
		break;

	};

	return strError;
}
