/*
 * scgui_win32.c: GUI Support for Belgium EID card
 *
 * Copyright (C) 2003, Zetes Belgium
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* README: using 1 big file ==> easier for integration with opensc.dll ... */

#include "scgui.h"
#include "resource.h"
#include <windows.h>

/* fixme: also defined in card-belpic.c */
#define BELPIC_PIN_MIN_LENGTH 4
#define BELPIC_PIN_MAX_LENGTH 12

typedef struct {
	char pin[BELPIC_PIN_MAX_LENGTH+1];
	char *msg;
	char *btn_ok;
	char *btn_cancel;
	char *title;
	char *pinTooShort;
	scgui_param_t iconSign;
} VerifyPinData;

typedef struct {
	char oldpin[BELPIC_PIN_MAX_LENGTH+1];
	char newpin[BELPIC_PIN_MAX_LENGTH+1];
	char confirmpin[BELPIC_PIN_MAX_LENGTH+1];
	char *msg;
	char *btn_ok;
	char *btn_cancel;
	char *title;
	char *pinTooShort;
	char *confirmPinError;
} ChangePinData;

typedef struct {
	char *short_msg;
	char *long_msg;
	scgui_ret_t ret; /* fixme: should use 2nd param of EndDialog to return button clicked */
	char *btn_ok;
	char *btn_cancel;
	char *title;
	char *pinpad_reader;
} AskMessageData;

static BOOL CALLBACK VerifyPinDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam);
static BOOL CALLBACK ChangePinDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam);
static BOOL CALLBACK MessageDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam);
static BOOL CALLBACK AskMessageDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam);

static HINSTANCE _hInst;

BOOL APIENTRY DllMain(HANDLE hModule, DWORD  ul_reason_for_call, LPVOID lpReserved)
{
	switch (ul_reason_for_call) {
	case DLL_PROCESS_ATTACH:
	case DLL_THREAD_ATTACH:
		_hInst = hModule;
		break;

	case DLL_THREAD_DETACH:
	case DLL_PROCESS_DETACH:
		break;
	}
	return TRUE;
}

/************************/
/* SCGUI API functions */
/************************/
SCGUI_API scgui_ret_t scgui_init(void)
{	
	return SCGUI_OK;
}

SCGUI_API scgui_ret_t scgui_enterpin(const char *title, const char *msg, char *pin, int *len, const char *btn_ok, const char *btn_cancel, const char *pinTooShort, scgui_param_t iconSign)
{
	VerifyPinData data;

	if (*len != BELPIC_PIN_MAX_LENGTH+1) {
		return SCGUI_BUFFER_TOO_SMALL;
	}

	data.msg = (char *)msg;
	data.pin[0] = '\0';
	data.btn_ok = (char *)btn_ok;
	data.btn_cancel = (char *)btn_cancel;
	data.title = (char *)title;
	data.pinTooShort = (char *)pinTooShort;
	data.iconSign = iconSign;

	if (iconSign== SCGUI_NO_ICON)
		DialogBoxParam(_hInst, MAKEINTRESOURCE(IDD_VERIFY_PIN), NULL, VerifyPinDlgProc, (LPARAM)&data);
	else if (iconSign == SCGUI_SIGN_ICON)
		DialogBoxParam(_hInst, MAKEINTRESOURCE(IDD_VERIFY_PIN_NONREP), NULL, VerifyPinDlgProc, (LPARAM)&data);
	else
		return SCGUI_BAD_PARAM;

	*len = strlen(data.pin);
	strcpy(pin, data.pin);

	if (*len > 0) {
		return SCGUI_OK;
	}

	return SCGUI_CANCEL;
}

SCGUI_API scgui_ret_t scgui_changepin(const char *title, const char *msg, char *oldpin, int *oldpinlen, char *newpin, int *newpinlen, const char *btn_ok, const char *btn_cancel, const char *pinTooShort, const char *confirmPinError)
{
	ChangePinData data;

	if (*oldpinlen != BELPIC_PIN_MAX_LENGTH+1) {
		return SCGUI_BUFFER_TOO_SMALL;
	}

	data.msg = (char *)msg;
	data.oldpin[0] = data.newpin[0] = data.confirmpin[0] = '\0';
	data.btn_ok = (char *)btn_ok;
	data.btn_cancel = (char *)btn_cancel;
	data.title = (char *)title;
	data.pinTooShort = (char *)pinTooShort;
	data.confirmPinError = (char *)confirmPinError;

	DialogBoxParam(_hInst, MAKEINTRESOURCE(IDD_CHANGE_PIN), NULL, ChangePinDlgProc, (LPARAM)&data);
	*newpinlen = strlen(data.newpin);
	*oldpinlen = strlen(data.oldpin);
	strcpy(oldpin, data.oldpin);
	strcpy(newpin, data.newpin);

	if (*newpinlen > 0) {
		return SCGUI_OK;
	}

	return SCGUI_CANCEL;
}

SCGUI_API scgui_ret_t scgui_display_message(const char *title, const char *short_msg, const char *long_msg, const char *btn_close, void **handle, scgui_param_t signIcon, const char *pinpad_reader)
{
	if (!handle) {
		return SCGUI_ERROR; /* fixme: different constant */
	}

	if (signIcon == SCGUI_NO_ICON) {
                if (btn_close) {
		        *handle = CreateDialog(_hInst, MAKEINTRESOURCE(IDD_DISPLAY_MESSAGE), NULL, MessageDlgProc);
                        SetDlgItemText((HWND)*handle, IDC_LABEL_SHORT_MSG, short_msg);
		        SetDlgItemText((HWND)*handle, IDC_LABEL_LONG_MSG, long_msg);
		        SetDlgItemText((HWND)*handle, IDC_LABEL_PINPAD, pinpad_reader);
                } else {
                        *handle = CreateDialog(_hInst, MAKEINTRESOURCE(IDD_DISPLAY_MESSAGE2), NULL, MessageDlgProc);
                        SetDlgItemText((HWND)*handle, IDC_LABEL_SHORT_MSG_2, short_msg);
		        SetDlgItemText((HWND)*handle, IDC_LABEL_LONG_MSG_2, long_msg);
		        SetDlgItemText((HWND)*handle, IDC_LABEL_PINPAD_2, pinpad_reader);
                }
	} else if (signIcon == SCGUI_SIGN_ICON) {
                if (btn_close) {
		        *handle = CreateDialog(_hInst, MAKEINTRESOURCE(IDD_DISPLAY_MESSAGE_NONREP), NULL, MessageDlgProc);
                        SetDlgItemText((HWND)*handle, IDC_LABEL_SHORT_MSG_SIGN, short_msg);
		        SetDlgItemText((HWND)*handle, IDC_LABEL_LONG_MSG_SIGN, long_msg);
		        SetDlgItemText((HWND)*handle, IDC_LABEL_PINPAD_SIGN, pinpad_reader);
                } else {
                        *handle = CreateDialog(_hInst, MAKEINTRESOURCE(IDD_DISPLAY_MESSAGE_NONREP2), NULL, MessageDlgProc);
                        SetDlgItemText((HWND)*handle, IDC_LABEL_SHORT_MSG_SIGN_2, short_msg);
		        SetDlgItemText((HWND)*handle, IDC_LABEL_LONG_MSG_SIGN_2, long_msg);
		        SetDlgItemText((HWND)*handle, IDC_LABEL_PINPAD_SIGN_2, pinpad_reader);
                }
	} else
		return SCGUI_BAD_PARAM;

	if (!*handle) {
		char buf[100];
		wsprintf (buf, "Error x%x", GetLastError ());
		MessageBox (0, buf, "CreateDialog", MB_ICONEXCLAMATION | MB_OK);
		return SCGUI_ERROR;
	}

	
	SendMessage((HWND)*handle, WM_SETTEXT, 0, (LPARAM)title);
        if (btn_close)
	        SendMessage(GetDlgItem((HWND)*handle, IDCLOSE), WM_SETTEXT, 0, (LONG)btn_close);

	ShowWindow(*handle, SW_SHOW);
	UpdateWindow(*handle);
	
	return SCGUI_OK;
}

SCGUI_API scgui_ret_t scgui_remove_message(void *handle)
{
	if (handle && IsWindow(handle))
		DestroyWindow(handle);
	
	return SCGUI_OK;
}

SCGUI_API scgui_ret_t scgui_ask_message(const char *title, const char *short_msg, const char *long_msg, const char *btn_ok, const char *btn_cancel, const char *pinpad_reader)
{
	AskMessageData data;
	data.short_msg = (char *)short_msg;
	data.long_msg = (char*)long_msg;
	data.ret = SCGUI_OK;
	data.btn_ok = (char *)btn_ok;
	data.btn_cancel = (char *)btn_cancel;
	data.title = (char *)title;
	data.pinpad_reader = (char *)pinpad_reader;

	if (btn_cancel)
		DialogBoxParam(_hInst, MAKEINTRESOURCE(IDD_ASK_MESSAGE), NULL, AskMessageDlgProc, (LPARAM)&data);
	else
		DialogBoxParam(_hInst, MAKEINTRESOURCE(IDD_ASK_MESSAGE2), NULL, AskMessageDlgProc, (LPARAM)&data);

	return data.ret;
}

/*******************/
/* private methods */
/*******************/

/* fixme: caret should go max BELPIC_PIN_MAX_LENGTH to the right, now simply truncated and no hscroll */
static BOOL CALLBACK VerifyPinDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static char *pin; /* static ! */
	static char *pinTooShort;
	HWND hPIN = GetDlgItem(hDlg, IDC_PIN);

	switch (msg) {
	case WM_INITDIALOG:
		{
		VerifyPinData *pData = (VerifyPinData *)lParam;
		pin = pData->pin;
		pinTooShort = pData->pinTooShort;
		if (pData->iconSign)
			SetDlgItemText(hDlg, IDC_LABEL_VERIFY_SIGN, pData->msg);
		else
			SetDlgItemText(hDlg, IDC_LABEL_VERIFY, pData->msg);
		SetFocus(hPIN); /* or put this in the .RC file ... */
		SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM)pData->title);
		SendMessage(GetDlgItem(hDlg, IDOK), WM_SETTEXT, 0, (LONG)pData->btn_ok);
		SendMessage(GetDlgItem(hDlg, IDCANCEL), WM_SETTEXT, 0, (LONG)pData->btn_cancel);
		return FALSE; /* need to return false, otherwise SetFocus doesn't work ! */
		}

	case WM_COMMAND:
		{
		int controlid = LOWORD(wParam);
		int command = HIWORD(wParam);
		
		switch (controlid) {
		case IDOK:
			if (strlen(pin) < BELPIC_PIN_MIN_LENGTH) {
				MessageBox(hDlg, pinTooShort, "Error", MB_ICONERROR | MB_OK);
				pin[0] = '\0';
				SetDlgItemText(hDlg, IDC_PIN, "");
				return FALSE;
			}
			EndDialog(hDlg, controlid);
			return TRUE;

		case IDCANCEL:
			pin[0] = '\0';
			EndDialog(hDlg, controlid);
			return TRUE;
		
		case IDC_PIN:		
			if (command == EN_CHANGE && strlen(pin) < BELPIC_PIN_MAX_LENGTH) {
				SendMessage(hPIN, WM_GETTEXT, (WPARAM)BELPIC_PIN_MAX_LENGTH, (LPARAM)pin);				
			}
			
			return FALSE;
		}
		break;
		}
	}

	return FALSE;
}

/* fixme: duplicate code ... */
static BOOL CALLBACK ChangePinDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static char *oldpin;
	static char *newpin;
	static char *confirmpin;
	static char *confirmPinError;
	static char *pinTooShort;

	HWND hOldPIN = GetDlgItem(hDlg, IDC_OLDPIN);
	HWND hNewPIN = GetDlgItem(hDlg, IDC_NEWPIN);
	HWND hConfirmPIN = GetDlgItem(hDlg, IDC_CONFIRMPIN);

	switch (msg) {
	case WM_INITDIALOG:
		{
		ChangePinData *pData = (ChangePinData *)lParam;
		oldpin = pData->oldpin;
		newpin = pData->newpin;
		confirmpin = pData->confirmpin;
		pinTooShort = pData->pinTooShort;
		confirmPinError = pData->confirmPinError;
		SetDlgItemText(hDlg, IDC_LABEL_CHANGEPIN, pData->msg);
		SetFocus(hOldPIN);
		SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM)pData->title);
		SendMessage(GetDlgItem(hDlg, IDOK), WM_SETTEXT, 0, (LONG)pData->btn_ok);
		SendMessage(GetDlgItem(hDlg, IDCANCEL), WM_SETTEXT, 0, (LONG)pData->btn_cancel);
		return FALSE;
		}

	case WM_COMMAND:
		{
		int controlid = LOWORD(wParam);
		int command = HIWORD(wParam);
		
		switch (controlid) {
		case IDOK:
			if (strlen(newpin) < BELPIC_PIN_MIN_LENGTH) {
				MessageBox(hDlg, pinTooShort, "Error", MB_ICONERROR | MB_OK);
				newpin[0] = confirmpin[0] = '\0';
				SetDlgItemText(hDlg, IDC_OLDPIN, "");
				SetDlgItemText(hDlg, IDC_NEWPIN, "");
				SetDlgItemText(hDlg, IDC_CONFIRMPIN, "");
				SetFocus(hOldPIN);
				return FALSE;
			}
			if (strcmp(confirmpin, newpin) != 0) {
				MessageBox(hDlg, confirmPinError, "Error", MB_ICONERROR | MB_OK);
				newpin[0] = confirmpin[0] = '\0';
				SetDlgItemText(hDlg, IDC_OLDPIN, "");
				SetDlgItemText(hDlg, IDC_NEWPIN, "");
				SetDlgItemText(hDlg, IDC_CONFIRMPIN, "");
				SetFocus(hOldPIN);
				return FALSE;
			}

			EndDialog(hDlg, controlid);
			return TRUE;
		
		case IDCANCEL:
			newpin[0] = confirmpin[0] = '\0';
			EndDialog(hDlg, controlid);
			return TRUE;
		
		case IDC_OLDPIN:			
			if (command == EN_CHANGE && strlen(oldpin) < BELPIC_PIN_MAX_LENGTH) {
				SendMessage(hOldPIN, WM_GETTEXT, BELPIC_PIN_MAX_LENGTH, (LPARAM)oldpin);
			}
			
			return FALSE;

		case IDC_NEWPIN:
			if (command == EN_CHANGE && strlen(newpin) < BELPIC_PIN_MAX_LENGTH) {
				SendMessage(hNewPIN, WM_GETTEXT, BELPIC_PIN_MAX_LENGTH, (LPARAM)newpin);
			}
			
			return FALSE;

		case IDC_CONFIRMPIN:
			if (command == EN_CHANGE && strlen(confirmpin) < BELPIC_PIN_MAX_LENGTH) {
				SendMessage(hConfirmPIN, WM_GETTEXT, BELPIC_PIN_MAX_LENGTH, (LPARAM)confirmpin);
			}
			
			return FALSE;
		}
		break;
		}
	}

	return FALSE;
}

static BOOL CALLBACK MessageDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg) {
	case WM_INITDIALOG:
		return TRUE;
		break;
	
	case WM_COMMAND:
		{
		int controlid = LOWORD(wParam);
		int command = HIWORD(wParam);
		
		switch (controlid) {
		case IDCLOSE:
			DestroyWindow(hDlg);
			return TRUE;
		}
		}
		break;
	}

	return FALSE;
}

static BOOL CALLBACK AskMessageDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
	HWND hPIN = GetDlgItem(hDlg, IDC_PIN);
	static AskMessageData *pData;

	switch (msg) {
	case WM_INITDIALOG:
		{
		pData = (AskMessageData *)lParam;
		/* todo: multiple IDC_LABEL_SHORT_MSG necessary ? */
		if (pData->btn_cancel) {
			SetDlgItemText((HWND)hDlg, IDC_LABEL_SHORT_MSG_OKCANCEL, pData->short_msg);
			SetDlgItemText((HWND)hDlg, IDC_LABEL_LONG_MSG_OKCANCEL, pData->long_msg);
			SetDlgItemText((HWND)hDlg, IDC_LABEL_PINPAD_OKCANCEL, pData->pinpad_reader);
		} else {
			SetDlgItemText((HWND)hDlg, IDC_LABEL_SHORT_MSG_OK, pData->short_msg);
			SetDlgItemText((HWND)hDlg, IDC_LABEL_LONG_MSG_OK, pData->long_msg);
			SetDlgItemText((HWND)hDlg, IDC_LABEL_PINPAD_OK, pData->pinpad_reader);
		}
		SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM)pData->title);
		SendMessage(GetDlgItem(hDlg, IDOK), WM_SETTEXT, 0, (LONG)pData->btn_ok);
		if (pData->btn_cancel)
			SendMessage(GetDlgItem(hDlg, IDCANCEL), WM_SETTEXT, 0, (LONG)pData->btn_cancel);
		return TRUE;
		}
		break;
	
	case WM_COMMAND:
		{
		int controlid = LOWORD(wParam);
		int command = HIWORD(wParam);

		switch (controlid) {
		case IDOK:
			pData->ret = SCGUI_OK;
			EndDialog(hDlg, controlid);
			return TRUE;

		case IDCANCEL:
			pData->ret = SCGUI_CANCEL;
			EndDialog(hDlg, controlid);
			return TRUE;
		}
		}
		break;
	}

	return FALSE;
}
