// CertifManager.cpp: implementation of the CCertifManager class.
//
//////////////////////////////////////////////////////////////////////

#include "CertifManager.h"
#include "eiddefines.h"
#include "Verify.h"
#include "eiderrors.h"
#include "Config.h"

#include <opensc/pkcs15.h>

//////////////////////

class CFindCertif
{
private:
    char m_szID[256];
public:
    explicit CFindCertif(const char *pszID)
    {
        memset(m_szID, 0, sizeof(m_szID));
        if(pszID != NULL)
        {
            strcpy(m_szID, pszID);
        }
    }

    bool operator() (eidlib::CCertif *pCertif)
    {
        if(0 == strcmp(pCertif->GetID(), m_szID))
        {
            return true;
        }
        return false;
    }
};


/////////////////////
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

#define CERT_COUNT 5

typedef struct
{
    unsigned char *pucID;
    char *pszLabel;
} tCertLabel;


const unsigned char CCertifManager::m_ucMF[] = {0x3F, 0x00};
const unsigned char CCertifManager::m_ucAID[] = {0xA0, 0x00, 0x00, 0x01, 0x77, 0x50, 0x4B, 0x43, 0x53, 0x2D, 0x31, 0x35};
const unsigned char CCertifManager::m_ucRN[] = {0x50, 0x3C};
const unsigned char CCertifManager::m_ucCA[] = {0x50, 0x3A};
const unsigned char CCertifManager::m_ucROOT[] = {0x50, 0x3B};
const unsigned char CCertifManager::m_ucAUTH[] = {0x50, 0x38};
const unsigned char CCertifManager::m_ucSIG[] = {0x50, 0x39};

const unsigned char *CCertifManager::m_ucCertifs[CERT_COUNT] = {CCertifManager::m_ucRN, CCertifManager::m_ucCA, CCertifManager::m_ucROOT, CCertifManager::m_ucAUTH, CCertifManager::m_ucSIG};

static tCertLabel CertLabels[] = 
{
    {(unsigned char *)CCertifManager::m_ucRN, "RN"},
    {(unsigned char *)CCertifManager::m_ucCA, "CA"},
    {(unsigned char *)CCertifManager::m_ucROOT, "Root"},
    {(unsigned char *)CCertifManager::m_ucAUTH, "Authentication"},
    {(unsigned char *)CCertifManager::m_ucSIG, "Signature"}
};


CCertifManager::CCertifManager()
{
    m_pCardReader = NULL;
}

CCertifManager::~CCertifManager()
{
    Cleanup();
}

void CCertifManager::Cleanup()
{
    for(unsigned int i = 0; i < m_Certifs.size(); ++i)
    {
        delete m_Certifs[i];
    }
    m_Certifs.clear();
}

void CCertifManager::SetCardReader(CCardReader *pReader)
{
    m_pCardReader = pReader;
}

long CCertifManager::ReadCertif(unsigned char *pucID, BEID_Status *ptStatus)
{
    long lRet = SC_NO_ERROR;
    if(m_pCardReader != NULL && pucID != NULL)
    {
        unsigned char ucCertif[BEID_MAX_CERT_LEN] = {0};
        unsigned long ulCertifLen = BEID_MAX_CERT_LEN;
        if(SC_NO_ERROR == (lRet = m_pCardReader->BeginTransaction(ptStatus)))
        {
            lRet = m_pCardReader->SelectFile(m_ucMF, sizeof(m_ucMF), 0x02, ptStatus);
            if(lRet == SC_NO_ERROR)
            {
                if(SC_NO_ERROR == (lRet = m_pCardReader->SelectFile(m_ucAID, sizeof(m_ucAID), 0x04, ptStatus)))
                {
                    if(SC_NO_ERROR == (lRet = m_pCardReader->SelectFile(pucID, 2, 0x02, ptStatus)))
                    {
                        if (SC_NO_ERROR == (lRet = m_pCardReader->ReadBinary(ucCertif, &ulCertifLen, ptStatus)))
                        {
                            char *pszLabel = PathToLabel(pucID);
                            if(pszLabel == NULL)
                            {
                                pszLabel = "Unknown";
                            }
                            eidlib::CCertif *pCertif = new eidlib::CCertif(ucCertif, ulCertifLen, pszLabel);
                            m_Certifs.push_back(pCertif);
                        }
                    }
                }
           }
            m_pCardReader->EndTransaction(ptStatus);
        }
    }
    return lRet;
}

long CCertifManager::ReadCertifs(BEID_Status *ptStatus)
{
    long lRet = SC_NO_ERROR;
    if(m_pCardReader != NULL)
    {
        Cleanup();
        if(SC_NO_ERROR == (lRet = m_pCardReader->BeginTransaction(ptStatus)))
        {
            for (int i = 0; i < CERT_COUNT; i++)
            {
                ReadCertif((unsigned char *)m_ucCertifs[i], ptStatus);
            }
            m_pCardReader->EndTransaction(ptStatus);
        }
    }
    return lRet;
}

long CCertifManager::ReadCertifsP15(BEID_Status *ptStatus, BOOL bAll /* TRUE */)
{
    long lRet = SC_NO_ERROR;
    if(m_pCardReader != NULL)
    {
        if(m_pCardReader->GetCard() != NULL)
        {
            Cleanup();

            struct sc_card *pCard = (sc_card *)m_pCardReader->GetCard(); 
            struct sc_pkcs15_card *p15Card = NULL;

            if(SC_NO_ERROR == (lRet = m_pCardReader->BeginTransaction(ptStatus)))
            {
                if(SC_NO_ERROR == (lRet = sc_pkcs15_bind(pCard, &p15Card)))
                {
                    if(bAll)
                    {
                        // List Certificates
                        int iNumber = 0;
                        struct sc_pkcs15_object *pObjs[32];
	                    iNumber = sc_pkcs15_get_objects(p15Card, SC_PKCS15_TYPE_CERT_X509, pObjs, 32);

                        for (int i = 0; i < iNumber; i++) 
                        {
		                    struct sc_pkcs15_cert_info *pCertInfo = (struct sc_pkcs15_cert_info *) pObjs[i]->data;
		                    struct sc_pkcs15_cert *pCert = NULL;
		                    if(SC_NO_ERROR != (lRet = sc_pkcs15_read_certificate(p15Card, pCertInfo, &pCert)))
                                break;
                            AddCertif(pCert->data, pCert->data_len, pObjs[i]->label);
		                    sc_pkcs15_free_certificate(pCert);
                        }
                    }
                }

                if(SC_NO_ERROR == lRet)
                {
                    // Read RN Certificate
                    if(NULL == GetCertif("RN"))
                    {
                        // First check cache
                        struct sc_path tPath = {0};
                        tPath.type = SC_PATH_TYPE_PATH;
                        strcpy((char *)tPath.value, (const char *)m_ucRN);
                        tPath.len = sizeof(m_ucRN);
                        tPath.count = -1;
                        unsigned char *pCache = new unsigned char[BEID_MAX_CERT_LEN];
                        memset(pCache, 0, BEID_MAX_CERT_LEN);
                        unsigned int iLen = BEID_MAX_CERT_LEN;
                        if(eidlib::CConfig::AllowCaching() && SC_NO_ERROR == sc_pkcs15_read_cached_file(p15Card, &tPath, &pCache, &iLen))
                        {
                            AddCertif(pCache, iLen, PathToLabel(m_ucRN));
                        }
                        else
                        {
                            lRet = ReadCertif((unsigned char *)CCertifManager::m_ucRN, ptStatus);
                            eidlib::CCertif *pRNCert = GetCertif("RN");
                            if(eidlib::CConfig::AllowCaching() && NULL != pRNCert)
                            {
                                sc_pkcs15_cache_file(p15Card, &tPath, pRNCert->GetData(), pRNCert->GetLength());  
                            }
                        }
                        delete pCache;
                    }
                }        
        
                if (p15Card != NULL)
                {
		            sc_pkcs15_unbind(p15Card);
                }
        
                m_pCardReader->EndTransaction(ptStatus);
            }
        }
        else
        {
            lRet = ReadCertifs(ptStatus);
        }
    }
    ConvertOpenSCError(lRet, ptStatus);
    return lRet;
}

void CCertifManager::AddCertif(unsigned char *pucData, unsigned long ulDataLen, char *pszLabel)
{
    if(pucData != NULL && ulDataLen > 0)
    {
        eidlib::CCertif *pCertif = new eidlib::CCertif(pucData, ulDataLen, pszLabel);
        m_Certifs.push_back(pCertif);
    }
}


void CCertifManager::FillCertifs(BEID_Certif_Check *pData)
{
    if(pData != NULL)
    {
        int iCertNumber = m_Certifs.size();
        if(iCertNumber > BEID_MAX_CERT_NUMBER)
        {
            iCertNumber = BEID_MAX_CERT_NUMBER;
        }
        pData->certificatesLength =  iCertNumber;
        for(int i = 0; i < iCertNumber; ++i)
        {
            eidlib::CCertif *pCertif = m_Certifs[i];
            if(pCertif != NULL)
            {
                memcpy(pData->certificates[i].certif, pCertif->GetData(), pCertif->GetLength()); 
                pData->certificates[i].certifLength = pCertif->GetLength();
                pData->certificates[i].certifStatus = pCertif->GetCertStatus();
                memset(pData->certificates[i].certifLabel, 0, BEID_MAX_CERT_LABEL_LEN);
                strcpy(pData->certificates[i].certifLabel, pCertif->GetID()); 
            }
        }
    }
}

eidlib::CCertif *CCertifManager::GetCertif(const char *pszID)
{
    eidlib::CCertif *pCertif = NULL;
    if(pszID != NULL)
    {
        ItCertifs it = std::find_if(m_Certifs.begin(), m_Certifs.end(), CFindCertif(pszID));
        if(it != m_Certifs.end())
        {
            pCertif = (*it);
        }
    }    
    return pCertif;
}

char *CCertifManager::PathToLabel(const unsigned char *pucID)
{
    char *pPath = NULL;

    if(pucID != NULL)
    {
        int k = sizeof(CertLabels) / sizeof(tCertLabel);
        
        for(int i = 0; i < k ; i++)
        {
            if(0== memcmp(CertLabels[i].pucID, pucID, 2))
            {
                pPath = CertLabels[i].pszLabel;
                break;
            }
        }
    }    
    return pPath;
}

long CCertifManager::VerifyCertsCRL()
{
    long lRet = CVerify::VerifyCRL(m_Certifs); 
    return lRet;
}

long CCertifManager::VerifyCertsOCSP()
{
    long lRet = CVerify::VerifyOCSP(m_Certifs); 
    return lRet;
}

void CCertifManager::UpdateCertStatus(void *pCertX509, long lStatus)
{
    CVerify::UpdateCertStatus(m_Certifs, pCertX509, lStatus);
}

bool CCertifManager::FindNotValidated()
{
    bool bRet = false;
    for(unsigned int i = 0; i < m_Certifs.size() && !bRet; ++i)
    {
        eidlib::CCertif *pCertif = m_Certifs[i];
        if(pCertif != NULL && BEID_CERTSTATUS_CERT_NOT_VALIDATED == pCertif->GetCertStatus())
        {
            // Found one
            bRet = true;
        }
    }
    return bRet;
}

