// PCSCManager.cpp: implementation of the CPCSCManager class.
//
//////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "PCSCManager.h"
#include "ApplicationManager.h"
#include "ReaderObject.h"
#include "ApplicationObject.h"
#include "TransactionStack.h"
#include "TLVBuffer.h"


#define TIMEOUTSTEP 100 // 100 milliseconds
#define KILLSTACKTIMEOUT 50 //5 seconds, 50 * 100 milliseconds 


#ifdef BELPIC_PIN_PAD

// PinPad stuff
#define SCR_INIT_ID          100
#define SCR_VERIFY_ID      101
#define SCR_CHANGE_ID    102
#define SCR_CARD_HANDLE          999
#define PINPADNUM 3

typedef enum 
{
	SCR_SUPPORT_OK = 0L,
	SCR_SUPPORT_INCOMPATIBLE_CALLING_VERSION = 1L,
	SCR_SUPPORT_INCOMPATIBLE_FIRMWARE = 2L,
	SCR_SUPPORT_INCOMPATIBLE_FIRMWARE_VERSION = 3L,
} SCR_SupportConstants;

typedef struct 
{
	BYTE *data;
	DWORD length;
} SCR_Bytes;

typedef struct 
{
	SCARDHANDLE hCard;
	char *language;
	SCR_Bytes id;
	void *pinFormat; /* reserved for future use */
} SCR_Card;

typedef struct 
{
	SCR_Bytes id;
	char *shortString;
	char *longString;
} SCR_Application;

typedef struct 
{
	long code;
	char *shortString;
	char *longString;
} SCR_PinUsage;


typedef long (*SCR_INIT_PROC)(LPCTSTR, DWORD, SCR_SupportConstants*);
typedef long (*SCR_VERIFYPIN_PROC)(const SCR_Card*, BYTE, const SCR_PinUsage*, const SCR_Application*, BYTE *);
typedef long (*SCR_CHANGEPIN_PROC)(const SCR_Card*, BYTE, const SCR_Application*, BYTE *);

#endif // BELPIC_PIN_PAD


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CPCSCManager::CPCSCManager()  : CManager ("PCSC")
{
	PASS_INITLOCK(&lock);
    m_hContext = 0;
    m_pTransActionStack = new CTransactionStack;
    if(NULL != m_pTransActionStack)
    {
        m_pTransActionStack->Start();
    }
#ifdef BELPIC_PIN_PAD
    m_pPinPadDlls = new PinPadObject[PINPADNUM];
    memset(m_pPinPadDlls, 0, sizeof(PinPadObject) * PINPADNUM);
#endif
}

CPCSCManager::~CPCSCManager()
{
	unsigned long ulTimeout = 0;

    if(NULL != m_pTransActionStack)
    {
        m_pTransActionStack->Stop();

        // Wait until it stops
	    ulTimeout = 0;
	    while (m_pTransActionStack->m_bCanBeDeleted == FALSE && ulTimeout < KILLSTACKTIMEOUT)
	    {
		    PASS_MILLISLEEP(TIMEOUTSTEP);
		    ulTimeout++;
	    }
	    // If reach timeout, kill the thread and delete it
	    if (ulTimeout == KILLSTACKTIMEOUT)
	    {
		    m_pTransActionStack->Kill();
	    }
        delete m_pTransActionStack;
        m_pTransActionStack = NULL;
    }

#ifdef BELPIC_PIN_PAD
    if(m_pPinPadDlls != NULL)
    {
        for(int i = 0; i < PINPADNUM; i++)
        {
            if(m_pPinPadDlls[i].pDll  != NULL)
            {
                delete m_pPinPadDlls[i].pDll;
            }
        }
        delete m_pPinPadDlls;
        m_pPinPadDlls = NULL;
    }
#endif

   	ULONG32 position = 0; 
    char szKey[256]; 
    CReaderObject *pObject = (CReaderObject *)m_Readers.GetFirst (&position, szKey, sizeof(szKey));

    while (pObject != NULL) 
    {   
        if(pObject->GetCardHandle () > 0)
        {
            // Disconnect Card
            SCardDisconnect(pObject->GetCardHandle (), SCARD_LEAVE_CARD);
        }
        delete pObject;
        pObject = (CReaderObject *)m_Readers.GetNext (&position, szKey, sizeof(szKey)); 
    }

     if(m_hContext != 0)
     {
         // Release Context
         SCardReleaseContext(m_hContext);
         m_hContext = 0;
     }

	PASS_DESTROYLOCK (&lock);
}

void CPCSCManager::Initialize()
{
     long lReturn;
     // Establish the context.
     if(m_hContext != 0)
     {
         // Release Context
         ::SCardReleaseContext(m_hContext);
         m_hContext = 0;
     }
     do
     {
        m_hContext = 0;
        lReturn = ::SCardEstablishContext(SCARD_SCOPE_USER, NULL, NULL, &m_hContext);
        if(SCARD_S_SUCCESS != lReturn)
        {
            PASS_MILLISLEEP(1000);
        }
     }
     while(SCARD_S_SUCCESS != lReturn);
}

SCARDCONTEXT CPCSCManager::GetContextHandle()
{
    return m_hContext;
}

long CPCSCManager::S_CardEstablishContext(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        SCARDCONTEXT hContext = 0;
        unsigned long lConnID = 0;
        pMessage->GetValueForKey("ConnID", &lConnID);
        CApplicationObject *pApplication = NULL;
        if(NULL != (pApplication = pApplicationManager->FindExisting(lConnID)))
        {
            pApplication->AddRef(); 
        }
        else
        {
            // New Application
            pApplication = pApplicationManager->CreateApplication(pMessage, lConnID);
        }
        if(pApplication)
        {
            hContext = pApplication->GetContext();
            lRet = SCARD_S_SUCCESS;
        }
        pMessage->AddValueForKey("Context", hContext);
    }        
    return lRet;
}

long CPCSCManager::S_CardReleaseContext(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        SCARDCONTEXT hContext = 0;
        pMessage->GetValueForKey ("Context", (unsigned long *)&hContext);
        if(NULL != pApplicationManager->FindApplication(hContext))
        {
            pApplicationManager->DeleteApplication(hContext);
            lRet = SCARD_S_SUCCESS;
        }
    }        
    return lRet;
}

long CPCSCManager::S_CardConnect(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        CApplicationObject *pApplication = NULL;
        SCARDCONTEXT hContext = 0;
        pMessage->GetValueForKey ("Context", (unsigned long *)&hContext);
        if(NULL != (pApplication = pApplicationManager->FindApplication(hContext)))
        {
            char szReader[256] = {0};
            pMessage->GetValueForKey ("Reader", szReader, sizeof(szReader));
            CReaderObject *pReader = (CReaderObject *)m_Readers.Lookup(szReader);
            if(pReader != NULL)
            {
                if(pReader->GetCardHandle() > 0)
                {
                    // Card in reader
                    SCARDHANDLE hLocal = pApplication->AddReader(pReader);
                    pMessage->AddValueForKey ("Card", hLocal);
                    pMessage->AddValueForKey ("ActiveProtocol", pReader->GetProtocol());                    
                    lRet = SCARD_S_SUCCESS;
                }
                else
                {
                    lRet = SCARD_E_NO_SMARTCARD;
                }
            }
            else
            {
                lRet = SCARD_E_UNKNOWN_READER;
            }
        }
    }
    return lRet;
}

long CPCSCManager::S_CardDisconnect(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        CApplicationObject *pApplication = NULL;
        SCARDHANDLE hCard = 0;
        unsigned long lConnID = 0;
        pMessage->GetValueForKey ("Card", (unsigned long *)&hCard);
        pMessage->GetValueForKey("ConnID", &lConnID);
        if(NULL != (pApplication = pApplicationManager->FindExisting(lConnID)))
        {        
            // Card in reader
            pApplication->DeleteReader(hCard);
            lRet = SCARD_S_SUCCESS;
        }
    }
    return lRet;
}

long CPCSCManager::S_CardStatus(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        CApplicationObject *pApplication = NULL;
        SCARDHANDLE hCard = 0;
        unsigned long lConnID = 0;
        pMessage->GetValueForKey ("Card", (unsigned long *)&hCard);
        pMessage->GetValueForKey("ConnID", &lConnID);
        if(NULL != (pApplication = pApplicationManager->FindExisting(lConnID)))
        {        
            // Card in reader
            CReaderObject *pReader = pApplication->GetReader(hCard);
            if(pReader != NULL)
            {
                char szReaderName[256] = {0};
                unsigned long cchLen = sizeof(szReaderName);
                unsigned long ulState = 0;
                unsigned long ulProtocol = 0;
                unsigned char ucAtr[32] = {0};
                unsigned long ulAtrLen = sizeof(ucAtr);
                SCARDHANDLE hReal = pApplication->GetOriginal(hCard);
                if(SCARD_S_SUCCESS == (lRet = SCardStatus(hReal, szReaderName, &cchLen, &ulState, &ulProtocol, ucAtr, &ulAtrLen)))
                {
                    pMessage->AddValueForKey("ReaderName", szReaderName);
                    pMessage->AddValueForKey("ReaderLen", cchLen);
                    pMessage->AddValueForKey("State", ulState);
                    pMessage->AddValueForKey("Protocol", ulProtocol);
                    pMessage->AddValueForKey("Atr", ucAtr, ulAtrLen);
                    pMessage->AddValueForKey("AtrLen", ulAtrLen);
                }
            }
        }
    }
    return lRet;
}

long CPCSCManager::S_CardListReaders(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        SCARDCONTEXT hContext = 0;
        pMessage->GetValueForKey ("Context", (unsigned long *)&hContext);
        if(NULL != pApplicationManager->FindApplication(hContext))
        {
            char *pszGroups = NULL;
            char szGroups[256] = {0};
            if(pMessage->GetValueForKey("Groups", szGroups, sizeof(szGroups)))
            {
                pszGroups = szGroups;
            }
            char szReaders[256] = {0};
            unsigned long ulLen = sizeof(szReaders);
            if(SCARD_S_SUCCESS == (lRet = SCardListReaders(m_hContext, pszGroups, szReaders, &ulLen)))
            {
                pMessage->AddValueForKey("Readers", (unsigned char *)szReaders, ulLen);
                pMessage->AddValueForKey("ReadersLen", ulLen);                
            }
        }
    }
    return lRet;
}

long CPCSCManager::S_CardGetStatusChange(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        SCARDCONTEXT hContext = 0;
        pMessage->GetValueForKey ("Context", (unsigned long *)&hContext);
        if(NULL != pApplicationManager->FindApplication(hContext))
        {
            unsigned long ulTimeout = 0;
            unsigned long ulReaders = 0;
            pMessage->GetValueForKey("Timeout", &ulTimeout);
            pMessage->GetValueForKey("ReadersLen", &ulReaders);
            SCARD_READERSTATE_A *prgReaderStates = new SCARD_READERSTATE_A[ulReaders];
            memset(prgReaderStates, 0, sizeof(SCARD_READERSTATE_A) * ulReaders);
            char szReaders[MAXIMUM_SMARTCARD_READERS][64] = {0};
            for(unsigned int i = 0; i < ulReaders; ++i)
            {
                char szReader[16] = {0};
                char szState[16] = {0};
                sprintf(szReader, "Reader%d", i); 
                sprintf(szState, "CurrentState%d", i);                
                pMessage->GetValueForKey(szReader, szReaders[i], 64);
                prgReaderStates[i].szReader = szReaders[i];
                pMessage->GetValueForKey(szState, &(prgReaderStates[i].dwCurrentState));
            }
            if(SCARD_S_SUCCESS == (lRet = SCardGetStatusChange(m_hContext, 0, prgReaderStates, ulReaders)))
            {
                for(unsigned int i = 0; i < ulReaders; ++i)
                {
                    char szEventstate[16] = {0};
                    char szAtrLen[16] = {0};
                    char szAtr[16] = {0};
                    sprintf(szEventstate, "EventState%d", i); 
                    sprintf(szAtrLen, "AtrLen%d", i); 
                    sprintf(szAtr, "Atr%d", i); 
                    pMessage->AddValueForKey (szEventstate, prgReaderStates[i].dwEventState);  
                    pMessage->AddValueForKey (szAtrLen, prgReaderStates[i].cbAtr);  
                    pMessage->AddValueForKey (szAtr, prgReaderStates[i].rgbAtr, prgReaderStates[i].cbAtr);  
                }
            }
            if(prgReaderStates != NULL)
            {
                delete prgReaderStates;
            }
        }
    }
    return lRet;
}

long CPCSCManager::S_CardBeginTransaction(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        CApplicationObject *pApplication = NULL;
        SCARDHANDLE hCard = 0;
        unsigned long lConnID = 0;
        pMessage->GetValueForKey ("Card", (unsigned long *)&hCard);
        pMessage->GetValueForKey("ConnID", &lConnID);
        if(NULL != (pApplication = pApplicationManager->FindExisting(lConnID)))
        {        
            // Card in reader
            CReaderObject *pReader = pApplication->GetReader(hCard);
            if(pReader != NULL && m_pTransActionStack != NULL)
            {
                m_pTransActionStack->AddHandle(hCard);
                int iAvail = 0;
                while (0 == (iAvail = m_pTransActionStack->IsHandleAvailable(hCard)) && pApplication->MatchOriginal(hCard))
                {
                    PASS_MILLISLEEP(1000);
                }
                SCARDHANDLE hReal = pApplication->GetOriginal(hCard);
                if(iAvail > 0)
                {
                    lRet = SCardBeginTransaction(hReal);
                }
                else
                {
                    m_pTransActionStack->DeleteAllHandle(hCard); 
                }
            }
        }
    }
    return lRet;
}


long CPCSCManager::S_CardEndTransaction(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        CApplicationObject *pApplication = NULL;
        SCARDHANDLE hCard = 0;
        unsigned long lConnID = 0;
        pMessage->GetValueForKey ("Card", (unsigned long *)&hCard);
        pMessage->GetValueForKey("ConnID", &lConnID);
        if(NULL != (pApplication = pApplicationManager->FindExisting(lConnID)))
        {        
            // Card in reader
            CReaderObject *pReader = pApplication->GetReader(hCard);
            if(pReader != NULL && m_pTransActionStack != NULL)
            {
                int iAvail = 0;
                while (0 == (iAvail = m_pTransActionStack->IsHandleAvailable(hCard)) && pApplication->MatchOriginal(hCard))
                {
                    PASS_MILLISLEEP(1000);
                }            

                SCARDHANDLE hReal = pApplication->GetOriginal(hCard);
                if(iAvail > 0)
                {
                    lRet = SCardEndTransaction(hReal, SCARD_LEAVE_CARD);
                    m_pTransActionStack->DeleteHandle(hCard);
                }
                else
                {
                    m_pTransActionStack->DeleteAllHandle(hCard); 
                }
            }
        }
    }
    return lRet;
}

long CPCSCManager::S_CardTransmit(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        CApplicationObject *pApplication = NULL;
        SCARDHANDLE hCard = 0;
        unsigned long lConnID = 0;
        pMessage->GetValueForKey ("Card", (unsigned long *)&hCard);
        pMessage->GetValueForKey("ConnID", &lConnID);
        if(NULL != (pApplication = pApplicationManager->FindExisting(lConnID)))
        {        
            // Card in reader
            CReaderObject *pReader = pApplication->GetReader(hCard);
            if(pReader != NULL && m_pTransActionStack != NULL)
            {
                if(m_pTransActionStack->IsHandleAvailable(hCard) > 0)
                {
                    SCARDHANDLE hReal = pApplication->GetOriginal(hCard);
                    SCARD_IO_REQUEST ioSend = {0};
                    SCARD_IO_REQUEST ioRecv = {0};
                    unsigned long ulSendLen = 0;
                    unsigned long ulRecvLen = 0;
                    pMessage->GetValueForKey("SendBufferLen", &ulSendLen);
                    BYTE *pSendBuffer = new BYTE[ulSendLen];
                    memset(pSendBuffer, 0, ulSendLen);
                    pMessage->GetValueForKey("SendBuffer", (BYTE *)pSendBuffer, ulSendLen);
                    pMessage->GetValueForKey("SendProtocol", &ioSend.dwProtocol);
                    pMessage->GetValueForKey("SendProtocolLen", &ioSend.cbPciLength);
                    SCARD_IO_REQUEST *pioRecv = NULL;
                    if(pMessage->GetValueForKey("RecvProtocol", &ioRecv.dwProtocol))
                    {
                        pMessage->GetValueForKey("RecvProtocolLen", &ioRecv.cbPciLength);
                        pioRecv = &ioRecv;
                    }

                    if(!pMessage->GetValueForKey("RecvLen", &ulRecvLen))
                    {
                        ulRecvLen = 256;
                    }
                    BYTE *pRecvBuffer = new BYTE[ulRecvLen];
                    memset(pRecvBuffer, 0, ulRecvLen);
                    if(SCARD_S_SUCCESS == (lRet = SCardTransmit(hReal, &ioSend, pSendBuffer, ulSendLen, pioRecv, pRecvBuffer, &ulRecvLen)))
                    {
                        if(pioRecv != NULL)
                        {
                            pMessage->AddValueForKey("RecvProtocol", ioRecv.dwProtocol);
                            pMessage->AddValueForKey("RecvProtocolLen", ioRecv.cbPciLength);
                        }
                        pMessage->AddValueForKey ("RecvLen", ulRecvLen);
                        pMessage->AddValueForKey ("RecvBuffer", (BYTE *)pRecvBuffer, ulRecvLen);
                    }
                    if(pSendBuffer != NULL)
                    {
                        delete pSendBuffer;
                    }
                    if(pRecvBuffer != NULL)
                    {
                        delete pRecvBuffer;
                    }
                }
            }
        }
    }
    return lRet;
}

long CPCSCManager::S_CardControl(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        CApplicationObject *pApplication = NULL;
        SCARDHANDLE hCard = 0;
        unsigned long lConnID = 0;
        pMessage->GetValueForKey ("Card", (unsigned long *)&hCard);
        pMessage->GetValueForKey("ConnID", &lConnID);
        if(NULL != (pApplication = pApplicationManager->FindExisting(lConnID)))
        {        
            // Card in reader
            CReaderObject *pReader = pApplication->GetReader(hCard);
            if(pReader != NULL && m_pTransActionStack != NULL)
            {
                if(m_pTransActionStack->IsHandleAvailable(hCard) > 0)
                {
                    SCARDHANDLE hReal = pApplication->GetOriginal(hCard);
                    BYTE *pSendBuffer = NULL;
                    unsigned long ulSendLen = 0;
                    unsigned long ulRecvLen = 0;
                    unsigned long ulControlCode = 0;
                    pMessage->GetValueForKey("ControlCode", &ulControlCode);
                    if(pMessage->GetValueForKey("SendBufferLen", &ulSendLen))
                    {
                        pSendBuffer = new BYTE[ulSendLen];
                        memset(pSendBuffer, 0, ulSendLen);
                        pMessage->GetValueForKey("SendBuffer", (BYTE *)pSendBuffer, ulSendLen);
                    }

                    if(!pMessage->GetValueForKey("RecvLen", &ulRecvLen))
                    {
                        ulRecvLen = 256;
                    }
                    BYTE *pRecvBuffer = new BYTE[ulRecvLen];
                    memset(pRecvBuffer, 0, ulRecvLen);
                    unsigned long ulBytesRet = 0;
//#ifdef _WIN32
// The API of this function changed. 
// In pcsc-lite 1.2.0 and before the API was not Windows PC/SC compatible. 
// This has been corrected. 
                    if(SCARD_S_SUCCESS == (lRet = SCardControl(hReal, ulControlCode, pSendBuffer, ulSendLen, pRecvBuffer, ulRecvLen, &ulBytesRet)))
/*
#else
                    ulBytesRet = ulRecvLen;
                    if(SCARD_S_SUCCESS == (lRet = SCardControl(hReal, pSendBuffer, ulSendLen, pRecvBuffer, &ulBytesRet)))
#endif
*/
                    {
                        pMessage->AddValueForKey ("RecvLen", ulBytesRet);
                        pMessage->AddValueForKey ("RecvBuffer", (BYTE *)pRecvBuffer, ulBytesRet);
                    }
                    if(pSendBuffer != NULL)
                    {
                        delete pSendBuffer;
                    }
                    if(pRecvBuffer != NULL)
                    {
                        delete pRecvBuffer;
                    }
                }
            }
        }
    }
    return lRet;
}


long CPCSCManager::S_CardControlPinPad(CGenericMessage *pMessage)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
#ifdef BELPIC_PIN_PAD
    CMasterManager *pMaster = CMasterManager::Instance();
	CApplicationManager *pApplicationManager = (CApplicationManager *)pMaster->GetManager ("Application");

    if(m_hContext != 0 && pApplicationManager != NULL)
    {
        CApplicationObject *pApplication = NULL;
        SCARDHANDLE hCard = 0;
        unsigned long lConnID = 0;
        pMessage->GetValueForKey ("Card", (unsigned long *)&hCard);
        pMessage->GetValueForKey("ConnID", &lConnID);
        if(NULL != (pApplication = pApplicationManager->FindExisting(lConnID)))
        {        
            // Card in reader
            CReaderObject *pReader = pApplication->GetReader(hCard);
            if(SCR_CARD_HANDLE == hCard || (pReader != NULL && m_pTransActionStack != NULL))
            {
                if(SCR_CARD_HANDLE == hCard || m_pTransActionStack->IsHandleAvailable(hCard) > 0)
                {
                    SCARDHANDLE hReal = pApplication->GetOriginal(hCard);
                    BYTE *pSendBuffer = NULL;
                    unsigned long ulSendLen = 0;
                    unsigned long ulRecvLen = 0;
                    unsigned long ulControlCode = 0;
                    pMessage->GetValueForKey("ControlCode", &ulControlCode);
                    if(pMessage->GetValueForKey("SendBufferLen", &ulSendLen))
                    {
                        pSendBuffer = new BYTE[ulSendLen];
                        memset(pSendBuffer, 0, ulSendLen);
                        pMessage->GetValueForKey("SendBuffer", (BYTE *)pSendBuffer, ulSendLen);
                    }

                    if(!pMessage->GetValueForKey("RecvLen", &ulRecvLen))
                    {
                        ulRecvLen = 256;
                    }
                    BYTE *pRecvBuffer = new BYTE[ulRecvLen];
                    memset(pRecvBuffer, 0, ulRecvLen);
                    unsigned long ulBytesRet = 0;
                    lRet = HandlePinPad(pSendBuffer, ulSendLen, pRecvBuffer, &ulBytesRet, hReal);
                    pMessage->AddValueForKey ("RecvLen", ulBytesRet);
                    if(ulBytesRet > 0)
                    {
                        pMessage->AddValueForKey ("RecvBuffer", (BYTE *)pRecvBuffer, ulBytesRet);
                    }
                    if(pSendBuffer != NULL)
                    {
                        delete pSendBuffer;
                    }
                    if(pRecvBuffer != NULL)
                    {
                        delete pRecvBuffer;
                    }
                }
            }
        }
    }
#endif // BELPIC_PIN_PAD
    return lRet;
}

void CPCSCManager::EventCardState(unsigned long ulState, const char *pszReaderName)
{
    SCARDHANDLE hCard = 0;
    unsigned long lReturn;
    unsigned long ulAP;
    CReaderObject *pReader = (CReaderObject *)m_Readers.Lookup((char *)pszReaderName);
    if(pReader == NULL)
    {
        pReader = new CReaderObject();
        m_Readers.Add((char *)pszReaderName, pReader);
        pReader->SetName((char *)pszReaderName); 
    }
    if(ulState == SCARD_STATE_PRESENT && pReader->GetCardHandle() == 0)
    {
        // Card Inserted -> Connect exclusive
        lReturn = SCardConnect(m_hContext, pszReaderName, SCARD_SHARE_EXCLUSIVE, SCARD_PROTOCOL_T0 | SCARD_PROTOCOL_T1 , &hCard, &ulAP);
        if(lReturn == SCARD_E_SHARING_VIOLATION)
        {
            // Try again Shared access
            lReturn = SCardConnect(m_hContext, pszReaderName, SCARD_SHARE_SHARED, SCARD_PROTOCOL_T0 | SCARD_PROTOCOL_T1 , &hCard, &ulAP);
        }
        if(lReturn == SCARD_S_SUCCESS)
        {
          /*  BYTE szBuffer[256] = {0};
            DWORD dwLen = sizeof(szBuffer);
            if(SCARD_S_SUCCESS == SCardGetAttrib(hCard, SCARD_ATTR_ATR_STRING, szBuffer, &dwLen))
            {
            }*/
            pReader->SetCardHandle(hCard);
            pReader->SetProtocol(ulAP); 
        }

    }
    else if(ulState == SCARD_STATE_EMPTY && pReader->GetCardHandle() > 0)
    {
        // Card pulled out -> Reinit everything
#ifndef _WIN32
            SCardDisconnect(pReader->GetCardHandle (), SCARD_LEAVE_CARD);
#endif
        pReader->SetCardHandle(hCard);
        pReader->SetProtocol(0);
    }
}

#ifdef BELPIC_PIN_PAD

long CPCSCManager::HandlePinPad(const void *lpInBuffer, DWORD nInBufferSize, void *lpOutBuffer, DWORD *lpBytesReturned, SCARDHANDLE hReal)
{
    long lRet = SCARD_F_INTERNAL_ERROR;
    *lpBytesReturned = 0;

    // Parse TLV
    CTLVBuffer oTLVBuffer;
    oTLVBuffer.ParseTLV((unsigned char *)lpInBuffer, nInBufferSize);

    long lOp = 0;
    char szPinPad[256] = {0};
    oTLVBuffer.FillLongData(0x01, &lOp); 
    oTLVBuffer.FillASCIIData(0x02, szPinPad);

    BOOL bFound = FALSE;
    int i = 0;
    int iIndex = -1;
    for(i = 0; i < PINPADNUM; i++)
    {
        if(0 == strcmp(m_pPinPadDlls[i].szPinPadDll, szPinPad))
        {
            bFound = TRUE;
            break;
        }
        if(m_pPinPadDlls[i].pDll == NULL && iIndex == -1)
        {
            iIndex = i;
            break;
        }
    }
    wxDynamicLibrary *pPinPadLoader = NULL;
    if (bFound)
    {
        pPinPadLoader = m_pPinPadDlls[i].pDll;
    }
    else
    {
        pPinPadLoader = new wxDynamicLibrary(szPinPad);
        m_pPinPadDlls[iIndex].pDll = pPinPadLoader;
        strcpy(m_pPinPadDlls[iIndex].szPinPadDll, szPinPad);
    }

    if(pPinPadLoader && pPinPadLoader->GetLibHandle() > 0)
    {
        if(SCR_INIT_ID == lOp)
        {
            char szReader[256] = {0};
            long lVersion = 0;
            oTLVBuffer.FillASCIIData(0x03, szReader);
            oTLVBuffer.FillLongData(0x04, &lVersion);
            SCR_SupportConstants supported;
            SCR_INIT_PROC scr_init = (SCR_INIT_PROC)pPinPadLoader->GetSymbol("SCR_Init");
            if(scr_init != NULL)
            {
                lRet = scr_init(szReader, lVersion, &supported);
                ((BYTE *)lpOutBuffer)[0] = supported;
                *lpBytesReturned = 1;
            }
        }
        else if(SCR_VERIFY_ID == lOp || SCR_CHANGE_ID == lOp)
        {
            unsigned long ulLen = 0;
            SCR_Card card = {0};
            SCR_PinUsage usage = {0};
            SCR_Application application = {0};
            BYTE pinID = 0;
            BYTE *pCardData = NULL;
            char *pszLanguage = NULL;
            char *pszShortUsage = NULL;
            char *pszLongUsage = NULL;
            BYTE *pAppData = NULL;
            char *pszShortApp = NULL;
            char *pszLongApp = NULL;
            BYTE ucRet[2] = {0};

            oTLVBuffer.FillLongData(0x03, (long *)&card.hCard);
            // We need real SCARD HANDLE here
            card.hCard = hReal;
            pszLanguage = new char[256];
            memset(pszLanguage, 0, 256);
            oTLVBuffer.FillASCIIData(0x04, pszLanguage);
            card.language = pszLanguage;
            if(oTLVBuffer.GetTagData(0x05))
            {
                pCardData = new BYTE[256];
                memset(pCardData, 0, 256);
                oTLVBuffer.FillBinaryData(0x05, pCardData, &ulLen);  
                card.id.data = pCardData;
                card.id.length = ulLen;
            }
            oTLVBuffer.FillBinaryData(0x06, &pinID, &ulLen);  
            if(oTLVBuffer.GetTagData(0x07))
            {
                oTLVBuffer.FillLongData(0x07, &usage.code);
                if(oTLVBuffer.GetTagData(0x08))
                {
                    pszShortUsage = new char[256];
                    memset(pszShortUsage, 0, 256);
                    oTLVBuffer.FillASCIIData(0x08, pszShortUsage);
                    usage.shortString = pszShortUsage;
                }
                if(oTLVBuffer.GetTagData(0x09))
                {
                    pszLongUsage = new char[256];
                    memset(pszLongUsage, 0, 256);
                    oTLVBuffer.FillASCIIData(0x09, pszLongUsage);
                    usage.longString = pszLongUsage;
                }
            }
            if(oTLVBuffer.GetTagData(0x0A))
            {
                pAppData = new BYTE[256];
                memset(pAppData, 0, 256);
                oTLVBuffer.FillBinaryData(0x0A, pAppData, &ulLen);  
                application.id.data = pAppData;
                application.id.length = ulLen;
            }
            if(oTLVBuffer.GetTagData(0x0B))
            {
                pszShortApp = new char[256];
                memset(pszShortApp, 0, 256);
                oTLVBuffer.FillASCIIData(0x0B, pszShortApp);
                application.shortString = pszShortApp;

            }
            if(oTLVBuffer.GetTagData(0x0C))
            {
                pszLongApp = new char[256];
                memset(pszLongApp, 0, 256);
                oTLVBuffer.FillASCIIData(0x0C, pszLongApp);
                application.longString = pszLongApp;
            }
            if(SCR_VERIFY_ID == lOp)
            {
                SCR_VERIFYPIN_PROC scr_verify = (SCR_VERIFYPIN_PROC)pPinPadLoader->GetSymbol("SCR_VerifyPIN");
                if(scr_verify != NULL)
                {
                    lRet = scr_verify(&card, pinID, &usage, &application, ucRet);
                    memcpy(lpOutBuffer, ucRet, 2);
                    *lpBytesReturned = 2;
                }
            }
            else if (SCR_CHANGE_ID == lOp)
            {
                SCR_CHANGEPIN_PROC scr_change = (SCR_CHANGEPIN_PROC)pPinPadLoader->GetSymbol("SCR_ChangePIN");
                if(scr_change != NULL)
                {
                    lRet = scr_change(&card, pinID, &application, ucRet);
                    memcpy(lpOutBuffer, ucRet, 2);
                    *lpBytesReturned = 2;
                }
            }
            if(pCardData)
                delete [] pCardData;
            if(pszLanguage)
                delete [] pszLanguage;
            if(pszShortUsage)
                delete [] pszShortUsage;
            if(pszLongUsage)
                delete [] pszLongUsage;
            if(pAppData)
                delete [] pAppData;
            if(pszShortApp)
                delete [] pszShortApp;
            if(pszLongApp)
                delete [] pszLongApp;
        }
    }
    return lRet;
}

#endif // BELPIC_PIN_PAD

BOOL CPCSCManager::Suspend()
{
   	ULONG32 position = 0; 
    char szKey[256] = {0}; 
    CReaderObject *pObject = (CReaderObject *)m_Readers.GetFirst (&position, szKey, sizeof(szKey));

    while (pObject != NULL) 
    {   
        if(pObject->GetCardHandle () > 0)
        {
            // Disconnect Card
            SCardDisconnect(pObject->GetCardHandle (), SCARD_LEAVE_CARD);
            pObject->SetCardHandle(0);
            pObject->SetProtocol(0);
        }
        pObject = (CReaderObject *)m_Readers.GetNext (&position, szKey, sizeof(szKey)); 
    }
    return TRUE;
}

/*BOOL CPCSCManager::CheckAccess(CApplicationObject* pApplication, BYTE *pSendBuffer, unsigned long ulSendLen)
{
    BOOL bRet = TRUE;

    if(ulSendLen > 1 && 0 == memcmp(pSendBuffer, ucSelect, 2) && pSendBuffer[4] > 0x05)
    {
        if(0 == memcmp(pSendBuffer + 5, ucFile, 5))
        {
            int iFile = pSendBuffer[10];
            switch(iFile)
            {
            case ucID:
            ::MessageBox(NULL, "ID read", "Hallo", MB_OK); 
                    break;
            case ucAddress:
            ::MessageBox(NULL, "Address read", "Hallo", MB_OK); 
                break;
            case ucPhoto:
            ::MessageBox(NULL, "Photo read", "Hallo", MB_OK); 
                break;
            }
        }
    }
    return  bRet;
}
*/
