// CRLScheduler.cpp: implementation of the CCRLScheduler class.
//
//////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "CRLScheduler.h"
#include "Bytearray.h"
#include "Config.h"
#include <openssl/ocsp.h>

// For compilers that support precompilation
#include "wx/wxprec.h"

#ifndef WX_PRECOMP
    #include <wx/wx.h>
#endif

#include "wx/filename.h"

#ifndef _WIN32
#define strnicmp strncasecmp
#define stricmp strcasecmp
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CCRLScheduler::CCRLScheduler(): CThread ("CRLScheduler")
{
	SetTickTime (300000);
}

CCRLScheduler::~CCRLScheduler()
{

}

void CCRLScheduler::Tick (void)
{	
    char *pszHttpCRLStore = (char *)CConfig::GetHttpCRLStorePath();
    if(pszHttpCRLStore != NULL)
    {
        BuildCRLList();
        CheckCRLList(pszHttpCRLStore);
    //void *pvData = RetrieveWebdata(pszHttpCRLStore, &iLen);
        //if(pvData != NULL)
        //{
            // Build CRL list
            //BuildCRLList((char *)pvData, iLen);
            //CheckCRLList(pszHttpCRLStore);
            //free(pvData);
        //}
    }
}


BIO *CCRLScheduler::Connect(char *pszHost, int iPort, int iSSL, SSL_CTX **ppSSLCtx) 
{
    BIO *pConn = NULL;
   
    if (iSSL) 
    {
        if (!(pConn = ConnectSSL(pszHost, iPort, ppSSLCtx))) goto error_exit;
        //SSL *pSSL = NULL;
        //BIO_get_ssl(pConn, &pSSL);
        //if (pSSL && !VerifyCertHostname(SSL_get_peer_certificate(pSSL), pszHost)) goto error_exit;
        //if (SSL_get_verify_result(pSSL) != X509_V_OK) goto error_exit;
        return pConn;
    }

    *ppSSLCtx = 0;
    if (!(pConn = BIO_new_connect(pszHost))) goto error_exit;
    BIO_set_conn_int_port(pConn, &iPort);
    if (BIO_do_connect(pConn) <= 0) goto error_exit;
    return pConn;
   
error_exit:
    if (pConn) BIO_free_all(pConn);
    return 0;
}

BIO *CCRLScheduler::ConnectSSL(char *pszHost, int iPort, SSL_CTX **ppSSLCtx) 
{
    BIO *pConn = NULL;
    OpenSSL_add_all_algorithms();
    *ppSSLCtx = SSL_CTX_new(SSLv23_client_method());
    
    if (!(pConn = BIO_new_ssl_connect(*ppSSLCtx))) goto error_exit;
    BIO_set_conn_hostname(pConn, pszHost);
    BIO_set_conn_int_port(pConn, &iPort);
   
    if (BIO_do_connect(pConn) <= 0) goto error_exit;  
    return pConn;
   
error_exit:
  if (pConn) BIO_free_all(pConn);
  if (*ppSSLCtx) 
  {
      SSL_CTX_free(*ppSSLCtx);
      *ppSSLCtx = NULL;
  }
  return 0;
}


void *CCRLScheduler::RetrieveWebdata(char *pszUri, int *piDataLen) 
{
    int iBytes = 0;
    int iContentLength = 0;
    int iHeaderLen = 0;
    int iSd = 0;
    int iSSL = 0;
    BIO *pConn = 0;
    SSL *pSSL = 0;
    SSL_CTX *pSSLCtx = 0;
    char szBuffer[1024] = {0};
    char *pszHeaders = 0;
    char *pszHost = 0;
    char *pszPath = 0;
    char *pszPort = 0;
    char *pszTmp = 0;
    void  *pvData = 0;
    fd_set  rmask, wmask;
    const char *pszConfigHost = CConfig::GetProxyHost();
    const char *pszConfigPort = CConfig::GetProxyPort() ;
   
    *piDataLen = 0;

    if(pszConfigHost != NULL && strlen(pszConfigHost) > 0)
    {
        pszHost = (char *)OPENSSL_malloc(strlen(pszConfigHost) + 1);
        memset(pszHost, 0, strlen(pszConfigHost) + 1);
        strcpy(pszHost, pszConfigHost); 
    }
    if(pszConfigPort != NULL && strlen(pszConfigPort) > 0)
    {
        pszPort = (char *)OPENSSL_malloc(strlen(pszConfigPort) + 1);
        memset(pszPort, 0, strlen(pszConfigPort) + 1);
        strcpy(pszPort, pszConfigPort); 
    }

     if(pszHost == NULL || pszPort == NULL)
    {
        if (!OCSP_parse_url(pszUri, &pszHost, &pszPort, &pszPath, &iSSL)) goto end_error;
     }
     else
     {
        pszPath = (char *)OPENSSL_malloc(strlen(pszUri) + 1);
        memset(pszPath, 0, strlen(pszUri) + 1);
        strcpy(pszPath, pszUri); 
     }

    if (!(pConn = Connect(pszHost, atoi(pszPort), iSSL, &pSSLCtx))) goto end_error;
   
    // Send the request for the data
    BIO_printf(pConn, "GET %s HTTP/1.0\r\nConnection: close\r\n\r\n", pszPath);
   
    // Put the socket into non-blocking mode
    BIO_get_fd(pConn, &iSd);
    BIO_socket_nbio(iSd, 1);
    if (iSSL) 
    {
        BIO_get_ssl(pConn, &pSSL);
        SSL_set_mode(pSSL, SSL_MODE_ENABLE_PARTIAL_WRITE | SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER);
    }
   
    // Loop reading data from the socket until we've got all of the headers
    for (;;) 
    {
        FD_ZERO(&rmask);
        FD_SET(iSd, &rmask);
        FD_ZERO(&wmask);
        if (BIO_should_write(pConn)) 
            FD_SET(iSd, &wmask);
        if (select(FD_SETSIZE, &rmask, &wmask, 0, 0) <= 0) continue;
        if (FD_ISSET(iSd, &wmask)) BIO_write(pConn, szBuffer, 0);
        if (FD_ISSET(iSd, &rmask)) 
        {
            if ((iBytes = BIO_read(pConn, szBuffer, sizeof(szBuffer))) <= 0) 
            {
                if (BIO_should_retry(pConn)) continue;
                goto end_error;
            }
            if (!(pszHeaders = (char *)realloc((pszTmp = pszHeaders), iHeaderLen + iBytes))) 
            {
                pszHeaders = pszTmp;
                goto end_error;
            }
            memcpy(pszHeaders + iHeaderLen, szBuffer, iBytes);
            iHeaderLen += iBytes;
            if ((pszTmp = strstr(pszHeaders, "\r\n\r\n")) != 0) 
            {
                *(pszTmp + 2) = '\0';
                *piDataLen = iHeaderLen - ((pszTmp + 4) - pszHeaders);
                iHeaderLen -= (*piDataLen + 2);
                if (*piDataLen > 0) 
                {
                    if (!(pvData = (char *)malloc(*piDataLen))) goto end_error;
                    memcpy(pvData, pszTmp + 4, *piDataLen);
                }
                break;
            }
        }
    }
  
    /* Examine the headers to determine whether or not to continue.  If we are to
    * continue, look for a content-length header to find out how much data we're
    * going to get.  If there is no content-length header, we'll have to read
    * until the remote server closes the connection.
    */
    if (0 == strnicmp(pszHeaders, "HTTP/1.", 7)) 
    {
        if (!(pszTmp = strchr(pszHeaders, ' '))) goto end_error;
        if (strncmp(pszTmp + 1, "200 ", 4) && strncmp(pszTmp + 1, "200\r\n", 5))
            goto end_error;
        for (pszTmp = strstr(pszHeaders, "\r\n");  pszTmp;  pszTmp = strstr(pszTmp + 2, "\r\n")) 
        {
            if (strnicmp(pszTmp + 2, "content-length: ", 16)) continue;
            iContentLength = atoi(pszTmp + 18);
            break;
        }
    } 
    else 
        goto end_error;
   
    /* Continuously read and accumulate data from the remote server.  Finish when
    * we've read up to the content-length that we received.  If we didn't receive
    * a content-length, read until the remote server closes the connection.
    */

    while (!iContentLength || *piDataLen < iContentLength) 
    {
        FD_ZERO(&rmask);
        FD_SET(iSd, &rmask);
        FD_ZERO(&wmask);
        if (BIO_should_write(pConn)) FD_SET(iSd, &wmask);
        if (select(FD_SETSIZE, &rmask, &wmask, 0, 0) <= 0) continue;
        if (FD_ISSET(iSd, &wmask)) BIO_write(pConn, szBuffer, 0);
        if (FD_ISSET(iSd, &rmask))
        if ((iBytes = BIO_read(pConn, szBuffer, sizeof(szBuffer))) <= 0) 
        {
            if (BIO_should_retry(pConn)) continue;
            break;
        }
        if (!(pvData = realloc((pszTmp = (char *)pvData), *piDataLen + iBytes))) 
        {
            pvData = pszTmp;
            goto end_error;
        }

        memcpy((char *)pvData + *piDataLen, szBuffer, iBytes);
        *piDataLen += iBytes;
    }
   
    if (iContentLength && *piDataLen != iContentLength) goto end_error;
        goto end;
   
end_error:
  if (pvData) 
  { 
      free(pvData);  
      pvData = 0;  
      *piDataLen = 0; 
  }
end:
  if (pszHeaders) free(pszHeaders);
  if (pConn) BIO_free_all(pConn);
  if (pszHost) OPENSSL_free(pszHost);
  if (pszPort) OPENSSL_free(pszPort);
  if (pszPath) OPENSSL_free(pszPath);
  if (pSSLCtx) SSL_CTX_free(pSSLCtx);
  return pvData;
}

void CCRLScheduler::BuildCRLList()
{
    m_oVecCRLList.clear();
    const char *pszCRLStorePath = CConfig::GetCRLStorePath();
    wxString fFile = wxFindFirstFile(wxString::Format("%s*.crl", pszCRLStorePath));
    while ( !fFile.IsEmpty() )
    {
        wxFileName fName(fFile);
        m_oVecCRLList.push_back(fName.GetFullName().GetData());
        fFile = wxFindNextFile();
    }

/*
    const char *pszSearch = "<A HREF=\"";
    int iSearchLen = strlen(pszSearch);
    char *pszDest = NULL;
    char *pszTemp = pszData;
    while (pszDest = strstr(pszTemp, pszSearch))
    {
        int iPos = strcspn(pszDest + iSearchLen, "\">");
        if(iPos > 0)
        {
            char szFile[256] = {0};
            strncpy(szFile, pszDest + iSearchLen, iPos);
            if(strstr(szFile, ".crl") != NULL)
            {
                m_oVecCRLList.push_back(szFile);
            }
        }
        pszTemp = pszDest + iSearchLen;
        pszDest = NULL;
    }
*/
}

void CCRLScheduler::CheckCRLList(char *pszHttpCRLStore)
{
    for(unsigned int i = 0; i < m_oVecCRLList.size(); ++i)
    {
        if(IsRunning())
        {
            FindLocalCRL(pszHttpCRLStore, (char *)m_oVecCRLList[i].c_str());
        }
    }
}

void CCRLScheduler::FindLocalCRL(char *pszHttpCRLStore, char *pCRLDistribution)
{
    X509_CRL *pX509CRL = NULL;
    bool bDownload = false;
    const char *pszCRLStorePath = CConfig::GetCRLStorePath();

    if(pCRLDistribution != NULL && pszCRLStorePath != NULL)
    {
        char *pTemp; 
        char *pFile = pCRLDistribution;
        while(NULL != (pTemp = strstr(pFile, "/")))
        {
            pFile = ++pTemp;
        }

        FILE *pf = NULL;
        char szFile[256] = {0};
        sprintf(szFile, "%s%s", pszCRLStorePath, pFile);
        pf = fopen(szFile, "r+b");
        if(NULL != pf)
        {
            unsigned char szBuffer[4096] = {0};
            CByteArray oByteArray;
            int numread = 0;
            while( !feof(pf) )
            {                    
                numread = fread(szBuffer, sizeof(unsigned char), sizeof(szBuffer), pf);
                if(numread > 0)
                {
                    oByteArray.Append(szBuffer, numread); 
                    numread = 0;
                }
            }
            if(oByteArray.GetSize() > 0)
            {
                unsigned char *pDummy = oByteArray.GetData();
                pX509CRL = d2i_X509_CRL(&pX509CRL, (unsigned char **)&pDummy, oByteArray.GetSize());
                if(pX509CRL != NULL)
                {
                    // Check period
                    int iCheck = 0;
	                iCheck = X509_cmp_time(X509_CRL_get_lastUpdate(pX509CRL), NULL);
	                if (iCheck >= 0)
		            {
                        // X509_V_ERR_CRL_NOT_YET_VALID or X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD error
                        bDownload = true;
		            }

	                if(X509_CRL_get_nextUpdate(pX509CRL))
		            {
		                iCheck = X509_cmp_time(X509_CRL_get_nextUpdate(pX509CRL), NULL);
		                if (iCheck <= 0)
			            {
                            // X509_V_ERR_CRL_HAS_EXPIRED or X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD error
                            bDownload = true;
			             }
		            }
                }
                else
                {
                    bDownload = true;
                }
            }
            fclose(pf);
        }
        else
        {
            bDownload = true;
        }
        if(pX509CRL != NULL)
        {
            X509_CRL_free(pX509CRL);
        }
        if(bDownload)
        {
            // Try to download CRL
            char szUri[256] = {0};
            strcpy(szUri, pszHttpCRLStore);
            strcat(szUri, pCRLDistribution);
            RetrieveCRL(szUri, szFile); 
        }
    }
}

void CCRLScheduler::RetrieveCRL(char *pszUri, char *pszFile) 
{
    int  iDataLen = 0;
    void  *pvData = 0;
    FILE *pf = 0;
    
    if (!(pvData = RetrieveWebdata(pszUri, &iDataLen))) 
    {
        goto end;
    }
   
    // Store on disk
    pf = fopen(pszFile, "w+b");
    if(NULL != pf)
    {
        int iNumWritten = 0;
        iNumWritten = fwrite(pvData, sizeof(unsigned char), iDataLen, pf);
        fclose(pf);
    }

end:
    if (pvData) free(pvData);
}

int CCRLScheduler::VerifyCertHostname(X509 *pCert, char *pszHostname)
{
    struct stack_st *pStack = NULL;
    BOOL bFound = FALSE;

    pStack = (struct stack_st *) X509_get_ext_d2i(pCert, NID_subject_alt_name, NULL, NULL);
    if(pStack != NULL)
    {
        int i, iLen1, iLen2;
	    char *pszDomain = NULL;
	    iLen1 = strlen(pszHostname);
	    pszDomain = strchr(pszHostname,  '.');
	    if (pszDomain)
        {
	    	iLen2 = iLen1 - (pszDomain - pszHostname);
        }

        for(i = 0; i < sk_GENERAL_NAME_num(pStack) && !bFound; i++) 
        {
            GENERAL_NAME *pName = sk_GENERAL_NAME_value(pStack, i);
            if(pName != NULL && pName->type == GEN_DNS )
            {
                char *pData = (char *)ASN1_STRING_data(pName->d.ia5); 
                int iLen = ASN1_STRING_length(pName->d.ia5);

                // Is this an exact match ?
                if ((iLen1 == iLen) && !strnicmp(pszHostname, pData, iLen1))
                {
                    bFound = TRUE;
                }
		        // Is this a wildcard match ?
		        else if ((*pData == '*') && pszDomain && (iLen2 == iLen - 1) && !strnicmp(pszDomain, pData + 1, iLen2))
                {
                    bFound = TRUE;
                }
            }
         }
        sk_GENERAL_NAME_free(pStack);
    }
    if(!bFound)
    {
        char pszName[256] = {0};
        X509_NAME *pxName = NULL;
        pxName = X509_get_subject_name(pCert);
        if(pxName &&  X509_NAME_get_text_by_NID(pxName, NID_commonName, pszName, sizeof(pszName)) > 0) 
        {
            pszName[sizeof(pszName) - 1] = '\0';
            if (!stricmp(pszName, pszHostname)) 
            {
                bFound = TRUE;
            } 
         }
    }

    return bFound;
}
