// ConnectionSender.cpp: implementation of the CConnectionSender class. 
// 
////////////////////////////////////////////////////////////////////// 
 
#include "../../Include/Comm/ConnectionSender.h" 
#include "../../Include/Comm/MessageStack.h" 
#include "../../Include/Comm/Message.h" 
 
#ifdef _WIN32 
#include <windows.h> 
#endif //_WIN32 
 
#ifdef linux 
#include <stddef.h> 
#include <unistd.h> 
#include <string.h> 
#include <arpa/inet.h> 
#include <stdio.h> 
#include <errno.h>
#ifndef MSG_DONTWAIT
    #define MSG_DONTWAIT 0
#endif 
#endif //linux 
         
#ifdef __SCO__ 
#include <stddef.h> 
#include <unistd.h> 
#include <string.h> 
#include <arpa/inet.h> 
#include <stdio.h> 
#include <errno.h> 
#endif // __SCO__ 
         
#ifdef __HPUX__ 
#include <stddef.h> 
#include <unistd.h> 
#include <string.h> 
#include <arpa/inet.h> 
#include <stdio.h> 
#include <errno.h> 
#endif // __HPUX__ 
         

////////////////////////////////////////////////////////////////////// 
// Construction/Destruction 
////////////////////////////////////////////////////////////////////// 
         
        int writen (int fd,const void *vptr, int n) 
        { 
        	int nleft; 
        	int nwritten; 
        	const char *ptr; 
         
        	ptr = (const char *)vptr; 
        	nleft = n; 
         
        	while (nleft > 0) 
        	{ 
#ifdef _WIN32 
        		if ((nwritten = send(fd,ptr,nleft,0)) <= 0) 
#endif // _WIN32 
#ifdef linux 
        		if ((nwritten = send(fd,ptr,nleft,MSG_DONTWAIT)) <= 0) 
#endif // linux 
#ifdef __SCO__ 
        		if ((nwritten = send(fd,ptr,nleft,0)) <= 0)  // Was MSG_DONTWAIT but changed to old form "0"
#endif // __SCO__ 
#ifdef __HPUX__ 
        		if ((nwritten = send(fd,ptr,nleft,0)) <= 0)
#endif // __HPUX__ 
        		{ 
#ifdef _WIN32 
				return -1; 
#endif // _WIN32 
#ifdef linux 
				// Get error 
				if (errno != EAGAIN) 
				{ 
        				return -1; 
				} 
				usleep (10); 
#endif // linux				 
#ifdef __SCO__ 
				// Get error 
				if (errno != EAGAIN) 
				{ 
        				return -1; 
				} 
				usleep (10); 
#endif // __SCO__				 
#ifdef __HPUX__ 
				// Get error 
				if (errno != EAGAIN) 
				{ 
        				return -1; 
				} 
				PASS_MILLISLEEP (10); 
#endif // __HPUX__				 
        		} 
			else 
			{ 
        			nleft -= nwritten; 
				printf ("byte written : %d left to write : %d\n",nwritten,nleft); 
        			ptr += nwritten; 
			} 
        	} 
        	return n; 
        } 
 
#ifdef PASS_SSL 
      int SSL_writen (SSL *sslfd,const void *vptr, int n) 
      { 
      	int nleft; 
      	int nwritten; 
      	const char *ptr; 
       
      	ptr = (const char *)vptr; 
      	nleft = n; 
       
      	while (nleft > 0) 
      	{ 
      		if ((nwritten = SSL_write(sslfd,ptr,nleft)) <= 0) 
      		{ 
      			return -1; 
      		} 
      		nleft -= nwritten; 
      		ptr += nwritten; 
      	} 
      	return n; 
      } 
#endif // PASS_SSL 
 
        CConnectionSender::CConnectionSender() 
        { 
        	m_RemoteServerIP = NULL; 
        	m_RemoteServerName = NULL; 
        	m_RemoteServerPort = 0; 
        	m_bInitOK = FALSE; 
        	m_SendStack = NULL; 
	  		sockfd = 0; 
 
#ifdef linux 
  		m_ClientSocket = 0; 
#endif // linux 
 
#ifdef PASS_SSL 
        	m_SSLctx = NULL; 
        	m_SSLssl = NULL; 
        	m_SSLmeth = NULL; 
        	m_szCertificateFile = NULL; 
	      	m_bSecuredConnection = FALSE; 
#endif //PASS_SSL 
		PASS_INITLOCK (&lock);
        } 
         
CConnectionSender::~CConnectionSender() 
{ 
#ifdef PASS_SSL 
	// Clean up secure socket 
	if (m_bSecuredConnection == TRUE) 
	{ 
		if (m_SSLssl != NULL) 
		{ 
			printf ("Try to shutdown SSL handle\n"); 
			SSL_shutdown(m_SSLssl); 
			printf ("SSL handle shutdown\n"); 
		} 
	} 
#endif //PASS_SSL 
 
	if (sockfd != 0) 
	{ 
		printf ("Socket not yet closed - force to shutdown socket\n"); 
		shutdown (sockfd,0x02); 
		printf ("Socket shutdown\n"); 
 
#ifdef _WIN32 
		closesocket (sockfd);       
#endif //_WIN32 
 
#ifdef linux 
		printf ("Try to close socket\n"); 
		close (sockfd); 
		printf ("Socket closed\n"); 
#endif //linux 
 
#ifdef __SCO__ 
		printf ("Try to close socket\n"); 
		close (sockfd); 
		printf ("Socket closed\n"); 
#endif // __SCO__ 
#ifdef __HPUX__ 
		printf ("Try to close socket\n"); 
		close (sockfd); 
		printf ("Socket closed\n"); 
#endif // __HPUX__ 
	} 
 
#ifdef PASS_SSL 
	// Clean up secure socket 
	if (m_bSecuredConnection == TRUE) 
	{ 
		printf ("Try to release SSL structure\n"); 
		if (m_SSLssl != NULL) 
		{ 
			SSL_free (m_SSLssl); 
		} 
		if (m_SSLctx != NULL) 
		{ 
			SSL_CTX_free (m_SSLctx); 
		} 
		printf ("SSL structures relaesed\n"); 
	} 
#endif //PASS_SSL 
 
	if (m_RemoteServerIP != NULL) 
	{ 
		delete [] m_RemoteServerIP; 
	} 
	if (m_RemoteServerName != NULL) 
	{ 
		delete [] m_RemoteServerName; 
	} 

	PASS_DESTROYLOCK (&lock); 
} 
         
        BOOL CConnectionSender::Init (void) 
        { 
         
        	if (m_bInitOK == FALSE) 
	       	{ 
#ifdef _WIN32 
 			int err; 
        		// Initialise WinSock 
         
	 		WORD wVersionRequested;WSADATA wsaData; 
        		wVersionRequested = MAKEWORD( 2, 2 );  
        		err = WSAStartup( wVersionRequested, &wsaData ); 
        		if ( err != 0 )  
        		{ 
        			/* Tell the user that we could not find a usable */ 
        			/* WinSock DLL.*/ 
        			m_bInitOK = FALSE; 
        			return FALSE; 
        		} 
        		 
        		/* Confirm that the WinSock DLL supports 2.2.*/ 
        		/* Note that if the DLL supports versions greater    */ 
        		/* than 2.2 in addition to 2.2, it will still return */ 
        		/* 2.2 in wVersion since that is the version we      */ 
        		/* requested.                                        */  
        		if ( LOBYTE( wsaData.wVersion ) != 2 || 
        				HIBYTE( wsaData.wVersion ) != 2 ) 
        		{ 
        			/* Tell the user that we could not find a usable */ 
        			/* WinSock DLL.                                  */ 
        			WSACleanup( ); 
        			m_bInitOK = FALSE; 
        			return FALSE; 
        		} 
#endif //_WIN32 
#ifdef PASS_EXCEPTION
        		try 
#endif // PASS_EXCEPTION
        		{ 
        			sockfd = socket (AF_INET, SOCK_STREAM, 0); 
         
#ifdef _WIN32 
        			if (sockfd == INVALID_SOCKET) 
#endif //_WIN32 
#ifdef linux 
		 			if (sockfd == -1) 
#endif //linux 
#ifdef __SCO__ 
		 			if (sockfd == -1) 
#endif // __SCO__ 
#ifdef __HPUX__ 
		 			if (sockfd == -1) 
#endif // __HPUX__ 
        			{ 
        				m_bInitOK = FALSE; 
        				return FALSE; 
        			} 
         
	 			if (m_RemoteServerIP != NULL) 
	 			{ 
	        			memset (&servaddr,0,sizeof(servaddr)); 
	         
	        			servaddr.sin_family = AF_INET; 
	        			servaddr.sin_port = htons (m_RemoteServerPort); 
		         
		 				printf ("IP address to connect back : %s\n",m_RemoteServerIP); 
	        			unsigned long inaddr = inet_addr (m_RemoteServerIP); 
	        			memcpy (&servaddr.sin_addr,&inaddr,sizeof (unsigned long)); 
	         
	 			} 
#ifdef linux 
	 			else 
	 			{ 
	 				if (m_ClientSocket != 0) 
	 				{ 
	 					socklen_t socketAddrLen = sizeof (struct sockaddr); 
	 					getpeername (m_ClientSocket,(struct sockaddr *)&servaddr,&socketAddrLen); 
	 					char buffer[256]; 
	 					memset (buffer,0,256); 
	 					strncpy (buffer,inet_ntoa (servaddr.sin_addr),256); 
	 					printf ("IP address to connect back : %s\n",buffer); 
	 				} 
	 			} 
#endif // linux 
  
#ifdef _WIN32 
       				if ((connect (sockfd, (struct sockaddr *)&servaddr,sizeof(servaddr))) == SOCKET_ERROR) 
#endif //_WIN32 
#ifdef linux 
		 			if ((connect (sockfd, (struct sockaddr *)&servaddr,sizeof(servaddr))) == -1) 
#endif //linux 
#ifdef __SCO__ 
		 			if ((connect (sockfd, (struct sockaddr *)&servaddr,sizeof(servaddr))) == -1) 
#endif // __SCO__ 
#ifdef __HPUX__ 
		 			if ((connect (sockfd, (struct sockaddr *)&servaddr,sizeof(servaddr))) == -1) 
#endif // __HPUX__ 
        			{ 
        				m_bInitOK = FALSE; 
        				return FALSE; 
        			} 
         
#ifdef PASS_SSL 
        			if (m_bSecuredConnection == TRUE) 
        			{ 
	  					printf ("Try to start a secure connection : CConnectionSender::Init\n"); 
	  					if (m_SSLctx == NULL) 
	  					{ 
	  						printf ("Can't create new SSL handle : m_SSLctx NULL\n"); 
	  						return FALSE; 
	  					} 
 
						int errSSL; 
 
        				m_SSLssl = SSL_new (m_SSLctx); 
        				SSL_set_cipher_list(m_SSLssl,"RC4-SHA"); 
        				SSL_set_fd (m_SSLssl, sockfd); 
	  					printf ("Before SSL_connect\n"); 
       	 				errSSL = SSL_connect (m_SSLssl); 
						printf ("After SSL_connect\n"); 
         
        				if (errSSL == -1) 
        				{ 
        					return FALSE; 
        				} 
         
        				SSL_CIPHER *SSLcipher = SSL_get_current_cipher(m_SSLssl); 
        				int alg_bits; 
		  				int currentBits; 
		  				currentBits = SSL_CIPHER_get_bits(SSLcipher, &alg_bits); 
         
        				char buffer[256]; 
        				X509*    server_cert; 
        				char*    str; 
         
        				/* Get server's certificate (note: beware of dynamic allocation) - opt */ 
         
        				server_cert = SSL_get_peer_certificate (m_SSLssl); 
         
        				str = X509_NAME_oneline (X509_get_subject_name (server_cert),0,0); 
        				sprintf (buffer,"\t subject: %s\n", str); 
        				free (str); 
         
        				str = X509_NAME_oneline (X509_get_issuer_name  (server_cert),0,0); 
        				sprintf (buffer,"\t issuer: %s\n", str); 
        				free (str); 
         
        				/* We could do all sorts of certificate verification stuff here before 
        				deallocating the certificate. */ 
         
        				X509_free (server_cert); 
        			} 
#endif //PASS_SSL 
        		}
#ifdef PASS_EXCEPTION 				
        		catch (...) 
        		{ 
        			m_bInitOK = FALSE; 
        			return FALSE; 
        		} 
#endif // PASS_EXCEPTION         
        		m_bInitOK = TRUE; 
        	} 
         
        	return TRUE; 
        } 
         
        void CConnectionSender::PreRun (void) 
        { 
        } 
         
        void CConnectionSender::PostRun (void) 
        { 
			printf ("CConnectionSender PostRun - Wait for all messages to be sent\n"); 
 
        } 
         
        void CConnectionSender::SetRemoteServerIP (char *serverIP) 
        { 
        	if (serverIP != NULL) 
        	{ 
        		if (m_RemoteServerIP != NULL) 
        		{ 
        			delete [] m_RemoteServerIP; 
        		} 
        		m_RemoteServerIP = new char [(strlen (serverIP)) + 1]; 
        		strcpy (m_RemoteServerIP,serverIP); 
        	} 
        } 
         
        void CConnectionSender::SetRemoteServerName (char *serverName) 
        { 
        	if (serverName != NULL) 
        	{ 
        		if (m_RemoteServerName != NULL) 
        		{ 
        			delete [] m_RemoteServerName; 
        		} 
        		m_RemoteServerName = new char [(strlen (serverName)) + 1]; 
        		strcpy (m_RemoteServerName,serverName); 
        	} 
        } 
         
        void CConnectionSender::SetRemoteServerPort (unsigned short serverPort) 
        { 
        	m_RemoteServerPort = serverPort; 
        } 
         
        void CConnectionSender::SetStack (CMessageStack *aStack) 
        { 
        	m_SendStack = aStack; 
        } 
         
        BOOL CConnectionSender::SendObject (struct SerializedObject *anObject) 
        { 
        	if (anObject == NULL) 
        	{ 
        		return FALSE; 
        	} 
         
        	if (m_bInitOK == TRUE) 
        	{ 
#ifdef PASS_SSL 
        		if (m_bSecuredConnection == TRUE) 
        		{ 
        			if (m_SSLssl == NULL) 
        			{ 
        				return FALSE; 
        			} 
        			else 
        			{ 
#ifdef PASS_EXCEPTION
						try
#endif // PASS_EXCEPTION 
        				{ 
      					if ((SSL_writen (m_SSLssl, (char *)&anObject->nameSize, 4)) == -1) 
        					{ 
        						return FALSE; 
        					} 
         
      					if ((SSL_writen (m_SSLssl, anObject->name, anObject->nameSize)) == -1) 
        					{ 
        						return FALSE; 
        					} 
         
      					if ((SSL_writen (m_SSLssl, (char *)&anObject->typeSize, 4)) == -1) 
        					{ 
        						return FALSE; 
        					} 
         
      					if ((SSL_writen (m_SSLssl, anObject->type, anObject->typeSize)) == -1) 
        					{ 
        						return FALSE; 
        					} 
         
      					if ((SSL_writen (m_SSLssl, (char *)&anObject->serializedObjectSize, 4)) == -1) 
        					{ 
        						return FALSE; 
        					} 
         
      					if ((SSL_writen (m_SSLssl, (char *)anObject->serializedObject, anObject->serializedObjectSize)) == -1) 
        					{ 
        						return FALSE; 
        					} 
         
        					return TRUE; 
        				}
#ifdef PASS_EXCEPTION 						
        				catch (...) 
        				{ 
        					return FALSE; 
        				} 
#endif // PASS_EXCEPTION 
        			} 
        		} 
        		else 
#endif //PASS_SSL 
        		{ 
#ifdef PASS_EXCEPTION 
        			try 
#endif // PASS_EXCEPTION 
        			{ 
		  				printf ("nameSize : %ld\n",anObject->nameSize); 
        				if ((writen (sockfd, &anObject->nameSize, 4)) == -1) 
        				{ 
        					return FALSE; 
        				} 
         
        				if ((writen (sockfd, anObject->name, anObject->nameSize)) == -1) 
        				{ 
        					return FALSE; 
        				} 
         
        				if ((writen (sockfd, &anObject->typeSize, 4)) == -1) 
        				{ 
        					return FALSE; 
        				} 
         
        				if ((writen (sockfd, anObject->type, anObject->typeSize)) == -1) 
        				{ 
        					return FALSE; 
        				} 
         
        				if ((writen (sockfd, &anObject->serializedObjectSize, 4)) == -1) 
        				{ 
        					return FALSE; 
        				} 
         
        				if ((writen (sockfd, anObject->serializedObject, anObject->serializedObjectSize)) == -1) 
        				{ 
        					return FALSE; 
        				} 
         
        				return TRUE; 
        			}
#ifdef PASS_EXCEPTION 					
        			catch (...) 
        			{ 
        				return FALSE; 
        			} 
#endif // PASS_EXCEPTION 
        		} 
        	} 
        	return FALSE; 
        } 
         
        void CConnectionSender::Tick (void) 
        {
		PASS_LOCK (&lock); 
#ifdef PASS_EXCEPTION
			try
#endif // PASS_EXCEPTION 
        	{ 
        		if (m_SendStack->IsMessageAvailable()) 
        		{ 
        			CMessage *aMessage = m_SendStack->GetMessage(); 
         
        			if (aMessage != NULL) 
        			{ 
        				// Multi pass serialization 
        				aMessage->SetSerialMultiPass(); 
        				aMessage->SetSerialMultiPassCount(); 
        				aMessage->Serialize (); 
         
        				aMessage->SetSerialMultiPassFill(); 
        				aMessage->Serialize (); 
         
        				SerializedObject *object = aMessage->GetSerializedObject(); 
 
						if (object == NULL) 
						{ 
							// Can't get serialized object 
							// Trace Error 
							delete aMessage; 

							PASS_UNLOCK (&lock);
							
							return; 
						} 
 
        				if (SendObject (object) == FALSE) 
						{ 
							// Can't send object 
							// Trace Error 
							delete aMessage; 

							PASS_UNLOCK (&lock);

							return; 
						} 
        				 
        				// Check if it's a close message 
        				if (aMessage->GetName() != NULL) 
        				{ 
        					if ((strcmp (aMessage->GetName(),"ConnectionMessage_Close")) == 0) 
        					{ 
#ifdef _WIN32 
        						//It's a Close message 
        						SetEvent (m_hCloseEvent); 
#endif //_WIN32 
#ifdef linux 
	  						pthread_mutex_lock (&m_InternalLock); 
	  						m_CloseFlag = TRUE; 
	  						pthread_mutex_unlock (&m_InternalLock); 
#endif //linux 
#ifdef __SCO__ 
	  						mutex_lock (&m_InternalLock); 
	  						m_CloseFlag = TRUE; 
	  						mutex_unlock (&m_InternalLock); 
#endif // __SCO__ 
#ifdef __HPUX__ 
	  						PASS_LOCK (&m_InternalLock); 
	  						m_CloseFlag = TRUE; 
	  						PASS_UNLOCK (&m_InternalLock); 
#endif // __HPUX__ 
        					} 
        				} 
        				delete aMessage; 
        			} 
        		} 
        	} 
#ifdef PASS_EXCEPTION 
        	catch (...) 
        	{ 
        		// Error, do something 
        	} 
#endif // PASS_EXCEPTION 
        
		PASS_UNLOCK (&lock); 
	} 
 
#ifdef linux 
  void CConnectionSender::SetClientSocket (int clientSocket) 
  { 
  	m_ClientSocket = clientSocket; 
  } 
#endif // linux   
 
#ifdef PASS_SSL 
        void CConnectionSender::SetSecuredConnection (BOOL flag) 
        { 
        	m_bSecuredConnection = flag; 
        	if (m_bSecuredConnection == TRUE) 
        	{ 
        		// Initialize SSL 
           
        		/* SSL preliminaries. We keep the certificate and key with the context. */ 
         
        		SSL_load_error_strings(); 
        		SSLeay_add_ssl_algorithms(); 
        		m_SSLmeth = SSLv3_client_method(); 
        		m_SSLctx = SSL_CTX_new (m_SSLmeth); 
         
        		if (!m_SSLctx) 
        		{ 
        			m_SSLctx = NULL; 
        		} 
        		else if (SSL_CTX_use_certificate_file(m_SSLctx, m_szCertificateFile, SSL_FILETYPE_PEM) <= 0) 
        		{ 
        			m_SSLctx = NULL; 
        		} 
        	} 
        } 
         
        void CConnectionSender::SetCertificateFile (char *file) 
        { 
        	if (file != NULL) 
        	{ 
        		if (m_szCertificateFile != NULL) 
        		{ 
        			delete [] m_szCertificateFile; 
        		} 
        		m_szCertificateFile = new char [(strlen (file)) + 1]; 
        		strcpy (m_szCertificateFile,file); 
        	} 
        } 
         
#endif //PASS_SSL 
