/* This header is made for cross platform compatibility */ 
 
#ifndef __PLATFORM__ 
#define __PLATFORM__ 

/************************************ WIN32 defines ***********************************/

#ifdef _WIN32
// Sleeping
#define PASS_MILLISLEEP(milliSec) Sleep(milliSec)

#include <windows.h>
#include <winbase.h>

// Locking
#define PASS_SYNCH CRITICAL_SECTION
#define PASS_INITLOCK(synch) InitializeCriticalSection(synch)
#define PASS_DESTROYLOCK(synch) DeleteCriticalSection(synch)
#define PASS_LOCK(synch) EnterCriticalSection(synch)
#define PASS_UNLOCK(synch) LeaveCriticalSection(synch)

// Class definition macros
#ifdef PASS_BASEDLL_EXPORT
#define PASS_BASEDLL_API __declspec(dllexport)
#endif

#ifdef PASS_BASEDLL_IMPORT
#define PASS_BASEDLL_API __declspec(dllimport)
#endif

#endif // _WIN32

#ifdef PASS_BASEDLL_API
#define PASS_CLASS_OBJECT PASS_BASEDLL_API CObject
#define PASS_CLASS_MANAGER PASS_BASEDLL_API CManager
#define PASS_CLASS_THREAD PASS_BASEDLL_API CThread
#else
#define PASS_CLASS_OBJECT CObject
#define PASS_CLASS_MANAGER CManager
#define PASS_CLASS_THREAD CThread
#endif

/************************************ LINUX defines ***********************************/

#ifdef linux 
typedef unsigned char BYTE;
#ifndef BOOL
typedef short BOOL;
#endif
#ifndef DWORD 
typedef unsigned long DWORD;
#endif
#ifndef ULONG32
typedef unsigned long ULONG32; 
#endif
#undef NULL 
#define NULL 0
#ifndef TRUE 
#define TRUE 1
#endif
#ifndef FALSE 
#define FALSE 0 
#endif

// function mapping
#define _snprintf snprintf
#define _vsnprintf vsnprintf

// Sleeping
#include <sys/unistd.h>
#define PASS_MILLISLEEP(milliSec) \
    if(milliSec < 1000000) \
        usleep(milliSec * 1000); \
    else \
        sleep(milliSec / 1000);

// Locking
#include <pthread.h>
#define PASS_SYNCH pthread_mutex_t
#define PASS_INITLOCK(synch) pthread_mutex_init (synch,0)
#define PASS_DESTROYLOCK(synch) pthread_mutex_destroy(synch)
#define PASS_LOCK(synch) pthread_mutex_lock(synch)
#define PASS_UNLOCK(synch) pthread_mutex_unlock(synch)

#endif // linux 
 
/************************************ SCO   defines ***********************************/

#ifdef __SCO__ 
typedef unsigned char BYTE; 
typedef unsigned char BOOL; 
typedef unsigned long DWORD; 
typedef unsigned long ULONG32; 
#define TRUE 1 
#define FALSE 0 
#undef NULL 
#define NULL 0 

// function mapping
#define _snprintf snprintf
#define _vsnprintf vsnprintf

// Sleeping
#define PASS_MILLISLEEP(milliSec) \
    if(milliSec < 1000000) \
        usleep(milliSec * 1000); \
    else \
        sleep(milliSec / 1000);

//Byte ordering
#include <sys/byteorder.h>

// Locking
#include <synch.h>
#define PASS_SYNCH mutex_t
#define PASS_INITLOCK(synch) mutex_init(synch,USYNC_THREAD,NULL)
#define PASS_DESTROYLOCK(synch) mutex_destroy(synch)
#define PASS_LOCK(synch) mutex_lock(synch)
#define PASS_UNLOCK(synch) mutex_unlock(synch)
#endif // __SCO__ 
 
/************************************ BEOS  defines ***********************************/

#ifdef __BEOS__
typedef unsigned char BYTE; 
typedef unsigned char BOOL; 
typedef unsigned long DWORD; 
typedef unsigned long ULONG32; 
#undef NULL 
#define NULL 0 
#define TRUE 1 
#define FALSE 0 

// function mapping
#define _snprintf snprintf
#define _vsnprintf vsnprintf

// Sleeping
#define PASS_MILLISLEEP(milliSec) snooze(milliSec * 1000)

// Locking
#include <OS.h>
#define PASS_SYNCH sem_id
#define PASS_INITLOCK(synch) *synch = create_sem(1,NULL)
#define PASS_DESTROYLOCK(synch) delete_sem(*synch)
#define PASS_LOCK(synch) acquire_sem(*synch)
#define PASS_UNLOCK(synch) release_sem(*synch)

#endif // __BEOS__

/************************************ HP-UX defines ***********************************/

#ifdef __HPUX__ 
typedef unsigned char BYTE; 
typedef unsigned char BOOL; 
typedef unsigned long DWORD; 
typedef unsigned long ULONG32; 
#undef NULL 
#define NULL 0 
#define TRUE 1 
#define FALSE 0 

// function mapping
#define _snprintf snprintf
#define _vsnprintf vsnprintf

// supporting internal function
int internal_usleep (unsigned int useconds);

// sleeping
#define PASS_MILLISLEEP(milliSec) internal_usleep(milliSec*1000)

// Locking
#include <pthread.h>
#define PASS_SYNCH pthread_mutex_t
#define PASS_INITLOCK(synch) pthread_mutex_init (synch,0)
#define PASS_DESTROYLOCK(synch) pthread_mutex_destroy(synch)
#define PASS_LOCK(synch) pthread_mutex_lock(synch)
#define PASS_UNLOCK(synch) pthread_mutex_unlock(synch)

#endif // __HPUX__ 
 
/************************ enable exception handling in supported OSes *****************/ 
#ifdef _WIN32 
#ifndef PASS_EXCEPTION
    #define PASS_EXCEPTION
#endif 
#endif 
#ifdef linux 
#ifndef PASS_EXCEPTION
    #define PASS_EXCEPTION
#endif 
#endif 
#ifdef __BEOS__ 
#ifndef PASS_EXCEPTION
    #define PASS_EXCEPTION
#endif 
#endif 
 
#endif // __PLATFORM__ 
 
