/***************************************************************************
                         -- win.cc --
                         -------------------
    begin                : 2001-09-01
    copyright            : (C) 2001 - 2002
    authors              : Jason Kasper (vanRijn)
                         : Ben Jansens (xOr)
    URL                  : http://bbconf.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 // E_O_H_VR
#include "win.h"
#include <qsplitter.h>
#include <qvariant.h>
#include <qmessagebox.h>
#include "config.h"

/* 
 *  Constructs a MainWindow which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 *
 *  The dialog will by default be modeless, unless you set 'modal' to
 *  TRUE to construct a modal dialog.
 */
MainWindow::MainWindow( QWidget* parent,  const char* name, WFlags fl ) :
		QWidget( parent, name, fl ) {
	listmap = new bbc_map<QListViewItem*, TreeNode<QWidget>*>;
	pluginmap = new bbc_map<QWidget*, plugin*>;
	if (!name)
		setName("MainWindow");
  setCaption(tr(PACKAGE));

	LayoutDialog = new QVBoxLayout(this); 
	LayoutDialog->setSpacing(6);
	LayoutDialog->setMargin(11);

  QSplitter *split = new QSplitter(this);

  ListView1 = new QListView(split, "ListView1");
  ListView1->addColumn(tr("Blackbox Configuration"));
  ListView1->header()->setClickEnabled(FALSE, ListView1->header()->count() - 1);
  ListView1->header()->setResizeEnabled(FALSE, ListView1->header()->count() - 1);
  ListView1->setGeometry(QRect(11, 11, 120, 424)); 
  ListView1->setMinimumSize(QSize(130, 0));
  ListView1->setMaximumSize(QSize(500, 32767));
  ListView1->setRootIsDecorated(TRUE);
  ListView1->setColumnWidthMode(0, QListView::Maximum);
  ListView1->setMinimumWidth(ListView1->columnWidth(0) + 6);
	// ListView1->setSorting(-1);

	WidgetStack1 = new QWidgetStack(split);
	WidgetStack1->setFrameShape(QFrame::StyledPanel);
	WidgetStack1->setFrameShadow(QFrame::Sunken);

  LayoutButtons = new QHBoxLayout; 
  LayoutButtons->setSpacing( 6 );
  LayoutButtons->setMargin( 0 );

  ButtonHelp = new QPushButton(this, "ButtonHelp");
  ButtonHelp->setText(tr("&Help"));
  LayoutButtons->addWidget(ButtonHelp);

	QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  LayoutButtons->addItem( spacer );

  ButtonApply = new QPushButton(this, "ButtonApply");
  ButtonApply->setText(tr("&Apply"));
  LayoutButtons->addWidget(ButtonApply);

  ButtonReset = new QPushButton(this, "ButtonReset");
  ButtonReset->setText(tr("&Reset"));
  LayoutButtons->addWidget(ButtonReset);

	LayoutDialog->addWidget(split);
	LayoutDialog->addLayout(LayoutButtons);

	// signals and slots connections
  connect(ButtonApply, SIGNAL(clicked()), this, SLOT(apply()));
  connect(ListView1, SIGNAL(selectionChanged(QListViewItem*)), this, SLOT(changeFrame(QListViewItem*)));
  connect(ButtonReset, SIGNAL(clicked()), this, SLOT(reset()));
  connect(ButtonHelp, SIGNAL(clicked()), this, SLOT(help()));
}

/*  
 *  Destroys the object and frees any allocated resources
 */
MainWindow::~MainWindow() {
	if (listmap) delete listmap;
	if (pluginmap) delete pluginmap;
}

void MainWindow::changeFrame(QListViewItem *item) {
	TreeNode<QWidget> *tree = listmap->find(item);
	if (tree) {
		widgetChanging(tree->object());
		WidgetStack1->raiseWidget(tree->object());
	}
}

void MainWindow::reset() {
	// reset the settings for the plugin of the currenly shown widget on the stack
	plugin *p = pluginmap->find(WidgetStack1->visibleWidget());
	p->load();
}

bool MainWindow::apply() {
	// save the settings for the plugin of the currenly shown widget on the stack
	plugin *p = pluginmap->find(WidgetStack1->visibleWidget());
	bool retry = true;
	bool saved = false;
	while (retry) {
		saved = p->save();
		if (saved)
			retry = false;
		else
			retry = showSaveFailed(p->get_configfile() );
	}
	return saved;
}

void MainWindow::promptSaveForget() {
	bool done = false;
	
	while (!done) {
		QMessageBox mb( "Warning! Settings changed!",
			"You have made changes to the properties of the\n"
			"current module, and unless you apply the changes\n"
			"you've made, you will lose them.\n\n"
			"Do you want to apply the changes you have made\n"
			"and save your changes to disk before running the \n"
			"new module, or forget them?",
			QMessageBox::Warning,
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::No | QMessageBox::Escape,
			0);
		mb.setButtonText( QMessageBox::Yes, "&Apply" );
		mb.setButtonText( QMessageBox::No, "&Forget" );
		switch( mb.exec() ) {
			case QMessageBox::Yes:
				// save and exit
				done = apply();
				break;
			case QMessageBox::No:
				// exit without saving
				reset();
				done = true;
				break;
		}
	}
}

void MainWindow::widgetChanging(QWidget *w) {
	plugin *oldplugin = pluginmap->find(WidgetStack1->visibleWidget());
	plugin *newplugin = pluginmap->find(w);
	if (oldplugin)
		if (newplugin != oldplugin)
			if (oldplugin->isModified()) {
				promptSaveForget();
			}
}

void MainWindow::closeEvent(QCloseEvent *ev) {
	plugin *p = pluginmap->find(WidgetStack1->visibleWidget());
	if (p && (p->isModified())) {
		promptSaveForget();
	}
	QWidget::closeEvent(ev);
}

void MainWindow::addPluginWidget(QWidget *w, plugin *p) {
	static int count = 0;
	WidgetStack1->addWidget(w, count++);
	pluginmap->add(w, p);
}

void MainWindow::addChildren(TreeNode<QWidget> *tree, QListViewItem *viewparent,
		plugin *p) {
	if (!tree) return;
	while (tree->next()) {
		tree = tree->next();
		QListViewItem *lastItem = new QListViewItem(viewparent, tree->object()->caption());
		addPluginWidget(tree->object(), p);
		listmap->add(lastItem, tree);
		addChildren(tree->child(), lastItem, p);
	}		
}

void MainWindow::setPlugins(QList<plugin> *plist, QString &start_plugin) {
	QListIterator<plugin> it(*plist);
	// parse the list in reverse order, because each one is added to the top of
	// the listview, so this way they are in the same order in the listview as in
	// 'plist'
	for (it.toLast(); it.current(); --it) {
		TreeNode<QWidget> *tree = it.current()->getMembers(); // get first dummy node
		if (tree) {
			tree = tree->next();// select the first plugin at the top level, and only
													// do this for the first, dont want people cluttering
													// the tree up!
			if (tree) {
				QListViewItem *lastItem = new QListViewItem(ListView1, tree->object()->caption());
				addPluginWidget(tree->object(), it.current());
				listmap->add(lastItem, tree);
				addChildren(tree->child(), lastItem, it.current());
				// set the item to be expanded by default
				ListView1->setOpen(lastItem, true);
			}
		}
	}
	// try find start_plugin, if it can't be found, default to the first plugin.
	bool found_start = false; // true if the start_plugin was found
	QListViewItemIterator list_it(ListView1);
	for (; list_it.current(); ++list_it) {
		if (list_it.current()->text(0).lower().contains (start_plugin.lower()) ) {
			ListView1->setSelected(list_it.current(), true);
			found_start = true;
			break;
		}
	}
	if (!found_start)
		ListView1->setSelected(ListView1->firstChild(), true);
}

void MainWindow::help() {
	QDialog *helpDialog = new QDialog(this, "HelpDialog", false, WDestructiveClose);
	helpDialog->setCaption(tr("Help"));
	helpDialog->setMinimumWidth(400);
	helpDialog->setMinimumHeight(300);

	QVBoxLayout *helpLayout = new QVBoxLayout(helpDialog);
	helpLayout->setSpacing(6);
	helpLayout->setMargin(11);

	QTextView *helpText = new QTextView( helpDialog );
	helpText->setText(
		tr("<h2>bbconf - version %1</h2><br>"
		"<p>bbconf is written by:<br>"
		"Jason Kasper (vanRijn) "
		"<a href='mailto:vR@movingparts.net'>vR at movingparts dot net</a><br> "
		"and<br> "
		"Ben Jansens (xOr) <a href='mailto:xor@orodu.net'>xor at orodu dot net "
		"</a>.</p> "
		"<br> "
		"<p> "
		"All artwork has been done by artwiz  "
		"(<a href='http://artwiz.artramp.org'>http://artwiz.artramp.org</a>) "
		"and by no means is a fair representation of his talent. All images  "
		"presented in this application are copyrighted under artwiz's ZIWTRA "
		"copyright, and all accolades and job offers should go to him.  =:) "
		"</p> "
		"<br> "
		"<p>bbconf lives in  "
		"<a href='http://sourceforge.net/cvs/?group_id=32563'>a "
		"sourceforge.net CVS repository at "
		"http://sourceforge.net/cvs/?group_id=32563</a> "
		"and you can get the latest snapshots of "
		"it from there.  Please visit bbconf's home page at  "
		"<a href='http://bbconf.sourceforge.net'>http://bbconf.sourceforge.net</a> "
		"for more information, screenshots, and documentation.... "
		"</p>\n").arg(VERSION) );
	helpLayout->addWidget(helpText);
	
	QHBoxLayout *helpHorizLayout = new QHBoxLayout;
	QSpacerItem *helpSpacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
	helpHorizLayout->addItem(helpSpacer);

	QPushButton *helpCloseButton = new QPushButton(helpDialog, "CloseButton");
	helpCloseButton->setText(tr("Close"));
	helpCloseButton->setDefault(true);
	helpHorizLayout->addWidget(helpCloseButton);
	
	helpLayout->addLayout(helpHorizLayout);

  connect(helpCloseButton, SIGNAL(clicked()), helpDialog, SLOT(close()));
	helpDialog->show();
}

bool MainWindow::showSaveFailed(QString rcfile) {
	QMessageBox mb("Save failed!",
		tr("Unable to save your changes! The current plugin\n"
		"attempted to save your changes to the file you have specified\n"
		"(%1), but was unable to do so. \n\n"
		"Please make sure that you are able to write to this file\n"
		"(make sure you have 'write permissions'). \n\n"
		"Check the permissions on this file and either click\n"
		"\"retry\" or cancel below.").arg(rcfile),
		QMessageBox::Critical,
		QMessageBox::Retry | QMessageBox::Default,
		QMessageBox::Cancel | QMessageBox::Escape,
		QMessageBox::NoButton);
	mb.setButtonText( QMessageBox::Retry, "&Retry" );
	mb.setButtonText( QMessageBox::Cancel, "&Cancel" );
	if (mb.exec() == QMessageBox::Retry)
		return true;
	return false;
}
