<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    utils.php                                               */
/* Author:      Paul Waite                                              */
/* Description: Various general utility routines.                       */
/*                                                                      */
/* ******************************************************************** */
/** @package utils */

// ----------------------------------------------------------------------
/**
* Insert vertical space in table
* Insert some vertical whitespace into a table.
* @param integer $height Whitespace height in pixels
* @param integer $cols   No. of columns in the target table
* @return string HTML for a row of the given height
*/
function vspace($height, $cols=1) {
  echo "<tr><td colspan=\"$cols\" height=\"$h\"></td></tr>";
} // vspace

// ----------------------------------------------------------------------
/**
* Send a message to the system logfile
* Send message to the syslog, prefixed with APP_NAME nicely. Saves
* having to prefix it each time.
* @param string $msg     Message to enter into the system log
* @param string $prefix  Prefix to the message
*/
function log_sys($msg, $prefix="") {
  if ($prefix == "") {
    if (defined("APP_NAME")) {
      $prefix = APP_NAME;
    }
    else {
      $prefix = "AXYL";
    }
  }
  error_log("$prefix: $msg", 0);
} // log_sys

// ----------------------------------------------------------------------
/**
* Exit the application with error message
* Echo the message and leave. Intended to handle 'emergencies'.
* @param string $heading Message heading, subject
* @param string $msg     Message in detail
*/
function error_exit($heading, $msg="") {
  global $RESPONSE;

  switch ($RESPONSE->browser_type) {
    case BROWSER_TYPE_XHTML:
    case BROWSER_TYPE_HTML:
      $RESPONSE->delete_cookie();
      echo "<h3>$heading</h3><strong>$msg</strong>";
      break;

    case BROWSER_TYPE_XHTMLMP:
    case BROWSER_TYPE_WML:
    case BROWSER_TYPE_WMLUP:
      include_once("wml-defs.php");
      $card = new WMLcard("error_exit", "System Error");
      $card->insert(
        "<p>" .
            "$heading<br/>" .
            "$msg" .
        "</p>"
        );
      // Create the card deck and output it..
      $deck = new WMLdeck($card);
      echo $deck->wml();
      break;

    default:
      $RESPONSE->delete_cookie();
      echo "<h3>$heading</h3><strong>$msg</strong>";
  } // switch
  exit;
} // error_exit

// ----------------------------------------------------------------------
/**
* Return an HTTP error message
* Returns a formatted HTTP Error
* @param integer $code HTTP error code
* @return string The formatted HTTP error code
*/
function HTTPError($code) {
  $s = "";
  switch ($code) {
    case 401:
      $s = "Unauthorised.";
      break;
    case 403:
      $s = "Forbidden.";
      break;
    case 404:
      $s = "Not Found.";
      break;
    case 405:
      $s = "Method Not Allowed.";
      break;
    case 406:
      $s = "Not Acceptable.";
      break;
    case 500:
      $s = "Infernal server error.";
      break;
    case 501:
      $s = "Not Implemented.";
      break;
    default:
      $s = "Unknown Server Error.";
      break;
  } // switch

  return "HTTP/1.1 $code: $s";
} // HTTPError

// ----------------------------------------------------------------------
/**
* Exit with HTTP error code
* Send a simple error code and then die.
* @param integer $code HTTP error code
*/
function errorcode_exit($code) {
  error_exit( HTTPError($code) );
} // errorcode_exit

// -----------------------------------------------------
/**
* Return a value which may be defaulted
* Returns the value of the variable, if it is valid, otherwise
* returns the specified default value.
* @param mixed $val     The value of interest
* @param mixed $default The default value to apply if $val is empty
* @return mixed Either the value, or the default
*/
function defaulted($val, $default) {
  if (empty($val) || $val == "") return $default;
  else return $val;
} // defaulted

// ----------------------------------------------------------------------
/**
* Format a string (dotted) form of an IP address. We make sure we have
* the full dotted quad, and if we have to pad it with zeroes, then we
* add the network spec (/24, /16 etc.) as well. If the given IP already
* has a netmask, then we don't change it, and assume the IP is already
* correctly formatted by someone who knows what they are doing!
*
* @param string $pi IP address in string dotted format
* @return string The padded IP address
*/
function ip_format($ip) {
  // Pad shorthand addresses with zeroes..
  $myipbits = explode("/", $ip);
  $myip = trim($myipbits[0]);
  $mynetmask = trim($myipbits[1]);

  $octets = explode(".", $myip);
  if (count($octets < 4)) {
    while ( count($octets) < 4) {
      $octets[] = "0";
    }
    $myip = implode(".", $octets);
  }

  // If not defined already, find the netmask bits,
  // by scanning for zeroes from the end..
  if ($mynetmask == "") {
    $mynetmask = 32;
    for ($ix=3; $ix >= 0; $ix--) {
      if ($octets[$ix] == 0) $mynetmask -= 8;
      else break;
    }
  }

  // Append netmask if a network..
  if ($mynetmask < 32) {
    $myip .= "/$mynetmask";
  }
  return $myip;
} // ip_format

// -----------------------------------------------------
/**
* Check string to see if it is in IPv4 format. This is
* pretty simplistic. It returns true if the string $ip
* is of the form 'n.n.n.n'.
* @return bool True if it is in IPv4 format
*/
function is_ipaddress($ip) {
  $isip = false;
  $myip = explode(".", $ip);
  if (count($myip) == 4) {
    // A dotted quad is close enough..
    $isip = true;
  }
  return $isip;
} // is_ipaddress

// -----------------------------------------------------
/**
 * Function which adds content to existing content in one
 * of three modes - the default of which is 'replace':
 *  'replace' - blows away what is there, and sets new value
 *  'append'  - adds new value to the end
 *  'prefix'  - adds new value to the end
 * You can also set the 'once' flag, which controls whether
 * we only append/prefix content which does not already exist,
 * with a case-insensitive comparison test. The default for
 * this flag is true.
 * There is also an optional delimiter, which if specified (ie.
 * non-blank) is used to separate appended/prefixed content.
 * @param string  $original The existing content
 * @param string  $new      The new content
 * @param string  $mode     Add mode: 'append' (default), 'prefix', or 'replace'
 * @param boolean $once     If true duplicate content is rejected, default is false
 * @param string  $delim    Optional delimiter separating added content
 */
define("ADD_ONCE_ONLY", true);
define("ADD_MULTIPLE",  false);
function add_content_to($original, $new, $mode="replace", $once=ADD_ONCE_ONLY, $delim="") {
  $res = $original;
  # assert the delimiter on new addition
  if ($delim != "") {
    if (substr($new, -1) != $delim) {
      $new .= $delim;
    }
  }
  # add in appropriate mode
  switch ($mode) {
    case "append":
      if ($original != "") {
        if ($once !== ADD_ONCE_ONLY || !stristr($original, $new)) {
          if ($delim != "" && substr($original, -1) != $delim) {
            $original .= $delim;
          }
          $res = $original . $new;
        }
      }
      else {
        $res = $new;
      }
      break;
      
    case "prefix":
      if ($original != "") {
        if ($once !== ADD_ONCE_ONLY || !stristr($original, $new)) {
          if ($delim != "" && substr($original, -1) != $delim) {
            $original = $delim . $original;
          }
          $res = $new . $original;
        }
      }
      else {
        $res = $new;
      }
      break;

    # Replace mode is the default
    default:
      $res = $new;
      break;
  } // switch

  // Return new content
  return $res;
  
} // add_content_to
// -----------------------------------------------------
/** This function is a utility to allow composite fragments of javascript
* to be replaced or concatenated together to form a new string of
* javascript. This is designed to be used for building short javascript
* statement strings to go in things such as onclick events for form
* elements etc.
 * @param string  $newscript The existing script content
 * @param string  $oldscript The new script content
 * @param string  $mode      Add mode: 'append' (default), 'prefix', or 'replace'
 * @param boolean $once      If true duplicate content is rejected, default is false
 * @param string  $delim     Optional delimiter separating added content
*/
/** Add script by replacing current content (default) */
define("SCRIPT_REPLACE", 0);
/** Add script by keeping current content and appending to the end */
define("SCRIPT_APPEND",  1);
/** Add script by keeping current content and prefixing to the beginning */
define("SCRIPT_PREFIX",  2);

function inline_script($newscript, $oldscript="", $mode=SCRIPT_REPLACE, $once=ADD_ONCE_ONLY, $delim=";") {
  $res = "";
  switch ($mode) {
    case SCRIPT_APPEND:
      $res = add_content_to($oldscript, $newscript, "append", $once, $delim);
      break;
    case SCRIPT_PREFIX:
      $res = add_content_to($oldscript, $newscript, "prefix", $once, $delim);
      break;
    default:
      $res = add_content_to($oldscript, $newscript, "replace", $once, $delim);
      break;
  } // switch

  // Return new inline script
  return $res;
} // inline_script

// -----------------------------------------------------
/**
* Resolve a foreign key value
* Resolve a simple single-level foreign key. The variable
* 'display_fields' can contain multiple fields separated
* by the "|" char. A literal is prefixed by "#".
* @param mixed  $fkey            The value of the foreign key field to look up
* @param string $tbl             The name of the table to look up in
* @param string $key_field       Name of the key field to look up
* @param string $display_fields  Name of the display fields to get data from
* @return mixed The returned value
*/
function resolveFK($fkey, $tbl, $key_field, $display_fields) {
  debug_trace("utils.php: resolveFK()");
  $res = "";
  $q = dbrecordset("SELECT * FROM $tbl WHERE $key_field = '$fkey'");
  if ($q->hasdata) {
    $dispflds = explode("|",$display_fields);
    for ($i=0; $i < count($dispflds); $i++) {
      if ($i > 0) $res .= " ";
      $fld = $dispflds[$i];
      if (substr($fld,0,1) == "#") $res .= substr($fld,1);
      else $res .= $q->field($fld);
    }
  }
  debug_trace();
  return $res;
} // resolveFK

// -----------------------------------------------------
/**
* DEPRECATED: This global utility function is now deprecated since the
* code is non-database-independent, and was written solely to support
* Postgresql sequences. Instead, please update your code to use the new
* 'get_next_sequencevalue()' global function, or 'next_sequencevalue()'
* methods on the 'dbupdate' and 'dbinsert' classes, or else use the new
* 'dbseq' class itself.
* Return next sequence value - get the next value of the given sequence
* @param string $seq The name of the sequence to get the next value of
* @return integer The next sequence value
*/
function next_sequencevalue($seq) {
  debug_trace("utils.php: next_sequencevalue()");
  $res = FALSE;
  $qseq = new dbrows("SELECT NEXTVAL('$seq')" );
  if ($qseq->valid) {
    $r = $qseq->get_current();
    $qseq->tidyup();
    $res = $r[0];
  }
  debug_trace();
  return $res;
} // next_sequencevalue

// -----------------------------------------------------
/**
* Detect a URL with the "xxx://" protocol prefix. Returns
* true if it does, else false.
* @param string $url URL to detect protocol prefix
* @return boolean True if it is a prefixed URL
*/
function protocol_prefixed($url){
  return (strpos($url, "://") !== false);
} // protocol_prefixed

// -----------------------------------------------------
/**
* Strip off any xxxx:// protocol prefix from a URL. Usually
* in Axyl, this is "http://", hence the name of this function.
* @param string $url URL to strip of protocol prefix
* @return string URL stripped of protocol prefix.
*/
function strip_http_prefix($url){
  $urlbits = explode("://", $url);
  return (count($urlbits) == 2) ? $urlbits[1] : $url;
} // strip_http_prefix

// -----------------------------------------------------
/**
* Make sure there is an http:// on a URL. If already
* present then string is returned untouched.
* @param string $url URL to add protocol prefix to
* @param string $ssl If true then add "https", else "http"
* @return string The URL prefixed with the protocol
*/
function add_http_prefix($url, $ssl=false){
  $s = strip_http_prefix($url);
  $prefix = ($ssl ? "https" : "http");
  $s = "$prefix://$s";
  return $s;
} // add_http_prefix

// -----------------------------------------------------
/**
* Add a parameter keyvalue pair to a URL. We check that
* it isn't already there, and use the right delimiter.
* @param string $href URL to append the new parm to
* @param string $pname Name of the parameter
* @param string $pvalue Value of the parameter
*/
function href_addparm($href, $pname, $pvalue) {
  if ($href != "") {
    $urlbits = explode("?", $href);
    $baseurl = $urlbits[0];
    $query = (isset($urlbits[1]) ? $urlbits[1] : "");
    $qvars = array();
    parse_str($query, $qvars);
    $qvars[$pname] = $pvalue;
    $q = array();
    foreach ($qvars as $name => $val) {
      $q[] = "$name=$val";
    }
    $query = implode("&", $q);
    $href = $baseurl . (($query != "") ? "?$query" : "");
  }
  return $href;
} // href_addparm

// -----------------------------------------------------
/**
* Remove a parameter keyvalue pair from a URL.
* @param string $href URL to remove the parm from
* @param string $pname Name of the parameter
*/
function href_delparm($href, $pname) {
  if ($href != "") {
    $urlbits = explode("?", $href);
    $baseurl = $urlbits[0];
    $query = (isset($urlbits[1]) ? $urlbits[1] : "");
    $qvars = array();
    parse_str($query, $qvars);
    if (isset($qvars[$pname])) {
      unset($qvars[$pname]);
    }
    $q = array();
    foreach ($qvars as $name => $val) {
      $q[] = "$name=$val";
    }
    $query = implode("&", $q);
    $href = $baseurl . (($query != "") ? "?$query" : "");
  }
  return $href;
} // href_delparm

// -----------------------------------------------------
/**
* Returns the name in the form SURNAME, Firstname.
* @param string $firstname First name
* @param string $lastname Last name(s)
* @return string The name in format SURNAME, Firstname
*/
function format_name($firstname, $lastname) {
  return strtoupper($lastname) . ", " . ucfirst($firstname);
} // format_name

// -----------------------------------------------------
/**
* Returns a value inside quotes. The type of quotes (single or
* double) are determined from the value content. If the content
* already has a double quote in it, then single quotes are used,
* else (default) double quotes are used.
*
* @return string The value string with appropriate quotes around it.
*/
function quoted_valuestring($val) {
  // Default the delimiter to double quote..
  $delim = chr(34);

  // Check content for quotes..
  $dq = strchr($val, chr(34)); // Double quote(s) in value
  $sq = strchr($val, chr(39)); // Single quote(s) in value
  // Both present in value, sanitise..
  if ($dq && $sq) {
    $val = str_replace(chr(34), chr(39), $val);
  }
  // Only double quotes in value, swap..
  elseif ($dq) {
    $delim = chr(39);
  }
  return $delim . $val . $delim;
} // quoted_valuestring

// -----------------------------------------------------
/**
* Returns a nicely formatted time string in '3d 12h 14m 33s'
* format, given a number of seconds. Leading elements are suppressed
* if they are zero.
*
* @param integer Number of seconds to convert to time string
* @return string A formatted time string eg: '12h 14m 33s'
*/
function nicetime($secs) {
  $days = (int) ($secs / 86400);
  $secs = $secs % 86400;
  $hrs  = (int) ($secs / 3600);
  $secs = $secs % 3600;
  $mins = (int) ($secs / 60);
  $secs = $secs % 60;
  $s = "";
  if ($days > 0) $s .= $days . "d ";
  if ($hrs > 0) $s .= $hrs . "h ";
  if ($mins > 0) $s .= $mins . "m ";
  $s .= $secs . "s";
  return $s;
} // nicetime

// -----------------------------------------------------
/**
* Returns a nicely formatted size string for displaying
* byte sizes. Eg. 1024 returns '1Kb', etc. Obviously this
* is a display-only string, not a number. You can specify
* the number of decimal places, and the thousands separator
* if you for example want nullstring instead of a comma.
*
* @param integer $bytes    Number of bytes.
* @param integer $decimals Number of decimal places to report
* @param string $thousep   Thousands separator
* @return string A formatted bytesize eg.
*/
function nicebytesize($bytes, $decimals=0, $thousep=",") {
  if ($bytes > GIGABYTE) {
    $s = number_format( ($bytes / GIGABYTE), $decimals, ".", $thousep) . "GB";
  }
  elseif ($bytes > MEGABYTE) {
    $s = number_format( ($bytes / MEGABYTE), $decimals, ".", $thousep) . "MB";
  }
  elseif ($bytes > KILOBYTE) {
    $s = number_format( ($bytes / KILOBYTE), $decimals, ".", $thousep) . "KB";
  }
  else {
    $s = number_format( $bytes, $decimals, ".", $thousep);
  }
  return $s;
} // nicebytesize

// -----------------------------------------------------------------------
/** Generate a precis from a bunch of text. The text can be of mostly
 * any format, and HTML etc. tags will be stripped as will newlines.
 * @param string $content The content to make the precis of 
 * @param integer $maxwords Maximum number of words for the precis
 * @param integer $minwords Minimum number of words for the precis
 * @return string The precis
 */
function make_precis($content, $maxwords=50, $minwords=5) {
  $precis = "";
  $patt = "(([\S]+[\s]+){" . $minwords . "," . $maxwords . "})";
  $matches = array();
  $content = scrub_string($content);
  preg_match("/$patt/", $content, $matches);
  if (isset($matches[1])) {
    $precis = $matches[1];
  }
  return $precis;
} // make_precis

// -----------------------------------------------------------------------
/**
 * Split a string of prose into more or less N equal parts, or into a
 * specified number of parts by percentage. Used for displaying news
 * stories in N columns. Optionally, care is taken to try and split on
 * a sentence-ending word (full-stop, exclamation, question-mark). This
 * will usually alter the percentages, if specified, but will always
 * tend to fill left-hand columns with more text, which looks better
 * than the other way around.
 * @param string $content The string content to split
 * @param mixed $chunks Number of chunks, or an array of percentages
 * @param boolean $split_on_sentences If true, try to split on sentence endings
 * @return array Array of N chunks of text
 */
function chunkify($content, $chunks, $split_on_sentences=true) {

  $retchunks = array();
 
  $allwords = array();
  $words = explode(" ", $content);
  foreach ($words as $word) {
    $word = ascii_string($word);
    $extrawords = explode(" ", $word);
    foreach ($extrawords as $extraword) {
      $allwords[] = $extraword;
    }
  }
  $nwords = count($allwords);
  
  // Generate percentage array, if N-chunks passed in
  if (!is_array($chunks)) {
    $newchunks = array();
    if (intval($chunks) == 0) {
      $chunks = 2;
    }
    $basepct = intval(ceil((1 / $chunks) * 100));
    $totpct = 0;
    for ($i = 0; $i < ($chunks - 1); $i++) {
      $newchunks[] = $basepct;
      $totpct += $basepct;
    }
    $newchunks[] = 100 - $totpct;
    $chunks = $newchunks;
  }    
  $nchunks = count($chunks);

  // Determine chunk starts in #words offsets
  $chunkstarts = array();
  $start_chunkpos = 0;
  $end_chunkpos = -1;
  foreach ($chunks as $chunkpct) {
    $chunkwords = intval(ceil(($nwords * $chunkpct)/100));
    $start_chunkpos = $end_chunkpos + 1;
    $end_chunkpos   = $start_chunkpos + $chunkwords - 1;
    if ($split_on_sentences) {
      for ($i = 0; $i < 30; $i++) {
        $word = $allwords[$end_chunkpos + $i];
        $lastchar = substr($word, -1);
        if (($end_chunkpos + $i) >= $nwords || preg_match("/[.?!\r\n<]/", $lastchar)) {
          $end_chunkpos += $i;
          break;
        }
      }
    }
    $chunkstarts[] = $start_chunkpos;
  }    
  $chunkstarts[] = $nwords;

  // Populate return array
  if (count($chunkstarts) > 0) {
    for ($i = 0; $i < $nchunks; $i++ ) {
      $s = "";
      for ($j = $chunkstarts[$i]; $j < $chunkstarts[$i + 1]; $j++) {
        $s .= $allwords[$j] . " ";
      }
      $retchunks[] = $s;      
    }
  }
  return $retchunks;
    
} // chunkify
  
// ----------------------------------------------------------------------
/**
 * Replace 'hard' paragraph markers (double newlines) with HTML paragaph
 * tags - '<p>'. You can optionally change the hard paragraph marker
 * pattern from the default of two contiguous CRLFs: '\r\n\r\n'.
 * Note: we add a leading space to preserve 'word' separation.
 * @param string $s The string to make the replacements in
 * @param string $regex The regular expression which marks paragraphs
 */
function paras2ptags($s, $regex="\r\n\r\n|<br \/><br \/>") {
  $s = str_replace("</p>", "", $s);
  return preg_replace("/$regex/", " <p>", trim($s));
} // paras2ptags

// ----------------------------------------------------------------------
/**
 * This is a right little scrubber. It performs some coommon cleanup
 * of a bunch of text which you want to be 'vaniila'. It removes line
 * endings, all markup tags (such as HTML), plus some 'funny' chars.
 * The 'ascii_only' option will remove ALL hi-value (>127) characters.
 * @param string $s The string to scrub
 * @param string $eol The string to replace end-of-lines with
 * @param boolean $asscii_only If true, remove all chars >127
 */
function scrub_string($s, $eol=" ", $ascii_only=false) {
  $res = $s;
  if ($res != "") {
    # apply some standard transforms
    $res = strip_tags($s);
    $res = ereg_replace( "[\r\n]+", $eol, $res); # line-ends
    $res = ereg_replace( "[\xAD]+", "", $res);  # soft hyphens
    # also deal to any hi-value chars, if requested
    if ($ascii_only) {
      $res = ascii_string($res);
    }
  }
  return $res;
} // scrub_string

// ----------------------------------------------------------------------
/**
 * Given a string, scan it for 'wikitable' formatting pattern and replace
 * these with the appropriate constructs for HTML table rendering
 * Wikitable formats (Axyl style) are exemplified here:
 * Eg. of a table with no heading row:
 * {table
 * |row1-cellA|row1-cellB|
 * |row2-cellA|row2-cellB|
 * }
 * Eg. of a table with a heading row:
 * {table
 * ^headingRow-CellA^headingRow-CellB^
 * |row1-cellA|row1-cellB|
 * |row2-cellA|row2-cellB|
 * }
 * Eg. of a table with a heading row spanning all columns, and a
 * column width setting:
 * {table
 * ~60%~40%~
 * #spanning heading##
 * |row1-cellA|row1-cellB|
 * |row2-cellA|row2-cellB|
 * }
 * Eg. of a table with custom width and alignment settings, a class
 * and a row which spans all columns. Notice that each of the properties
 * in the first line are terminated with colons ';'. These are required.
 * {table width=350px; align=center; class=mytable;
 * #spanning heading##
 * @spanning cell content@@
 * |row1-cellA|row1-cellB|
 * |row2-cellA|row2-cellB|
 * }
 * @param string $wt The string which may contain some wikitable(s)
 * @return string The same string with wikitable replaced by HTML table
 */
function wikitable2html($wt, $css="") {
  if ($wt != "" && strstr($wt, "{")) {
    $wt = preg_replace("/<\/p>/i", "", $wt);
    $wt = preg_replace("/<p>/i", "<br /><br />", $wt);
    $wt = preg_replace("/[\n]|<br>/i", "<br />", $wt);
    $rows = explode("<br />", $wt);
    $newrows = array();
    $intable = false;
    $wprofile = "";
    $t_align = "";
    foreach ($rows as $row) {
      if (!$intable) {
        if (strstr($row, "{table")) {
          $t = new table("wikitable2html", $css);
          if (preg_match("/width=(.*?);/", $row, $matches)) {
            $t->setwidth($matches[1]);
          }
          if (preg_match("/align=(.*?);/", $row, $matches)) {
            $t->setalign($matches[1]);
            $t_align = $matches[1];
          }
          if (preg_match("/style=(.*?);/", $row, $matches)) {
            $t->setstyle($matches[1]);
          }
          if (preg_match("/class=(.*?);/", $row, $matches)) {
            $t->setclass($matches[1]);
          }
          $t->setpadding(3);
          $wprofile = "";
          $intable = true;
        }
        else {
          $newrows[] = $row;
        }
      }
      else {
        $rowtype = substr($row, 0, 1);
        $row = substr($row, 1);
        $row = substr($row, 0, -1);
        switch ($rowtype) {
          #end of table definition
          case "}":
            if ($wprofile != "") {
              $t->set_width_profile($wprofile);
            }
            $t->autojustify();
            $t_html = $t->render(); 
            if ($t_align != "") {
              $t_html = "<div style=\"text-align:$t_align\">" . $t_html . "</div>";
            }
            $newrows[] = $t_html;
            $intable = false;
            break;
          #width profile
          case "~":
            $wprofile = trim(str_replace("~", ",", $row));
            break;
          #column headings
          case "^":
            $t->thead("axyl_rowstripe_dark");
            $t->tr();
            $cells = explode("^", $row);
            foreach ($cells as $cell) {
              $t->th($cell);
              $t->th_contentcss($css);
            }
            $t->tbody();
            break;
          #single heading spanning all columns
          case "#":
            $t->thead("axyl_rowstripe_dark");
            $t->tr();
            $cells = explode("#", $row);
            $t->th($cells[0]);
            $t->th_contentcss($css);
            $t->th_colspan( count($cells) );
            $t->tbody();
            break;
          #single cell spanning all columns
          case "@":
            $t->tr();
            $cells = explode("@", $row);
            $t->td($cells[0]);
            $t->td_colspan( count($cells) );
            $t->td_alignment("", "top");
            $t->td_contentcss($css);
            break;
          #standard table cells
          case "|":
            $t->tr();
            $cells = explode("|", $row);
            foreach ($cells as $cell) {
              if (trim($cell) == "") {
                $cell = "&nbsp;";
              }
              $t->td($cell);
              $t->td_alignment("", "top");
              $t->td_contentcss($css);
            }
            break;
        } // switch
      }
    }
    $s = implode("<br />", $newrows);
    $s = str_replace("<br /><br />", "<p>", $s);
  }
  else {
    $s = $wt;
  }
  return $s;
} // wikitable2html

// ----------------------------------------------------------------------
/**
 * Removes ALL hi-value (>127) characters in the given string and
 * returns the result.
 * @param string $s The string to scrub
 * @return string An ascii-only representation of the string
 */
function ascii_string($s) {
  $res = "";
  if ($s != "") {
    for ($i = 0; $i < strlen($s); $i++) {
      $ch = substr($s, $i, 1);
      if (ord($ch) <= 127 || ord($ch) >= 20) {
        $res .= $ch;
      }
    }
  }
  return $res;
} // ascii_string

// -----------------------------------------------------
// Compat funcs - ie. if the Php version doesn't provide these
// then we implement them ourselves..

if (!function_exists("pg_escape_string")) {
  function pg_escape_string($str) {
    return addslashes($str);
  }
}

/**
 * Return a copy (clone) of the object.
 * The Php5 assignment operator returns object handles (refs) instead of
 * the value of the object. Php5 provides the 'clone' function (can also be
 * used as a keyword like 'new') to return a cloned copy, hence this, to
 * provide compatibility code for Php4. Whenever we require a copy of an
 * object we should write it like this:
 *   $my_cloned_obj = clone($obj);
 * @param object $object The object you want to be copied/cloned
 * @return object The object copy (same object)
 */ 
if (version_compare(phpversion(), '5.0') < 0) {
  eval('function clone($object) {
          return $object;
        }'
  );
}
// -----------------------------------------------------
?>