<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    json-defs.php                                           */
/* Author:      Paul Waite                                              */
/*              (from code by Cesar D. Rodas)                           */
/*                                                                      */
/* Description: Definitions for converting to/from JSON formatted data. */
/*                                                                      */
/* ******************************************************************** */

define('STATE_NOWHERE',  0);
define('STATE_STRING',   1);
define('STATE_OBJECT',   2);
define('STATE_ATOMIC',   3);
define('STATE_ASSIGN',   4);
define('STATE_ENDSTMT',  5);
define('STATE_ARRAY',    6);

/**
 * Constructor.
 */
class JSONconverter {
  /** Error code, if any. */
  var $error = false;
  
  function JSONconverter() {
    $this->error = false;
  } // JSONconverter
  // ....................................................................
  /**
   * Jsonize a Php object or array. 
   * param mixed $obj Object or array to convert into JSON.
   * return string JSON.
   */
  function toJSON($obj) {
    if (is_object($obj) ) {
      $e = get_object_vars($obj);
      $properties = array();
      foreach ($e as $k => $v) {
        $properties[] = $this->_convert_to_json($k, $v);
      }
      return "{" . implode(",", $properties) . "}";
    }
    elseif (is_array($obj)) {
      return $this->_convert_to_json("", $obj);
    }
  } // toJSON
  // ....................................................................
  /**
   * Transform an JSON text into a PHP object
   * and return it.
   * @access public 
   * @param string $text JSON text
   * @return mixed PHP Object, array or false.
   */
  function fromJSON($text) {
    $this->error = false;
     return !$this->error ? $this->_convert_from_json($text) : false;
  } // fromJSON  
  // ....................................................................
  /**
   * Transform an JSON text into a PHP object
   * and return it.
   * @access private 
   * @param string $text JSON text
   * @return mixed PHP Object, array or false.
   */
  function _convert_from_json($text) {
    $ret = new stdClass;
     
    while (  $f = $this->getNextToken($text, $i, $type)  ) {
      switch ( $type ) {
        case STATE_ARRAY:
          $tmp = $this->_convert_from_jsonArray($text);
          $ret = $tmp[0];
          break;
          
        case STATE_OBJECT:
          $g=0;
          do  {
            $varName = $this->getNextToken($f, $g, $xType);
            if ($xType != STATE_STRING)  {
                return false; /* error parsing */
            }
            $this->getNextToken($f,$g,$xType);
            if ($xType != STATE_ASSIGN) {
              return false;
            }
            $value = $this->getNextToken($f, $g, $xType);
            
            if ($xType == STATE_OBJECT) {
              $ret->$varName = $this->fromJSON( "{" . $value . "}" );
              $g--;
            }
            elseif ($xType == STATE_ARRAY) {
              $ret->$varName = $this->_convert_from_jsonArray($value);
              $g--;
            }
            else {
              $ret->$varName = $value;
            }
            $this->getNextToken($f, $g, $xType);
            
          } while ( $xType == STATE_ENDSTMT);
          break;
          
        default:
          $this->error = true;
          break 2;
      }
    }
    return $ret;
  } // _convert_from_json
  // ....................................................................
  /**
   * Array Parser. Transform a json-array into a PHP array
   * @access private
   * @param string $text String to parse
   * @return Array PHP Array
   */
  function _convert_from_jsonArray($text) {
    $r = array();
    do {
      $f = $this->getNextToken($text, $i, $type);
      switch ($type) {
        case STATE_STRING:
        case STATE_ATOMIC:
          $r[] = $f;
          break;
          
        case STATE_OBJECT:
          $r[] = $this->fromJSON("{" . $f . "}");
          $i--;
          break;
          
        case STATE_ARRAY: 
          $r[] = $this->_convert_from_jsonArray($f);
          $i--;
          break;
      }
      $this->getNextToken($text, $i, $type);
    } while ($type == STATE_ENDSTMT);
    
    return $r;
  } // _convert_from_jsonArray
  // ....................................................................
  /**
   *  Tokenizer. Return to the Parser the next valid token and the type     
   *  of the token. If the tokenizer fails it returns false.
   *  @access private
   *  @param string $e Text to extract token
   *  @param integer $i  Start position to search next token
   *  @param integer $state Variable to get the token type
   *  @return string|bool Token in string or false on error.
   */
  function getNextToken($e, &$i, &$state) {
    $state = STATE_NOWHERE;
    $end = -1;
    $start = -1;
    while ( $i < strlen($e) && $end == -1 ) {
      switch( $e[$i] ) {
        /* objects */
        case "{":
        case "[":
          $_tag = $e[$i]; 
          $_endtag = $_tag == "{" ? "}" : "]";
          if ( $state == STATE_NOWHERE ) {
            $start = $i+1;
            switch ($state) {
              case STATE_NOWHERE:
                $aux = 1; /* for loop objects */
                $state = $_tag == "{" ? STATE_OBJECT : STATE_ARRAY;
                break;
                
              default:
                break 2; /* exit from switch and while */
            } // switch
            while ( ++$i && $i < strlen($e) && $aux != 0 ) {
              switch( $e[$i] ) {
                case $_tag:
                  $aux++;
                  break;
                  
                case $_endtag:
                  $aux--;
                  break;
              } // switch
            }
            $end = $i-1;
          }
          break;
        
        case '"':
        case "'":
            $state = STATE_STRING;
            $buf = "";
            while ( ++$i && $i < strlen($e) && $e[$i] != '"' ) {
              if ( $e[$i] == "\\") { 
                $i++;
              }
              $buf .= $e[$i];
            }
            $i++;
            return eval('return "'.str_replace('"','\"',$buf).'";');
            break;
            
        case ":":
            $state = STATE_ASSIGN;
            $end = 1;
            break;
            
        case "n":
            if ( substr($e, $i, 4) == "null" ) {
                $i=$i+4;
                $state = STATE_ATOMIC;
                return NULL;
            }
            else {
              break 2; /* exit both switch and while */
            }
            
        case "t":
            if ( substr($e, $i, 4) == "true") {
                $state = STATE_ATOMIC;
                $i=$i+4;
                return true;
            }
            else {
              break 2; /* exit from switch and while */
            }
            break;
            
        case "f":
            if ( substr($e, $i, 4) == "false") {
                $state = STATE_ATOMIC;
                $i=$i+4;
                return false;
            }
            else break 2; /* exit both switch and while */
            break;
                
        case ",":
            $state = STATE_ENDSTMT;
            $end = 1;
            break;
            
        case " ":
        case "\t":
        case "\r":
        case "\n":
            break;
        case "+":
        case "-":
        case 0:
        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
        case 8:
        case 9:
        case '.':
            $state = STATE_ATOMIC;
            $start = (int)$i;
            if ( $e[$i] == "-" || $e[$i] == "+") {
                $i++;
            }
            for ( ;  $i < strlen($e) && (is_numeric($e[$i]) || $e[$i] == "." || strtolower($e[$i]) == "e") ;$i++) {
                $n = $i+1 < strlen($e) ? $e[$i+1] : "";
                $a = strtolower($e[$i]);
                if ( $a == "e" && ($n == "+" || $n == "-"))
                    $i++;
                else if ( $a == "e") 
                    $this->error = true;
            } // for
            $end = $i;
            break 2; /* break while too */
        default: 
            $this->error = true;
      }
      $i++;
    }
    return $start == -1 || $end == -1 ? false : substr($e, $start, $end - $start);
  } // getNextToken
  // ....................................................................  
  /** 
   * Internal converter from php to JSON
   * @param string $key Variable name
   * @param mixed $value Value of the variable
   * @access private
   * @return string Serialized variable
   */
  function _convert_to_json($key="", &$value) {
      $r = "";
      if ( $key != ""){
        $r .= "\"${key}\" : ";
      }
      if ( is_numeric($value) ) {
        $r .= $value;
//        $r .= '"' . $this->toString($value) . '"';
      }
      else if (is_string($value) ) {
        $r .= '"' . $this->toString($value) . '"';
      }
      else if (is_object($value) ) {
        $r .= $this->toJSON($value);
      }
      else if (is_null($value) ) {
        $r .= "null";
      }
      else if (is_bool($value) ) {
        $r .= $value ? "true":"false";
      }
      else if (is_array($value) ) {
        $f = array();
        foreach ($value as $k => $v) {
          $f[] = $this->_convert_to_json("", $v);
        }
        $r .= "[" . implode(",", $f) . "]";
        unset($f);
      }
      return $r;
  } // _convert_to_json
  // ....................................................................  
  /** 
   * Convert String variables
   * @param string $e Variable with an string value
   * @access private
   * @return string Serialized variable
   */
  function toString($e) {
      $rep = array("\\","\r","\n","\t","'",'"');
      $val = array("\\\\",'\r','\n','\t','\'','\"');
      $e = str_replace($rep, $val, $e);
      return $e;
  } // toString

} // class JSONconverter
?>