<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    debugger.php                                            */
/* Author:      Paul Waite                                              */
/* Description: Definitions for the debugger module.                    */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

// Define classes of debug message here. This allows
// some control over what messages appear on output..

/** Dummy value */
define("DBG_UNDEFINED",  -1);
/** No debugging (redundant) */
define("DBG_NONE",        0);
/** Ad-hoc debugging output */
define("DBG_DEBUG",       1);
/** Diagnostic output */
define("DBG_DIAGNOSTIC",  2);
/** SQL queries to database */
define("DBG_SQL",         4);
/** SQL SELECT data from database */
define("DBG_SQLDATA",     8);
/** Dump HTTP and PHP page vars */
define("DBG_DUMP",       16);
/** Provide debug traceback info */
define("DBG_TRACE",      32);
/** Show table borders and validation checks */
define("DBG_TABLES",     64);
/** Show debugging profiler output */
define("DBG_PROFILE",   128);
/** Show site authentication output */
define("DBG_AUTH",      256);
/** DEBUG and DIAGNOSTICS as default setting */
define("DBG_DEFAULT",     3);
/** Everything. Warning, this can be VERBOSE! */
define("DBG_ALL",       511);

// Defined output options. Where debug output goes..
/** No output */
define("DBG_O_NONE",      0);
/** Output stored in $content */
define("DBG_O_STORED",    1);
/** Output via direct echo */
define("DBG_O_ECHO",      2);
/** Output for CLI: echoed raw ASCII, LF end-of-line */
define("DBG_O_CLI",       4);
/** Output to system log */
define("DBG_O_LOG",       8);
/** To logfile (not implemented yet) */
define("DBG_O_LOGFILE",  16);
/** Default output mode (stored) */
define("DBG_O_DEFAULT",   1);
/** Ubiquitous output mode */
define("DBG_O_ALL",     255);

//-----------------------------------------------------------------------
/**
* The debugger class. Responsible for accumulating, filtering and then
* rendering debug content to various outputs. This class is automatically
* instantiated by the system as $RESPONSE->debugger; and is used by various
* modules (eg. query) to log information. It can also be used as either
* an ad-hoc debugging aid, or as an analysis tool where debug output
* statements are inserted into code and left there permanently, their output
* only switched on when problems arise. Various classes of debug output can
* be enabled in any combination by OR'ing the following constants when the
* debug_on() function is called:
*    DBG_DEBUG          Ad-hoc debugging output
*    DBG_DIAGNOSTIC     Diagnostic output
*    DBG_SQL            SQL queries to database
*    DBG_SQLDATA        SQL SELECT data from database
*    DBG_DUMP           Dump HTTP and PHP page vars
*    DBG_TRACE          Provide debug traceback info
*    DBG_TABLES         Show table borders and validate cells
*    DBG_PROFILES       Show process profiling information
*    DBG_AUTH           Authentication messages
*    DBG_DEFAULT        DEBUG and DIAGNOSTICS only
*    DBG_ALL            Everything (verbose!)
* eg. debug_on(DBG_SQL | DBG_DIAGNOSTIC | DBG_DUMP);
* @package core
*/
class webdebugger extends RenderableObject {
  // Public
  /** Accumulated debugging output content */
  var $content = "";
  /** Classes of debugging to accumulate */
  var $debugclass = DBG_DEFAULT;
  /** Status of the debugger */
  var $enabled = false;
  /** Modes of output for debugging content */
  var $output_mode = DBG_O_DEFAULT;

  // Private
  /** Array of traced locations
      @access private */
  var $trace = array();
  /** Depth of tracing
      @access private */
  var $tracedepth = 0;
  /** Microtimer object for profiling
      @access private */
  var $profile_timer;
  /** Profiler data array
      @access private */
  var $profile = array();

  // ................................................................
  /**
  * Constructor
  * @param integer $debugclass The classes of output to render
  */
  function webdebugger($debugclass=DBG_DEFAULT) {
    $this->debug_class($debugclass);
  }
  // ................................................................
  /**
  * Return whether we have any content, true or false.
  */
  function debug_hascontent() {
    return (
      $this->enabled &&
      (
        (strlen($this->content) > 0)
        || ($this->debugclass & DBG_DUMP)
        || ($this->profiling() && count($this->profile) > 0)
      )
    );
  }
  // ................................................................
  /**
  * Set debug output mode
  * This function starts/stops the debugger from echoing
  * the content directly as well as storing it for a
  * subsequent display by webpage-defs.php. The values which
  * can be OR'ed together are as follows:
  *    DBG_O_NONE       No output
  *    DBG_O_STORED     Output stored in $content
  *    DBG_O_ECHO       Output via direct echo
  *    DBG_O_CLI        Output for CLI: echoed tag-stripped ASCII, LF end-of-line
  *    DBG_O_LOG        Output to system log
  *    DBG_O_LOGFILE    To logfile (not implemented yet)
  *    DBG_O_DEFAULT    Default output mode (echoed)
  *    DBG_O_ALL        Ubiquitous output mode
  * @param integer $mode  Output mode of debugger
  */
  function debug_output($mode=DBG_O_DEFAULT) {
    $this->output_mode = $mode;
  }
  // ................................................................
  /**
  * Set debug output mode
  * Register content for debugging output. Check that it is
  * of an acceptable class and direct it all to the right place(s).
  * @param string  $content Content to add to debug output buffer
  * @param integer $debugclass Class of output
  */
  function debug_write($content, $debugclass=DBG_DIAGNOSTIC) {
    if ($this->enabled) {
      // Are we providing traceback info..
      if ($this->debugclass & DBG_TRACE) {
        if ($this->tracedepth > 0) {
          $content .= "[ " . $this->traceback() . " ] ";
        }
      }
      // Direct debugging content where we should..
      if ($debugclass & $this->debugclass) {
        if ($this->output_mode & DBG_O_STORED) {
          $this->content .= $content;
        }
        if ($this->output_mode & DBG_O_ECHO) {
          echo $content;
        }
        if ($this->output_mode & DBG_O_CLI) {
          $s = str_replace("<br>", "\n", html_entity_decode($content));
          if (!strstr($content, "<pre>")) {
            $s =  strip_tags($s);
          }
          echo $s;
        }
        if ($this->output_mode & DBG_O_LOG) {
          error_log(strip_tags(html_entity_decode($content)), 0);
        }
      }
    }
    return $this;
  }
  // ................................................................
  /**
  * Set debug class(es) to accept
  * @param integer $debugclass Class of output
  */
  function debug_class($debugclass) {
    if (isset($debugclass)) {
      $this->debugclass = $debugclass & DBG_ALL;
      if ($this->debugclass == DBG_NONE)
        $this->enabled = false;
    }
    return $this;
  }
  // ................................................................
  /**
  * Set debug mode
  * Set debug output to be enabled(default)/disabled
  * @param boolean  $enabled  If true enable debugger, else disable it
  * @param integer $debugclass     Class of output to accept
  */
  function debug_mode($enabled=true, $debugclass=DBG_UNDEFINED) {
    $this->enabled = $enabled;
    if ($enabled) {
      $this->debug_class($debugclass);
    }
    return $this;
  }
  // ................................................................
  /**
  * Flag debugger to dump global variables
  * Sets a flag for the debugger to dump all global vars at output time.
  * @access private
  */
  function debug_dump() {
    $this->debugclass |= DBG_DUMP;
    return $this;
  }
  // ................................................................
  /**
  * Profile breakpoint.
  * Profiling allows you to ascertain the elapsed time between two
  * breakpoints. To do this just call this method twice with the same
  * label.
  * @param string $label A label for this profile breakpoint
  */
  function profile($label) {
    if (!isset($this->profile_timer)) {
      $this->profile_timer = new microtimer();
      $this->profile_timer->start();
    }
    $ms = $this->profile_timer->millisecs();
    if (isset($this->profile[$label])) {
      $bits = explode("|", $this->profile[$label]);
      $elapsed  = $bits[0];
      $started  = $bits[1];
      $finished = $bits[2];
      $calls    = $bits[3];
      // Ending a profiling section
      if ($finished == "") {
        $finished = $ms;
        $elapsed += ($finished - $started);
        $calls += 1;
      }
      // Starting a profiling section..
      else {
        $started = $ms;
        $finished = "";
      }
    }
    else {
      $elapsed = 0;
      $started = $ms;
      $finished = "";
      $calls = 0;
    }
    $this->profile[$label] = "$elapsed|$started|$finished|$calls";
  }
  // ................................................................
  /** This makes sure the profiles are all finished.
  * @access private
  */
  function profile_close() {
    if (isset($this->profile_timer)) {
      $this->profile_timer->stop();
      foreach ($this->profile as $label => $prof) {
        $bits = explode("|", $prof);
        if ($bits[2] == "") {
          $this->profile($label);
        }
      }
    }
  }
  // ................................................................
  /** Returns true if this debugger is in profiling mode. */
  function profiling() {
    return $this->debugclass & DBG_PROFILE;
  }
  // ................................................................
  /**
  * List classes debugger is accepting
  * @return string List of classes acceptable to the debugger
  */
  function report_classes() {
    $rpt = array();
    if ($this->debugclass & DBG_TRACE)      $rpt[] = "TRACEBACK";
    if ($this->debugclass & DBG_SQL)        $rpt[] = "SQL";
    if ($this->debugclass & DBG_DEBUG)      $rpt[] = "DEBUG";
    if ($this->debugclass & DBG_DIAGNOSTIC) $rpt[] = "DIAGNOSTIC";
    if ($this->debugclass & DBG_DUMP)       $rpt[] = "DUMP";
    if ($this->debugclass & DBG_TABLES)     $rpt[] = "TABLES";
    if ($this->debugclass & DBG_PROFILE)    $rpt[] = "PROFILER";
    if ($this->debugclass & DBG_AUTH)       $rpt[] = "AUTH";
    return implode(",", $rpt);
  }
  // ................................................................
  /**
  * Push a trace on stack
  * Pushes a traceback label onto our trace stack. We return the
  * depth of trace we are at.
  * @param  mixed $traceobj  Thing to push on the stack
  * @return integer  Depth of tracing so far
  * @access private
  */
  function pushtrace($traceobj) {
    if ($this->debugclass & DBG_TRACE) {
      // First determine the trace label..
      if (is_object($traceobj)) {
        $label = "<font color=blue>" . get_class($traceobj) . "</font>";
      }
      elseif (is_string($traceobj)) {
        $label = "<font color=green>" . $traceobj . "</font>";
      }
      else {
        $label = "unknown";
      }
      array_push($this->trace, $label);
      $this->tracedepth = count($this->trace);
      // Automatic trace profiling..
      if ($this->profiling()) {
        $this->profile(strip_tags($label));
      }
    }
    return $this->tracedepth;
  }
  // ................................................................
  /**
  * Pop a trace off stack
  * Pop a trace label off our trace stack
  * @access private
  */
  function poptrace() {
    if ($this->debugclass & DBG_TRACE) {
      if ($this->tracedepth > 0) {
        $label = array_pop($this->trace);
        if ($this->profiling()) {
          $this->profile(strip_tags($label));
        }
        return $label;
      }
      else {
        return "";
      }
      $this->tracedepth = count($this->trace);
    }
  }
  // ................................................................
  /**
  * Traceback list
  * Return a string of the form "label->label->label"
  * forming the traceback level labels in order.
  * @return string  Traceback list
  */
  function traceback() {
    $traceback = "";
    if ($this->tracedepth > 0) {
      $ditto = "";
      foreach ($this->trace as $tracelabel) {
        if ($tracelabel != $ditto) {
          if ($traceback != "") $traceback .= "->";
          $traceback .= $tracelabel;
          $ditto = $tracelabel;
        }
      }
    }
    return $traceback;
  }
  // ................................................................
  /**
  * Use render() to render the debugger output.
  * Renders the debugger output content as HTML.
  * @see render()
  * @return string  HTML rendering of debug output content
  */
  function html() {
    if (!$this->debug_hascontent()) {
      return "";
    }
    // Header stuff.,
    $header  = "<h4>Axyl Debug Output</h4>";
    $header .= "<p>";
    $header .= "<b>File: $PHP_SELF</b><br>";
    $header .= "<b>Debug filter: " . $this->report_classes() . "</b><br>";
    $header .= "</p>";
    $footer = "+++ END OF DEBUG OUTPUT +++<br>";
    if ($this->debugclass & DBG_DUMP) {
      $s  ="<p>";
      $s .= "<table border=1 cellpadding=2 cellspacing=0>";
      if (isset($_COOKIE)) {
        $s .= "<tr><td colspan=2><h4>Cookie Vars</h4></td></tr>";
        foreach ($_COOKIE as $key => $val) {
          $s .= "<tr><td>$key</td><td>" . displayvar($val) . "</td></tr>";
        }
      }
      if (isset($_GET)) {
        $s .= "<tr><td colspan=2><h4>GET Vars</h4></td></tr>";
        foreach ($_GET as $key => $val) {
          $s .= "<tr><td>$key</td><td>" . displayvar($val) . "</td></tr>";
        }
      }
      if (isset($_POST)) {
        $s .= "<tr><td colspan=2><h4>POSTed Vars</h4></td></tr>";
        foreach ($_POST as $key => $val) {
          $s .= "<tr><td>$key</td><td>" . displayvar($val) . "</td></tr>";
        }
      }
      if (isset($_SESSION)) {
        $s .= "<tr><td colspan=2><h4>Session Vars</h4></td></tr>";
        foreach ($_SESSION as $key => $val) {
          $s .= "<tr><td>$key</td><td>" . displayvar($val) . "</td></tr>";
        }
      }
      if (isset($_FILES)) {
        $s .= "<tr><td colspan=2><h4>Uploaded Files</h4></td></tr>";
        foreach ($_FILES as $key => $val) {
          $s .= "<tr><td>$key</td><td>" . displayvar($val) . "</td></tr>";
        }
      }
      $s .= "</table>";
      $s .= "</p>";
      $this->content .= $s;
    }
    // Profiler output..
    if ($this->profiling()) {
      $s  ="<p>";
      $s .= "<table border=1 cellpadding=2 cellspacing=0>\n";
      $s .= "<tr><td colspan=6><h4>Profiler</h4></td></tr>\n";
      $s .= "<tr>";
      $s .= "<td>Label</td>";
      $s .= "<td align=center>Calls</td>";
      $s .= "<td align=center>mS/Call</td>";
      $s .= "<td align=center>mS</td>";
      $s .= "<td align=right>Rel%</td>";
      $s .= "<td>&nbsp</td>";
      $s .= "</tr>\n";

      // Make sure all profiles are finished..
      $this->profile_close();

      // Find max value first..
      $maxms = 0;
      $profiles = array();
      $calls = array();
      foreach ($this->profile as $label => $prof) {
        $bits = explode("|", $prof);
        if (isset($bits[0])) {
          $ms = (float) $bits[0];
          $profiles[$label] = $ms;
          if ($ms > $maxms) $maxms = $ms;
          $calls[$label] = $bits[3];
        }
      }
      $maxwidth = 50;
      if ($maxms > 0) {
        $scalefactor = $maxwidth / $maxms;
      }
      else {
        $scalefactor = 1;
      }
      //arsort($profiles);
      foreach ($profiles as $label => $ms) {
        $calltot = $calls[$label];
        $mspercall = number_format($ms/$calltot, 1);
        $width = (int) ceil($scalefactor * $ms);
        $pct = number_format(($ms / $maxms) * 100, 1);
        $bar = str_repeat("*", $width);
        $ms = number_format($ms, 2);
        $s .= "<tr>";
        $s .= "<td>$label</td>";
        $s .= "<td align=right>$calltot</td>";
        $s .= "<td align=right>$mspercall</td>";
        $s .= "<td align=right>$ms</td>";
        $s .= "<td align=right>$pct</td>";
        $s .= "<td valign=middle>$bar</td>";
        $s .= "</tr>\n";
      }
      $s .= "</table>\n";
      $s .= "</p>";
      $this->content .= $s;
    }
    return $header . $this->content . $footer;
  }
  // ................................................................
  /**
  * Use render() to render the debugger output.
  * Renders the debugger output content as WML.
  * @see render()
  * @return string  WML rendering of debug output content
  */
  function wml() {
    if (!$this->debug_hascontent()) return "";

    // Get names of globals we will use. These depend on the
    // version of Php, since we won't rely on the old ones
    // sticking around forever..
    if (version_compare(phpversion(), "4.1.0", "ge")) {
      $vname_get = "_GET";
      $vname_post = "_POST";
    }
    else {
      $vname_get = "HTTP_GET_VARS";
      $vname_post = "HTTP_POST_VARS";
    }
    global $$vname_get;
    global $$vname_post;
    global $GLOBALS;
    $s = "";
    if ($this->debugclass & DBG_DUMP) {
      $s .= "<b>Variables</b><br/>";
      if (isset($$vname_get)) {
        $s .= "GET Vars:<br/>";
        reset($$vname_get);
        while (list($key, $val) = each($$vname_get)) {
          $s .= "$key=" . displayvar($val) . "<br/>";
        }
      }
      if (isset($$vname_post)) {
        $s .= "POSTed Vars:<br/>";
        reset($$vname_post);
        while (list($key, $val) = each($$vname_post)) {
          $s .= "$key=" . displayvar($val) . "<br/>";
        }
      }
      if (isset($GLOBALS)) {
        $s .= "Globals:<br/>";
        reset($GLOBALS);
        while (list($key, $val) = each($GLOBALS)) {
          $s .= "$key=" . displayvar($val) . "<br/>";
        }
      }
    }
    // Massage content to make it vanilla WML compliant..
    $content = $this->content . $s;
    $content = str_replace("<br>", "<br/>", $content);
    $content = strip_tags($content, "<br/><p><b><small>");
    $this->content = $content;
    return $this->content;
  }
  // ................................................................
  /**
  * Send debug output to a file.
  * @param string $name  Filename to put output into
  * @param string $dir   Directory/path to put file into
  */
  function send_to_file($name, $dir="") {
    if (!$this->debug_hascontent()) return;
    $s = $this->render();
    $f = new outputfile($name, $dir);
    if ($f->opened) {
      $f->write($this->content);
      $f->closefile();
    }
  }
} // webdebugger class

// ------------------------------------------------------------------
// FUNCTIONS DEBUGGER INTERFACE
// Debugging functions. These are just an interface
// to our debugger for the client..
// ------------------------------------------------------------------
/**
* Add content to debug content
* Adds the string content to the debugger output. This is done
* in raw fashion without any <br> or linefeed chars appended.
* The output will then appear as per the output settings.
* @see debugbr()
* @param string  $content  Content to add to debug output buffer
* @param integer $debugclass    Class of output
*/
function debug($content, $debugclass=DBG_DIAGNOSTIC) {
  global $RESPONSE;
  if (isset($RESPONSE)) {
    $RESPONSE->debugger->debug_write($content, $debugclass);
  }
}
// ..................................................................
/**
* Add content to debug content with <br>
* Adds the string content to the debugger output and appends
* <br> to it. This is intended for output to HTML pages.
* @see debug()
* @param string  $content  Content to add to debug output buffer
* @param integer $debugclass    Class of output
*/
function debugbr($content, $debugclass=DBG_DIAGNOSTIC) {
  debug($content . "<br>", $debugclass);
}
// ..................................................................
/**
* Add string to debug content as hexdump
* Add string content to output as a hexdump. This is for
* string data only. use this when you want to see 'inside'
* a string variable and view the characters as hexadecimal
* highlighted according to hex range. Chars are highlighted
* as follows:
*     ASCII value  = 32 (Spaces) ......... Blue
*     ASCII values < 32 (Control chars) .. Red
*     ASCII values > 127 ................. Green
* @param string  $str    String to view as hex
* @param string  $msg    Message to title the dump with
* @param integer $debugclass    Class of output
*/
function debug_hex($str, $msg="", $debugclass=DBG_DIAGNOSTIC) {
  if (is_string($str)) {
    if ($msg != "") {
      debugbr($msg, $debugclass);
    }
    debug("<pre>", $debugclass);
    for ($i=0; $i < strlen($str); $i++) {
      $c = ord(substr($str, $i, 1));
      $xc = sprintf("%02x", $c);
      if ($c == 32)      $xc = "<font color=blue>$xc</font>";
      elseif ($c < 32)  $xc = "<font color=red>$xc</font>";
      elseif ($c > 127) $xc = "<font color=green>$xc</font>";
      debug("$xc|", $debugclass);
      if (($i + 1) % 16 == 0) debugbr("", $debugclass);
    }
    debug("</pre><br>", $debugclass);
  }
}
// ..................................................................
/**
* Set debugging class(es)
* Sets the class or classes (OR'ed together) of debug output
* which will be accepted/rendered on ouput.
* @param integer $debugclass    Class of output
*/
function debug_class($debugclass="") {
  global $RESPONSE;
  $currentclass = DBG_UNDEFINED;
  if (isset($RESPONSE)) {
    if ($debugclass != "") {
      $RESPONSE->debugger->debug_class($debugclass);
    }
    $currentclass = $RESPONSE->debugger->debugclass;
  }
  // Return current class setting..
  return $currentclass;
}
// ..................................................................
/**
* Set debugging on
* Sets the debugging on. Sets class(es) to accept.
* @param integer $debugclass    Class of output
*/
function debug_on($debugclass=DBG_DEFAULT) {
  global $RESPONSE;
  if (isset($RESPONSE)) {
    $RESPONSE->debugger->debug_mode(ON, $debugclass);
  }
}
// ..................................................................
/**
* Set debugging off
*/
function debug_off() {
  global $RESPONSE;
  if (isset($RESPONSE)) {
    $RESPONSE->debugger->debug_mode(OFF);
  }
}
// ..................................................................
/**
* Sets flag for debugger to include global vars in output.
*/
// Causes page vars to be dumped..
function debug_dump() {
  global $RESPONSE;
  if (isset($RESPONSE)) {
    $RESPONSE->debugger->debug_dump();
  }
}
// ..................................................................
/**
* Render the debug output as a string.
* Normally debug output is taken care of by the system, however
* you might need to get hold of the output for some reason, and
* this is the function to do it.
* @return string  Debugger content as a string
*/
function debug_render() {
  global $RESPONSE;
  if (isset($RESPONSE) && debugging()) {
    return $RESPONSE->debugger->render();
  }
  else return "";
}
// ..................................................................
/**
* Return debugger status
* Function for external routines to determine
* whether debugging is enabled or not..
* @return boolean  True if debugging is enabled
*/
function debugging() {
  global $RESPONSE;
  if (isset($RESPONSE)) {
    return $RESPONSE->debugger->enabled;
  }
}
// ..................................................................
/**
* Set debugger output mode
* This function allows setting of the debugger output
* mode which determines where output goes.
* @param integer $mode  The debugger output mode
*/
function debug_output($mode=DBG_O_DEFAULT) {
  global $RESPONSE;
  if (isset($RESPONSE)) {
    $RESPONSE->debugger->debug_output($mode);
  }
}
// ..................................................................
/**
* Calls Php phpinfo() function
*/
function debug_phpinfo() {
  phpinfo(INFO_GENERAL|INFO_ENVIRONMENT|INFO_VARIABLES);
}
// ..................................................................
/**
* Display a variable nicely
* Variables might be other than simple scalars. This function
* is used internally by the debugger to make sure we show
* them off in their best light.
* @param mixed $var  The variable to show off
*/
function displayvar($var) {
  $s = "";
  if (is_array($var)) {
    foreach ($var as $elem) {
      if (is_scalar($elem)) {
        $s .= "[$elem] ";
      }
      else {
        $s .= "[non-scalar]";
      }
    }
  }
  elseif (is_bool($var)) {
    if ($var === true) $s .= "true";
    else $s .= "false";
  }
  else $s .= $var;
  return $s;
}
// ..................................................................
/**
* Insert an entry into the profiler. This label will have a line in
* the profile output against the time elapsed since the last line.
*/
function debug_profile($label) {
  global $RESPONSE;
  if (isset($RESPONSE) && $RESPONSE->debugger->profiling()) {
    $RESPONSE->debugger->profile($label);
  }
}
// ..................................................................
/** Returns true if the RESPONSE is in profiling mode */
function debug_profiling() {
  global $RESPONSE;
  return ( isset($RESPONSE) && $RESPONSE->debugger->profiling() );
}
// ..................................................................
/**
* DEBUG TRACEBACK
* Usage: In your function you bracket the statements you want to
* label for tracing as in the following example..
*  function thing() {
*    debug_trace("mymodule"); // pushes "mymodule" on trace stack
*    ...blah blah             // program statements
*    debug_trace();           // pops current trace off trace stack
*  }
* NOTE: You can also use the object identifier for a class instead
* of a simple string like "mymodule". For example:
*  function thing() {
*    debug_trace($this);      // pushes name of object class on trace stack
*    ...blah blah             // program statements
*    debug_trace();           // pops current trace off trace stack
*  }
* @param mixed $traceobj  Optional object trace is called in
*/
function debug_trace($traceobj="") {
  global $RESPONSE;
  if (isset($RESPONSE)) {
    if ($traceobj == "") {
      return $RESPONSE->debugger->poptrace();
    }
    else {
      return $RESPONSE->debugger->pushtrace($traceobj);
    }
  }
}

// ------------------------------------------------------------------
?>