#! /bin/bash
#
# Rebuild the Axyl website database CSV file. We do this by first of
# all acquiring all of the DocumentRoots from Apache (or they can be
# passed in), then for each one we go and extract data from the
# Axyl application.xml file. Having done that, we again cycle through
# each one, and extract the VirtualHost, ServerName and ServerAdmin
# info from its Apache virtualhost configuration. Then we write out
# the Axyl database record to the CSV file.
#
# P Waite
set -e

# Name of this script
SCRIPTNAME=${0##*/}

# Get common funcs and config vars etc.
# Acquire the local configuration..
CONFDIR=/etc/axyl
CONF=${CONFDIR}/axyl.conf
if [ ! -f $CONF ]
then
  echo "Axyl configuration file $CONF not found!"
  exit 2
else
  . $CONF
  if [ ! -d $AXYL_HOME ] ; then
    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
    exit 6
  fi
  . ${AXYL_HOME}/install/axyl-common.sh
fi

# Parameters. If any are passed, they form a list of
# document roots to scan for Axyl applications..
DOCROOTS=""
if [ $# -gt 0 ] ; then
  DOCROOTS=$*
fi

# Add a docroot to our list, avoiding duplicates, and
# ensuring the docroot physically exists on filesystem..
add_docroot () {
  if [ "$1" != "" ] ; then
    X=`echo $DOCROOTS | perl -ne "m;^.*[\s]{1,1}$1[\s]{1,1}.*$; && print;"`
    if [ "$X" = "" ] ; then
      if [ -e $1 ] ; then
        DOCROOTS="$DOCROOTS $1 "
      fi
    fi
  fi
}

# Scan the file given in $1 for virtualhosts. Add those
# docroots found to our list..
scan_virtualhosts () {
  if [ "$1" != "" ] ; then
    candidates=`perl -ne 'm;^[\s]*DocumentRoot ;i && s/^\s+DocumentRoot // && print;' $1`
    if [ "$candidates" != "" ] ; then
      for candidate in $candidates ; do
        DOCROOT=`echo $candidate | cut -d' ' -f2`
        add_docroot $DOCROOT
      done
    fi
  fi
}

# Scan all apache configuration files for a virtualhost
# identified by docroot in $1. Sets $vhdata if found.
vhdata=""
scan_conffiles () {
  vhdata=""
  if [ "$1" != "" ] ; then
    DR=$1
    case $APACHE_CONFSYS in
      ap2|confd)
        if [ "$APACHE_CONFSYS" = "ap2" ] ; then
          # Try for sites-enabled
          if [ -e ${APACHE_CONFDIR}/sites-enabled ] ; then
            CONFD_FILES=`ls ${APACHE_CONFDIR}/sites-enabled`
            if [ "$CONFD_FILES" != "" ] ; then
              for CONFD_FILE in $CONFD_FILES ; do
                CONF=${APACHE_CONFDIR}/sites-enabled/${CONFD_FILE}
                if [ -e $CONF ] ; then
                  vhdata=`${AXYL_HOME}/install/acquire-virtualhost-config.sh $CONF -d $DR`
                  [ "$vhdata" != "" ] && break
                fi
              done
            fi
          fi
        fi
        # Always fall back to conf.d
        if [ "$vhdata" = "" ] ; then
          if [ -e ${APACHE_CONFDIR}/conf.d ] ; then
            CONFD_FILES=`ls ${APACHE_CONFDIR}/conf.d`
            if [ "$CONFD_FILES" != "" ] ; then
              for CONFD_FILE in $CONFD_FILES ; do
                CONF=${APACHE_CONFDIR}/conf.d/${CONFD_FILE}
                if [ -e $CONF ] ; then
                  vhdata=`${AXYL_HOME}/install/acquire-virtualhost-config.sh $CONF -d $DR`
                  [ "$vhdata" != "" ] && break
                fi
              done
            fi
          fi
        fi
        ;;
      httpd)
        CONF=${APACHE_CONFDIR}/httpd.conf
        if [ -r $CONF ] ; then
          # First the parent httpd.conf file..
          vhdata=`${AXYL_HOME}/install/acquire-virtualhost-config.sh $CONF -d $DR`
        fi
        ;;
    esac    
        
  fi
}

# If the document roots weren't passed in, then try to
# determine these from Apache config files..
if [ "$DOCROOTS" = "" ] ; then

  # The old httpd.conf
  if [ -e ${APACHE_CONFDIR}/httpd.conf ] ; then
    scan_virtualhosts $CONF
  fi
  
  # The conf.d directory
  if [ -e ${APACHE_CONFDIR}/conf.d ] ; then
    for CONFD_FILE in `ls ${APACHE_CONFDIR}/conf.d` ; do
      scan_virtualhosts ${APACHE_CONFDIR}/conf.d/${CONFD_FILE}
    done
  fi
  
  # The sites-enabled directory
  if [ -e ${APACHE_CONFDIR}/sites-enabled ] ; then
    for CONFD_FILE in `ls ${APACHE_CONFDIR}/sites-enabled` ; do
      scan_virtualhosts ${APACHE_CONFDIR}/sites-enabled/${CONFD_FILE}
    done
  fi
  
fi

# Now we have all the docroots, let's look at them..
if [ "$DOCROOTS" != "" ] ; then
  regcnt=0
  for docroot in $DOCROOTS ; do
    app_name=
    dbname=
    dbuser=
    dbtype=
    dbpasswd=
    dbhost=
    dbport=
    vhost=
    vservername=
    admin_email=
    # Only process Axyl websites..
    if [ -e ${docroot}/application.xml ] ; then
      # Attempt to acquire mandatory info first..
      app_name=`perl -ne 's/^[\s]*<definition name="APP_PREFIX">([\w]+)<.*$/$1/ && print;' ${docroot}/application.xml`
      dbname=`perl -ne 's/^[\s]*<parameter name="name" type="string">([\w]+)<.*$/$1/ && do {print; exit;}' ${docroot}/application.xml`
      dbuser=`perl -ne 's/^[\s]*<parameter name="user" type="string">([\w]+)<.*$/$1/ && do {print; exit;}' ${docroot}/application.xml`
      if [ "$app_name" != "" -a "$dbname" != "" -a "$dbuser" != "" ] ; then
        # Report this website..
        echo -n " ..${app_name}"
        # Extract the rest of the info we need..
        dbtype=`perl -ne 's/^[\s]*<parameter name="type" type="string">([\w]+)<.*$/$1/ && do {print; exit;}' ${docroot}/application.xml`
        dbpasswd=`perl -ne 's/^[\s]*<parameter name="password" type="string">([\w]+)<.*$/$1/ && do {print; exit;}' ${docroot}/application.xml`
        dbhost=`perl -ne 's/^[\s]*<parameter name="host" type="string">([\w]+)<.*$/$1/ && do {print; exit;}' ${docroot}/application.xml`
        dbport=`perl -ne 's/^[\s]*<parameter name="port" type="string">([\w]+)<.*$/$1/ && do {print; exit;}' ${docroot}/application.xml`
        admin_email=`perl -ne 's/^[\s]*<global name="WEBMASTER_EMAIL">([\w]+)<.*$/$1/ && print;' ${docroot}/application.xml`
        # Fallback defaults..
        [ "$dbtype" = "" ] && dbtype=postgres
        [ "$dbpasswd" = "" ] && dbpasswd=none
        [ "$dbhost" = "" ] && dbhost=direct
        [ "$dbport" = "" ] && dbport=5432
        [ "$vhost" = "" ] && vhost=localhost
        [ "$vservername" = "" ] && vservername=$app_name
        [ "$admin_email" = "" ] && admin_email="webmaster@$vservername"
        # Get virtualhost data and register the whole website..
        scan_conffiles $docroot
        ${AXYL_HOME}/install/register-axyl-website.sh $app_name $dbtype $dbname $dbuser $dbpasswd $dbhost $dbport $vhdata
        regcnt=`expr $regcnt + 1`
      fi
    fi
  done
  # Linefeed off the end of our reported list..
  if [ $regcnt -gt 0 ] ; then
    echo ""
  fi
fi

exit 0

# END