<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    paging-defs.php                                         */
/* Author:      Paul Waite                                              */
/* Description: Definitions for paging content. Handles the             */
/*              splitting of a text block into pages of a given         */
/*              size, and provides the Next/Previous urls.              */
/*                                                                      */
/* ******************************************************************** */
/** @package cm */

/** Button widgets */
include_once("button-defs.php");

// ----------------------------------------------------------------------
/**
* Pages class
* A class for managing pages of text. Initially written to provide
* support for sending text to WAP phones, where the content had to
* be paged into small chunks due to the limited capacity of the
* phones.
* @package cm
*/
class pages extends RenderableObject {
   /** Pages content */
  var $content = "";
   /** Size of each page in characters */
  var $pagesize = 800;
   /** Number of chars to look ahead */
  var $lookahead = 250;
   /** Number of chars to look back */
  var $lookback = 120;
   /** Current page number */
  var $pageno = 0;
   /** Text for current page */
  var $pagetext = "";
   /** URL for getting next page */
  var $paginghref = "";
  //.....................................................................
  /**
  * Constructor
  * Creates the basic text paging object.
  * @param string  $content     Text content for pages
  * @param string  $paginghref  URL for getting next page (back or forward)
  * @param integer $pagesize    Max. size of each page in characters
  * @param integer $lookahead   No. characters to look ahead
  * @param integer $pagesize    No. characters to look back
  */
  function pages($content="", $paginghref="", $pagesize=800, $lookahead=250, $lookback=120) {
    $this->content    = $content;
    $this->paginghref = $paginghref;
    $this->pagesize   = $pagesize;
    $this->lookahead  = $lookahead;
    $this->lookback   = $lookback;
  } // pages
  //.....................................................................
  /**
  * Deliver page N
  * Delivers the text content for the specified page.
  * @param integer $pgno Number of the page to deliver
  * @param string $brtag Line-break tag to us (defaults to WML)
  * @return string Text content for specified page
  */
  function page($pgno, $brtag="<br/>") {
    if ( strlen($this->content) < ($this->pagesize + $this->lookahead) ) {
      $this->pageno = 1;
      $this->pagetext = $this->content;
    }
    else {
      $epos = 0;
      $fpos = strlen($this->content);
      for ( $spos=0, $tot_pages=1; $epos < $fpos; $tot_pages++ ) {
        $epos = $spos + $this->pagesize;
        if ( $epos > ($fpos - ($this->lookahead + $this->lookback)) ) $epos = $fpos;
        else {
          // Look forward for start of new paragraph...
          $tpos = strpos( substr($this->content,$epos), $brtag );
          if ( $tpos && $tpos > 0 && $tpos < $this->lookahead ) $epos += $tpos;
          else {
            // Look backward for start of paragraph...
            $tpos = strpos( strrev(substr($this->content, $epos - $this->lookback, $this->lookback)), strrev($brtag) );
            if ( $tpos && $tpos > 0 && $tpos < $this->lookback ) $epos -= $tpos;
            else {
              // Desperate: look forward for first space...
              $tpos = strpos( substr($this->content,$epos), " " );
              if ( $tpos && $tpos > 0 && $tpos < $this->lookahead ) $epos += $tpos;
            }
          }
        }
        // If we're at the required page, assign and stop..
        if ( $tot_pages == $pgno ) {
          $this->pageno = $pgno;
          $this->pagetext = substr($this->content, $spos, $epos - $spos);
          if ( $epos < $fpos ) {
            if ($this->paginghref != "") {
              $npg = $this->pageno + 1;
              if (strstr($this->paginghref, "?")) $href = $this->paginghref . "&amp;pg=$npg";
              else $href = $this->paginghref . "?pg=$npg";
              $pglink = new Link($href, " More..");
            }
          }
        }
        $spos = $epos;
      }
      // Add the page notification and link..
      $this->pagetext .= " [$pgno/" . ($tot_pages - 1) . "]";
      if (isset($pglink)) {
         $this->pagetext .= " " . $pglink->render();
      }
    }
    return $this->pagetext;
  } // page
  //.....................................................................
  /** Deliver WML format page */
  function wml($pgno=1) {
    return $this->page($pgno, "<br/>");
  } // wml
  //.....................................................................
  /** Deliver HTML format page */
  function html($pgno=1) {
    return $this->page($pgno, "<br>");
  } // html
} // pages class

// ----------------------------------------------------------------------
?>