/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import nz.net.catalyst.Log;
import org.apache.lucene.index.IndexReader;

/**
 * Run basic health check operation(s) and return status.
 */

public class HealthCheck implements IPackage, Constants {
  private final Transmission input;
  private Application application;

  HealthCheck(Transmission transmission) {
    input = transmission;
  }

  /**
   * 
   * Summary:
   * 
   * Make some check(s) on the system and return the control response
   * either with ERROR or as normal response with total number of
   * indexed documents contained in it.
   * 
   * All this checks is that an IndexReader can be created on the application
   * index, and that the numDocs() method can be executed on it. No search
   * activity is performed, in order to keep the load low as possible.
   * 
   * @return
   */

  Transmission execute() {
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());

    String appName = input.get(APPLICATION, NO_APP);

    try {
      application = Application.getAppOrDefault(appName);
    }
    catch (ApplicationMissingException e) {
      return error(e.getMessage());
    }

    // Get an Index Reader..
    IndexReader reader = null;
    input.setApplication(application);
    File luceneStoreDir = Application.getIndexDirectory(application);
	if (luceneStoreDir == null) {
        String message = "Application index directory undefined";
        return error(message);
	}
    try {
        reader = IndexReader.open(luceneStoreDir);
    }
    catch (FileNotFoundException e) {
        String message = "No index";
        return error(message);
    }
    catch (IOException e) {
        String message = "IO exception: " + e.toString();
        return error(message);
    }
    finally {
        if (reader != null) {
            try {
                // We must always close the IndexReader!
                reader.close();
            }
            catch (Throwable e) {
                String message = "Error closing IndexReader: " + e.toString();
                return error(message);
            }
        }
    }
    // If we get here, all is ok, add the total indexed docs..
    String message = "healthy";
    response.add("Status", message);
    response.add("NumDocs", String.valueOf(reader.numDocs()));
    Log.info("healthcheck: " + message);

    return response;
  }

  /**
   * Build an error response for sending back to the client.
   *
   * @param message The text of the error message
   * @return An INDEX-RESPONSE Transmission
   */

  private Transmission error(String message)
  {
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    Log.error("healthcheck: " + message);
    return response;
  }
}
