/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.ref.SoftReference;
import java.net.Socket;
import java.util.BitSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;


public final class Util implements IPackage
{
  private Util(){} // Prevent instantiation.

  /**
   * The default prefix length used by {@link #clean} (80).
   */
  public static final int DEFAULT_CLEAN_LENGTH = 80;
  
  private static final BitSet DEFAULT_PLAIN = new BitSet(128);
  
  static
  {
    for (int i = ' '; i <= '~'; ++i)
      DEFAULT_PLAIN.set(i);

    DEFAULT_PLAIN.clear('%');
  }

  /**
   * Cache of Soft-references to Plain-Sets.  Theses are modified
   * bitsets of plain characters keyed by Pair(encoding, extraCoded).
   * The value objects are SoftReferences to BitSets.
   */

  private static final Map PLAIN_CACHE = new HashMap();

  public static final String DEFAULT_DELIMITERS = " \r\n\t\f,";

  /**
   * Obtain package name for a class by removing the class's base name
   * from its full name.
   *
   * @param cl The class for which the package component of the name
   *           is required.
   *
   * @return The package component of the class name, including
   *         trailing space.  If the class is a top-level class, then
   *         the returned string is blank.  If the class is for a
   *         non-object or array, then the result is undefined.
   */

  public static String getPackage(Class cl)
  {
    String className =  cl.getName();
    int lastDot = className.lastIndexOf('.');
    return className.substring(0, lastDot + 1);
  }

  /**
   * Simple-minded String splitter.  Splits at white-space.
   * None of the returned strings will be zero-length, that is,
   * multiple delimiter characters are treated as a single delimiter. <p>
   *
   * Cannot handle any form of embedding of delimiters within the
   * values such as using quoted strings or escape characters.
   *
   * @param s A String containing multiple values to be split apart

   * @return Return a String array containing the tokens found between
   *         the delimiters, or null if the original string was
   *         null.
   */
  public static String[] split(String s)
  {
    return split(s, " \r\f\t\n");
  }

  /**
   * Simple-minded CSV splitter.  Splits at white-space or commas.
   * None of the returned strings will be zero-length, that is,
   * multiple delimiter characters are treated as a single delimiter. <p>
   *
   * Cannot handle any form of embedding of delimiters within the
   * values such as using quoted strings or escape characters.
   *
   * @param csv A String containing multiple values to be split apart

   * @return Return a String array containing the tokens found between
   *         the delimiters, or null if the original csv string was
   *         null.
   */
  public static String[] getCsv(String csv)
  {
    return split(csv, DEFAULT_DELIMITERS);
  }

  /**
   * Simple-minded CSV splitter.  Splits at the specified characters.
   * None of the returned strings will be zero-length, that is,
   * multiple delimiter characters are treated as a single delimiter. <p>
   *
   * Cannot handle any form of embedding of delimiters within the
   * values such as using quoted strings or escape characters.
   *
   * @param s A String containing multiple values to be split apart
   * @param delim The separator characters to look for when splitting
   *              the string
   * @return Return a String array containing the tokens found between
   *         the delimiters, or null if the original string was
   *         null.
   */
  public static String[] split(String s, String delim)
  {
    if (s == null)
      return null;

    StringTokenizer st = new StringTokenizer(s, delim);
    String[] result = new String[st.countTokens()];

    for (int i = 0; i < result.length; ++i)
      result[i] = st.nextToken();

    return result;
  }

  /**
   * Return a string identifying the local and remote ends of a socket.
   *
   * @param socket A connected socket
   * @return A string detailing the ip-address and ports of the end-points.
   */

  public static String name(Socket socket)
  {
    return
      //socket.getLocalAddress().getHostAddress() +
      //":" +
      //    socket.getLocalPort() +
      //" <-> " +
      socket.getInetAddress().getHostAddress() +
      ":" +
      socket.getPort();
  }

  /**
   * Do a simplified form of URL-encoding on a string.  This is less
   * overbearing than the standard URL-encoding in that it passes
   * through all printable ASCII characters (including space) except
   * '%'.  It does not translate space to '+'.   <p>
   *
   * The string to be encoded is converted to a byte-array using the
   * "UTF-8" character mapping.  That byte stream is encoded
   * and the result is a string containing only the printable ASCII
   * characters.
   *
   * @param s The string to encoded.
   *
   * @return The resulting string of printable ASCII characters.
   */

  public static String textEncode(String s)
  {
    try
    {
      return textEncodeWithCodeset(s, null, "UTF-8");
    }
    catch (UnsupportedEncodingException e)
    {
      throw new RuntimeException("Could not create string with default encoding ???", e);
    }
  }

  /**
   * Do a simplified form of URL-encoding on a string.  This is less
   * overbearing than the standard URL-encoding in that it passes
   * through all printable ASCII characters (including space) except
   * '%'.  It does not translate space to '+'.   <p>
   *
   * The string to be encoded is converted to a byte-array using the
   * "UTF-8" character mapping.  That byte stream is encoded
   * and the result is a string containing only the printable ASCII
   * characters.
   *
   * @param s The string to encoded.
   * @param alsoEncode Additioanl characters that should be encoded
   *                   and not passed through unchanged.
   *
   * @return The resulting string of printable ASCII characters.
   */

  public static String textEncode(String s, String alsoEncode)
  {
    try
    {
      return textEncodeWithCodeset(s, alsoEncode, "UTF-8");
    }
    catch (UnsupportedEncodingException e)
    {
      throw new RuntimeException("Could not create string with default encoding ???", e);
    }
  }


  /**
   * Do a simplified form of URL-encoding on a string.  This is less
   * overbearing than the standard URL-encoding in that it passes
   * through all printable ASCII characters (including space) except
   * '%' and the characters specified in the <tt>alsoEncode</tt>
   * string.  It does not translate space to '+'.
   *
   * The string to be encoded is converted to a byte-array using the
   * provided character mapping.  That byte stream is encoded
   * and the result is a string containing only the printable ASCII
   * characters.
   *
   * @param s The string to encoded.
   * @param encoding The Character Encoding to use to map the source
   *                 string to a byte array, or null to use the
   *                 platform standard mapping.
   *
   * @return The resulting string of printable ASCII characters.
   */

  public static String textEncodeWithCodeset(String s, String encoding)
    throws UnsupportedEncodingException
  {
    return textEncodeWithCodeset(s, null, encoding);
  }

  /**
   * Do a simplified form of URL-encoding on a string.  This is less
   * overbearing than the standard URL-encoding in that it passes
   * through all printable ASCII characters (including space) except
   * '%' and the characters specified in the <tt>alsoEncode</tt>
   * string.  It does not translate space to '+'.
   *
   * The string to be encoded is converted to a byte-array using the
   * default platform character mapping.  That byte stream is encoded
   * and the result is a string containing only the printable ASCII
   * characters.
   *
   * @param s The string to encoded.
   * @param alsoEncode Additioanl characters that should be encoded
   *                   and not passed through unchanged.
   * @param encoding The Character Encoding to use to map the source
   *                 string to a byte array, or null to use the
   *                 platform standard mapping.
   *
   * @return The resulting string of printable ASCII characters.
   */

  public static String textEncodeWithCodeset(String s, String alsoEncode, String encoding)
    throws UnsupportedEncodingException
  {
  	
  	//hack by hamish
  	boolean hack = true;
  	if (hack) return s;
  	//end hack by hamish
  	
    BitSet plain;
    if (alsoEncode == null)
      plain = DEFAULT_PLAIN;
    else
    {
      Pair pair = new Pair(encoding, alsoEncode);

      synchronized(PLAIN_CACHE)
      {
        plain = (BitSet)getSoftMap(PLAIN_CACHE, pair, false);

        if (plain == null)
        {
          plain = (BitSet)DEFAULT_PLAIN.clone();
          byte[] extra = encoding == null
            ? alsoEncode.getBytes()
            : alsoEncode.getBytes(encoding);

          for (int i = 0; i < extra.length; ++i)
            plain.clear((int)extra[i] & 0xFF);

          putSoftMap(PLAIN_CACHE, pair, plain, false);
        }
      }
    }

    byte[] text = encoding == null ? s.getBytes() : s.getBytes(encoding);
    byte[] output = new byte[text.length * 3]; // Maximum length possible
    int pos = 0;

    for (int i = 0; i < text.length; ++i)
    {
      int ch = (int) text[i] & 0xFF;
      if (plain.get(ch))
        output[pos++] = text[i];
      else
      {
        output[pos++] = (byte)'%';
        for (int shift = 4; shift >= 0; shift -= 4)
        {
          int nibble = (ch >> shift) & 0xF;
          if (nibble >= 10)
            output[pos++] = (byte)(nibble + 'A');
          else
            output[pos++] = (byte)(nibble + '0');
        }
      }
    }
    try
    {
      return new String(output, 0, pos, "UTF-8");
    }
    catch (UnsupportedEncodingException e)
    {
      throw new RuntimeException("Could not create string with UTF-8 ???", e);
    }
  }

  /**
   *
   */

  public static String textDecode(String s)
  {
    try
    {
      return textDecodeWithCodeset(s, "UTF-8");
    }
    catch (UnsupportedEncodingException e)
    {
      throw new RuntimeException("Could not create string with default encoding ???", e);
    }
  }

  public static String textDecodeWithCodeset(String s, String encoding)
    throws UnsupportedEncodingException
  {
    char[] ch = s.toCharArray();
    byte[] decoded = new byte[ch.length];
    int pos = 0;

    for (int i = 0; i < ch.length; ++i)
    {
      if (ch[i] == '%' && i + 2 < ch.length)
      {
        try
        {
          int value = Integer.parseInt(s.substring(i+1, i+3), 16);
          decoded[pos++] = (byte) value;
          i += 2;
        }
        catch (NumberFormatException e)
        {
          decoded[pos++] = '%';
        }
      }
      else
        decoded[pos++] = (byte)ch[i];
    }
    return encoding == null
      ? new String(decoded, 0, pos)
      : new String(decoded, 0, pos, encoding);
  }

  /**
   * Get an entry from a Map containing SoftReference objects and
   * optionally purge all where the SoftReference has relinquished its
   * grip.
   *
   * The values in the map are assumed to be SoftReference objects.
   *
   * @param map The Map containing the SoftReference objects.
   * @param key The key of the entry to be retrieved.
   * @param purge Whether the Map should be purged of reclaimed entries.
   *
   * @return The value associated with the key, or null if it is not
   *         present or has been reclaimed.
   */

  public static Object getSoftMap(Map map, Object key, boolean purge)
  {
    SoftReference ref = (SoftReference)map.get(key);
    Object result = ref == null ? null : ref.get();
    if (result == null)
      map.remove(key);

    if (purge)
    {
      for (Iterator it = map.values().iterator(); it.hasNext(); )
      {
        ref = (SoftReference)it.next();
        if (ref == null || ref.get() == null)
          it.remove();
      }
    }
    return result;
  }
  /**
   * Put an entry into a Map inside a SoftReference container.
   * Optionally purge all where the SoftReference has relinquished its
   * grip.
   *
   * The values in the map are assumed to be SoftReference objects.
   *
   * @param map The Map containing the SoftReference objects.
   * @param key The key of the entry to be set.
   * @param value The value to be stored for the key.  If null, the key entry will be removed
   * @param purge Whether the Map should be purged of reclaimed entries.
   *
   * @return the former value stored at this key, if any.
   */

  public static Object putSoftMap(Map map, Object key, Object value, boolean purge)
  {
    Object result = null;
    SoftReference ref = (SoftReference)map.get(key);
    if (ref != null)
      result = ref.get();

    map.put(key, new SoftReference(value));

    if (purge)
    {
      for (Iterator it = map.values().iterator(); it.hasNext(); )
      {
        ref = (SoftReference)it.next();
        if (ref == null || ref.get() == null)
          it.remove();
      }
    }
    return result;
  }

  /**
   * Try to get the canonical path for a file.  If it fails, just fall
   * back to the relative path.
   *
   * @param File file the File for which the path is required.
   * @return Usually the canonical path, but otherwise the relative
   * path if that is the best we can do.
   */

  public static String getFullPath(File file)
  {
    try
    {
      return file.getCanonicalPath();
    }
    catch (IOException e)
    {
      return file.getPath();
    }
  }

  /**
   * Truncate and remove unprintable characters from a string.
   * 
   * Return the first {@link #DEFAULT_CLEAN_LENGTH} characters of the
   * string, and clean them so that they are safe to display as debug
   * output, i.e. replace all non-printable characters with '?'
   *
   * @param rawString The string to be cleaned.
   * @return The beginning of the parameter string with control and
   * non-ascii characters replaced with '?'
   */
  
  public static String clean(String rawString)
  {
    return clean(rawString, DEFAULT_CLEAN_LENGTH);
  }

  /**
   * Clean a string for debug ouput, truncating and removing non
   * printable characters.
   *  
   * Return the first <code>prefix</code> characters of the string,
   * and clean them so that they are safe to display as debug output,
   * i.e. replace all non-printable characters with '?'.  If the
   * string is longer than <code>prefix</code> characters, the length
   * of the string will be appended to the rturn value.
   *
   * @param rawString The string to be cleaned.
   * @param prefix The length of prefix to return.
   * @return The beginning of the parameter string with control and
   * non-ascii characters replaced with '?'
   */
  
  public static String clean(String rawString, int prefix)
  {
    int strlen = rawString.length();
    int length = Math.min(strlen, prefix);
    char[] ch = rawString.substring(0, length).toCharArray();

    for (int i = 0; i < ch.length; ++i) {
      if (ch[i] < ' ' || ch[i] > '~')
        ch[i] = '?';
    }
    if (length == strlen)
      return new String(ch);

    StringBuffer sb = new StringBuffer(ch.length + 50);
    sb.append(ch);
    sb.append(" + ");
    sb.append(strlen - length);
    sb.append(" character");
    if (strlen - length != 1)
      sb.append('s');
    return sb.toString();
  }
}
