/*
 * SearcherCache.java
 * 
 * See package.html for licence
 * 
 * Created on 2002. j???lius 14., 15:43
 */

package nz.net.catalyst.lucene.cache;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Filter; 
import org.apache.lucene.search.HitCollector;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Searcher;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.Explanation;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.TopFieldDocs;

/**
 * A <code>SearcherCache</code> manages searches over an index. A Searcher
 * instance is reusable and should not be closed as far as the index isn't
 * touched (unmodified). The caller can request a <code>Searcher</code> with the
 * <code>getSearcher</code> method. This class returns a <code>CachedSearcher</code>
 * instance that is subclass of Lucene's built in <code>Searcher</code> class
 * and delegates searching job to a real <code>Searcher</code>.<br/>
 *
 * The caller should call the <code>close</code> method of the <code>CachedSearcher</code>
 * after using. The <code>close</code> method doesn't close the files
 * (and the real searcher) but notifies the cache that it <i>can</i> close it.<br/>
 *
 * When the caller requests a searcher this class checks whether the index was
 * modified or not. If not then returns the actual cached searcher. If yes then
 * the actual searcher is put in a table of old searchers and a new searcher is
 * created.<br/>
 *
 * When the class is notified that a searcher can be closed, it is closed if
 * and only if no one is using it and it is old.<br/>
 *
 * Please notice:
 * <li>after the first call of <code>getSearcher()</code> one <code>Searcher</code>
 * is always opened --> the <code>release</code> method should be called to close
 * all searcher</li>
 * <li>this is not a pool this is a cache: in some circumstances a lot of searchers
 * (limited by the OS) can be opened; when there is a lot of searcher threads that
 * use a searcher object for a long time and there is writer thread that modifies
 * the index continually</li>
 *
 * @author  Peter Halacsy
 */

public abstract class SearcherCache implements SearcherSource {
    // about the implementation:
    // after the first call of getSearcher there is always an actual
    // searcher that is opened and returned by getSearcher

    // if a searcher gets old because the index was modified, it is put
    // to a table of old searchers (m_oldSearchersInfo)

    // there can be more than one old searcher:
    // Thread I: request a searcher -> searcher I is created, this is the actual
    // Thread II: request a searcher -> the actual (searcher I) is returned
    // Thread III: modifies the index
    // Thread IV: request a searcher -> the actual (searcher I) is put to
    //                                  oldies table, new actual is created
    //                                  (searcher II)
    // Thread II: closes its searcher -> searcher I is not closed because
    //                                   Thread I has reference
    // Thread III: modifies the index
    // Thread II: request a searcher -> the actual (searcher II) is put to
    //                                  oldies table, new actual is created
    //                                  (searcher III)
    // Thread I: closes the its searcher -> searcher I is closed and deleted
    //                                      from the table

    /** debug variable. if true then debug comments will be outputted to system.out */
    private boolean debug = false;

    /** info about the actual Searcher */
    protected CachedSearcher m_actualCachedSearcher = null;

    /* info about Searcher that should be closed because the index has been
      modified since this searcher was created */
    protected HashMap m_oldSearchersInfo = new HashMap();


    /** Closes the last opened Searcher that must be opened becauese we always
     *  hold a Searcher opened. This method should be called if the application
     * is shutdowned.
     *
     *  for the sake of safety the <code>m_oldSearchersInfo</code> table is
     * checked for not closed searcher
     * @throws IOException If the index searcher cannot be released.
     */
    public synchronized void release() throws IOException {

        if(m_actualCachedSearcher != null) {
            m_actualCachedSearcher.getRealSearcher().close();
            m_actualCachedSearcher = null;
        }

        Iterator oldOpenSearchers = m_oldSearchersInfo.keySet().iterator();
        while(oldOpenSearchers.hasNext()) {
            CachedSearcher searcher  = (CachedSearcher) oldOpenSearchers.next();
            searcher.getRealSearcher().close();
        }
        m_oldSearchersInfo.clear();
    }


    /** get for search.
     * @throws IOException If searcher cannot be created
     * @return The cached searcher
     */
    public synchronized Searcher getSearcher() throws IOException {
        if (m_actualCachedSearcher == null || isSearcherOld(m_actualCachedSearcher) ) {
            // need new searcher but first check whether we have an old one
            if ( m_actualCachedSearcher != null ) {
                // that means : directory.lastModified() > searcherInfo.creationTime
                // this searcher won't be returned any more but others may be using it
                if (m_actualCachedSearcher.getCheckoutCount() > 1) // searcher has other references
                {
                    if (debug) System.out.println("a searcher signed as old");
                    m_oldSearchersInfo.put(m_actualCachedSearcher, m_actualCachedSearcher);
                }
                else // last reference to searcher
                {
                    // there is no other reference to actualCachedSearcher and
                    // it is old --> must be closed
                    if (debug) System.out.println("closing a searcher before creating new");
                    m_actualCachedSearcher.getRealSearcher().close();
                }
            }
            //if (debug) System.out.println("creating new searcher");
            m_actualCachedSearcher = createNewCachedSearcher();

        }
        else {
            // use existing actual searcher --> remember how many times it was
            // checked out
            if (debug) System.out.println("returning existing searcher");
            m_actualCachedSearcher.checkout();

        }

        return m_actualCachedSearcher;
    }

    /** This is called by getSearcher to decide whether a new searcher instance
     * has to be created. Subclasses has to overload it.
     * @param actualCachedSearcher The [cached]searcher to test the age of.
     * @return Age status of the searcher
     */
    protected abstract boolean isSearcherOld(CachedSearcher actualCachedSearcher);

    /** This method is called by getSearcher if fresh searcher is needed.
     * @throws IOException
     * @return a new CachedSearch
     */
    protected abstract CachedSearcher createNewCachedSearcher() throws IOException ;


    /** Thes method is called by the CachedSearcher which close method was closed
     * @param searcher
     * @throws IOException  */
    protected synchronized void searcherClosed(CachedSearcher searcher) throws IOException {
        //if (debug) System.out.println(Thread.currentThread().getName() + " searcherClosed event received");
        releaseSearcher(searcher);
    }

    /**
     * @param searcherToRelease
     * @throws IOException If searcher cannot be released.
     */    
    protected synchronized void releaseSearcher(CachedSearcher searcherToRelease) throws IOException {

        if(searcherToRelease.getCheckoutCount() > 1) { // searcher has other references
             // do not close
            if (debug) System.out.println("searcher checked in but not closed");
            searcherToRelease.checkin();
        } else if (m_oldSearchersInfo.containsKey(searcherToRelease)) { // last reference to searcher and it's old
            // close
            if (debug) System.out.println("old searcher closed");
            searcherToRelease.getRealSearcher().close();
            m_oldSearchersInfo.remove(searcherToRelease);

        } else {
            // what does it mean? someone released a searcher that has no other
            // reference and it is not old --> searcherToRelease == m_actualCachedSearcher
            // it musn't be closed

        }
    }

    /** The cached searcher implementation that wraps a normal lucene searcher.
     */    
    protected class CachedSearcher extends Searcher {
        int m_checkoutCount = 1;
        long m_creationTime;
        /** The real search the work is delegated */
        Searcher m_searcher;

        /** Creates new CachedSearcher */
        CachedSearcher(Searcher searcher) {
            m_searcher = searcher;
            m_creationTime = System.currentTimeMillis();
            m_checkoutCount = 1;
        }
        long getCreationTime() {
            return m_creationTime;
        }

        Searcher getRealSearcher() {
            return m_searcher;
        }


        int getCheckoutCount() {
            return m_checkoutCount;
        }

        void checkout() {
            m_checkoutCount++;
        }

        void checkin() {
            m_checkoutCount--;
        }

        /** Test equality of objects.
         * @param other The object compare with.
         * @return Whether equal or not
         */        
        public boolean equals(Object other) {
            if(other instanceof CachedSearcher) {
                CachedSearcher c = (CachedSearcher) other;
                 return (c.m_creationTime == this.m_creationTime &&
                   c.m_checkoutCount == this.m_checkoutCount &&
                   c.m_searcher == this.m_searcher);
            }
            return false;

        }

        /** Create a hashcode of this object
         * @return A hashcode of the object
         */        
        public int hashCode() {
            return m_searcher.hashCode() * m_checkoutCount;
        }

        /** This method must be called by the client application if the searcher
         *  won't be used any more. This method doesn't close the searcher but
         *  notifies the manager.
         * @throws IOException If the searcher cannot be closed.
         */
        public void close() throws IOException {
            // do not close the searcher but notify listener
            searcherClosed(this);
        }

        // implementing methods of Searcher

        /** For use by {@link HitCollector} implementations.
         * @param i The index number for the document. <B>This is not a unique identifier!</b>
         * Use this number only in the context of this session.
         * @throws IOException If the document cannot be retrieved.
         * @return The document requested.
         */
        public Document doc(int i) throws IOException {
            return m_searcher.doc(i);
        }

        /** Perform the search operation
         * <br>See org.apache.lucene.search.Searcher for more information.
         * @param query The query to perform
         * @param filter Filter to use on query results?
         * @param results The results from the search
         * @throws IOException If unable to search.
         */        
        public void search(Query query, Filter filter, HitCollector results) throws IOException {
            m_searcher.search(query, filter, results);
        }
        /** Perform a TopDocs search.
         * See org.apache.lucene.search.Searcher for more information.
         * @param query The query to perform
         * @param filter Filter to use on results?
         * @param n The number of results to retrieve.
         * @throws IOException If unable to perform search.
         * @return The TopDocs from the search
         */        
        public TopDocs search(Query query, Filter filter, int n) throws IOException {
            return m_searcher.search(query, filter, n);
        }
        
        /** Perform a TopDocs sorted search.
         * See org.apache.lucene.search.Searcher for more information.
         * @param query The query to perform
         * @param filter Filter to use on results?
         * @param n The number of results to retrieve.
         * @param sort The Sort criteria to use
         * @throws IOException If unable to perform search.
         * @return The TopDocs from the search
         */        
        public TopFieldDocs search(Query query, Filter filter, int n, Sort sort) throws IOException {
            return m_searcher.search(query, filter, n, sort);
        }

        /** This command performs a query to determine the frequency of a given term.
         * Use to answer questions like "How many times is 'cat' indexed in our website?".
         * <br>See org.apache.lucene.search.Searcher for more information.
         *
         * @param term Term to get frequency for.
         * @throws IOException If unable to read index.
         * @return The frequency of the specified term.
         */        
        public int docFreq(Term term) throws IOException {
            return m_searcher.docFreq(term);
        }

        /** <br>See org.apache.lucene.search.Searcher for more information.
         * @throws IOException If unable to search
         * @return The maxDoc value.
         */        
        public int maxDoc() throws IOException {
            return m_searcher.maxDoc();
        }

		public Query rewrite(Query original) throws IOException {
		  return m_searcher.rewrite(original);
		}

		public Explanation explain(Query query, int doc) throws IOException {
		  return m_searcher.explain(query, doc);
		}     
    }
}
