/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import nz.net.catalyst.Log;
import nz.net.catalyst.Util; 
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Hits;
import org.apache.lucene.search.RangeQuery;
import org.apache.lucene.search.Searcher;
import org.apache.lucene.search.TermQuery;

/**
 * Execute a Lucene Query.
 * 
 * With the query response the following fields are ALWAYS returned:<p>
 * I: a counter representing the document's place in the result set.<p>
 * RANK: the document's rank relative to the other documents<p>
 * Domian: the domain the document belongs too.  Don't know why this is
 * returned!<p>
 * [your optional fields here]<p>
 */

public class Query implements IPackage, Constants
{
  private final Transmission input;
  private Application application;

  Query(Transmission transmission)
  {
    input = transmission;
  }

  Transmission execute()
  {
    long queryStart = System.currentTimeMillis();
    Transmission response = new Transmission(ECommand.QUERY_RESPONSE);
    response.setSerial(input.getSerial());

    String appName = input.get(APPLICATION, NO_APP);

    try
    {
      application = Application.getAppOrDefault(appName);
    }
    catch (ApplicationMissingException e)
    {
      return error(e.getMessage());
    }
    input.setApplication(application);

    Analyzer analyzer = Application.getAnalyzer(input); // Decide upon an analyzer
    													// THIS IS WHERE WE COULD BOLT IN DIFFERENT ANALYZERS.

    String defaultField = input.get(DEFAULT_FIELD);
    if (defaultField == null)
      defaultField = "UNDEFINED_FIELD"; // Make one up -- it won't matter!

    // Get list of field definitions.
    Map fieldMap = mapFields(input.getFields());

    long beforeParse = System.currentTimeMillis();

    org.apache.lucene.search.Query query;
    try {
        query = buildQuery(analyzer, defaultField, fieldMap);
    } catch (QueryException e) {
        return error(e.getMessage());
    } 

  	long beforeOpenSearcher = System.currentTimeMillis();

    File luceneStoreDir = Application.getIndexDirectory(application);
    
    Searcher searcher = null;
    String exceptionMessage;

    List returnFields = getReturnFields(fieldMap); // List of FieldDef

    
    //hu.origo.lucenetools.iac.IndexAccessControl iac;
  	long beforeSearch;
  	long beforeSort;
  	long beforeOutput;
    try {
      exceptionMessage = "closing idle IndexWriter";
      WriterControl.closeIdleWriter(luceneStoreDir);
      exceptionMessage = "opening Index";
      
      //searcher = new IndexSearcher(luceneStoreDir.getPath());
      //TESTING USING INDEX SEARCHER CONTROL.  THIS SHOULD LIMIT NUMBER OF FILES OPENED.
      
      //This one (iac) had problems when accessed from multiple threads.
      //iac = hu.origo.lucenetools.iac.IndexAccessControl.getInstance(luceneStoreDir);
      //searcher = iac.getSearcher();
      
      try {
	      nz.net.catalyst.lucene.cache.IndexSearcherCache isc = nz.net.catalyst.lucene.cache.IndexSearcherCache.getInstance(luceneStoreDir);
	      searcher = isc.getSearcher();
      } catch (IOException e) {
      		Log.error("Query performed on non-existing index!");
      		Log.error("An index needs to have at least one document successfully indexed before Querying.");
      		Log.error(e.getMessage());
      		String message = "Error during query: Index not found. Index documents first or check settings.";
    	  	Log.error(message);
	      	return error(message);
      } 
      beforeSearch = System.currentTimeMillis();      
      exceptionMessage = "searching Index";
      
      Hits hits = searcher.search(query);
      int hitCount = hits.length();

	  Log.debug("Found " + hitCount + " results in " + (System.currentTimeMillis() - beforeSearch) + "ms");

      // Find out how many results to return.  Default is all of them
      // need to do this now to ensure the Sort-Limit values make sense.
      int limit = -1;
      try	{
    	  limit = Integer.parseInt(input.get(LIMIT));
    	  if (limit < 0) limit = -1;
      }  catch (Exception e) { /* Ignore */ }

    	// Find out where to starting returning results from.
    	// Default is from result 1 (the first).
    	int first = 1;
    	try {
    	  first = Integer.parseInt(input.get(FIRST));
    	  if (first < 1)
    		first = 1;
    	} catch (Exception e) { /* Ignore */ }
    	--first;

      beforeSort = System.currentTimeMillis();	  

    	//SoftReference[] docs = null; //no longer using softreferences
    	SortObject[] docs = null; //USE HARD REFERENCES NOW
    	Sort[] sortList = getSortList();

      if (sortList != null) {
		//calculate sort limits.
      	int sortLimitSystem = -1;
      	int sortLimitUser   = -1;
      	try {
      	  sortLimitSystem = Integer.parseInt(System.getProperty(PACKAGE + "SortLimitSystem"));
      	  sortLimitUser   = Integer.parseInt(input.get("Sort-Limit"));
      	} catch (NumberFormatException e) { /* ignore */ }

      	Log.debug("sortLimitSystem: " + sortLimitSystem);
      	Log.debug("sortLimitUser: " + sortLimitUser);

		//validate limits.
      	if (sortLimitSystem != -1 && sortLimitUser > sortLimitSystem) {
      	  sortLimitUser = sortLimitSystem;
      	  Log.debug("Resetting Sort-Limit sent from client because was greater than the system sort limit. Sort-Limit=" + sortLimitUser + " Sort-Limit=" + sortLimitUser);
      	}
      	if (sortLimitUser != -1 && limit > sortLimitUser) {
      	  sortLimitUser = -1;
      	  Log.debug("Ignoring Sort-Limit sent from client because it is less than their specified limit. Limit=" + limit + " Sort-Limit=" + sortLimitUser);
      	}
		//finished calculating sort limits

      	//check the number of results against the sortLimitXXXXs to see if we should sort.
      	if (sortLimitUser != -1 && hitCount > sortLimitUser) {
      		Log.debug("Number of results exceeds user sort limit. HitCount=" + hitCount + " Sort-Limit=" + sortLimitUser);
      		return error("Number of results exceeds user sort limit. Please restrict search or do not sort results.");
      	}
      	if (sortLimitSystem != -1 && hitCount > sortLimitSystem) {
      		Log.debug("Number of results exceeds system sort limit. HitCount=" + hitCount + " System Sort-Limit=" + sortLimitSystem);
      		return error("Number of results exceeds system sort limit. Please restrict search or do not sort results.");
      	}
      		
      	//there is some sorting criteria.
        if (Log.isDebugEnabled()) {
	          //output sort criteria
	          Log.debug("Sort override requested -- will retrieve all matching documents into memory for sorting.");
	          StringBuffer sb = new StringBuffer();
	          for (int i = 0; i < sortList.length; ++i) {
		            if (i > 0) sb.append(", ");
		            sb.append(sortList[i].field == null ? RANK : sortList[i].field);
		            sb.append(':');
		            sb.append(sortList[i].descending ? "Desc" : "Asc");
	          }
	          Log.debug("Sort order is: " + sb);
        }

        // Use Soft-References to hold the sort data.  If we run out
        // of memory, then some of them will "evaporate".  In that
        // case, the sort routime will notice and throw an
        // "Out-of-Memory" message back to the client.
		// we have stopped using soft-references now as sorting was failing
		// because the softreferences were claimed before increasing the 
		// JVM heap which wasn't good (TM).
        //docs = new SoftReference[hitCount];
      	docs = new SortObject[hitCount]; //USE HARD REFERENCES NOW

        exceptionMessage = "building sort keys";

									/*
									//BEGIN FILTER-SORT
											if (hitCount > limit) {
												Log.debug("about to try filter-sort approach. hitcount="+hitCount+" limit="+limit);
												java.util.Vector filteredHits = new java.util.Vector(limit);
												
												Comparator comparator1 = new DocCompare(sortList);
												
												for (int i = 0; i < limit; i++) {
												  SortObject item = makeSortObject(sortList, hits, i);
												  filteredHits.add(item);
												}
									
												java.util.Collections.sort(filteredHits, comparator1);
												Log.debug("initial sort completed. filteredHits.size() = " + filteredHits.size());
									      		SortObject limitItem = (SortObject) filteredHits.get(limit-1);
									      		
										        for (int i = limit; i < hitCount; i++) {
										          //docs[i] = new SoftReference(makeSortObject(sortList, hits, i));
										      	  
										      	  SortObject item = makeSortObject(sortList, hits, i);
										          int result = comparator1.compare(item, limitItem);
										          //Log.debug(String.valueOf(result));
										      	  if ( result > 0) {
										      	  	 //Log.debug("Just added item to the list");
										      	  	 filteredHits.add(item);
										      	  }
										      	  //leave this out for now.
										      	  //java.util.Collections.sort(filteredHits, comparator1);	 
										        }
												Log.debug("Full sort completed. filteredHits.size() = " + filteredHits.size());
												
												SortObject[] a = new SortObject[1];
												
										        docs = (SortObject[]) filteredHits.toArray(a);
										        
										        Log.debug("docs.length="+docs.length);
									        		
										    } //end of if hitCount > limit
										    else {
										    	Log.debug("using standard sort approach (i.e. no filter)");
										    	for (int i = 0; i < hitCount; i++)
										    	  //docs[i] = new SoftReference(makeSortObject(sortList, hits, i));
									  	    	  docs[i] = makeSortObject(sortList, hits, i);
										    }        
									//END FILTER-SORT        
									*/

		long beforeResultsRetrieve = System.currentTimeMillis();
		
		Log.debug("about to put result set into array for sorting");
      	for (int i = 0; i < hitCount; i++) {
      	  //docs[i] = new SoftReference(makeSortObject(sortList, hits, i)); 
      	  docs[i] = makeSortObject(sortList, hits, i); //USE HARD REFERENCES NOW
      	  
      	}
      	
      	long afterResultsRetrieve = System.currentTimeMillis();
      	
      	Log.debug("Time taken to retrieve results into docs array: " + (afterResultsRetrieve - beforeResultsRetrieve) + "ms.");
        
        Comparator comparator = new DocCompare(sortList);
     	
        try {
          exceptionMessage = "sorting results";
          Arrays.sort(docs, comparator);
        } catch (QueryException e) {
          return error("Ran out of memory while sorting " + hitCount +
                       " search results.  Please restrict search or do not sort them.");
        } catch (OutOfMemoryError e) {
        	Log.error("OutOfMemoryError - while sorting " + hitCount + " documents.");
        	return error("Ran out of memory while sorting " + hitCount +
        				 " search results.  Please restrict search or do not sort them.");
        }
        
      	long afterSort = System.currentTimeMillis();
      	Log.debug("Sort took (in ms):" + (afterSort - beforeSort));
      }

	 beforeOutput = System.currentTimeMillis();

      exceptionMessage = "retrieving results";
      int last = first + limit;
      response.add(COUNT, String.valueOf(hitCount));

      for (int i = first; i != last && i < hitCount; ++i) {
        int index = i;

        if (docs != null) {
          //SortObject sort = (SortObject)docs[i].get(); //no longer using soft references.
          SortObject sort = (SortObject)docs[i]; //USE HARD REFERENCES NOW
          if (sort == null)
            return error("Ran out of memory while retrieving sorted " +
                         "documents after sorting " + hitCount +
                         " search results.  Please restrict search " +
                         "or do not sort them.");
            index = sort.index;  //seems redundant (unreachable)
        }
        Document d = hits.doc(index);
        float score = hits.score(index);
        response.add(I, String.valueOf(i+1)); 
        response.add(RANK, String.valueOf(score));
        for (Iterator fld = returnFields.iterator(); fld.hasNext(); ) {
          FieldDef returnField = (FieldDef)fld.next();
          String value = d.get(returnField.name);

          if (value == null) {
            value = "";
          } else if (returnField.date) {
	            try {
		            long msec = DateField.stringToTime(value);
		            value = String.valueOf(msec / 1000);
	            } catch (java.lang.NumberFormatException e) {
	            	String errorMsg = "Invalid Timestamp for datefield:" + returnField.name + ". " + e.getMessage();
	            	Log.error(errorMsg);
	            	return error(errorMsg);
	            }
          }
          response.add(returnField.name, value);
        }
      }

      // for (int i = 0 ; i < hits.length() && i < 10; i++)
      // {
      //   Document d = hits.doc(i);
      //   Log.debug("Found " + DOMAIN + '=' + d.get(DOMAIN) + ", " +
      //                      ID + '=' + d.get(ID));
      // }

    } catch (IOException e) {
      String message = "Error during query: " + e.toString();
      Log.error(message);
      return error(message);
    } finally {
      if (searcher != null) {
        try {
          // We must always close the IndexReader!
          searcher.close();
        } catch (Throwable e) {
          String message = "Error while closing IndexReader: " + e.toString();
          Log.error(message);
        }
      }
    }

	 long end = System.currentTimeMillis();
	 
  	 Log.debug("ReadQuery:" + String.valueOf(beforeParse - queryStart));
  	 Log.debug("Parse/BuildQuery:" +  String.valueOf(beforeOpenSearcher - beforeParse));
  	 Log.debug("OpenSearcher:" + String.valueOf(beforeSearch - beforeOpenSearcher));
  	 Log.debug("Search: " + String.valueOf(beforeSort - beforeSearch));
  	 Log.debug("Sort: " +   String.valueOf(beforeOutput - beforeSort));
  	 Log.debug("Output: " +String.valueOf(end - beforeOutput));

    return response;
  }

  /**
   * Generate a BooleanQuery from all the Query terms.
   * Field parameter is the default field if not otherwise specified.
   */
  private org.apache.lucene.search.Query buildQuery(
    Analyzer a, String field, Map fieldMap)
  {

	try {
	  int clauseLimitSystem = Integer.parseInt(System.getProperty(PACKAGE + "ClauseLimitSystem"));
	  BooleanQuery.setMaxClauseCount(clauseLimitSystem);
	  
	  Log.debug("setting max clause count to:" + clauseLimitSystem);
	} catch (NumberFormatException e) { 
	  Log.error("invalid setting for ClauseLimitSystem:" 
	            + System.getProperty(PACKAGE + "ClauseLimitSystem"));
	}

    QueryParser qp = new QueryParser(field, a);
    BooleanQuery result = new BooleanQuery();
    boolean anyTerms = false;

    String domain = input.get(DOMAIN);
    if (domain != null)
    {
      Term domainSearch = new Term(DOMAIN, domain);
      result.add(new TermQuery(domainSearch), true, false);
      anyTerms = true;
    }
    
    String query = input.get(QUERY);
    Log.debug("Query (Unparsed): " + query);
    if (query != null)
    {
      try {
        result.add(qp.parse(query), true, false);
        anyTerms = true;
	  } catch (BooleanQuery.TooManyClauses e) {
		//Log the error.
		Log.error("Too many clauses in query: " + query);
		
		org.apache.lucene.search.BooleanClause[] b = result.getClauses();
		Log.error(" - Number of clauses before error: " + b.length);
		Log.error(" - Try increasing ClauseLimitSystem in Server.config.");
		throw new QueryException("Too many clauses in query \"" + query +
								 "\"", e);
      } catch (ParseException e) {
      	//Log the error.
      	Log.warn("Error parsing query: " + query);
      	Log.info(e.getMessage());
      	
		org.apache.lucene.search.BooleanClause[] b = result.getClauses();
		Log.error(" Number of clauses before error: " + b.length);
      	
        throw new QueryException("Error parsing query \"" + query +
                                 "\": " + e.getMessage(), e);
      } catch (org.apache.lucene.queryParser.TokenMgrError e) {
    	  //Log the error.
    	  Log.warn("Error parsing query: " + query);
    	  Log.info(e.getMessage());
    	  throw new QueryException("Error parsing query \"" + query +
    							   "\": " + e.getMessage(), e);
      }
      
    }

    for (Iterator it = getRangeQueries(fieldMap).iterator(); it.hasNext(); )
    {
		org.apache.lucene.search.Query theRange = null;
	  try {
		theRange = (org.apache.lucene.search.Query)it.next();
		result.add(theRange, true, false);
		anyTerms = true;
	  } catch (BooleanQuery.TooManyClauses e) {
		//Log the error.
		Log.error("Too many clauses in query when adding range: " + theRange);
		org.apache.lucene.search.BooleanClause[] b = result.getClauses();
		Log.error(" - Number of clauses before error: " + b.length);
		Log.error(" - Try increasing ClauseLimitSystem in Server.config.");
		throw new QueryException("Too many clauses in query when adding range \"" + theRange +
								 "\"", e);
	  }
    }

    if (!anyTerms)
      throw new QueryException("No search expression!");

  	if (Log.willDebug()) {
		Log.debug("Query (Parsed):   " + result.toString(""));
		org.apache.lucene.search.BooleanClause[] b = result.getClauses();
		Log.debug(" Number of clauses in query: " + b.length);
  	}
    
    return result;
  }

  private Map mapFields(List fieldList)
  {
    Map result = new HashMap();
    for (Iterator it = fieldList.iterator(); it.hasNext(); )
    {
      FieldDef field = (FieldDef)it.next();
      result.put(field.name, field);
    }
    return result;
  }
  /**
   * Obtain a List of FieldDef objects defining the fields to return.
   * If the list of fields names to return includes any that are not
   * in the list of defined fields, then a default FieldDef is made.
   *
   * @param fieldMap The Map of FieldDef objects describing the
   *                  definedfields.
   * @return A List of FieldDef object of the items to be returned to
   *         the client.  These should all be fields that were stored
   *         in the index.
   */

  private List getReturnFields(Map fieldMap)
  {
    // Get names of fields to return.
    List requestedFields = Arrays.asList(input.get(RETURN, USE_APP, SPLIT));
    List returnFields = new LinkedList(requestedFields);

    // Check for mandatory return field "Id". If not present then add.
    if (!returnFields.contains(ID))
      returnFields.add(0, ID);

    // Build a list of FieldDef objects corresponding to field names.
    List result = new LinkedList();

    // Now scan through the list of field names to return looking for
    // the field definitions for those field names.  If they don't
    // exist, then a new, default field definition is created.
    for (Iterator it = returnFields.iterator(); it.hasNext(); ) {
      String name = (String)it.next();
      FieldDef field = (FieldDef)fieldMap.get(name);

      if (field == null)
        field = new FieldDef(name);

      result.add(field);
    }

    // Return a List of field definitions corresponding to the "Return"
    // header values.
    return result;
  }

  /**
   * Obtain a List of Lucene-Query objects representing the Range searches
   * requested.
   */

  private List getRangeQueries(Map fieldMap) {
    List result = new LinkedList();

    for (Iterator it = input.getRanges().iterator(); it.hasNext(); ) {
      RangeDef range = (RangeDef)it.next();
      if (range.from == null && range.to == null)
        throw new QueryException("Missing From or To value for Range field." + range.name);

      FieldDef field = (FieldDef)fieldMap.get(range.name);
      boolean date = field != null && field.date;

      try {
        String from = range.from;
        if (date && from != null)
          from = DateField.dateToString(Application.makeDate(from));

        String to = range.to;
        if (date && to != null)
          to = DateField.dateToString(Application.makeDate(to));

        result.add(new RangeQuery(
                     from == null ? null : new Term(range.name, from),
                     to == null ? null : new Term(range.name, to),
                     true));
      } catch (IllegalArgumentException e) {
        throw new QueryException("Error in range field \"" + range.name +
                                 "\": " + e.getMessage());
      }
    }
    return result;
  }

  /**
   * Obtain a sort-key specification.  Returns null if we're doing a
   * default descending RANK sort.
   */

  private Sort[] getSortList()
  {
    String[] fieldSpecs = input.get(SORT, USE_APP, SPLIT);
    if (fieldSpecs.length == 0)
      return null;
    
    Sort[] result = new Sort[fieldSpecs.length];

    for (int i = 0; i < result.length; ++i)
    {
      String[] subSpec = Util.split(fieldSpecs[i], ":");

      if (subSpec.length < 1 || subSpec.length > 2)
        throw new QueryException("Invalid sort spec: \"" + input.get(SORT) + '"');

      String field = subSpec[0];

      if (field.equals(RANK))
        field = null;

      boolean descending = false;

      if (subSpec.length == 2)
      {
        String direction = subSpec[1].toLowerCase();
        if (direction.startsWith("d"))
          Log.debug("Setting sort order for field(" + field + ") to descending.");
          descending = true;
      }
      result[i] = new Sort(field, descending);
    }

    // See if we're doing a descending RANK sort (the default)

//    for (int i = 0; i < result.length; ++i)
//    {
//      if (result[i].field != null || !result[i].descending)
//        return result;
//    }
//    return null;
	//rewrote this to try and make it clearer.
	if (result.length == 1 && result[0].field == null && result[0].descending == false) {
		Log.debug("Sort term specified was default term.  Ignoring sort request.");
		return null;
	} else {
		return result;
	}

  }

  /**
   * Build an error response for sending back to the client.
   *
   * @param message The text of the error message
   * @return An INDEX-RESPONSE Transmission
   */

  private Transmission error(String message)
  {
    Transmission response = new Transmission(ECommand.QUERY_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    return response;
  }

  /**
   * Build an object containing all the sort keys for one document.
   * This will be sorted using a custom Comparator
   */

  private SortObject makeSortObject(Sort[] sortList, Hits hits, int index)
    throws IOException
  {
    SortObject result = new SortObject(index, sortList.length);
    Document d = hits.doc(index);

    for (int i = 0; i < sortList.length; ++i)
    {
      if (sortList[i].field == null)
        result.key[i] = new Float(hits.score(index));
      else
      {
        String key = d.get(sortList[i].field);
        if (key == null)
          key = "";
        result.key[i] = key;
        
        //Log.debug("Key = " + key);
        
      }
    }
    return result;
  }

  private static class DocCompare implements Comparator
  {
    Sort[] sortList;  // The key to the correct ordering.

    DocCompare(Sort[] sortList)
    {
      this.sortList = sortList;
    }

    // We're comparing objects stored within a soft-reference.  If any
    // of them have evaporated, then that means we've run out of
    // memory.

    public int compare(Object objA, Object objB)
    {
      //SortObject a = (SortObject)((SoftReference)objA).get();
      //SortObject b = (SortObject)((SoftReference)objB).get();
      SortObject a = (SortObject) objA; //USE HARD REFERENCES NOW
	  SortObject b = (SortObject) objB; //USE HARD REFERENCES NOW
    		  
      if (a == null || b == null)
        throw new QueryException("Not enough memory to hold all results");

      for (int i = 0; i < a.key.length; ++i)
      {
        int result = a.key[i].compareTo(b.key[i]);
        if (result != 0)
        {
          if (sortList[i].descending)
            result = -result;

          return result;
        }
      }
      return 0;
    }
  }

  /**
   * This object contains the sort keys for one document, plus a
   * reference to the document.  Once all objects have been
   * appropriately sorted, then they can be refetched and returned to
   * the client.
   */
  private static class SortObject
  {
    final int index;
    final Comparable[] key;

    SortObject(int index, int keyCount)
    {
      this.index = index;
      this.key = new Comparable[keyCount];
    }
  }

  private static class Sort
  {
    // The field name to sort on.  Null signifies the RANK
    final String field;

    // Sort direction
    final boolean descending;

    Sort(String field, boolean descending)
    {
      this.field = field;
      this.descending = descending;
    }
  }
}

