# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package Arch::Session;

use base 'Arch::Storage';

use Arch::Util qw(run_tla _parse_revision_descs);
use Arch::TempFiles qw(temp_dir_name);
use Arch::Changeset;
use Arch::Log;
use Arch::Tree;

sub new ($%) {
	my $class = shift;
	my %init = @_;
	my $self = $class->SUPER::new(%init);
	$self->clear_cache;
	return $self;
}

sub archives ($) { 
	my $self = shift;
	$self->{archives} ||= [ run_tla("archives -n") ];
	return $self->{archives};
}
 
*is_archive_registered = *Arch::Storage::is_archive_managed;
*is_archive_registered = *is_archive_registered;

sub categories ($;$) {
	my $self = shift;
	my $archive  = @_ > 0? shift: $self->{archive};
	die "No working archive" unless defined $archive;

	unless ($self->{categories}->{$archive}) {
		$self->{categories}->{$archive} = [ run_tla("categories", $archive) ];
	}
	return $self->{categories}->{$archive};
}

sub branches ($;$$) {
	my $self = shift;
	my $archive  = @_ > 1? shift: $self->{archive};
	my $category = @_ > 0? shift: $self->{category};
	die "No working archive" unless defined $archive;
	die "No working category" unless defined $category;

	my $full_category =
		$archive
		. '/' . $category;

	unless ($self->{branches}->{$full_category}) {
		$self->{branches}->{$full_category} = [ run_tla("branches", $full_category) ];
	}
	return $self->{branches}->{$full_category};
}

sub versions ($;$$$) {
	my $self = shift;
	my $archive  = @_ > 2? shift: $self->{archive};
	my $category = @_ > 1? shift: $self->{category};
	my $branch   = @_ > 0? shift: $self->{branch};
	$branch = "" unless defined $branch;  # support branchless revisions
	die "No working archive"  unless defined $archive;
	die "No working category" unless defined $category;
	die "No working branch"   unless defined $branch;

	my $full_branch =
		$archive
		. '/' . $category
		. ($branch ne '' ? '--' : '') . $branch;

	unless ($self->{versions}->{$full_branch}) {
		$self->{versions}->{$full_branch} = [ run_tla("versions", $full_branch) ];
		$self->{versions}->{$full_branch} = [ map { s/--/----/; $_ } grep !/--.*--/, @{$self->{versions}->{$full_branch}} ]
			if $branch eq '';
	}
	return $self->{versions}->{$full_branch};
}

sub revisions ($;$$$$) {
	my $self = shift;
	my $archive  = @_ > 3? shift: $self->{archive};
	my $category = @_ > 2? shift: $self->{category};
	my $branch   = @_ > 1? shift: $self->{branch};
	my $version  = @_ > 0? shift: $self->{version};
	die "No working archive"  unless defined $archive;
	die "No working category" unless defined $category;
	die "No working branch"   unless defined $branch;
	die "No working version"  unless defined $version;

	my $full_version =
		$archive 
		. '/' . $category
		. ($branch ne '' ? '--' : '') . $branch
		. '--' . $version;

	unless ($self->{revisions}->{$full_version}) {
		$self->{revisions}->{$full_version} = [ run_tla("revisions", $self->working_name) ];
	}
	return $self->{revisions}->{$full_version};
}

sub get_revision_descs ($;$$$$) {
	my $self = shift;
	my $archive  = @_ > 3? shift: $self->{archive};
	my $category = @_ > 2? shift: $self->{category};
	my $branch   = @_ > 1? shift: $self->{branch};
	my $version  = @_ > 0? shift: $self->{version};
	die "No working archive"  unless defined $archive;
	die "No working category" unless defined $category;
	die "No working branch"   unless defined $branch;
	die "No working version"  unless defined $version;

	my $full_version =
		$archive 
		. '/' . $category
		. ($branch ne '' ? '--' : '') . $branch
		. '--' . $version;

	unless ($self->{revision_descs}->{$full_version}) {
		my $version = $full_version;
		$version =~ s|^.*/||;

		# $ok is used to work around the tla bug with branchless version
		# $prev_line is used to track revisions with no (empty) summary
		my $ok = 0;
		my $prev_line = "";

		my @revision_lines = map { s/^        //? $_: undef }
			grep {
				$ok = /^      \Q$version\E$/ if /^      [^ ]/;
				my $end = ($prev_line =~ /^        /) && ($_ eq "");
				$prev_line = $_;
				($end || /^        /) && $ok
			}
			run_tla("abrowse --desc", $full_version);

		my $revision_descs = _parse_revision_descs(2, \@revision_lines);
		$self->{revision_descs}->{$full_version} = $revision_descs;
		$self->{revisions}->{$full_version} = [ map { $_->{name} } @$revision_descs ];
	}
	return $self->{revision_descs}->{$full_version};
}

*revision_details = *get_revision_descs; *revision_details = *revision_details;

sub clear_cache ($) {
	my $self = shift;

	$self->{archives} = undef;
	$self->{categories} = {};
	$self->{branches} = {};
	$self->{versions} = {};
	$self->{revisions} = {};
	$self->{revision_descs} = {};
}

# [
#   [ category1, [
#     [ branch1, [
#       [ version1, start_revision1, end_revision1 ],
#       [ version2, start_revision2, end_revision2 ],
#     ] ],
#     [ branch2, [
#       [ version3, start_revision3, end_revision3 ],
#       [ version4, start_revision4, end_revision4 ],
#     ] ],
#     ...,
#   ] ],
# ]

sub expanded_archive_info ($;$$) {
	my $self = shift;
	die "expanded_archive_info: no working archive\n" unless defined $self->{archive};
	my $archive_name = $self->working_name || shift;
	my $full_listing = shift || 0;  # currently ignored

	my $infos = [];
	my @category_infos = split(/^\b/m, join('',
		map { s/^  //; "$_\n" } grep { /^  / }
			run_tla("abrowse $archive_name")
	));

	my $error = 0;
	CATEGORY_ITEM:
	foreach (@category_infos) {
		my ($category, $branch_infos) = /^([^\s]+)\n(  .*)$/s;
		push @$infos, [ $category, [] ];
		unless (defined $category) {
			$error = 1; next CATEGORY_ITEM;
		}

		my @branch_infos = split(/^\b/m, join('',
			map { s/^  // or $error = 1; "$_\n" }
				split("\n", $branch_infos)
		));
		$error = 1 unless @branch_infos;
		foreach (@branch_infos) {
			my ($branch, $version_infos) = /^\Q$category\E(?:--([^\s]+))?\n(  .*)$/s;
			$branch = "" if defined $version_infos && !defined $branch;
			unless (defined $branch) {
				$error = 1; next CATEGORY_ITEM;
			}
			push @{$infos->[-1]->[1]}, [ $branch, [] ];

			my @version_infos = split(/^\b/m, join('',
				map { s/^  // or $error = 1; "$_\n" }
					split("\n", $version_infos)
			));
			$error = 1 unless @version_infos;
			foreach (@version_infos) {
				my ($version, $revision0, $revisionl) = /^\Q$category\E(?:--)?\Q$branch\E--([^\s]+)(?:\n  ([^\s]+)(?: \.\. ([^\s]+))?\n)?$/s;
				unless (defined $version) {
					$error = 1; next CATEGORY_ITEM;
				}
				# TODO: consider $full_listing here
				$revision0 = '' unless defined $revision0;
				$revisionl = '' unless defined $revisionl;
				push @{$infos->[-1]->[1]->[-1]->[1]}, [ $version, $revision0, $revisionl ];
			}
		}
	} continue {
		if ($error) {
			warn "Unexpected abrowse output, skipping:\n$_\n";
			pop @$infos;
			$error = 0;
		}
	}
	return $infos;
}

sub get_revision_changeset ($$;$) {
	my $self = shift;
	my $revision = shift;
	my $dir = defined $_[0]? shift: temp_dir_name("arch-changeset");
	die "get_changeset: incorrect dir ($dir)\n" unless $dir && !-d $dir;

	run_tla("get-changeset", $revision, $dir);
	return Arch::Changeset->new($revision, $dir);
}

sub get_changeset ($;$) {
	my $self = shift;
	my $dir = shift;
	my $full_revision = $self->working_name;
	return $self->get_revision_changeset($full_revision, $dir);
}

sub get_revision_log ($$) {
	my $self = shift;
	my $revision = shift || die "get_revision_log: No revision given\n";
	my $message = run_tla("cat-archive-log", $revision);
	die "Can't get log of $revision from archive.\n"
		. "Unexisting revision or system problems.\n"
		unless $message;
	return Arch::Log->new($message);
}

sub get_log ($) {
	my $self = shift;
	die "get_log: no working revision\n" unless defined $self->{revision};
	return $self->get_revision_log($self->working_name);
}

sub get_tree ($;$$) {
	my $self = shift;
	my $revision = shift || $self->working_name || die "get_tree: no b|v|r\n";
	my $dir = shift || temp_dir_name("arch-tree");
	die "get_tree: no directory name (internal error?)\n" unless $dir;
	die "get_tree: directory already exists ($dir)\n" if -d $dir;

	run_tla("get --silent --no-pristine", $revision, $dir);
	die "Can't get revision $revision from archive.\n"
		. "Unexisting revision or system problems.\n"
		unless -d $dir;
	return Arch::Tree->new($dir);
}

sub init_tree ($$;$) {
	my $self = shift;
	my $version = shift || $self->working_name || die "init_tree: no version\n";
	my $dir = shift || ".";

	run_tla("init-tree", "-d", $dir, $version);
	return undef unless $? == 0;
	return Arch::Tree->new($dir);
}

sub my_id ($;$) {
	my $self = shift;
	my $userid = shift;

	if (defined $userid) {
		return 0 unless $userid =~ /<.+\@.*>/;
		run_tla("my-id", $userid);
		return !$?;
	} else {
		($userid) = run_tla("my-id");
		return $userid;
	}
}

1;

__END__

=head1 NAME

Arch::Session - access arch archives

=head1 SYNOPSIS

    use Arch::Session;

    my $session = Arch::Session->new;

    my $rev  = 'migo@homemail.com--Perl-GPL/arch-perl--devel--0--patch-1';
    my $log  = $session->get_revision_log($rev);
    my $cset = $session->get_revision_changeset($rev);
    my $tree = $session->get_tree($rev);

=head1 DESCRIPTION

Arch::Session provides an interface to access changesets and logs
stored in arch archives.

=head1 METHODS

The following methods are available:

B<new>,
B<archives>,
B<categories>,
B<branches>,
B<versions>,
B<revisions>,
B<get_revision_descs>,
B<clear_cache>,
B<expanded_archive_info>,
B<get_revision_changeset>,
B<get_changeset>,
B<get_revision_log>,
B<get_log>,
B<get_tree>,
B<init_tree>,
B<my_id>.

=over 4

=item B<new> [I<args>]

Create a new Arch::Session object.

=item B<archives>

Returns a list of registered archives.

=item B<categories> [I<archive>]

=item B<branches>   [[I<archive>] I<category>]

=item B<versions>   [[[I<archive>] I<category>] I<branch>]

=item B<revisions>  [[[[I<archive>] I<category>] I<branch>] I<version>]

Returns a list of categories, branches, versions or revisions
respectively. Unspecified parent elements default to the output of
B<working_names>.

=item B<get_revision_descs> [[[[I<archive>] I<category>] I<branch>] I<version>]

Returns describing hash for every revision in the specified version,
which defaults to B<working_names> if omitted.

The revision hashes have the following fields:

=over 4

=item B<name>

The revision name (i.e. C<base-0>, C<patch-X>, C<version-X> or C<versionfix-X>)

=item B<summary>

The revision's commit log's summary line

=item B<creator>

The name part of the committers C<tla my-id> (i.e. C<John Hacker>)

=item B<email>

The email address part of the committers C<tla my-id>
(i.e. C<jhacker@nowhere.org>)

=item B<date>

The revisions commit date in C<%Y-%m-%d %H:%M:%S %Z> format (see
L<strftime(3)>)

=item B<kind>

The kind of revision (i.e. one of C<tag>, C<import>, C<cset> or C<unknown>)

=back

=item B<clear_cache>

For performance reasons, most method results are cached. Use this
method to explicitly request this cache to be cleared.

=item B<expanded_archive_info> [I<archive>]

Returns a tree of categories, branches and versions in the
archive. The archive defaults to B<working_names>.

Returns a reference to a list of categories. Every category is a list
consisting of the category name and a list of branches. Every branch
is a list consisting of the branch name and a list of versions. Every
version is list consisting of the version number and the first and
last revision name.

    [
      [ "category1", [
        [ "branch1", [
          [ "version1", "first_revision1", "last_revision1" ],
          [ "version2", "first_revision2", "last_revision2" ],
          ...
        ],
        ...
      ],
      ...
    ]

=item B<get_revision_changeset> I<revision> [I<dir>]

=item B<get_changeset> [I<dir>]

Fetches the changeset for I<revision> or B<working_name> and returns
an Arch::Changeset for it. If I<dir> is specified, it will be used to
store the contents of the changeset. Otherwise a new temporary
directory will be created.

=item B<get_revision_log> I<revision>

=item B<get_log>

Fetch the log for the I<revision> or B<working_name>. Returns an
Arch::Log object.

=item B<get_tree> [I<revision> [I<dir>]]

Construct a working tree for I<revision> or B<working_name> in
I<dir>. If I<dir> is not specified, a new temporary directory is
automatically created.

=item B<init_tree> I<dir>

Run C<tla init-tree> in I<dir>.

=item B<my_id> [I<newid>]

Get or set C<tla my-id>.

=back

=head1 BUGS

No known bugs.

=head1 AUTHORS

Mikhael Goikhman (migo@homemail.com--Perl-GPL/arch-perl--devel).

Enno Cramer (uebergeek@web.de--2003/arch-perl--devel).

=head1 SEE ALSO

For more information, see L<tla>, L<Arch::Storage>, L<Arch::Library>.

=cut
