/*****************************************************************************
 *
 * uniar2.c: Universal Archive Utility
 *
 * Author: Martin N Dunstan (martin@nag.co.uk)
 *
 * Copyright (c) 1990-2007 Aldor Software Organization Ltd (Aldor.org).
 *
 * Bugs:
 *  - if we die during uniWriteArchive() then we MUST restore the original
 *    archive ... that means adding a signal handler etc and remembering
 *    the original size of the archive (so we don't copy over the bits that
 *    we appended during rq operations).
 *  - we assume that (time_t)0 corresponds to 1 Jan 1970 UTC: this is true
 *    for UNIX systems but is unlikely to be true for any other OS.
 *  - we silently ignore references to member "/" in UCB archives
 *  - we have no checks that files to be added are actually files 
 *
 * Notes:
 *  - the generic algorithm is to update the TOC structure within the main
 *    archive structure and then call uniWriteArchive(). The frilly bits
 *    are there to avoid unnecessary calls to uniWriteArchive() because we
 *    don't like creating temporary files (too many ways for things to go
 *    wrong leaving a trashed archive).
 *  - quick-append on AIX/AIX4 archives wastes space because the old table
 *    of contents is left in the archive (unlinked). Since we always keep
 *    the index last we could overwrite it with the appended members.
 *
 * Future work:
 *  - add -h, -help and --help options
 *  - add a --format={ucb,ar,aix,aix4,cms} option to allow the user to
 *    specify the type of archive that they want to create. This option
 *    would be ignored if the archive already existed
 *  - add a --translate={ucb,ar,aix,aix4,cms} option to allow users to
 *    convert archives from one format to another
 *  - see if we can find out what archive format MSVC++ uses, ideally
 *    without breaking copyright (no reverse engineering)
 *  - implement "d" under AIX properly: members are simply unlinked and
 *    placed on the free list. We do not need to rebuild the archive.
 *  - merge common code more: AIX and AIX-big archives are basically
 *    the same and could be handled by the same code using a look-up
 *    table for archive-specific data. Likewise, we could converge our
 *    internal archive format towards AIX-big format and factor out
 *    more code that way.
 *  - provide support for old UCB archives (those for IRIX 4 systems).
 *    This means truncating filenames when appending and not including
 *    a directory table. We warn users if the archive isn't in old UCB.
 *  - allow the "t" option to take a list of members whose details will
 *    be displayed (really only useful in conjunction with "v").
 *
 ****************************************************************************/

/*
 * Get basic configuration information
 */

#define _ALL_SOURCE

#include <ctype.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include "platform.h"

#if !defined(CONFIG)
        /* this should stop things eventually */
        "Cannot determine your configuration. Please edit uniar2.c."
#endif

#if defined(OS_WIN32) && defined(CC_MICROSOFT)
#include <sys\stat.h>
#define stat _stat
static char *uniTempName(char *);
#endif

#if defined(OS_WIN32) && !defined(CC_MICROSOFT)
#include <sys\stat.h>
static char *uniTempName(char *);
#endif

#if defined(OS_MS_DOS) || defined(OS_IBM_OS2)
#include <sys\stat.h>
#endif

#if defined(OS_UNIX)
#include <unistd.h>
#include <sys/stat.h>
static char *uniTempName(char *);
static char *temporary_name(char *, char *);
#endif

#if defined(OS_VMS)
#include <stat.h>
#endif

#if defined(OS_CMS)
struct stat {
    unsigned long st_mode;
    unsigned long st_uid;
    unsigned long st_gid;
    unsigned long st_size;      /* File size in bytes */
    unsigned long st_mtime;     /* Time of last data modification */
};

static unsigned long osFileSize(char *);
static int stat(char *, struct stat *);
static void normalise(char *);
#else
#define normalise(x)
#endif


/*****************************************************************************
 *
 * :: Archive file formats
 *
 ****************************************************************************/
/*
 * All archive formats handled by uniar are basically the same: an archive
 * header record followed by zero or more file segments. Each segment begins
 * with a file header resembling a UNIX stat() buffer and is followed by the
 * file data. Headers and data are aligned on 2-byte boundaries. The first
 * member begins immediately after the archive header for UCB/CMS archives
 * with each member appearing one after the other. For AIX archives, members
 * may appear anywhere within the file: members are stored in a doubly-linked
 * list with previous/next pointers in the headers.
 *
 * File headers contain information such as the name of the member file, its
 * creation date, UID/GID etc. CMS archives can handle names of up to 20
 * characters while AIX archives appear to cope with anything up to 9999.
 * UCB archives handle names of up to 14 characters just like the other
 * archive formats; file names longer than 14 characters are recorded in
 * a directory entry (member name "//") and each member is given the name
 * "/N" where N is the character offset into the directory member where the
 * long filename can be found.
 *
 * With the exception of file access modes (ar_mode field), all numbers are
 * stored as left-justified ASCII decimals padded on the right with spaces.
 * Access modes are left-justified ASCII octals similarly padded.
 *
 * UCB/!<arch>
 * -----------
 *
 * The format of each item header is:
 *    struct ar_hdr
 *    {
 *       char ar_name[16]; // Member file name, sometimes / terminated.
 *       char ar_date[12]; // File date, decimal seconds since Epoch.
 *       char ar_uid[6];   // User ID, in ASCII decimal.
 *       char ar_gid[6];   // Group ID, in ASCII decimal.
 *       char ar_mode[8];  // File mode, in ASCII octal.
 *       char ar_size[10]; // File size, in ASCII decimal.
 *       char ar_fmag[2];  // Always contains ARFMAG ("`\n").
 *    }
 *
 * Member names are usually terminated by "/" and can be up to 15 characters
 * long (including the "/" terminator). Longer names are represented by the
 * name "/" followed by an offset in ASCII. The offset refers to an offset
 * into a special directory table which is stored as an archive member with
 * the name "//". The offset is in characters. In practice we only allow 14
 * characters in filenames. Names in the directory table are terminated by
 * "/\n". Alignment padding at the end of member data is taken as "\n".
 *
 * The fields used in the header for the directory table may differ: under
 * Linux only ar_size is present while under Solaris and IRIX all fields are
 * present. On older IRIX systems (before SysVr4) the directory table is not
 * supported and filenames must be truncated to 15 characters.
 *
 * Archives with a symbol table have a special member called "/".
 *
 * Note that under VMS the UID field will not fit into the five character
 * ar_uid slot so we use UID=100 and MODE=0.
 *
 * CMS
 * ---
 * The format of each item header is:
 *    struct ar_hdr
 *    {
 *       char ar_name[20]; // Member name, format unknown
 *       char ar_date[12]; // File date, decimal seconds since Epoch.
 *       char ar_uid[6];   // User ID, in ASCII decimal.
 *       char ar_gid[6];   // Group ID, in ASCII decimal.
 *       char ar_mode[8];  // File mode, in ASCII octal.
 *       char ar_size[10]; // File size, in ASCII decimal.
 *       char ar_fmag[2];
 *    }
 *
 * File sizes are (apparently) not to be trusted under CMS ...
 *
 * AIX
 * ---
 * The format of the archive header is:
 *    struct ar_ar_hdr
 *    {
 *       char fl_magic[8];    // Archive magic number "<aiaff>\n"
 *       char fl_memoff[12];  // Position of index table in ASCII decimal
 *       char fl_gstoff[12];  // Position of global symbol table
 *       char fl_fstmoff[12]; // Position of first member in ASCII decimal
 *       char fl_lstmoff[12]; // Position of last member in ASCII decimal
 *       char fl_freeoff[12]; // Position of first member on free list
 *    }
 *
 * Offset 32 (fl_fstmoff) of the archive header contains a 12-byte ASCII
 * decimal holding the position of the first member in the archive.
 *
 * The format of each item header is:
 *    struct ar_hdr
 *    {
 *       char ar_size[12]; // File size in ASCII decimal
 *       char ar_next[12]; // Position of next item in ASCII decimal
 *       char ar_prev[12]; // Position of previous item in ASCII decimal
 *       char ar_date[12]; // File date, decimal seconds since Epoch (?)
 *       char ar_uid[12];  // User ID, in ASCII decimal.
 *       char ar_gid[12];  // Group ID, in ASCII decimal.
 *       char ar_mode[12]; // File mode, in ASCII octal.
 *       char ar_nlen[4];  // Length of item name, in ASCII decimal.
 *    }
 * The name of the item follows immediately for ar_nlen characters padded
 * with NULs and with "'\n" as the ARFMAG terminator. This provides support
 * for member names of up to 255 characters.
 *
 * The index table has a header with ar_date, ar_uid, ar_gid and ar_mode
 * all zero. The index table has no name so ar_nlen is also zero.
 *
 * The format of the index table is:
 *    struct ar_idx
 *    {
 *       char ar_num[12]; // Number of entries in the table
 *       char ar_pos1[12]; // Position of member 1 in file (ASCII decimal)
 *       char ar_pos2[12]; // Position of member 2 in file (ASCII decimal)
 *       ...
 *       char ar_name1[];  // NUL terminated name of member 1
 *       char ar_name2[];  // NUL terminated name of member 2
 *       ...
 *    }
 *
 * Note that the index table is linked into the archive as a member with
 * no name. Use the fl_memoff[] field of the archive header to find it. It
 * is padded with NUL at the end to ensure correct alignment and (under AIX)
 * is kept at the end of the archive along with the global symbol table.
 *
 * Note that member order is determined by the linked list NOT by the order
 * in which members are listed in the index table.
 *
 * Deleted members are simply unlinked and placed on the free list. Their
 * contents are left alone and may be overwritten by future append/replace
 * operations if they are big enough.
 *
 * Member data is padded to even-byte boundaries with NULs.
 *
 * AIX4/AIX-big
 * ------------
 * This is essentially the same as standard AIX archives except that
 * member pointers and sizes are stored as 20-byte ASCII decimals.
 *
 * Offset 68 of the archive header contains a 20-byte ASCII decimal
 * holding the position of the first member in the archive.
 *
 * The format of each item header is:
 *    struct ar_hdr
 *    {
 *       char ar_size[20]; // File size in ASCII decimal
 *       char ar_next[20]; // Position of next item in ASCII decimal
 *       char ar_prev[20]; // Position of previous item in ASCII decimal
 *       char ar_date[12]; // File date, decimal seconds since Epoch (?)
 *       char ar_uid[12];  // User ID, in ASCII decimal.
 *       char ar_gid[12];  // Group ID, in ASCII decimal.
 *       char ar_mode[12]; // File mode, in ASCII octal.
 *       char ar_nlen[4];  // Length of item name, in ASCII octal.
 *    }
 * The name of the item follows immediately for ar_nlen[] characters
 * with an additional two (unknown/unspecified) characters as terminator.
 */

/* Number of characters in the magic number */
#define SARMAGLEN	8


/* Useful definitions */
#ifndef SEEK_SET
#define SEEK_SET	0
#endif
#ifndef SEEK_END
#define SEEK_END	2
#endif


/* Useful shorthands */
typedef struct stat StatBuf;


typedef struct {
	int filec;
	char **filev;
} FileVec;


typedef enum arFmtTag {
    AR_START,
        AR_Arch = AR_START,
        AR_AIX,
        AR_AIX4,
        AR_CMS,
    AR_LIMIT
} ArFmtTag;


typedef struct ar_info {
        ArFmtTag	tag;
        char		*magic;
        int		align;
        int		hdrsz;
} ArInfo;


/* UCB member header */
typedef struct ucb_ar_hdr {
	char ar_name[17];	/* Member name from header */
	long ar_date;		/* File date, decimal seconds since Epoch */
	long ar_uid;		/* User ID, in ASCII decimal */
	long ar_gid;		/* Group ID, in ASCII decimal */
	long ar_mode;		/* File mode, in ASCII octal */
	long ar_size;		/* File size, in ASCII decimal */
	char ar_fmag[3];	/* Always contains ARFMAG ("`\n") */

	/* Extra data for ourselves */
	char *ar_rname;		/* Actual member name */
	long ar_pos;		/* Pointer to start of member data */
} UCB_ArHdr;


/* CMS member header */
typedef struct cms_ar_hdr {
	char ar_name[21];	/* Member name, format unknown */
	long ar_date;		/* File date, decimal seconds since Epoch */
	long ar_uid;		/* User ID, in ASCII decimal */
	long ar_gid;		/* Group ID, in ASCII decimal */
	long ar_mode;		/* File mode, in ASCII octal */
	long ar_size;		/* File size, in ASCII decimal */
	char ar_fmag[3];	/* Unknown ARFMAG */
} CMS_ArHdr;


/* AIX member header */
typedef struct aix_ar_hdr {
	long ar_size;		/* File size in ASCII decimal */
	long ar_next;		/* Position of next item in ASCII decimal */
	long ar_prev;		/* Position of previous item */
	long ar_date;		/* File date, decimal seconds since Epoch */
	long ar_uid;		/* User ID, in ASCII decimal */
	long ar_gid;		/* Group ID, in ASCII decimal */
	long ar_mode;		/* File mode, in ASCII octal */
	long ar_nlen;		/* Length of member name, in ASCII octal */
	char *ar_name;		/* Member name of variable length */
	char ar_fmag[3];	/* Unknown ARFMAG */

	/* Extra data for ourselves */
	long ar_pos;		/* Pointer to start of member data */
	long ar_hpos;		/* Pointer to start of member header */
	struct aix_ar_hdr *pre;	/* Pointer to previous member */
	struct aix_ar_hdr *nex;	/* Pointer to next member */
} AIX_ArHdr;


/* AIX4 member header */
typedef struct aix4_ar_hdr {
	char ar_size[20]; /* File size in ASCII decimal */
	char ar_next[20]; /* Position of next item in ASCII decimal */
	char ar_prev[20]; /* Position of previous item in ASCII decimal */
	char ar_date[12]; /* File date, decimal seconds since Epoch (?) */
	char ar_uid[12];  /* User ID, in ASCII decimal. */
	char ar_gid[12];  /* Group ID, in ASCII decimal. */
	char ar_mode[12]; /* File mode, in ASCII octal. */
	char ar_nlen[4];  /* Length of member name, in ASCII octal. */
	char *ar_name;    /* Member name of variable length */
	char ar_fmag[3];  /* Unknown ARFMAG: assume same as UCB (unlikely) */
} AIX4_ArHdr;


/* Structure representing UCB-format archive table-of-contents */
typedef struct ucb_ar_toc {
	int membs;		/* Number of slots in membv */
	int membc;		/* Number of members in archive */
	UCB_ArHdr **membv;	/* Pointer to member headers */
	UCB_ArHdr *stab;	/* Pointer to symbol table header */
} UCB_ArToc;


/* Structure representing AIX-format archive table-of-contents */
typedef struct aix_ar_toc {
	int membs;		/* Number of slots in membv */
	int membc;		/* Number of members in archive */
	AIX_ArHdr **membv;	/* Pointer to member headers (not index) */
	AIX_ArHdr *index;	/* Pointer to index member header */
	long *posv;		/* File addresses of member headers (unused) */
} AIX_ArToc;


/* Generic archive structure */
typedef struct archive {
	char *name;		/* Name of the archive */
	FILE *handle;		/* Handle of the open archive */
	StatBuf statbuf;	/* Filing system details of the archive */
	ArInfo *info;		/* Describes the archive */
	union {
		UCB_ArToc ucb;
		AIX_ArToc aix;
	} toc;
} Archive;


/*
 * These are the "magic number" strings identifying the various sorts
 * of archive file formats.  These need to be given in numeric form,
 * rather than toAscii of a string, since the "\n" is not guaranteed to
 * translate to the same code in a portable way. Currently we assume
 * that all archives use the same magic number to mark the end of the
 * member headers.
 */
static char arstrArch []={33, 60, 97,114, 99,104, 62, 10, 0}; /* !<arch>\n */
static char arstrAIX  []={60, 97,105, 97,102,102, 62, 10, 0}; /* <aiaff>\n */
static char arstrAIX4 []={60, 98,105,103, 97,102, 62, 10, 0}; /* <bigaf>\n */
static char arstrCMSaf[]={60, 67, 77, 83, 97,102, 62, 10, 0}; /* <CMSaf>\n */
static char arstrMagic[]={96, 10,  0};                        /* `\n */


/* Table of information about different archive formats */
ArInfo arInfoTable[] = {
	/* Type    Magic            Align    Header */
        {AR_Arch,  arstrArch,           2,       8},
        {AR_AIX,   arstrAIX,            2,      68},
        {AR_AIX4,  arstrAIX4,           2,     128},
        {AR_CMS,   arstrCMSaf,          2,       8},
	{AR_LIMIT, 0,			0,	 0}
};


/* Buffer for preparing error messages in */
static char errmsg[1024];


/* Buffer for reading/writing into */
static char buffer[1024];


/* Updated by the command-line arguments */
static int opt_delete		= 0;
static int opt_quick_append	= 0;
static int opt_replace		= 0;
static int opt_toc		= 0;
static int opt_extract		= 0;
static int opt_print		= 0;
static int opt_create_quietly	= 0;
static int opt_verbose		= 0;


/* These could be exported for use in libraries */
Archive *	uniOpenArchive		(char *, char *, char *);
ArInfo *	uniArchiveType		(FILE *);
void		uniWriteArchive		(Archive *);
void		uniCreateArchive	(char *, ArFmtTag);
long		uniFirstMember		(FILE *, ArFmtTag);
void		uniListArchive		(Archive *);
void		uniExtractMembers	(Archive *, FileVec *);
void		uniPrintMembers		(Archive *, FileVec *);
void		uniAppendMembers	(Archive *, FileVec *);
void		uniReplaceMembers	(Archive *, FileVec *);
void		uniDeleteMembers	(Archive *, FileVec *);


/* Forward declarations */
static void	process_flags		(int, char **);

/* UCB archive handling */
static void	uniOpenArchiveUCB	(Archive *);
static void	uniCreateArchiveUCB	(char *);
static void	uniWriteArchiveUCB	(Archive *, FILE *);
static void	uniListArchiveUCB	(Archive *);
static void	uniExtractUCB		(Archive *, FileVec *, int);
static void	uniAppendUCB		(Archive *, FileVec *);
static void	uniReplaceUCB		(Archive *, FileVec *);
static void	uniDeleteUCB		(Archive *, FileVec *);
static void	uniAppendTocUCB		(Archive *, UCB_ArHdr *);
static void	uniListMemberUCB	(UCB_ArHdr *);
static void	uniExtractMemberUCB	(UCB_ArHdr *, FILE *, int);
static void	uniAppendMemberUCB	(Archive *, UCB_ArHdr *, char);
static long	uniWriteHeaderUCB	(Archive *, UCB_ArHdr *);
static long	uniFirstMemberUCB	(FILE *);
static long	uniReadHeaderUCB	(FILE *, UCB_ArHdr *, long, long, int);
static char *	uniReadDirUCB		(FILE *, long);

/* AIX archive handling */
static void	uniOpenArchiveAIX	(Archive *);
static void	uniCreateArchiveAIX	(char *);
static void	uniWriteArchiveAIX	(Archive *, FILE *);
static void	uniListArchiveAIX	(Archive *);
static void	uniExtractAIX		(Archive *, FileVec *, int);
static void	uniAppendAIX		(Archive *, FileVec *);
static void	uniReplaceAIX		(Archive *, FileVec *);
static void	uniDeleteAIX		(Archive *, FileVec *);
static void	uniAppendTocAIX		(Archive *, AIX_ArHdr *);
static void	uniListMemberAIX	(AIX_ArHdr *);
static void	uniExtractMemberAIX	(AIX_ArHdr *, FILE *, int);
static void	uniAppendMemberAIX	(Archive *, AIX_ArHdr *, char);
static void	uniAppendIndexAIX	(Archive *);
static long	uniFirstMemberAIX	(FILE *);
static long	uniIndexMemberAIX	(FILE *);
static long	uniWriteHeaderAIX	(Archive *, AIX_ArHdr *);
static void	uniPatchHeadersAIX	(Archive *);
static long	uniReadHeaderAIX	(FILE *, AIX_ArHdr *, long);

/* AIX4 (big) archive handling */
static void	uniOpenArchiveAIX4	(Archive *);
static void	uniCreateArchiveAIX4	(char *);
static void	uniWriteArchiveAIX4	(Archive *, FILE *);
static void	uniListArchiveAIX4	(Archive *);
static void	uniExtractAIX4		(Archive *, FileVec *, int);
static void	uniAppendAIX4		(Archive *, FileVec *);
static void	uniReplaceAIX4		(Archive *, FileVec *);
static void	uniDeleteAIX4		(Archive *, FileVec *);
static long	uniFirstMemberAIX4	(FILE *);

/* CMS archive handling */
static void	uniOpenArchiveCMS	(Archive *);
static void	uniCreateArchiveCMS	(char *);
static void	uniWriteArchiveCMS	(Archive *, FILE *);
static void	uniListArchiveCMS	(Archive *);
static void	uniExtractCMS		(Archive *, FileVec *, int);
static void	uniAppendCMS		(Archive *, FileVec *);
static void	uniReplaceCMS		(Archive *, FileVec *);
static void	uniDeleteCMS		(Archive *, FileVec *);
static long	uniFirstMemberCMS	(FILE *);

/* General utility functions */
static int	uniReadString		(FILE *, char *, int);
static void	uniReadText		(FILE *, char *, int);
static void	uniReadNumber		(FILE *, long *, int, int);
static void	uniReadDecimal		(FILE *, long *, int);
static void	uniReadOctal		(FILE *, long *, int);
static char *	uniModeUNIX		(long);
static char *	uniTemporaryName	(char *);
static void	uniWriteData		(FILE *, FILE *, long);
static char *	dup_str			(char *);
static FILE *	open_or_error		(char *, char *, char *);
static void	option_error		(char *);
static void	fatal_error		(char *);
static void	message			(char *);
static void	syntax			(void);
static int	fileExists		(char *);


int
main(int argc, char **argv)
{
	FileVec	fv;
	char	*arname;
	Archive *archive;

	/* Deal with the main flags */
	process_flags(argc, argv);


	/* Extract the name of the archive and normalise it */
	arname = dup_str(argv[2]);
	normalise(arname);


	/* Check for the specified archive, creating if necessary */
	if (!fileExists(arname)) {
		/* Quick append or replace create archives if not there */
		if (opt_quick_append || opt_replace) {
			uniCreateArchive(arname, AR_Arch);

			if (!opt_create_quietly) {
				char *fmt = "created archive %s";
				(void)sprintf(errmsg, fmt, arname);
				message(errmsg);
			}
		}
		else {
			char *fmt = "archive '%s' must already exist";
			(void)sprintf(errmsg, fmt, arname);
			fatal_error(errmsg);
		}
	}


	/* Open the archive for reading */
	archive = uniOpenArchive(arname, "rb", "for reading");


	/* Skip past the command name, options and archive name */
	argc -= 3, argv += 3;
	fv.filec = argc, fv.filev = argv;


	/* Perform the specified action */
	if (opt_toc) {
		if (argc) option_error("too many arguments for 't' option");
		uniListArchive(archive);
	}
	else if (opt_extract)
		uniExtractMembers(archive, &fv);
	else if (opt_print)
		uniPrintMembers(archive, &fv);
	else if (opt_quick_append)
		uniAppendMembers(archive, &fv);
	else if (opt_replace)
		uniReplaceMembers(archive, &fv);
	else if (opt_delete)
		uniDeleteMembers(archive, &fv);
	else
		fatal_error("operation not supported yet");


	/* Be tidy - close the archive */
	if (archive->handle) (void)fclose(archive->handle);


	/* Return quietly */
	return 0;
}


static void
process_flags(int argc, char **argv)
{
	char	*flags = 0;
	int	disjointOptions = 0;

	/*
	 * At least two arguments are always required. The first is the set
	 * of options, no hyphens, no blanks; the second is the archive; the
	 * others are member names.
	 */
	if (argc < 3) option_error("two arguments are required");


	/* Process the flags */
	for (flags = argv[1]; *flags; flags++) {
		switch (*flags) {
		/* We must have precisely one of the following */
		case 'd': opt_delete         = 1; disjointOptions++; break;
		case 'p': opt_print          = 1; disjointOptions++; break;
		case 'q': opt_quick_append   = 1; disjointOptions++; break;
		case 'r': opt_replace        = 1; disjointOptions++; break;
		case 't': opt_toc            = 1; disjointOptions++; break;
		case 'x': opt_extract        = 1; disjointOptions++; break;

		/* We can have any number of the following */
		case 'c': opt_create_quietly = 1; break;
		case 'v': opt_verbose        = 1; break;

		default:
			(void)sprintf(errmsg, "unknown option '%c'", *flags);
			option_error(errmsg);
		}
	}


	/* Safety check */
	if (disjointOptions != 1)
		option_error("exactly one of [pqrtx] must be specified");
}

/*****************************************************************************
 *
 * :: Generic interface to archive handling code
 *
 ****************************************************************************/

ArInfo *
uniArchiveType(FILE *handle)
{
	int i;
	char magic[SARMAGLEN + 1];

	/* Make sure that we are at the start of the file */
	if (fseek(handle, 0L, SEEK_SET))
		fatal_error("unable to seek to start of archive");


	/* Read the magic number */
	if (uniReadString(handle, magic, SARMAGLEN))
		fatal_error("unable to determine archive type");


	/* Compare it with the magic numbers that we recognise */
	for (i = 0; arInfoTable[i].tag != AR_LIMIT; i++)
		if (!strcmp(magic, arInfoTable[i].magic))
			return &arInfoTable[i];


	/* Not recognised */
	fatal_error("unsupported archive format");
	return (ArInfo *)0;
}


Archive *
uniOpenArchive(char *name, char *mode, char *why)
{
	FILE *handle = open_or_error(name, mode, why);
	ArInfo *info = uniArchiveType(handle);
	Archive *result = (Archive *)malloc(sizeof(*result));


	/* Check that we have enough memory */
	if (!result) fatal_error("not enough memory for archive structure");


	/* Check the existence of the specified archive */
	if (stat(name, &(result->statbuf))) fatal_error("can't stat archive");


	/* Initialise the archive structure */
	result->name = name;
	result->handle = handle;
	result->info = info;


	/* Format-specific code */
	switch (info->tag) {
        case AR_Arch:	uniOpenArchiveUCB(result); break;
        case AR_AIX:	uniOpenArchiveAIX(result); break;
        case AR_AIX4:	uniOpenArchiveAIX4(result); break;
        case AR_CMS:	uniOpenArchiveCMS(result); break;
	default:	fatal_error("unhandled case");
	}


	/* Return the updated archive structure */
	return result;
}


void
uniWriteArchive(Archive *ar)
{
	FILE *backup;

	/* Create a temporary name for the backup and open for writing */
	char *backup_name = uniTemporaryName("ar");
	if (!backup_name) fatal_error("no spare names for temporary file");
	backup = open_or_error(backup_name, "wb", "for writing");


	/* Re-open the archive for reading and copy to the backup */
	(void)fclose(ar->handle);
	ar->handle = open_or_error(ar->name, "rb", "for reading");
	uniWriteData(ar->handle, backup, ar->statbuf.st_size);


	/* Close the backup and re-open for reading */
	(void)fclose(backup);
	backup = open_or_error(backup_name, "rb", "for reading");


	/* Close the original archive and recreate */
	(void)fclose(ar->handle);
	uniCreateArchive(ar->name, ar->info->tag);
	ar->handle = open_or_error(ar->name, "ab", "for appending");


	/* Get the details of the new empty archive */
	if (stat(ar->name, &(ar->statbuf))) fatal_error("can't stat archive");


	/* Format-specific code */
	switch (ar->info->tag) {
        case AR_Arch:	uniWriteArchiveUCB(ar, backup); break;
        case AR_AIX:	uniWriteArchiveAIX(ar, backup); break;
        case AR_AIX4:	uniWriteArchiveAIX4(ar, backup); break;
        case AR_CMS:	uniWriteArchiveCMS(ar, backup); break;
	default:	fatal_error("unhandled case");
	}


	/* Close the backup and remove it */
	(void)fclose(backup);
	(void)unlink(backup_name);
}


void
uniCreateArchive(char *name, ArFmtTag fmt)
{
	switch (fmt) {
        case AR_Arch:	uniCreateArchiveUCB(name); break;
        case AR_AIX:	uniCreateArchiveAIX(name); break;
        case AR_AIX4:	uniCreateArchiveAIX4(name); break;
        case AR_CMS:	uniCreateArchiveCMS(name); break;
	default:	fatal_error("unhandled case");
	}
}


void
uniListArchive(Archive *ar)
{
	switch (ar->info->tag) {
        case AR_Arch:	uniListArchiveUCB (ar); break;
        case AR_AIX:	uniListArchiveAIX (ar); break;
        case AR_AIX4:	uniListArchiveAIX4(ar); break;
        case AR_CMS:	uniListArchiveCMS (ar); break;
	default:	fatal_error("unhandled case");
	}
}


void
uniExtractMembers(Archive *ar, FileVec *fv)
{
	switch (ar->info->tag) {
        case AR_Arch:	uniExtractUCB (ar, fv, 0); break;
        case AR_AIX:	uniExtractAIX (ar, fv, 0); break;
        case AR_AIX4:	uniExtractAIX4(ar, fv, 0); break;
        case AR_CMS:	uniExtractCMS (ar, fv, 0); break;
	default:	fatal_error("unhandled case");
	}
}


void
uniPrintMembers(Archive *ar, FileVec *fv)
{
	switch (ar->info->tag) {
        case AR_Arch:	uniExtractUCB (ar, fv, 1); break;
        case AR_AIX:	uniExtractAIX (ar, fv, 1); break;
        case AR_AIX4:	uniExtractAIX4(ar, fv, 1); break;
        case AR_CMS:	uniExtractCMS (ar, fv, 1); break;
	default:	fatal_error("unhandled case");
	}
}


void
uniAppendMembers(Archive *ar, FileVec *fv)
{
	switch (ar->info->tag) {
        case AR_Arch:	uniAppendUCB (ar, fv); break;
        case AR_AIX:	uniAppendAIX (ar, fv); break;
        case AR_AIX4:	uniAppendAIX4(ar, fv); break;
        case AR_CMS:	uniAppendCMS (ar, fv); break;
	default:	fatal_error("unhandled case");
	}
}


void
uniReplaceMembers(Archive *ar, FileVec *fv)
{
	switch (ar->info->tag) {
        case AR_Arch:	uniReplaceUCB (ar, fv); break;
        case AR_AIX:	uniReplaceAIX (ar, fv); break;
        case AR_AIX4:	uniReplaceAIX4(ar, fv); break;
        case AR_CMS:	uniReplaceCMS (ar, fv); break;
	default:	fatal_error("unhandled case");
	}
}


void
uniDeleteMembers(Archive *ar, FileVec *fv)
{
	switch (ar->info->tag) {
        case AR_Arch:	uniDeleteUCB (ar, fv); break;
        case AR_AIX:	uniDeleteAIX (ar, fv); break;
        case AR_AIX4:	uniDeleteAIX4(ar, fv); break;
        case AR_CMS:	uniDeleteCMS (ar, fv); break;
	default:	fatal_error("unhandled case");
	}
}


/* Jump to the first member of the archive */
long
uniFirstMember(FILE *handle, ArFmtTag fmt)
{
	switch (fmt) {
        case AR_Arch:	return uniFirstMemberUCB(handle); break;
        case AR_AIX:	return uniFirstMemberAIX(handle); break;
        case AR_AIX4:	return uniFirstMemberAIX4(handle); break;
        case AR_CMS:	return uniFirstMemberCMS(handle); break;
	default:	fatal_error("unhandled case"); break;
	}
	return 0L;
}

/*****************************************************************************
 *
 * :: UCB-style archives
 *
 ****************************************************************************/


/* Current directory table for UCB archives (used by uniOpenArchiveUCB) */
static char *ucb_dir_table = 0;
static long ucb_dir_table_len = 0;


static void
uniOpenArchiveUCB(Archive *ar)
{
	long next;
	UCB_ArHdr *hdr;
	FILE *handle = ar->handle;
	long arsize = (long)(ar->statbuf.st_size);
	int align = arInfoTable[AR_Arch].align;
	int want = 64; /* Start with a TOC with 64 slots */


	/* Make sure that we start at the first member */
	next = uniFirstMemberUCB(handle);


	/* Initial TOC size */
	ar->toc.ucb.membs = want;
	ar->toc.ucb.membc = 0;
	ar->toc.ucb.membv = (UCB_ArHdr **)malloc(want*sizeof(UCB_ArHdr *));
	if (!ar->toc.ucb.membv) fatal_error("not enough memory for toc");
	ar->toc.ucb.stab = 0;


	/* Read the header of every entry in the archive */
	while (next < arsize) {
		/* Make sure that we are at the correct position */
		if (fseek(handle, next, SEEK_SET))
			fatal_error("cannot seek to member");


		/* Allocate storage for the header */
		hdr = (UCB_ArHdr *)malloc(sizeof(*hdr));
		if (!hdr) fatal_error("not enough memory for member header");


		/* Read the header for this member */
		next = uniReadHeaderUCB(handle, hdr, next, arsize, align);
		if (!next) break; /* Soft-error during read */


		/* Store the symbol table separately */
		if (!hdr->ar_rname)
			ar->toc.ucb.stab = hdr;
		else
			uniAppendTocUCB(ar, hdr);
	}


	/* Finished with the directory table */
	if (ucb_dir_table) (void)free(ucb_dir_table);
	ucb_dir_table = (char *)0;
	ucb_dir_table_len = 0;
}


static void
uniCreateArchiveUCB(char *name)
{
	FILE *handle = open_or_error(name, "wb", "for writing");
	(void)fprintf(handle, "%s", arInfoTable[AR_Arch].magic);
	(void)fclose(handle);
}


static void
uniWriteArchiveUCB(Archive *ar, FILE *backup)
{
	int membc;
	UCB_ArHdr **membv;
	long dir_len = 0;
	int align = arInfoTable[AR_Arch].align;


	/* Do we need to create a directory table? */
	membc = ar->toc.ucb.membc, membv = ar->toc.ucb.membv;
	for (; membc; membc--, membv++) {
		UCB_ArHdr *hdr = *membv;
		int slen = strlen(hdr->ar_rname);


		/* Record the length of long names */
		if (slen > 14) dir_len += slen + 2; /* +2 for "/ " */
	}


	/* Create the directory if necessary */
	if (dir_len) {
		int dir_pos = 0;
		char *dir, *dirp;
		UCB_ArHdr hdr;


		/* Allocate an aligned directory block */
		if (dir_len % align) dir_len += align - (dir_len % align);
		dir = dirp = (char *)malloc((dir_len + 1)*sizeof(char));
		if (!dir) fatal_error("not enough memory for directory");


		/* Create the directory and update the TOC */
		membc = ar->toc.ucb.membc, membv = ar->toc.ucb.membv;
		for (; membc; membc--, membv++) {
			UCB_ArHdr *hdr = *membv;
			int slen = strlen(hdr->ar_rname);


			/* Ignore members with short names */
			if (slen <= 14) continue;


			/* Create a directory entry and update the TOC */
			(void)sprintf(dirp, "%s/%c", hdr->ar_rname, 10);
			(void)sprintf(hdr->ar_name, "/%d", dir_pos);


			/* Move onto the next entry */
			dir_pos += slen + 2;
			dirp += slen + 2;
		}


		/* Create a header record for the directory */
		hdr.ar_rname = "//"; /* dup_str not needed here */
		(void)strcpy(hdr.ar_name, hdr.ar_rname);
		hdr.ar_date = ar->statbuf.st_mtime;
		hdr.ar_uid  = ar->statbuf.st_uid;
		hdr.ar_gid  = ar->statbuf.st_gid;
		hdr.ar_mode = ar->statbuf.st_mode;
		hdr.ar_size = (long)dir_len;


		/* Write out the directory header */
		(void)uniWriteHeaderUCB(ar, &hdr);


		/* Write out the directory entry itself */
		if (fwrite(dir, dir_len, 1, ar->handle) != 1)
			fatal_error("failed to write directory table");
	}


	/* Write the old symbol table if present */
	if (ar->toc.ucb.stab) {
		UCB_ArHdr *hdr = ar->toc.ucb.stab;
		long siz = hdr->ar_size;


		/* Write out the header for this member */
		(void)uniWriteHeaderUCB(ar, hdr);


		/* Seek to the data in the backup */
		if (fseek(backup, hdr->ar_pos, SEEK_SET))
			fatal_error("failed to seek in backup for stab");


		/* Copy over the data */
		uniWriteData(backup, ar->handle, siz);


		/* Align the member */
		for (; siz % align; siz++)
			(void)fputc('\n', ar->handle);
	}


	/*
	 * Use the table of contents to read members from the backup
	 * and write them to the archive. We ignore member headers
	 * in the backup and use the table of contents.
	 */
	membc = ar->toc.ucb.membc, membv = ar->toc.ucb.membv;
	for (; membc; membc--, membv++) {
		UCB_ArHdr *hdr = *membv;
		long siz = hdr->ar_size;


		/* Write out the header for this member */
		(void)uniWriteHeaderUCB(ar, hdr);


		/* Seek to the data in the backup */
		if (fseek(backup, hdr->ar_pos, SEEK_SET))
			fatal_error("failed to seek in backup for member");


		/* Copy over the data */
		uniWriteData(backup, ar->handle, siz);


		/* Align the member */
		for (; siz % align; siz++)
			(void)fputc('\n', ar->handle);
	}
}


static void
uniListArchiveUCB(Archive *ar)
{
	int membc = ar->toc.ucb.membc;
	UCB_ArHdr **membv = ar->toc.ucb.membv;


	/* Display the details of each member */
	for (; membc; membc--, membv++)
		uniListMemberUCB(*membv);
}


static void
uniExtractUCB(Archive *ar, FileVec *fv, int print)
{
	/* If no files specified, extract them all */
	if (!fv->filec) {
		int membc = ar->toc.ucb.membc;
		UCB_ArHdr **membv = ar->toc.ucb.membv;

		for (; membc; membc--, membv++)
			uniExtractMemberUCB(*membv, ar->handle, print);
		return;
	}


	/* Extract in the order specified, not archive order! */
	for (; fv->filec; fv->filec--, fv->filev++) {
		char *name = *fv->filev;
		int membc = ar->toc.ucb.membc;
		UCB_ArHdr **membv = ar->toc.ucb.membv;


		/* Ensure that the name has been normalised */
		normalise(name);


		/* Search for this member */
		for (; membc; membc--, membv++) {
			UCB_ArHdr *hdr = *membv;


			/* Only extract if the name matches */
			if (!strcmp(name, hdr->ar_rname)) {
				uniExtractMemberUCB(hdr, ar->handle, print);
				break;
			}
		}


		/* Complain if member not found */
		if (!membc) {
			(void)sprintf(errmsg, "%s not found in archive", name);
			fatal_error(errmsg);
		}
	}
}


static void
uniAppendUCB(Archive *ar, FileVec *fv)
{
	int filec;
	char **filev;


	/* Must specify at least one file */
	if (!fv->filec) fatal_error("no files to append to archive");


	/* We will always append the files to the archive */
	(void)fclose(ar->handle);
	ar->handle = open_or_error(ar->name, "ab", "for appending");


	/* Validate the lengths of the file names to be added */
	for (filec = fv->filec, filev = fv->filev; filec; filec--, filev++)
		if (strlen(*filev) > 14) break;


	/*
	 * If all the files to be added have short names then we
	 * can perform this action without using a temporary file.
	 */
	if (!filec) {
		/* We re-use the same header structure */
		UCB_ArHdr hdr;


		/* Append each file in turn */
		filec = fv->filec, filev = fv->filev;
		for (; fv->filec; fv->filec--, fv->filev++) {
			/* Extract and normalise the name */
			char *name = *filev;
			normalise(name);


			/* Initialise the header input fields */
			hdr.ar_rname = name; /* dup_str not needed here */
			(void)strcpy(hdr.ar_name, name);


			/* Append the member */
			uniAppendMemberUCB(ar, &hdr, 'a');
		}
		return;
	}


	/* Append data to the current archive and update our TOC */
	for (filec = fv->filec, filev = fv->filev; filec; filec--, filev++) {
		char *name = *filev;
		UCB_ArHdr *hdr;


		/* Normalise the name */
		normalise(name);


		/* Allocate storage for the header */
		hdr = (UCB_ArHdr *)malloc(sizeof(*hdr));
		if (!hdr) fatal_error("not enough memory for member header");


		/*
		 * Initialise the header input fields: when we rewrite
		 * the archive the header fields will be recreated from
		 * our internal TOC so the member name is irrelevent if
		 * it is longer than 14 characters.
		 */
		hdr->ar_rname = dup_str(name);
		if (strlen(hdr->ar_rname) > 14)
			(void)strcpy(hdr->ar_name, "/0");
		else
			(void)strcpy(hdr->ar_name, hdr->ar_rname);


		/* Append the member with its fake name */
		uniAppendMemberUCB(ar, hdr, 'a');
		uniAppendTocUCB(ar, hdr);
	}


	/* Rebuild the whole archive from the TOC */
	uniWriteArchive(ar);
}


static void
uniReplaceUCB(Archive *ar, FileVec *fv)
{
	char **filev;
	UCB_ArHdr **membv;
	int i, filec, membc;
	int must_replace = 0;
	int *replace;


	/* Must specify at least one file */
	if (!fv->filec) fatal_error("no files to replace");


	/* We will always append the files to the archive */
	(void)fclose(ar->handle);
	ar->handle = open_or_error(ar->name, "ab", "for appending");


	/* Record slots of members already in the archive */
	replace = (int *)malloc(fv->filec*sizeof(int));
	if (!replace) fatal_error("not enough memory for internal array");
	for (i = 0; i < fv->filec; i++)
		replace[i] = -1;


	/* Validate the lengths of the file names to be added */
	for (filec = fv->filec, filev = fv->filev; filec; filec--, filev++) {
		char *name = *filev;


		/* Normalise the name */
		normalise(name);


		/* Check the length */
		if (strlen(name) > 14) must_replace = 1;


		/* Check to see if we are replacing or appending */
		membc = ar->toc.ucb.membc, membv = ar->toc.ucb.membv;
		for (i = 0; membc; i++, membc--, membv++) {
			UCB_ArHdr *hdr = *membv;


			/* Check for existence of the file in the archive */
			if (!strcmp(hdr->ar_rname, name)) {
				must_replace = 1;
				replace[fv->filec - filec] = i;
				break;
			}
		}
	}


	/*
	 * If all the files to be added have short names and none
	 * exist in the archive already then we can perform this
	 * action without using a temporary file.
	 */
	if (!must_replace) {
		/* We re-use the same header structure */
		UCB_ArHdr hdr;


		/* Append each file in turn */
		for (; fv->filec; fv->filec--, fv->filev++) {
			/* Extract the name (already normalised) */
			char *name = *fv->filev;


			/* Initialise the header input fields */
			hdr.ar_rname = name; /* dup_str not needed here */
			(void)strcpy(hdr.ar_name, name);


			/* Append the member */
			uniAppendMemberUCB(ar, &hdr, 'a');
		}
		return;
	}


	/* Append data to the current archive and update our TOC */
	for (filec = fv->filec, filev = fv->filev; filec; filec--, filev++) {
		char *name = *filev; /* Already normalised */
		int member = replace[fv->filec - filec];
		UCB_ArHdr *hdr;


		/* Allocate storage for the header */
		hdr = (UCB_ArHdr *)malloc(sizeof(*hdr));
		if (!hdr) fatal_error("not enough memory for member header");


		/*
		 * Initialise the header input fields: when we rewrite
		 * the archive the header fields will be recreated from
		 * our internal TOC so the member name is irrelevent for
		 * long names.
		 */
		hdr->ar_rname = dup_str(name);
		if (strlen(hdr->ar_rname) > 14)
			(void)strcpy(hdr->ar_name, "/0");
		else
			(void)strcpy(hdr->ar_name, hdr->ar_rname);


		/* Decide if we are replacing or appending */
		if (member < 0) {
			/* Appending: append the member with its fake name */
			uniAppendMemberUCB(ar, hdr, 'a');
			uniAppendTocUCB(ar, hdr);
		}
		else {
			/* Replacing: append the member with its fake name */
			uniAppendMemberUCB(ar, hdr, 'r');


			/* Release the original TOC entry */
			(void)free(ar->toc.ucb.membv[member]->ar_rname);
			(void)free(ar->toc.ucb.membv[member]);


			/* Replace the TOC entry */
			ar->toc.ucb.membv[member] = hdr;
		}
	}


	/* Rebuild the whole archive from the TOC */
	uniWriteArchive(ar);
}


static void
uniDeleteUCB(Archive *ar, FileVec *fv)
{
	char *delete;
	char **filev;
	UCB_ArHdr **membv;
	int i, filec, membc, from, to;


	/* Must specify at least one file */
	if (!fv->filec) fatal_error("no members to delete");


	/* Mark members which are to be deleted */
	delete = (char *)malloc((ar->toc.ucb.membc + 1)*sizeof(char));
	if (!delete) fatal_error("not enough memory for internal bitmap");
	for (i = 0; i < ar->toc.ucb.membc; i++)
		delete[i] = 0;


	/* Record the entries in the TOC which are to be deleted */
	for (filec = fv->filec, filev = fv->filev; filec; filec--, filev++) {
		char *name = *filev;


		/* Normalise the name */
		normalise(name);


		/* Locate the member in the TOC */
		membc = ar->toc.ucb.membc, membv = ar->toc.ucb.membv;
		for (i = 0; membc; i++, membc--, membv++) {
			UCB_ArHdr *hdr = *membv;


			/* Check for existence of the file in the archive */
			if (!strcmp(hdr->ar_rname, name)) {
				delete[i] = '*';
				break;
			}
		}


		/* Did we find the specified member? */
		if (!membc) {
			char *fmt = "%s not found in archive";
			(void)sprintf(errmsg, fmt, name);
			fatal_error(errmsg);
		}
	}


	/* Compress the TOC */
	membc = ar->toc.ucb.membc, membv = ar->toc.ucb.membv;
	for (to = from = 0; membc; from++, membc--, membv++) {
		UCB_ArHdr *hdr = ar->toc.ucb.membv[from];

		if (delete[from]) {
			/* Tell them what we are doing */
			if (opt_verbose)
				(void)printf("d - %s\n", hdr->ar_rname);


			/* Release storage and decrement member count */
			(void)free(hdr->ar_rname);
			(void)free(hdr);
			ar->toc.ucb.membc--;
		}
		else
			ar->toc.ucb.membv[to++] = hdr;
	}


	/* Rebuild the whole archive from the TOC */
	uniWriteArchive(ar);
}


static void
uniAppendTocUCB(Archive *ar, UCB_ArHdr *hdr)
{
	/* Do we need to increase the TOC size? */
	if (ar->toc.ucb.membc == ar->toc.ucb.membs) {
		/* Double the table size */
		int  i, want = ar->toc.ucb.membs + ar->toc.ucb.membs;
		UCB_ArHdr **newv, **oldv = ar->toc.ucb.membv;


		/* Allocate a new table */
		newv = (UCB_ArHdr **)malloc(want*sizeof(UCB_ArHdr *));
		if (!newv) fatal_error("not enough memory for toc");


		/* Copy the old contents across */
		for (i = 0; i < ar->toc.ucb.membs; i++)
			newv[i] = oldv[i];


		/* Release storage for the old table */
		(void)free(oldv);


		/* Start using the new table */
		ar->toc.ucb.membs = want;
		ar->toc.ucb.membv = newv;
	}


	/* Add to the TOC */
	ar->toc.ucb.membv[ar->toc.ucb.membc++] = hdr;
}


static long
uniFirstMemberUCB(FILE *handle)
{
	/* Simply skip past the header */
	long pos = (long)arInfoTable[AR_Arch].hdrsz;
	if (fseek(handle, pos, SEEK_SET))
		fatal_error("unable to seek to first member");


	/* Return the current position */
	return pos;
}


static void
uniListMemberUCB(UCB_ArHdr *hdr)
{
	char stime[30];
	char uid_gid[30];
	struct tm *ltime;


	/*
	 * Convert epoch time into local time. Assumes that
	 * the platform on which the archive was created uses
	 * the same epoch as this platform. [FIXME]
	 */
	ltime = localtime(&(hdr->ar_date));
	strftime(stime, 30, "%b %d %H:%M %Y", ltime);


	/* Format the UID/GID */
	(void)sprintf(uid_gid, "%ld/%ld", hdr->ar_uid, hdr->ar_gid);


	/* Display the details */
	if (opt_verbose) {
		(void)printf("%s %11s %6ld %s %s\n",
			uniModeUNIX(hdr->ar_mode), uid_gid,
			hdr->ar_size, stime, hdr->ar_rname);
	}
	else
		(void)printf("%s\n", hdr->ar_rname);
}


/* Extract a single archive member */
static void
uniExtractMemberUCB(UCB_ArHdr *hdr, FILE *ar_handle, int print)
{
	FILE *handle = stdout;


	/* Seek to the position of the member data */
	if (fseek(ar_handle, hdr->ar_pos, SEEK_SET)) {
		char *fmt = "cannot seek to %s in archive";
		(void)sprintf(errmsg, fmt, hdr->ar_rname);
		fatal_error(errmsg);
	}


	/* Work out where we are extracting the member to */
	if (!print) {
		/* Open/create the output file (clobber existing file) */
		handle = fopen(hdr->ar_rname, "wb");
		if (!handle) {
			char *fmt = "unable to open %s for writing";
			(void)sprintf(errmsg, fmt, hdr->ar_rname);
			fatal_error(errmsg);
		}
	}


	/* Tell them what we are doing */
	if (opt_verbose) (void)printf("x - %s\n", hdr->ar_rname);


	/* Write all the data */
	uniWriteData(ar_handle, handle, hdr->ar_size);
	if (!print) (void)fclose(handle);
}


static void
uniAppendMemberUCB(Archive *ar, UCB_ArHdr *hdr, char mark)
{
	long hsize;
	FILE *source;
	FILE *handle = ar->handle;
	StatBuf *statbuf = &(ar->statbuf);
	StatBuf mstatbuf;
	int align = arInfoTable[AR_Arch].align;


	/* Get details of the file we are appending */
	if (stat(hdr->ar_rname, &mstatbuf)) {
		(void)sprintf(errmsg, "can't stat %s", hdr->ar_rname);
		fatal_error(errmsg);
	}


	/* File UID's under VMS won't fit into 5 characters */
#if defined(OS_VMS)
	mstatbuf.st_uid = 100;
	mstatbuf.st_mode = 0;
#endif


	/* Fill in the header details */
	hdr->ar_date = (long)mstatbuf.st_mtime;
	hdr->ar_uid  = (long)mstatbuf.st_uid;
	hdr->ar_gid  = (long)mstatbuf.st_gid;
	hdr->ar_mode = (long)mstatbuf.st_mode;
	hdr->ar_size = (long)mstatbuf.st_size;


	/* Open the file for reading */
	source = open_or_error(hdr->ar_rname, "rb", "for reading");


	/* Seek to the end of the archive */
	if (fseek(handle, 0L, SEEK_END)) fatal_error("seek to end failed");


	/* Tell them what we are doing */
	if (opt_verbose) printf("%c - %s\n", mark, hdr->ar_rname);


	/*
	 * We trust the alignment of the archive header and all members
	 * except for the last one. Use "statbuf" to ensure alignment.
	 */
	for (; statbuf->st_size % align; statbuf->st_size++)
		(void)fputc('\n', handle);


	/* Write out a header for this member */
	hsize = uniWriteHeaderUCB(ar, hdr);


	/* Record the offset of the data */
	hdr->ar_pos = statbuf->st_size += hsize;


	/* Append the file data */
	uniWriteData(source, handle, mstatbuf.st_size);


	/* Align the member */
	for (; mstatbuf.st_size % align; mstatbuf.st_size++)
		(void)fputc('\n', handle);


	/* Update the archive size */
	statbuf->st_size += mstatbuf.st_size;
}


static long
uniWriteHeaderUCB(Archive *ar, UCB_ArHdr *hdr)
{
	char name[17];
	char first = *hdr->ar_name;

	/* Format the name: trailing / for normal names */
	(void)sprintf(name, "%s%c", hdr->ar_name, first == '/' ? ' ' : '/');


	/* Create a header for this member */
	(void)sprintf(buffer, "%-15s %-11lu %-5lu %-5lu %-7lo %-9lu %2s",
		name, hdr->ar_date, hdr->ar_uid, hdr->ar_gid,
		hdr->ar_mode, hdr->ar_size, arstrMagic);


	/* Write the header */
	if (fwrite(buffer, 60, 1, ar->handle) != 1)
		fatal_error("failed to write member header");


	/* Return the header size */
	return 60L;
}


/*
 * uniReadHeaderUCB(handle, hdr, next, max, align) reads the header of
 * the member at file position "next" from stream "handle" and writes the
 * results into "hdr". The header alignment is "align" and there are at
 * most "max" bytes in the archive. If a directory entry is found then it
 * is read and we recurse to read the header of the entry that follows.
 * Similarly, if a long filename is not found in the directory table then
 * the member is skipped and we recurse to read the header of the next
 * member. If the end of the archive is reached after a bad entry or after
 * a directory scan then we return zero. Otherwise we return the position
 * of the next member.
 */
static long
uniReadHeaderUCB(FILE *handle, UCB_ArHdr *hdr, long next, long amax, int align)
{
	int i;
	static char fname[1024];


	/* Jump to the start of the next member */
	if (fseek(handle, next, SEEK_SET)) fatal_error("seek failed");


	/* Read the header record */
	uniReadText   (handle, hdr->ar_name, 16);
	uniReadDecimal(handle, &(hdr->ar_date), 12);
	uniReadDecimal(handle, &(hdr->ar_uid),   6);
	uniReadDecimal(handle, &(hdr->ar_gid),   6);
	uniReadOctal  (handle, &(hdr->ar_mode),  8);
	uniReadDecimal(handle, &(hdr->ar_size), 10);
	uniReadText   (handle, hdr->ar_fmag,  2);


	/* Normalise the filename */
	for (i = 15; i >= 0 && hdr->ar_name[i] == ' '; i--)
		hdr->ar_name[i] = 0;


	/* Compute the start of the member data */
	next += 16+12+6+6+8+10+2;
	hdr->ar_pos = next;


	/* Compute the start of the next member */
	next += hdr->ar_size;
	if (next % align != 0) next += align - next % align;


	/* Is this the directory table? */
	if (!strcmp(hdr->ar_name, "//")) {
		/* Yes: release our old table (never happens?) */
		if (ucb_dir_table) (void)free(ucb_dir_table);


		/* Read the directory and move onto the next member */
		ucb_dir_table_len = hdr->ar_size;
		ucb_dir_table = uniReadDirUCB(handle, ucb_dir_table_len);


		/* Recurse if we can */
		if (next >= amax) return 0; /* End-of-archive */
		return uniReadHeaderUCB(handle, hdr, next, amax, align);
	}


	/*
	 * More name mangling: names of the form /N for some decimal
	 * N are indirect references to names in the directory table.
	 * The special names "/" and "//" refer to the symbol table
	 * and directory table respectively. Note that we have already
	 * processed the directory table above. The symbol table is
	 * highlighted by the fact that it's ar_rname field is null.
	 */
	if (hdr->ar_name[0] == '/' && hdr->ar_name[1]) {
		/* Look up redirected names */
		long offset = strtol(hdr->ar_name + 1, 0, 10);
		char *sptr = ucb_dir_table + offset;
		char *dptr = fname;


		/* Safety check */
		if (offset >= ucb_dir_table_len) {
			/* Hopefully impossible */
			message("bad archive: name not found in directory");
			if (next >= amax) return 0;
			return uniReadHeaderUCB(handle, hdr, next, amax, align);
		}

		/* Copy characters up to / or newline */
		while ((*sptr != '\n') && (*sptr != '/') && *sptr)
			*dptr++ = *sptr++;
		*dptr = 0;


		/* This is the name to display/use */
		hdr->ar_rname = dup_str(fname);
	}
	else if (hdr->ar_name[0] != '/') {
		/* Remove any trailing "/" */
		int endp = strlen(hdr->ar_name) - 1;
		if (hdr->ar_name[endp] == '/')
			hdr->ar_name[endp] = 0;


		/* This is the name to display/use */
		hdr->ar_rname = dup_str(hdr->ar_name);
	}


	/* Normalise the name */
	normalise(hdr->ar_rname);


	/* Return the position of the next member */
	return next;
}


/*
 * uniReadDirUCB(fil,len) reads "len" bytes from "fil" corresponding
 * to the directory table.
 */
static char *
uniReadDirUCB(FILE *handle, long need)
{
	char *result = (char *)malloc((need + 1)*sizeof(char));


	/* Check that we managed to get the memory we needed */
	if (!result) fatal_error("not enough memory for directory table");


	/* Read the entire table into our buffer and return it */
	uniReadText(handle, result, need);
	return result;
}


/*****************************************************************************
 *
 * :: AIX-style archives
 *
 ****************************************************************************/

static void
uniOpenArchiveAIX(Archive *ar)
{
	int i;
	AIX_ArHdr *hdr;
	long next, index;
	FILE *handle = ar->handle;
	long membc;
	AIX_ArHdr **membv;
	long *posv;


	/* Get the address of the index record */
	index = uniIndexMemberAIX(handle);


	/* Allocate storage for the header */
	hdr = (AIX_ArHdr *)malloc(sizeof(*hdr));
	if (!hdr) fatal_error("not enough memory for member header");


	/* Read the index header and update the TOC */
	(void)uniReadHeaderAIX(handle, hdr, index);
	ar->toc.aix.index = hdr;


	/* Get the number of members in the archive */
	uniReadDecimal(handle, &membc, 12);


	/* Allocate a fully-sized TOC */
	membv = (AIX_ArHdr **)malloc(membc*sizeof(AIX_ArHdr *));
	if (!membv) fatal_error("not enough memory for toc");


	/* We also need a file-address lookup table */
	posv = (long *)malloc(membc*sizeof(long));
	if (!posv) fatal_error("not enough memory for address table");


	/* Initialise the TOC */
	ar->toc.aix.membs = ar->toc.aix.membc = membc;
	ar->toc.aix.membv = membv;
	ar->toc.aix.posv  = posv;


	/* Read the member offsets and allocate headers */
	for (i = 0; i < membc; i++, posv++, membv++) {
		/* Get the file-address of this member */
		uniReadDecimal(handle, posv, 12);


		/* Allocate a header for the TOC */
		hdr = (AIX_ArHdr *)malloc(sizeof(*hdr));
		if (!hdr) fatal_error("not enough memory for member header");


		/* Update the TOC */
		*membv = hdr;
	}


	/* Reset our local state */
	membv = ar->toc.aix.membv;
	posv  = ar->toc.aix.posv;


	/* Start at the first member */
	next = uniFirstMemberAIX(handle);


	/* Read the header of every entry in the archive */
	while (next) {
		AIX_ArHdr tmp;


		/* Make sure that we are at the correct position */
		if (fseek(handle, next, SEEK_SET))
			fatal_error("cannot seek to member");


		/* Ignore the index record */
		if (next == index) {
			next = ar->toc.aix.index->ar_next;
			continue;
		}


		/* Find our internal header for this member */
		for (i = 0, hdr = 0; i < membc; i++) {
			if (posv[i] == next) {
				hdr = membv[i];
				break;
			}
		}


		/* Safety check */
		if (!hdr) {
			message("found an unindexed member");
			hdr = &tmp;
		}


		/* Read the member header */
		next = uniReadHeaderAIX(handle, hdr, next);
		hdr->pre = hdr->nex = 0;


		/* Unlink the index record */
		if (hdr->ar_next == index)
			hdr->ar_next = ar->toc.aix.index->ar_next;
		if (hdr->ar_prev == index)
			hdr->ar_prev = ar->toc.aix.index->ar_prev;


		/* Patch our internal prev/next pointers */
		for (i = 0, hdr->pre = hdr->nex = 0; i < membc; i++) {
			if (posv[i] == hdr->ar_next) hdr->nex = membv[i];
			if (posv[i] == hdr->ar_prev) hdr->pre = membv[i];
		}
	}


	/* Unlink the index member completely */
	ar->toc.aix.index->nex = ar->toc.aix.index->pre = 0;
	ar->toc.aix.index->ar_next = ar->toc.aix.index->ar_prev = 0;
}


static void
uniCreateArchiveAIX(char *name)
{
	ArInfo *info = &arInfoTable[AR_AIX];
	char *magic = info->magic;
	FILE *handle = open_or_error(name, "wb", "for writing");


	/* Write out a possibly-valid header */
	(void)fprintf(handle, "%s%-12lu%-12lu%-12lu%-12lu%-12lu",
		magic, 0L, 0L, 0L, 0L, 0L);


	/* Finished */
	(void)fclose(handle);
}


static void
uniWriteArchiveAIX(Archive *ar, FILE *backup)
{
	int membc;
	long hsize;
	AIX_ArHdr **membv;
	int align = arInfoTable[AR_AIX].align;
	StatBuf *statbuf = &(ar->statbuf);

	/*
	 * Use the table of contents to read members from the backup
	 * and write them to the archive. We ignore member headers
	 * in the backup and use the table of contents.
	 */
	membc = ar->toc.aix.membc, membv = ar->toc.aix.membv;
	for (; membc; membc--, membv++) {
		AIX_ArHdr *hdr = *membv;
		long siz = hdr->ar_size;


		/* Store the new header position */
		hdr->ar_hpos = statbuf->st_size;


		/* Update the pointers to this record */
		if (hdr->nex) hdr->nex->ar_prev = hdr->ar_hpos;
		if (hdr->pre) hdr->pre->ar_next = hdr->ar_hpos;


		/* Write out the header for this member */
		hsize = uniWriteHeaderAIX(ar, hdr);


		/* Update the archive size */
		statbuf->st_size += hsize;


		/* Seek to the data in the backup */
		if (fseek(backup, hdr->ar_pos, SEEK_SET))
			fatal_error("failed to seek in backup for member");


		/* Copy over the data and update archive size */
		uniWriteData(backup, ar->handle, siz);
		statbuf->st_size += siz;


		/* Align the member */
		for (; siz % align; siz++, statbuf->st_size++)
			(void)fputc('\n', ar->handle);
	}


	/* Append the new table of contents */
	uniAppendIndexAIX(ar);


	/* Now we have to fix the pointers in member headers */
	uniPatchHeadersAIX(ar);
}


static void
uniListArchiveAIX(Archive *ar)
{
	/*
	 * Member order is determined by the linked list NOT
	 * by the order of members in the index table.
	 */
	int membc = ar->toc.aix.membc;
	AIX_ArHdr *hdr;
	AIX_ArHdr **membv = ar->toc.aix.membv;
	long first = uniFirstMemberAIX(ar->handle);


	/* If no members, do nothing */
	if (!membc) return;


	/* Locate the header of the first member */
	for (; membc; membc--, membv++)
		if ((*membv)->ar_hpos == first) break;


	/* Safety check */
	if (!membc) fatal_error("unable to find first member in table");


	/* Display the details of each member */
	for (hdr = *membv; hdr; hdr = hdr->nex)
		uniListMemberAIX(hdr);
}


static void
uniExtractAIX(Archive *ar, FileVec *fv, int print)
{
	/* If no files specified, extract them all */
	if (!fv->filec) {
		int membc = ar->toc.aix.membc;
		AIX_ArHdr **membv = ar->toc.aix.membv;

		for (; membc; membc--, membv++)
			uniExtractMemberAIX(*membv, ar->handle, print);
		return;
	}


	/* Extract in the order specified, not archive order! */
	for (; fv->filec; fv->filec--, fv->filev++) {
		char *name = *fv->filev;
		int membc = ar->toc.aix.membc;
		AIX_ArHdr **membv = ar->toc.aix.membv;


		/* Ensure that the name has been normalised */
		normalise(name);


		/* Search for this member */
		for (; membc; membc--, membv++) {
			AIX_ArHdr *hdr = *membv;


			/* Only extract if the name matches */
			if (!strcmp(name, hdr->ar_name)) {
				uniExtractMemberAIX(hdr, ar->handle, print);
				break;
			}
		}


		/* Complain if member not found */
		if (membc < 0) {
			(void)sprintf(errmsg, "%s not found in archive", name);
			fatal_error(errmsg);
		}
	}
}


static void
uniAppendAIX(Archive *ar, FileVec *fv)
{
	int filec;
	char **filev;
	AIX_ArHdr *hdr;


	/* Must specify at least one file */
	if (!fv->filec) fatal_error("no files to append to archive");


	/*
	 * Currently we always append the files to the archive. However,
	 * we ought to examine the free list and try to use that space
	 * first before increasing the archive size.
	 */
	(void)fclose(ar->handle);
	ar->handle = open_or_error(ar->name, "ab", "for appending");


	/* Append each file in turn */
	for (; fv->filec; fv->filec--, fv->filev++) {
		/* Extract and normalise the name */
		char *name = *fv->filev;
		normalise(name);


		/* Allocate a new header */
		hdr = (AIX_ArHdr *)malloc(sizeof(*hdr));
		if (!hdr) fatal_error("no memory for member header");


		/* Initialise member name and then append it */
		hdr->ar_name = name; /* dup_str not needed here */
		uniAppendMemberAIX(ar, hdr, 'a');


		/* Add to the TOC */
		uniAppendTocAIX(ar, hdr);
	}


	/* Append the new table of contents */
	uniAppendIndexAIX(ar);


	/* Now we have to fix the pointers in member headers */
	uniPatchHeadersAIX(ar);
}


static void
uniReplaceAIX(Archive *ar, FileVec *fv)
{
	char **filev;
	AIX_ArHdr **membv;
	int i, filec, membc;


	/* Must specify at least one file */
	if (!fv->filec) fatal_error("no files to replace");


	/* We will always append the files to the archive */
	(void)fclose(ar->handle);
	ar->handle = open_or_error(ar->name, "ab", "for appending");


	/* Append data to the current archive and update our TOC */
	for (filec = fv->filec, filev = fv->filev; filec; filec--, filev++) {
		AIX_ArHdr *hdr;
		char *name = *filev;


		/* Normalise the name */
		normalise(name);


		/* Allocate storage for the header */
		hdr = (AIX_ArHdr *)malloc(sizeof(*hdr));
		if (!hdr) fatal_error("not enough memory for member header");


		/* Initialise the header input fields */
		hdr->ar_name = dup_str(name);


		/* Search for the member in the archive */
		membc = ar->toc.aix.membc, membv = ar->toc.aix.membv;
		for (i = 0; membc; i++, membc--, membv++)
			if (!strcmp((*membv)->ar_name, name)) break;


		/* Decide if we are replacing or appending */
		if (!membc) {
			/* Appending */
			uniAppendMemberAIX(ar, hdr, 'a');
			uniAppendTocAIX(ar, hdr);
		}
		else {
			AIX_ArHdr *old_hdr = ar->toc.aix.membv[i];


			/* Replacing: append the member */
			uniAppendMemberAIX(ar, hdr, 'r');


			/*
			 * Unlink the newly appended member: note that
			 * hdr->nex and hdr->ar_next are always zero.
			 */
			hdr->ar_next = hdr->ar_prev = 0;
			if (hdr->pre) {
				hdr->pre->nex = hdr->nex;
				hdr->pre->ar_next = 0;
			}


			/* Re-link it into its correct position */
			if ((hdr->pre = old_hdr->pre)) {
				hdr->pre->nex = hdr;
				hdr->pre->ar_next = hdr->ar_hpos;
				hdr->ar_prev = hdr->pre->ar_hpos;
			}
			if ((hdr->nex = old_hdr->nex)) {
				hdr->nex->pre = hdr;
				hdr->nex->ar_prev = hdr->ar_hpos;
				hdr->ar_next = hdr->nex->ar_hpos;
			}


			/* Release the original TOC entry */
			(void)free(old_hdr->ar_name);
			(void)free(old_hdr);


			/* Replace the TOC entry */
			ar->toc.aix.membv[i] = hdr;
		}
	}


	/* Rebuild the whole archive from the TOC */
	uniWriteArchive(ar);
}


static void
uniDeleteAIX(Archive *ar, FileVec *fv)
{
	char *delete;
	char **filev;
	AIX_ArHdr **membv;
	int i, filec, membc, from, to;


	/* Must specify at least one file */
	if (!fv->filec) fatal_error("no members to delete");


	/* Mark members which are to be deleted */
	delete = (char *)malloc((ar->toc.aix.membc + 1)*sizeof(char));
	if (!delete) fatal_error("not enough memory for internal bitmap");
	for (i = 0; i < ar->toc.aix.membc; i++)
		delete[i] = 0;


	/* Record the entries in the TOC which are to be deleted */
	for (filec = fv->filec, filev = fv->filev; filec; filec--, filev++) {
		/* Normalise the name */
		normalise(*filev);


		/* Locate the member in the TOC */
		membc = ar->toc.aix.membc, membv = ar->toc.aix.membv;
		for (i = 0; membc; i++, membc--, membv++) {
			AIX_ArHdr *hdr = *membv;


			/* Check for existence of the file in the archive */
			if (!strcmp(hdr->ar_name, *filev)) {
				/* Mark for deletion */
				delete[i] = '*';


				/* Unlink the member from the TOC chain */
				if (hdr->pre) {
					hdr->pre->ar_next = hdr->ar_next;
					hdr->pre->nex = hdr->nex;
				}
				if (hdr->nex) {
					hdr->nex->ar_prev = hdr->ar_prev;
					hdr->nex->pre = hdr->pre;
				}
				break;
			}
		}


		/* Did we find the specified member? */
		if (!membc) {
			char *fmt = "%s not found in archive";
			(void)sprintf(errmsg, fmt, *filev);
			fatal_error(errmsg);
		}
	}


	/* Compress the TOC */
	membc = ar->toc.aix.membc, membv = ar->toc.aix.membv;
	for (to = from = 0; membc; from++, membc--, membv++) {
		AIX_ArHdr *hdr = ar->toc.aix.membv[from];

		if (delete[from]) {
			/* Tell them what we are doing */
			if (opt_verbose)
				(void)printf("d - %s\n", hdr->ar_name);


			/* Release storage and decrement member count */
			(void)free(hdr->ar_name);
			(void)free(hdr);
			ar->toc.aix.membc--;
		}
		else
			ar->toc.aix.membv[to++] = hdr;
	}


	/*
	 * Rebuild the whole archive from the TOC. Strictly speaking
	 * we ought to unlink the member and place it on the free
	 * list for use by later append/replace operations.
	 */
	uniWriteArchive(ar);
}


static void
uniListMemberAIX(AIX_ArHdr *hdr)
{
	char stime[30];
	char uid_gid[30];
	struct tm *ltime;


	/*
	 * Convert epoch time into local time. Assumes that
	 * the platform on which the archive was created uses
	 * the same epoch as this platform. [FIXME]
	 */
	ltime = localtime(&(hdr->ar_date));
	strftime(stime, 30, "%b %d %H:%M %Y", ltime);


	/* Format the UID/GID */
	(void)sprintf(uid_gid, "%ld/%ld", hdr->ar_uid, hdr->ar_gid);


	/* Display the details */
	if (opt_verbose) {
		(void)printf("%s %11s %6ld %s %s\n",
			uniModeUNIX(hdr->ar_mode), uid_gid,
			hdr->ar_size, stime, hdr->ar_name);
	}
	else
		(void)printf("%s\n", hdr->ar_name);
}


/* Extract a single archive member */
static void
uniExtractMemberAIX(AIX_ArHdr *hdr, FILE *ar_handle, int print)
{
	FILE *handle = stdout;


	/* Seek to the position of the member data */
	if (fseek(ar_handle, hdr->ar_pos, SEEK_SET)) {
		char *fmt = "cannot seek to %s in archive";
		(void)sprintf(errmsg, fmt, hdr->ar_name);
		fatal_error(errmsg);
	}


	/* Work out where we are extracting the member to */
	if (!print) {
		/* Open/create the output file (clobber existing file) */
		handle = fopen(hdr->ar_name, "wb");
		if (!handle) {
			char *fmt = "unable to open %s for writing";
			(void)sprintf(errmsg, fmt, hdr->ar_name);
			fatal_error(errmsg);
		}
	}


	/* Tell them what we are doing */
	if (opt_verbose) (void)printf("x - %s\n", hdr->ar_name);


	/* Write all the data */
	uniWriteData(ar_handle, handle, hdr->ar_size);
	if (!print) (void)fclose(handle);
}


static void
uniAppendTocAIX(Archive *ar, AIX_ArHdr *hdr)
{
	/* Do we need to increase the TOC size? */
	if (ar->toc.aix.membc == ar->toc.aix.membs) {
		/* Double the table size */
		int  i, want = ar->toc.aix.membs + ar->toc.aix.membs;
		AIX_ArHdr **newv, **oldv = ar->toc.aix.membv;


		/* Allocate a new table */
		newv = (AIX_ArHdr **)malloc(want*sizeof(AIX_ArHdr *));
		if (!newv) fatal_error("not enough memory for toc");


		/* Copy the old contents across */
		for (i = 0; i < ar->toc.aix.membs; i++)
			newv[i] = oldv[i];


		/* Release storage for the old table */
		(void)free(oldv);


		/* Start using the new table */
		ar->toc.aix.membs = want;
		ar->toc.aix.membv = newv;
	}


	/* Add to the TOC */
	ar->toc.aix.membv[ar->toc.aix.membc++] = hdr;
}


/*
 * This function appends the file hdr->ar_name to the archive. The
 * remaining fields of "hdr" are updated and the header is linked
 * into the list of TOC members (but NOT added to our TOC). Since
 * this action invalidates the ar_next field of existing members we
 * expect archive to be fixed in a separate pass.
 */
static void
uniAppendMemberAIX(Archive *ar, AIX_ArHdr *hdr, char mark)
{
	long hsize;
	int i, membc = ar->toc.aix.membc;
	AIX_ArHdr **membv = ar->toc.aix.membv;
	FILE *source;
	FILE *handle = ar->handle;
	StatBuf *statbuf = &(ar->statbuf);
	StatBuf mstatbuf;
	int align = arInfoTable[AR_AIX].align;


	/* Get details of the file we are appending */
	if (stat(hdr->ar_name, &mstatbuf)) {
		(void)sprintf(errmsg, "can't stat %s", hdr->ar_name);
		fatal_error(errmsg);
	}


	/* Fill in the header details */
	hdr->ar_size = (long)mstatbuf.st_size;
	hdr->ar_next = 0L; /* To be filled in later */
	hdr->ar_date = (long)mstatbuf.st_mtime;
	hdr->ar_uid  = (long)mstatbuf.st_uid;
	hdr->ar_gid  = (long)mstatbuf.st_gid;
	hdr->ar_mode = (long)mstatbuf.st_mode;
	hdr->ar_nlen = (long)strlen(hdr->ar_name);


	/* Open the file for reading */
	source = open_or_error(hdr->ar_name, "rb", "for reading");


	/* Seek to the end of the archive */
	if (fseek(handle, 0L, SEEK_END)) fatal_error("seek to end failed");


	/* Tell them what we are doing */
	if (opt_verbose) printf("%c - %s\n", mark, hdr->ar_name);


	/*
	 * We trust the alignment of the archive header and all members
	 * except for the last one. Use "statbuf" to ensure alignment.
	 */
	for (; statbuf->st_size % align; statbuf->st_size++)
		(void)fputc('\n', handle);


	/* Find the last member */
	for (i = 0; i < membc; i++)
		if (!membv[i]->ar_next) break;


	/* Link our member to it */
	membv[i]->nex = hdr;
	membv[i]->ar_next = hdr->ar_hpos = statbuf->st_size;
	hdr->pre = membv[i];
	hdr->ar_prev = membv[i]->ar_hpos;
	hdr->nex = 0;
	hdr->ar_next = 0;


	/* Write out a header for this member */
	hsize = uniWriteHeaderAIX(ar, hdr);


	/* Record the offset of the data */
	hdr->ar_pos = statbuf->st_size += hsize;


	/* Append the file data */
	uniWriteData(source, handle, mstatbuf.st_size);


	/* Align the member */
	for (; mstatbuf.st_size % align; mstatbuf.st_size++)
		(void)fputc('\n', handle);


	/* Update the archive size */
	statbuf->st_size += mstatbuf.st_size;
}


/*
 * This function appends the archive TOC and updates the ar_next
 * pointer of the last member in the archive.
 */
static void
uniAppendIndexAIX(Archive *ar)
{
	long hsize;
	int i, membc = ar->toc.aix.membc;
	AIX_ArHdr **membv = ar->toc.aix.membv;
	FILE *handle = ar->handle;
	StatBuf *statbuf = &(ar->statbuf);
	int align = arInfoTable[AR_AIX].align;
	AIX_ArHdr *hdr = ar->toc.aix.index;


	/* Fill in the header details */
	hdr->ar_size = 0L; /* Updated shortly */
	hdr->ar_next = 0L; /* We put the index last */
	hdr->ar_nlen = 0L;
	hdr->ar_name = "";


	/* Compute the size of the index record */
	hdr->ar_size += 12; /* #entries */
	hdr->ar_size += 12*membc; /* file positions */
	for (i = 0; i < membc; i++) /* filenames */
		hdr->ar_size += strlen(membv[i]->ar_name) + 1;


	/* Seek to the end of the archive */
	if (fseek(handle, 0L, SEEK_END)) fatal_error("seek to end failed");


	/*
	 * We trust the alignment of the archive header and all members
	 * except for the last one. Use "statbuf" to ensure alignment.
	 */
	for (; statbuf->st_size % align; statbuf->st_size++)
		(void)fputc('\n', handle);


	/* Find the last member */
	for (i = 0; i < membc; i++)
		if (!membv[i]->ar_next) break;


	/* Link our member to it */
	hdr->ar_hpos = statbuf->st_size;
	hdr->ar_prev = membv[i]->ar_hpos;


	/* Write out a header for this member */
	hsize = uniWriteHeaderAIX(ar, hdr);


	/* Record the offset of the data */
	hdr->ar_pos = statbuf->st_size += hsize;


	/* Append the index data: start with the number of members */
	(void)fprintf(handle, "%-12lu", (unsigned long)membc);


	/* Now write out the positions of each member */
	for (i = 0; i < membc; i++)
		(void)fprintf(handle, "%-12lu", membv[i]->ar_hpos);


	/* Finally the NUL-terminated names of each member */
	for (i = 0; i < membc; i++) {
		(void)fprintf(handle, "%s", membv[i]->ar_name);
		(void)fputc(0, handle);
	}


	/* Update the archive size */
	statbuf->st_size += hdr->ar_size;


	/* Align the index */
	for (; statbuf->st_size % align; statbuf->st_size++)
		(void)fputc(0, handle);
}


static long
uniWriteHeaderAIX(Archive *ar, AIX_ArHdr *hdr)
{
	int align = arInfoTable[AR_AIX].align;
	int slen = strlen(hdr->ar_name);
	char *fmt = "%-12lu%-12lu%-12lu%-12lu%-12lu%-12lu%-12lo%-4lu%s";


	/* Emit the main part of the header */
	(void)fprintf(ar->handle, fmt,
		hdr->ar_size, hdr->ar_next, hdr->ar_prev, hdr->ar_date,
		hdr->ar_uid, hdr->ar_gid, hdr->ar_mode, hdr->ar_nlen,
		hdr->ar_name);


	/* Align the name */
	for (;slen % align; slen++)
		(void)fputc(0, ar->handle);


	/* Terminator */
	(void)fprintf(ar->handle, "%s", arstrMagic);


	/* Return the header size */
	return 88L + slen + 2L;
}


/*
 * After adding new members to the archive the pointers in the headers
 * of some existing members will become obsolete. Furthermore the new
 * members can be appended more quickly if their headers are not linked
 * to the other members at the time of writing.
 *
 * This function fixes all member pointers in the file based on the
 * internal archive table.
 */
static void
uniPatchHeadersAIX(Archive *ar)
{
	int i, first = 0, last = 0;
	int membc = ar->toc.aix.membc;
	AIX_ArHdr **membv = ar->toc.aix.membv;
	AIX_ArHdr *idx_hdr = ar->toc.aix.index;


	/* Find the first and last members */
	for (i = 0; i < membc; i++) {
		if (!membv[i]->ar_prev) first = i;
		if (!membv[i]->ar_next) last = i;
	}


	/* Open for random-access updates */
	(void)fclose(ar->handle);
	ar->handle = open_or_error(ar->name, "rb+", "for update");


	/* Update the index pointer */
	if (fseek(ar->handle, 8L, SEEK_SET)) fatal_error("seek failed");
	(void)fprintf(ar->handle, "%-12lu", idx_hdr->ar_hpos);


	/* Update the pointers to first and last members */
	if (fseek(ar->handle, 32L, SEEK_SET)) fatal_error("seek failed");
	(void)fprintf(ar->handle, "%-12lu", membv[first]->ar_hpos);
	(void)fprintf(ar->handle, "%-12lu", membv[last]->ar_hpos);


	/* Now fix each member header in turn */
	for (i = 0; i < membc; i++) {
		/* Skip past the ar_size field */
		if (fseek(ar->handle, membv[i]->ar_hpos + 12L, SEEK_SET))
			fatal_error("seek failed");


		/* Update the next field: last member points to index */
		if (i == last)
			(void)fprintf(ar->handle, "%-12lu", idx_hdr->ar_hpos);
		else
			(void)fprintf(ar->handle, "%-12lu", membv[i]->ar_next);


		/* Update the prev field */
		(void)fprintf(ar->handle, "%-12lu", membv[i]->ar_prev);
	}


	/* Finally fix the index */
	if (fseek(ar->handle, idx_hdr->ar_hpos + 12L, SEEK_SET))
			fatal_error("seek failed");


	/* Update the next and prev fields */
	(void)fprintf(ar->handle, "%-12lu", 0L);
	(void)fprintf(ar->handle, "%-12lu", membv[last]->ar_hpos);
}


/*
 * uniReadHeaderAIX(handle, hdr, next) reads the header of the member at
 * file position "next" from stream "handle" and writes the results into
 * "hdr". We return the position of the next member.
 */
static long
uniReadHeaderAIX(FILE *handle, AIX_ArHdr *hdr, long next)
{
	int align = arInfoTable[AR_AIX].align;

	/* Jump to the start of the next member */
	if (fseek(handle, next, SEEK_SET)) fatal_error("seek failed");


	/* Store the address of this header */
	hdr->ar_hpos = next;


	/* Read the header record */
	uniReadDecimal(handle, &(hdr->ar_size), 12);
	uniReadDecimal(handle, &(hdr->ar_next), 12);
	uniReadDecimal(handle, &(hdr->ar_prev), 12);
	uniReadDecimal(handle, &(hdr->ar_date), 12);
	uniReadDecimal(handle, &(hdr->ar_uid),  12);
	uniReadDecimal(handle, &(hdr->ar_gid),  12);
	uniReadOctal  (handle, &(hdr->ar_mode), 12);
	uniReadDecimal(handle, &(hdr->ar_nlen),  4);
	uniReadText   (handle, buffer, hdr->ar_nlen);
	uniReadText   (handle, hdr->ar_fmag,  2);


	/* Allocate the member name on the heap and normalise */
	hdr->ar_name = dup_str(buffer);
	normalise(hdr->ar_name);


	/* Compute the start of the member data */
	next += 12+12+12+12+12+12+12+4 + hdr->ar_nlen + 2;
	if (next % align != 0) next += align - next % align;
	hdr->ar_pos = next;


	/* Ensure that we are at the start of the member data */
	if (fseek(handle, next, SEEK_SET)) fatal_error("cannot seek to data");


	/* Return the position of the next member */
	return hdr->ar_next;
}


static long
uniFirstMemberAIX(FILE *handle)
{
	long pos;

	/* Skip to the header "first-member" pointer */
	if (fseek(handle, 32L, SEEK_SET))
		fatal_error("unable to seek to first member pointer");


	/* Read the decimal pointer */
	uniReadDecimal(handle, &pos, 12);


	/* Now jump to the first member */
	if (fseek(handle, pos, SEEK_SET))
		fatal_error("unable to seek to first member");


	/* Return the position */
	return pos;
}


static long
uniIndexMemberAIX(FILE *handle)
{
	long pos;

	/* Skip to the header "index-member" pointer */
	if (fseek(handle, 8L, SEEK_SET))
		fatal_error("unable to seek to index member pointer");


	/* Read the decimal pointer */
	uniReadDecimal(handle, &pos, 12);


	/* Now jump to the index member */
	if (fseek(handle, pos, SEEK_SET))
		fatal_error("unable to seek to index member");


	/* Return the position */
	return pos;
}

/*****************************************************************************
 *
 * :: AIX4-style archives
 *
 ****************************************************************************/

static void
uniOpenArchiveAIX4(Archive *ar)
{
	fatal_error("uniOpenArchiveAIX4 not implemented yet");
}


static void
uniCreateArchiveAIX4(char *name)
{
	int i;
	ArInfo *info = &arInfoTable[AR_AIX4];
	char *magic = info->magic;
	int hdrsize = info->hdrsz;
	FILE *handle = open_or_error(name, "wb", "for writing");

	/* Start with the magic number */
	(void)fprintf(handle, "%s", magic);

	/* FIXME: must produce valid header contents */
	for (i = strlen(magic); i < hdrsize; i++)
		putc(' ', handle);

	/* Finished */
	(void)fclose(handle);
}


static void
uniWriteArchiveAIX4(Archive *ar, FILE *backup)
{
	fatal_error("uniWriteArchiveAIX4 unimplemented");
}


static void
uniListArchiveAIX4(Archive *ar)
{
	fatal_error("uniListArchiveAIX4 unimplemented");
}


static void
uniExtractAIX4(Archive *ar, FileVec *fv, int print)
{
	fatal_error("uniExtractAIX4 unimplemented");
}


static void
uniAppendAIX4(Archive *ar, FileVec *fv)
{
	fatal_error("uniAppendAIX4 unimplemented");
}


static void
uniReplaceAIX4(Archive *ar, FileVec *fv)
{
	fatal_error("uniReplaceAIX4 unimplemented");
}


static void
uniDeleteAIX4(Archive *ar, FileVec *fv)
{
	fatal_error("uniDeleteAIX4 unimplemented");
}


static long
uniFirstMemberAIX4(FILE *handle)
{
	long pos;

	/* Skip to the header "first-member" pointer */
	if (fseek(handle, 68L, SEEK_SET))
		fatal_error("unable to seek to first member pointer");


	/* Read the decimal pointer */
	uniReadDecimal(handle, &pos, 20);


	/* Now jump to the first member */
	if (fseek(handle, pos, SEEK_SET))
		fatal_error("unable to seek to first member");


	/* Return the position */
	return pos;
}

/*****************************************************************************
 *
 * :: CMS-style archives
 *
 ****************************************************************************/

static void
uniOpenArchiveCMS(Archive *ar)
{
	fatal_error("uniOpenArchiveCMS not implemented yet");
}


static void
uniCreateArchiveCMS(char *name)
{
	FILE *handle = open_or_error(name, "wb", "for writing");
	(void)fprintf(handle, "%s", arInfoTable[AR_CMS].magic);
	(void)fclose(handle);
}


static void
uniWriteArchiveCMS(Archive *ar, FILE *backup)
{
	fatal_error("uniWriteArchiveCMS unimplemented");
}


static void
uniListArchiveCMS(Archive *ar)
{
	fatal_error("uniListArchiveCMS unimplemented");
}


static void
uniExtractCMS(Archive *ar, FileVec *fv, int print)
{
	fatal_error("uniExtractCMS unimplemented");
}


static void
uniAppendCMS(Archive *ar, FileVec *fv)
{
	fatal_error("uniAppendCMS unimplemented");
}


static void
uniReplaceCMS(Archive *ar, FileVec *fv)
{
	fatal_error("uniReplaceCMS unimplemented");
}


static void
uniDeleteCMS(Archive *ar, FileVec *fv)
{
	fatal_error("uniDeleteCMS unimplemented");
}


static long
uniFirstMemberCMS(FILE *handle)
{
	/* Simply skip past the header */
	long pos = (long)arInfoTable[AR_CMS].hdrsz;
	if (fseek(handle, pos, SEEK_SET))
		fatal_error("unable to seek to first member");


	/* Return the current position */
	return pos;
}

/*****************************************************************************
 *
 * :: Utility functions
 *
 ****************************************************************************/

/*
 * uniReadString(fil,buf,len) reads precisely "len" characters from
 * the stream "fil" and writes them into "buf" followed by a NUL.
 * Returns -1 on failure (short read or read error), 0 on success.
 */
static int
uniReadString(FILE *handle, char *buffer, int len)
{
	if (fread(buffer, 1, len, handle) != len) return -1;
	buffer[len] = 0;
	return 0;
}


/*
 * uniReadText(fil,buf,len) executes uniReadString(fil,buf,len) but
 * dies with a fatal error if the read fails.
 */
static void
uniReadText(FILE *handle, char *dst, int len)
{
	if (!uniReadString(handle, dst, len)) return;
	fatal_error("truncated archive");
}


/*
 * uniReadNumber(fil,plong,len,base) executes uniReadString(fil,tmp,len)
 * on a temporary buffer "tmp" and dies with a fatal error if the read
 * fails. If the read was successful then the ASCII number in base "base"
 * stored in "tmp" is written to "*plong". Requires that "len <= 1024".
 */
static void
uniReadNumber(FILE *handle, long *plong, int len, int base)
{
	char *endp;

	/* First read the text */
	if (uniReadString(handle, buffer, len))
		fatal_error("truncated archive");


	/* Must have something in the buffer */
	if (!*buffer) fatal_error("invalid number in archive");


	/* Convert ASCII into binary */
	*plong = strtol(buffer, &endp, base);


	/* Validate the conversion (ignore ERANGE errors for now) */
	if (!*endp || *endp == ' ') return;
	fatal_error("invalid number in archive");
}


/*
 * uniReadDecimal(fil,plong,len) executes uniReadNumber(fil,plong,len,10).
 * Requires that "len <= 1024".
 */
static void
uniReadDecimal(FILE *handle, long *plong, int len)
{
	uniReadNumber(handle, plong, len, 10);
}


/*
 * uniReadOctal(fil,plong,len) executes uniReadNumber(fil,tmp,len,8).
 * Requires that "len <= 1024".
 */
static void
uniReadOctal(FILE *handle, long *plong, int len)
{
	uniReadNumber(handle, plong, len, 8);
}


/*
 * uniModeUNIX(mode) returns a pointer to the UNIX file permission string
 * corresponding to the access mode "mode". The result will be re-used by
 * the next call to this function.
 */
static char *
uniModeUNIX(long mode)
{
	static char result[10];

	/* Process the mode from right to left */
	result[6] = (mode & 4) ? 'r' : '-';
	result[7] = (mode & 2) ? 'w' : '-';
	result[8] = (mode & 1) ? 'x' : '-';
	mode = mode >> 3;

	result[3] = (mode & 4) ? 'r' : '-';
	result[4] = (mode & 2) ? 'w' : '-';
	result[5] = (mode & 1) ? 'x' : '-';
	mode = mode >> 3;

	result[0] = (mode & 4) ? 'r' : '-';
	result[1] = (mode & 2) ? 'w' : '-';
	result[2] = (mode & 1) ? 'x' : '-';
	mode = mode >> 3;

	result[9] = 0;
	return result;
}


static char *
uniTemporaryName(char *suf)
{
	char *tmp, *result;
	char *base = "uniar";


	/* Platform-specific temporary file creation */
	tmp = uniTempName(base);


	/* Return immediately if failed or no suffix requested */
	if (!tmp || !suf) return tmp;


	/* Append the suffix, free the temp-name and return */
	result = (char *)malloc((strlen(tmp) + strlen(suf) + 2)*sizeof(char));
	if (result) (void)sprintf(result, "%s.%s", tmp, suf);
	free(tmp);
	return result;
}


/*
 */
static void
uniWriteData(FILE *from, FILE *to, long nbytes)
{
	void *ptr = (void *) buffer;
	long got, want, siz = sizeof(char);
	long wrote = 0L;


	/* Transfer as many full buffers as possible */
	for (want = sizeof(buffer)/siz; nbytes > want; nbytes -= want)
	{
		/* Assume fwrite() always succeeds */
		got = fread(ptr,siz,want,from);
		(void)fwrite(ptr,siz,got,to);
		wrote += got;
		if (got != want) {
			char *fmt = "member truncated (%d/%d)";
			(void)sprintf(errmsg, fmt, wrote, nbytes);
			fatal_error(errmsg);
		}
	}


	/* Transfer any shortfall */
	if (nbytes > 0) {
		/* Assume fwrite() always succeeds */
		got = fread(ptr,siz,nbytes,from);
		(void)fwrite(ptr,siz,got,to);
		wrote += got;
		if (got != nbytes) {
			char *fmt = "member truncated (%d/%d)";
			(void)sprintf(errmsg, fmt, wrote, nbytes);
			fatal_error(errmsg);
		}
	}


	/* Flush the output file */
	(void)fflush(to);
}


static char *
dup_str(char *s)
{
	char *result = malloc(strlen(s) + 1);
	if (!result) return (char *)0;
	return strcpy(result, s);
}


static int
fileExists(char *fileName)
{
    FILE *handle = fopen(fileName, "r");
    if (!handle) return 0;
    fclose(handle);
    return 1;
}


static FILE *
open_or_error(char *file, char *mode, char *why)
{
	FILE *handle = fopen(file, mode);
	if (!handle) {
		(void)sprintf(errmsg, "unable to open '%s' %s", file, why);
		fatal_error(errmsg);
	}
	return handle;
}


static void
option_error(char *msg)
{
	message(msg);
	syntax();
	exit(1);
}


static void
fatal_error(char *msg)
{
	message(msg);
	exit(1);
}


static void
message(char *msg)
{
	(void)fprintf(stderr,"(uniar): %s.\n", msg);
}

static void
syntax(void)
{
	(void)fprintf(stderr,"Usage: uniar {dpqrtx}[cv] archive member...\n");
	(void)fprintf(stderr,"    Commands (choose one):\n");
	(void)fprintf(stderr,"       d - delete\n");
	(void)fprintf(stderr,"       p - print (extract to stdout)\n");
	(void)fprintf(stderr,"       q - quick append\n");
	(void)fprintf(stderr,"       r - replace\n");
	(void)fprintf(stderr,"       t - table of contents\n");
	(void)fprintf(stderr,"       x - extract\n");
	(void)fprintf(stderr,"    Modifiers (choose as many as you want):\n");
	(void)fprintf(stderr,"       c - do not warn if archive is created\n");
	(void)fprintf(stderr,"       v - verbose\n");
}

/*****************************************************************************
 *
 * :: UNIX-specific code
 *
 ****************************************************************************/

#if defined(OS_UNIX)

static char *
uniTempName(char *base)
{
	/* Start with /tmp */
	char *tmp = temporary_name("/tmp", base);


	/* If that failed, try /var/tmp, /usr/tmp and then . */
	tmp = tmp ? tmp : temporary_name("/var/tmp", base);
	tmp = tmp ? tmp : temporary_name("/usr/tmp", base);
	tmp = tmp ? tmp : temporary_name(".",        base);


	/* Use the ANSI function as the last resort */
	return dup_str(tmp ? tmp : tmpnam((char *)0));
}


/*
 * temporary_name(dir, pre) returns the name of a file which can be created
 * in the directory "dir". If successful, no file of this name will exist in
 * "dir" and nor will a file of this name be returned by subsequent calls.
 * Filenames will have the format <dir>/<pre><pid>X<cno> where <dir> is the
 * directory name "dir", <pre> is the prefix "pre", <pid> is the (hexadecimal)
 * process ID of the caller and <cno> is a unique (hexadecimal) integer which
 * is different for each call. Returns (char *)0 on failure.
 *
 * This function will fail if the user does not have search permissions on
 * "dir" or if the resulting filename is too long. See stat() for details.
 *
 * This function guarantees that, at the time of the call, the returned
 * name does not correspond to an existing file. This condition cannot be
 * guaranteed after the call returns although it almost certainly will.
 *
 * This function is intended to replace tempnam() (which isn't ANSI) and
 * tmpnam() (which is too vague).
 */
static char *
temporary_name(char *dir, char *pre)
{
	static long cno = 0L;
	static char result[1024];
	StatBuf buf;

	/* Generate a new (and hopefully) unique name */
	(void)sprintf(result, "%s/%s%lxX%lx", dir, pre, (long)getpid(), cno++);


	/* See if we can get any details on this file */
	if (stat(result, &buf) && (errno==ENOENT)) return result; /* Success */


	/* Already exists or some other error */
	return (char *)0;
}

#endif

/*****************************************************************************
 *
 * :: Win32-specific code
 *
 ****************************************************************************/

#if defined(OS_WIN32)

static char *
uniTempName(char *base)
{
	return dup_str(tmpnam((char *)0));
}

#endif

/*****************************************************************************
 *
 * :: CMS-specific code
 *
 ****************************************************************************/

#if defined(OS_CMS)
#include <ctype.h>

static long
osFileSize(char *name)
{
        FILE *file = fopen(name, "rb");
        long  size = 0;

        if (file) {
                int ch;
                ch = fgetc(file);
                while (ch != EOF) {
                        ++size;
                        ch = fgetc(file);
                }
                fclose(file);
        }
        return size;
}


static int
stat(char *path, StatBuf *buf)
{
    buf->st_mode    = 0;
    buf->st_uid     = 0;
    buf->st_gid     = 0;
    buf->st_mtime   = 0;
    buf->st_size    = osFileSize(path);
    if (!buf->st_size) return -1;
    return 0;
}


static void
normalise(char *fn)
{
    if (fn)
        while (*fn) {
            if (*fn == '.')
                *fn = ' ';
            else if (islower(*fn))
                *fn = toupper(*fn);
            fn++;
        }
}
#endif

