----------------------------- alg_sup0.as ----------------------------------
-- Copyright (c) Marc Moreno Maza 2002
-- Copyright (c) 1990-2007 Aldor Software Organization Ltd (Aldor.org).
-- Copyright (c) INRIA (France), USTL (France), UWO (Ontario) 2002
-----------------------------------------------------------------------------

#include "algebra"

#if ALDOC
\thistype{SparseUnivariatePolynomial0}
\History{Marc Moreno Maza}{13/05/2002}{created}
\History{Marc Moreno Maza}{21/06/2002}{last update}
\Usage{ import from \this~R\\ import from \this(R, x) }
\Params{
{\em R} & \altype{ExpressionType} & The coefficient domain\\
        & \altype{ArithmeticType} &\\
{\em x} & \altype{Symbol} & The variable name (optional)\\
}
\Descr{\this(R, x) implements free modules over an
arbitrary arithmetic system {\em R},
with respect to free monoid generated by {\em x}.
Its elements are assumed to have finite support.
The representation is sparse.}
\begin{exports}
\category{\altype{IndexedFreeModule} (R,Integer)}\\
\end{exports}
#endif

macro {
	Z == Integer;
	NNI == Integer;
	}

SparseUnivariatePolynomial0(R:Join(ArithmeticType, ExpressionType),
	avar:Symbol == new()): IndexedFreeModule(R,Z) with { CopyableType; 

} == add {

	------------------------
	--% Rep and imports  %--
	------------------------

	Term == Record(co: R, ex: NNI);
	Rep	== List Term;
		-- always ordered terms, decreasing wrt exponent's order
		-- and non-zero coefficients
	import from Term, Rep, R, Z;

	-----------------------
	--% local constanta %--
	-----------------------

	local integralDomain?  == {
		b1: Boolean == (R has IntegralDomain);
		if b1 then {import from R pretend IntegralDomain};
		b1;
	}
	local numberSystem? == (R has PrimeFieldCategory0) or _
				(R has IntegerCategory) or _
				(R has with {denominator:    % -> R;});
	local orderedArithmeticType? == {
		b2: Boolean == (R has OrderedArithmeticType);
		if b2 then {import from R pretend OrderedArithmeticType};
		b2;
	}
	local characteristicZero? == {
		b3: Boolean == (R has CharacteristicZero);
		if b3 then {import from R pretend CharacteristicZero};
		b3;
	}
        local goodRing? == integralDomain? and characteristicZero?;

	------------------------
	--% local functions  %--
	------------------------
	local equalTerm?(tx: Term, ty: Term): Boolean == {
		tx.ex = ty.ex and tx.co = ty.co;
	}
        local -(t:Term):Term	== [-t.co, t.ex]$Term;
	local neg!(tt: Term): Term  == {
		tt.co := -tt.co;
		tt;
	}
	local copyterm(t: Term): Term == [explode t];
	local deepCopy(l:List Term):List Term == {
		cl: List Term := empty;
		for t in l repeat {
			cl := cons([t.co, t.ex], cl);
		}
		reverse! cl;
	}

	local gen(l:List Term):Generator Cross(R, Z) == generate {
		import from Boolean;
		while ~empty? l repeat {
			t := first l;
			l := rest l;
			yield(t.co, t.ex);
		}
	}
	local makeCanonical!(xx: Rep): Rep == {
		repeat {
			empty? xx => break;
			tx := first xx;
			zero? tx.co => xx := rest xx;
			break;
		}
		empty? xx => xx;
		local current, next: Rep;
		current := xx;
		next := rest xx;
		repeat {
			repeat {
				empty? next => break;
				tx := first next;
				zero? tx.co => next := rest next;
				break;
			}
			setRest!(current,next);
			empty? next => break;
			current := next;
			next := rest next;
		}
		xx;
	}


	-------------------------------
	--% Exports as CopyableType %--
	-------------------------------
        -- copy!: (%, %) -> %
	copy(p:%):% == {
		zero? p => p;
		per deepCopy rep p;
	}

	--------------------------------
	--% Exports as FreeModule(R) %-- 
	-------------------------------- 
	-- FreeLinearCombinationType(R)
	-- if (R has Ring) then Module(R pretend Ring)
        -- term?: % -> Boolean
        -- reductum!: % -> %
        -- nonZeroCoefficients: % -> Generator(R)
        -- if (R has GcdDomain) then 
        --        content: % -> R
        --        primitive: % -> (R, %)
        --        primitive!: % -> (R, %)
        --        primitivePart: % -> %
        --        primitivePart!: % -> %
	-- if (R has Field) then 
        --        monic: % -> %
        --        monic!: % -> %


	ground? (x: %): Boolean == {
		empty? rep x => true;
		not empty? rest(rep x) => false;
		zero? ((first rep x).ex);
	}
	leadingCoefficient(x: %): R == {
		empty? rep x => 0;
		(first rep x).co;
	}
        trailingCoefficient(x: %): R == {
 		xx:= rep x;
                empty? xx => 0;
                while (not empty? rest xx) repeat xx := rest xx;
                (first xx).co;
        }       
	reductum(p:%):%			== { zero? p => 0; per rest rep p }
        support(x: %): Generator(Cross(R, %)) == generate {
           -- every pair returned has a non-zero coefficient
           xx := rep x;
           while not empty? xx repeat {
               t := first xx;
               xx := rest xx;
               yield(t.co,term(1,t.ex));
           }
        }
        if (R has HashType) then {
           import from R pretend HashType;
           hash(x: %): MachineInteger == {
               h: MachineInteger := 0;
               xx: Rep := rep x;
               i: MachineInteger := 1;
               for tx in xx repeat {
                   h := h + hash(tx.co) * hash(tx.ex) * i;
                   i := next(i);
               }
               h;
           }
        }
        if (R has SerializableType) then {
           import from R pretend SerializableType;
                        (port:BinaryWriter) << (p:%):BinaryWriter == {
                                -- g: Generator Cross(R, Z) := terms(p); SHOULD BE BETTER ??
                                g: Generator Cross(R, Z) := generator(p);
                                for tx in g repeat {
                                        (c: R, e: Z) := tx;
                                        port := port << c << e;
                                }
                                port << 0@R << 0@Z;
                        }
                        << (port:BinaryReader):% == {
                                p:% := 0;
                                local e:Z;
                                local r:R;
                                repeat {
                                   r := << port;
                                   e := << port;
                                   zero? r => break;
                                   p := add!(p, r, e);
                                }
                                p;
                        }
        }

	-----------------------------------------
	--% Exports as IndexedFreeModule(R,Z) %--
	-----------------------------------------
	-- FreeModule(R)
	-- IndexedFreeLinearCombinationType(R,Z)
	-- if (R has HashType) then HashType
	-- if (R has SerializableType) then SerializableType
        -- leadingMonomial: % -> %
        -- trailingMonomial: % -> %

	degree(x: %): NNI == {
		xx := rep x;
		if empty? xx then 0@NNI else (first xx).ex;
	}
	trailingDegree(x: %): NNI == {
 		xx:= rep x;
		empty? xx => 0;
		while (not empty? rest xx) repeat xx := rest xx;
		(first xx).ex;
	}
	leadingTerm(p:%):(R, Z) == {
		zero? p => (0, -1);
		t := first rep p;
		(t.co, t.ex);
	}
	trailingTerm(p:%):(R, Z) == {
 		xx:= rep p;
		empty? xx => (0@R, 0@NNI);
		while (not empty? rest xx) repeat xx := rest xx;
		((first xx).co, (first xx).ex);
	}
	generator(p:%):Generator Cross(R, Z)	== gen rep p;
	terms(p:%):Generator Cross(R, Z)	== gen reverse rep p;
	if R has Ring then {
		(i: Integer) * (x: %) : % == { 
			zero? i => 0;
			one?  i => x;
			zero? (r:= i :: R) => 0;
			if goodRing? then {
				per [[i*tx.co, tx.ex] for tx in rep x];
			}
			else {
				per [[r1, tx.ex] for tx in rep x | 
					not zero? (r1 := i*tx.co) ];
			}
		}
	}

	---------------------------------------------------
	--% Exports as IndexedFreeLinearCombinationType %-- 
	---------------------------------------------------
        -- monomial: Z -> %

	term(r:R, e:Z): % == {
		assert(e >= 0);
		zero? r => 0;
		per [[r,e]$Term];
	}
	coefficient(x:%, e:Z):R == {
		assert(e >= 0);
		for tx in rep x repeat {
			e = tx.ex => return tx.co;
			e > tx.ex => return 0;
		}
		0;
	}
	setCoefficient!(x:%, e:Z, r:R):% == {
		empty?(xx := rep x) => term(r,e);
		(empty? rest xx) and (zero? ((first xx).ex )) => 
			x - term(coefficient(x,e),e) + term(r,e);
		while ((not empty? xx) and ((first xx).ex > e)) repeat xx := rest xx;
		(empty? xx) or ((first xx).ex) < e => add!(x,r,e);
		(first xx).co := r;
		x;
	}
	add!(x:%, r:R, e:Z):% == {
		zero? r => x;
		zero? x => per [[r,e]$Term];
		local s: R; local d: NNI;
		l: Rep := rep x;
		t: Term := first(l);
		(s, d) := explode(t);
		zero? d => {
			zero? e => {
				s := s + r;
				zero? s => return 0;
				return (per [[s,d]$Term]);
			}
			return (per cons([r,e]$Term,l));
		}
		e > d => {
			per cons([r,e]$Term,l);
		}
		e = d => {
			r := r + s;
			zero? r => return per rest(l);
			per cons([r,e]$Term,rest(l));
		}
		empty? rest(l) => {
			setRest!(l,[[r,e]$Term]);
			per l;
		}
		local k: Rep;     
		repeat {
			k := l;
			l := rest l;
			t := first(l);
			t.ex <= e => break;
			empty? rest(l) => break;
		}
		t.ex = e => {
			r := r + t.co;
			zero? r => {
				setRest!(k,rest(l));
				return x;
			}
			t.co := r;
			return x;
		}
		t.ex < e => {
			setRest!(k,cons([r,e]$Term,l));
			return x;
		}
		setRest!(l,[[r,e]$Term]);
		x;
	}

	-----------------------------------------------
	--% Exports as FreeLinearCombinationType(R) %--  
	-----------------------------------------------
	-- map: R -> R -> % -> %
        -- map!: R -> R -> % -> %

	map! (f: R -> R, x: %): % == {
	   	xx := rep x;
	   	empty? xx => x;
	   	while (not empty? xx) repeat {
	       		tx := first xx;
	       		tx.co := f(tx.co);
	       		xx := rest xx;
	   	}
	   	per makeCanonical!(rep x);
	}

	-------------------------------------------
	--% Exports as LinearCombinationType(R) %--
	-------------------------------------------
	times!(r: R, x: %): % == {
	      	zero? r => per [];
		one?  r => copy x;
		zero? x => copy x;
		ground? x => r * x;
		if integralDomain? then {
			times!(tx:Term):Term == {tx.co:= r*tx.co;tx};
			per map(times!)( rep x);
		}
		else {
			local res, last, newend : Rep;
			xx := rep x;
			res := empty;
			while not empty? xx repeat {
				tx := first xx;
				tx.co := r * tx.co;
				if zero? tx.co then {
					xx := rest xx;
					iterate;
				}
				else {
					newend := xx;
					xx := rest xx;
				}
				if empty? res then {
					res := newend;
					last := res;
				}
				else {
					-- WAS  last.rest := newend;
					setRest!(last,newend); 
					last := newend;
				}
			}
			per res;
		}
	}
	(r: R) * (x: %) : % == {
		zero? r => 0;
		if integralDomain? then {
			per [[r*tx.co,tx.ex] for tx in rep x];
		}
		else {
			per [[rr, tx.ex] for tx in rep x | 
				not zero? (rr: R := r*tx.co)];
		}
	}
	add!(x:%, d:R, y:%):% == {
		zero? d => x;
		one? d => add!(x, y);
		zero? x => times!(d, copy y);
		add!(x,times!(d, y));
	}

	---------------------------------
	--% Exports as ExpressionType %--
	---------------------------------
	local dummy:Symbol == { import from String; -"dummy"; }
	local	apply(p:%, x:ExpressionTree):ExpressionTree == {
			import from Boolean, R, List ExpressionTree;
			import from UnivariateMonomial(R, dummy);
			zero? p => extree(0@R);
			l:List(ExpressionTree) := empty;
			for term in rep(p) repeat {
				m := monomial(term.co, term.ex)@UnivariateMonomial(R,dummy);
				l := cons(m x, l);
			}
			assert(~empty? l);
			empty? rest l => first l;
			ExpressionTreePlus reverse! l;
		}
	extree(p:%):ExpressionTree	== p extree avar;
	relativeSize(p:%):MachineInteger== #(rep p);

	-------------------------------
	--% Exports as AdditiveType %--
	-------------------------------
	0:%				== per empty;
	zero?(p:%):Boolean		== empty? rep p;

	- (x: %) : % == per map(-)( rep x);

	minus!(x: %): % == per map(neg!)( rep x);

	add!(x: %, y: %): % == {
		zero? x => copy y;
		ground? x =>  x + y;
		local res, last, newend: Rep;
		xx := rep x;
		yy := rep y;
		res := empty;
		repeat {
			empty? xx => break;
			empty? yy => break;
			tx := first xx;
			ty := first yy;
			if tx.ex > ty.ex then {
				newend := xx;
				xx := rest xx;
			}
			else if ty.ex > tx.ex then {
				newend := [copyterm ty];
				yy := rest yy;
			}
			else {
				r: R := tx.co + ty.co;
				yy := rest yy;
				if zero? r then {
					xx := rest xx;
					iterate;
				}
				else {
					tx.co := r;
					newend := xx;
					xx:= rest xx;
				}
			}
			if empty? res then {
				res := newend;
				last := res;
			}
			else {
				-- WAS last.rest := newend;
				setRest!(last,newend); 
				last := newend;
			}
		}
		newend := if ~empty?(xx) then xx else map(copyterm)(yy);
		empty? res => per newend;
		-- WAS  last.rest := newend;
		setRest!(last,newend); 
		per res;
	}
	(x: %) - (y: %) : %  == {
		xx := rep x;
		yy := rep y;
		res :DoubleEndedList Term := empty();
		repeat {
			empty? xx => break;
			empty? yy => break;
			tx: Term := first xx;
			ty: Term := first yy;
			if tx.ex > ty.ex then {
				concat!(res, tx);
				xx := rest xx;
			}
			else if ty.ex > tx.ex then {
				concat!(res, -ty);
				yy := rest yy;
			}
			else {
				r: R := tx.co - ty.co;
				if not zero? r then {
					concat!(res, [r, tx.ex]);
				}
				xx := rest xx;
				yy := rest yy;
			}
		}
		tt: Rep := if ~empty?(xx) then xx else map(-)(yy);
		empty? firstCell(res) => per tt;
		-- WAS lastCell(res).rest := tt;
		setRest!(lastCell(res),tt);
		per firstCell(res);
	}
	(x: %) + (y: %) : %  == {
		xx := rep x;
		yy := rep y;
		res: DoubleEndedList Term := empty();
		repeat {
			empty? xx => break;
			empty? yy => break;
			tx := first xx;
			ty := first yy;
			if tx.ex > ty.ex then {
				concat!(res, tx);
				xx := rest xx;
			}
			else if ty.ex > tx.ex then {
				concat!(res, ty);
				yy := rest yy;
			}
			else {
				r:R := tx.co + ty.co;
				if not zero? r then {
					concat!(res, [r, tx.ex]);
				}
				xx := rest xx;
				yy := rest yy;
			}
		}
		tt: Rep := if ~empty?(xx) then xx else yy;
		empty? firstCell(res) => per tt;
		-- WAS lastCell(res).rest := tt;
		setRest!(lastCell(res),tt);
		per firstCell(res);
	}


	-----------------------------
	--% Exports as OutputType %--
	-----------------------------
	(port:TextWriter) << (x:%):TextWriter	== {
		import from String;
		local writeTerm(t: Term): () == {
			numberSystem? => {
			    zero? t.ex => port << t.co;
			    if t.co ~= 1 then port << t.co << "*";
			    one?  t.ex => port << avar;
			    port << avar << "^" << t.ex;
			}
			zero? t.ex => port << t.co;
			if t.co ~= 1 then port << "(" << t.co << ")*";
			one?  t.ex => port << avar;
			port << avar << "^" << t.ex;
		}
		xx := rep x;
		empty? xx => port << 0$R;
		if ((first(xx)).co = -1) and characteristicZero?  then {
			port << " -";
			writeTerm(-(first xx));
		}
		else {
			writeTerm(first xx);
		}
		xx := rest xx;
		if orderedArithmeticType? then {
			while not empty? xx repeat {
				tx := first xx;
				if tx.co >= 0 then {
					port << " + ";
					writeTerm(tx);
				}
				else {
					port << " - ";
					writeTerm(-tx);
				}
				xx := rest xx;
			}
		} else {
			while not empty? xx repeat {
				port << " + ";
				writeTerm(first xx);
				xx := rest xx;
			}
		}
		port;
	}

	--------------------------------
	--% Exports as PrimitiveType %--
	--------------------------------
	(x:%) = (y:%):Boolean	== {
		xx := rep x;
		yy := rep y;
		repeat {
			empty? xx => return(empty? yy);
			empty? yy => return(false);
			tx := first xx;
			ty := first yy;
			not equalTerm?(tx,ty) => return(false);
			xx := rest xx; 
			yy := rest yy;
		}
		true;
	}
}

#if ALDORTEST
---------------------- test sup0.as ---------------------------
#include "algebra"
#include "aldortest"

-- #include "algebra"
-- #include "aldorinterp"
-- #include "aldortest"

X: Symbol == -"x";
Y: Symbol == -"y";


macro {
        Z == Integer;
        Zx == SparseUnivariatePolynomial0 (Z,X);
        Zxt == SparseUnivariatePolynomial0 (Zx,Y);
}

primitiveType():Boolean == {
        import from Z, Zx;
	x := term(1,1);
	p := term(1,2) - term(1,0);
	pp := term(1,2) - term(1,0);
	q := term(1,2) + x - term(1,0);
	(p = pp) and (p ~= q);
}

additiveType():Boolean == {
        import from Z, Zx;
	x := term(1,1);
	p: Zx := 0;
	q: Zx := term(1,2);
	r: Zx := term(1,0);
	s: Zx := x;
	t: Zx := -(term(1,2) + s + term(1,0));
	u := add!(p,q);
	add!(u,s);
	add!(u,r);
	minus!(u);
	add!(r,r);
	add!(r,u);
	(zero? p) and (q = term(1,2)) and (u = t) and (u = -(q+s+r)) and (r = term(1,0)) and (one? leadingCoefficient(r)) and zero? (degree(r));
}

linearCombinationType():Boolean == {
        import from Z, Zx;
        x := term(1,1);
	p := term(1,2) +x + term(1,0);
	t: Zx := - copy(p);
	times!(2,t);
	zero?(t + 2 * p);
}

freeLinearCombinationType():Boolean == {
        import from Z, Zx;
        x := term(1,1);
	f(z:Z):Z == z rem 2;
	p := 2*term(1,4) + term(1,3) + 2*term(1,2) + x + 2 * term(1,0);
	fp := term(1,3) + x;
	q := term(1,4) + 2*term(1,3) + term(1,2) + 2*x + term(1,0);
	fq := term(1,4) + term(1,2) + term(1,0);
	r: Zx := 2*x + 4 * term(1,0);
	fr : Zx := 0;
	s := term(1,4) + 2*term(1,3) + 4*term(1,2) + 2*x + 6 * term(1,0);
	fs := term(1,4);
	t := 2*term(1,4) + 2*term(1,3) + 4*term(1,2) + 2*x + 5 * term(1,0);
	ft: Zx := term(1,0);
	(map(f)(p) =  fp) and (map(f)(q) =  fq) and (map(f)(r) =  fr) and (map(f)(s) =  fs) and (map(f)(t) =  ft);
}

indexedFreeLinearCombinationType():Boolean == {
        import from Z, Zx;
        x := term(1,1);
	p := 2*term(1,4) + 2*term(1,2);
	
	bool: Boolean := (coefficient(p,4) = 2) and (coefficient(p,3) = 0) and (coefficient(p,2) = 2) and (coefficient(p,1) = 0) and (coefficient(p,0) = 0);
	
	p := setCoefficient!(p,5,10);
	setCoefficient!(p,4,8);
	setCoefficient!(p,3,6);
	setCoefficient!(p,2,4);
	setCoefficient!(p,1,2);
	setCoefficient!(p,0,-1);
	q := 10*term(1,5) + 8*term(1,4) + 6*term(1,3) + 4*term(1,2) + 2*x -term(1,0);
	bool := bool and (p = q);
	p := 0;
	add!(p,2,2);
	bool := bool and (zero? p);
	p := -4*term(1,4) - 1*term(1,2);
	p := add!(p,-5,5);
	add!(p,-3,3);
	add!(p,-1,2);
	z: Z := 0;
	add!(p,1,z);
	q := -5*term(1,5)  -4*term(1,4) -3*term(1,3) - 2*term(1,2) + term(1,0);
	bool := bool and (p = q);
}

indexedFreeModule():Boolean == {
        import from Z, Zx;
        x := term(1,1);
        p := term(1,3) - term(1,1);
        (degree p = 3) and (leadingCoefficient p = 1) and (trailingDegree p = 1) and (trailingCoefficient p = -1) and (reductum p = term(-1,1));
}

stdout << "Testing alg__sup0..." << endnl;
aldorTest("primitiveType",primitiveType );
aldorTest("additiveType",additiveType );
aldorTest("linearCombinationType",linearCombinationType );
aldorTest("freeLinearCombinationType",freeLinearCombinationType );
aldorTest("indexedFreeLinearCombinationType",indexedFreeLinearCombinationType );
aldorTest("indexedFreeModule",indexedFreeModule );
stdout << endnl;



#endif
