-------------------------------------------------------------------------------
--
-- sm_midrl.as: Degree-reverse-lexicographical exponent vectors 
--               of machine integers.
--
--------------------------------------------------------------------------------
--  Copyright (c) 1990-2007 Aldor Software Organization Ltd (Aldor.org).
-- Copyright: INRIA, UWO and University of Lille I, 2001
-- Copyright: Marc Moreno Maza
--------------------------------------------------------------------------------

-- dom MachineIntegerDegreeReverseLexicographicalExponent

#include "algebra"

macro { 
      Z == Integer;
      MI == MachineInteger;
}

+++ `MachineIntegerDegreeReverseLexicographicalExponent(V)' implements the 
+++ exponents of the monomials (or power products) generated by the finite 
+++ set of variables `V'. Such an exponent is represented by a primtive 
+++ array of machine integers with length `dim = n + 1' where `n' is the
+++ number of elements in `V'. Given `e' in `%' and `i' in `1..n' the
+++ degree of `e' in `variable(i)$V' is stored in slot `i'. Slot `0'
+++ is used to store the total degree of `e' that is the sum of the content 
+++ of all the other slots. An exponent `a' is greater than an exponent `b' 
+++ if `a' is greater than `b' w.r.t. the degree-reverse-lexicographical 
+++ ordering induced by `V' (by comparing `a.i' and `b.i' for `i' in 
+++ `lastIndex..firstIndex' after comparing the total degrees of `a' and `b').
+++ Author: Marc Moreno Maza
+++ Date Created: 09/07/01
+++ Date Last Update: 10/07/01


MachineIntegerDegreeReverseLexicographicalExponent(V: FiniteVariableType): 
 MachineIntegerExponentVectorCategory(V) == DirectProduct(next(#$V)$MI, MI) add { 
      Rep == PrimitiveArray(MI);
      import from V, Rep;
      local n__V: MI == #$V;
      local dim: MI == next(n__V);
      local a: MI == 0;
      local b: MI == prev(dim);
      exponent(v: V): % == {
            xx: Rep := new(dim,0$MI);
            set!(xx, index(v), 1);
            set!(xx, 0, 1);
            per xx; 
      }
      exponent(v: V, z: Z): % == {
            mi: MI := machine(z);
            assert(mi >= 0);
            xx: Rep := new(dim,0$MI);
            set!(xx, index(v), mi);
            set!(xx, 0, mi);
            per xx; 
      }
      exponent(g: Generator Cross(V, Z)): % == {
            xx: Rep := new(dim,0$MI);
            local v: V; local z: Z;
            td__x: MI := 0; local mi: MI;
            for cross in g repeat {
                (v,z) := cross;
                mi := machine(z);
                assert(mi >= 0);
                set!(xx, index(v), mi);
                td__x := td__x + mi;
             }
             set!(xx, 0, td__x);
             per xx;
      }
      exponent(lv: List(V), lz: List(Z)): % == {
            import from List(V), List(Z);
            assert(#lv = #lz);
            xx: Rep := new(dim,0$MI);
            local v: V; local z: Z;
            td__x: MI := 0; local mi: MI;
            while not empty? lv repeat {
                v := first lv; lv := rest lv;
                z := first lz; lz := rest lz;
                mi := machine(z);
                assert(mi >= 0);
                set!(xx, index(v), mi);
                td__x := td__x + mi;
             }
             set!(xx, 0, td__x);
             per xx;
      }
      exponent(t: Tuple MI): % == {
             assert(n__V = length(t));
             xx: Rep := new(dim);
             td__x: MI := 0; local mi: MI;
             for i in 1..b repeat {
                 mi := element(t,i);
                 assert(mi >= 0);
                 set!(xx, i, mi);
                 td__x := td__x + mi;
             }
             set!(xx, 0, td__x);
             per xx;
      }
      exponent(p: PrimitiveArray MI): % == {
             for i in a..b repeat {
                 assert(p.i >= 0);
             }
             per p;
      }
      parray(x: %): PrimitiveArray MI == {
             rep x;
      }
      terms(x: %): Generator Cross(V, Z) == generate {
             xx: Rep := rep(x);
             import from Z;
             for i in 1..b repeat {
                  if not zero? xx.i then yield (variable i, xx.i::Z);
             }
      }
      copy!(y: %, x:%): % == {
            yy := rep y;
            zero? yy.0 => copy x;
            xx := rep x;
            for i in a..b repeat set!(yy, i, xx.i);
            y;
      }
      hash(x: %): MI == {
            h: MI := 0;
            xx := rep x;
            for i in 1..b repeat {
                h := h + i * xx.i;
            }
            h;
      }
      0: % == per new(dim,0$MI);

      zero? (x: %): Boolean == zero? ((rep x).0);

      (x: %) + (y: %) : % == {
             xx := rep x;
             yy := rep y;
             zz: Rep := new(dim);
             for i in a..b repeat set!(zz, i, xx.i + yy.i);
             per zz;
      }
      add!(x: %, y: %): % == {
             xx := rep x;
             zero? xx.0 => copy y;
             yy := rep y;
             for i in a..b repeat set!(xx, i, xx.i + yy.i);
             x;
      }
      add!(x: %, v: V, z: Z): % == {
             xx: Rep := rep(x);
             zero? xx.0 => copy exponent(v,z);
             mi: MI := machine(z);
             assert(mi >= 0);
             i__v: MI := index(v);
             set!(xx, i__v, xx.(i__v) + mi);
             set!(xx, 0, xx.0 + mi);
             per xx;
      }
      times(z: Z, x: %): % == {
             mi: MI := machine(z);
             assert(mi >= 0);
             mi = 0 => 0;
             mi = 1 => x;
             xx: Rep  := rep x;
             yy: Rep := new(dim);
             for i in a..b repeat {
                 set!(yy, i, mi * xx.i);
             }
             per(yy);
      }
      (x: %) < (y: %) : Boolean == {
             xx: Rep  := rep x;
             yy: Rep  := rep y;
             xx.0 < yy.0 => true;
             xx.0 > yy.0 => false;
             import from Boolean;
             for i in b..1 by -1 repeat {
                 xx.i > yy.i => return true;
                 xx.i < yy.i => return false;
             }
             false;
      }
      cancel? (x: %, y: %): Boolean == {
             xx: Rep := rep x;
             yy: Rep  := rep y;
             import from Boolean;
             for i in a..b repeat {
                 xx.i < yy.i => return false;
             }
             true;
      }
      cancel (x: %, y: %): % == {
             xx: Rep  := rep x;
             yy: Rep  := rep y;
             zz: Rep := new(dim);
             for i in a..b repeat {
                 -- set!(zz, i, cancel(xx.i,yy.i)); GENERICALLY
                 assert (xx.i >= yy.i);
                 set!(zz, i, xx.i - yy.i);
             }
             per zz;
      }
      cancelIfCan (x: %, y: %): Partial % == {
             import from Partial(%);
             xx: Rep := rep x;
             yy: Rep  := rep y;
             zz: Rep := new(dim);
             for i in a..b repeat {
                 xx.i >= yy.i => set!(zz, i, xx.i - yy.i);
                 free!(zz);
                 return failed;
             }
             [per zz];
      }
      variables(x: %): List(V) == {
             xx: Rep := rep(x);
             lv__x: List(V) := empty;
             for i in b..1 by -1 repeat {
                 0 < xx.i => lv__x := cons(variable(i), lv__x);
             }
             lv__x;
      }
      mainVariable(x: %): Partial(V) == {
             import from  Partial(V);
             xx: Rep := rep(x);
             zero? xx.0 => failed;
             for i in 1..b repeat {
                 0 < xx.i => return [variable(i)];
             }
             never;
      }
      degree(x: %, v: V): Z == {
            mi: MI := (rep x).(index(v));
            mi::Z;
      }
      totalDegree(x: %): Z == {
            mi: MI := (rep x).0;
            mi::Z;
      }
      totalDegree(x: %, lv: List(V)): Z == {
            xx: Rep := rep(x);
            d: MI := 0;
            for v in lv repeat {
                d := d + xx.(index(v));
            }
            d::Z;
      }
      gcd(x: %, y:%): % == {
            xx := rep x;
            yy := rep y;
            zz: Rep := new(dim);
            d: MI := 0;
            local mi: MI;
            for i in 1..b repeat {
                 mi := min(xx.i,yy.i);
                 set!(zz, i, mi);
                 d := d + mi;
            }
            set!(zz, 0, d);
            per zz;
      }
      lcm(x: %, y:%): % == {
            xx := rep x;
            yy := rep y;
            zz: Rep := new(dim);
            d: MI := 0;
            local mi: MI;
            for i in 1..b repeat {
                 mi := max(xx.i,yy.i);
                 set!(zz, i, mi);
                 d := d + mi;
            }
            set!(zz, 0, d);
            per zz;
      }
      syzygy(x: %, y:%): (%, %) == {
            xx := rep x;
            yy := rep y;
            zz__x: Rep := new(dim,0$MI);
            zz__y: Rep := new(dim,0$MI);
            td__zz__x: MI := 0;
            td__zz__y: MI := 0;
            local mi: MI;
            for i in 1..b repeat {
                if xx.i > yy.i then {
                   mi := xx.i - yy.i;
                   set!(zz__y, i, mi);
                   td__zz__y := td__zz__y + mi;
                } else {
                   mi := yy.i - xx.i;
                   set!(zz__x, i, mi);
                   td__zz__x := td__zz__x + mi;
                }
            }
            set!(zz__x, 0, td__zz__x);
            set!(zz__y, 0, td__zz__y);
            (per(zz__x), per(zz__y));
      }
}   



#if ALDOC
\thistype{MachineIntegerDegreeReverseLexicographicalExponent}
\History{Marc Moreno Maza}{08/07/01}{created}
\Usage{import from \this~V}
\Params{
{\em V} & \altype{FiniteVariableType} & The type of the variables \\
}
\Descr{\this~V implements the 
 exponents of the mo\-no\-mials (or power products) generated by the finite 
 set of variables $V.$ Such an exponent is represented by a primtive 
 array of machine integers with length $dim = n + 1$ where $n$ is the
 number of elements in $V.$ Given $e$ in \% and $i$ in $1 \cdots n$ the
 degree of $e$ w.r.t. the $i$-th variable of $V$ is stored in slot $i$. Slot $0$
 is used to store the total degree of $e$, that is the sum of the content 
 of all the other slots. An exponent $a$ is greater than an exponent $b$ 
 if $a$ is greater than $b$ w.r.t. the degree-reverse-lexicographical 
 ordering induced by $V$ (by comparing $a_i$ and $b_i$ for $i$ running
 from $n$ to $1$, after comparing the total degrees of $a$ and $b$). }
\begin{exports}
\category{\altype{MachineIntegerExponentVectorCategory} V} \\
\end{exports}
#endif




