(* color to string encoder/decoder module type *)
module type ENCODE =
  sig
    type t
    val bytes_per_pixel : int
    val get : string -> int -> t
    val set : string -> int -> t -> unit
    val make : t -> string
  end

(* low level image module type *)
module type RAWIMAGE = sig
  module Encode : ENCODE
  type elt
  and bitmap
  and t = { width : int; height : int; bitmap : bitmap; } 
  val width : t -> int
  val height : t -> int
  val dump : t -> string
  val create_with : int -> int -> string -> t
  val create : int -> int -> t
  val make : int -> int -> elt -> t
  val unsafe_access : t -> int -> int -> string * int
  val get_strip : t -> int -> int -> int -> string
  val set_strip : t -> int -> int -> int -> string -> unit
  val get_scanline : t -> int -> string
  val set_scanline : t -> int -> string -> unit
  val unsafe_get : t -> int -> int -> elt
  val unsafe_set : t -> int -> int -> elt -> unit
  val get : t -> int -> int -> elt
  val set : t -> int -> int -> elt -> unit
  val destroy : t -> unit
  val copy : t -> t
  val sub : t -> int -> int -> int -> int -> t
  val blit : t -> int -> int -> t -> int -> int -> int -> int -> unit
  val map : (elt -> elt -> elt) -> 
    t -> int -> int -> t -> int -> int -> int -> int -> unit
end

(* low level image creater *)
module MakeRawImage : functor (E : ENCODE) -> (RAWIMAGE with type elt = E.t)

(* low image container module type *)
module type CONTAINER = sig
  type container 
  type rawimage
  val rawimage : container -> rawimage 
  val create_default : int -> int -> rawimage -> container
  val create_duplicate : container -> int -> int -> rawimage -> container
end

module type IMAGE = sig
  type t
  type elt
  val create : int -> int -> t
  val make : int -> int -> elt -> t
  val dump : t -> string
  val unsafe_access : t -> int -> int -> string * int
  val get_strip : t -> int -> int -> int -> string
  val set_strip : t -> int -> int -> int -> string -> unit
  val get_scanline : t -> int -> string
  val set_scanline : t -> int -> string -> unit
  val unsafe_get : t -> int -> int -> elt
  val unsafe_set : t -> int -> int -> elt -> unit
  val get : t -> int -> int -> elt
  val set : t -> int -> int -> elt -> unit
  val destroy : t -> unit
  val copy : t -> t
  val sub : t -> int -> int -> int -> int -> t
  val blit : t -> int -> int -> t -> int -> int -> int -> int -> unit
  val map : (elt -> elt -> elt) -> 
    t -> int -> int -> t -> int -> int -> int -> int -> unit
end

module Make : functor (RI : RAWIMAGE) -> 
              functor (CON : CONTAINER with type rawimage = RI.t) -> 
  IMAGE with type t = CON.container
	and  type elt = RI.elt

module type CONTAINER_INDEXED = sig
  type container
  type rawimage
  type mapelt
  val rawimage : container -> rawimage
  val create_default : int -> int -> rawimage -> container
  val create_duplicate : container -> int -> int -> rawimage -> container
  val colormap : container -> mapelt Color.map
end

module type IMAGEINDEXED = sig
  type t
  type elt
  type mapelt
  val create : int -> int -> t
  val make : int -> int -> elt -> t
  val dump : t -> string
  val unsafe_access : t -> int -> int -> string * int
  val get_strip : t -> int -> int -> int -> string
  val set_strip : t -> int -> int -> int -> string -> unit
  val get_scanline : t -> int -> string
  val set_scanline : t -> int -> string -> unit
  val unsafe_get : t -> int -> int -> elt
  val unsafe_set : t -> int -> int -> elt -> unit
  val get : t -> int -> int -> elt
  val set : t -> int -> int -> elt -> unit
  val unsafe_get_color : t -> int -> int -> mapelt
  val get_color : t -> int -> int -> mapelt
  val destroy : t -> unit
  val copy : t -> t
  val sub : t -> int -> int -> int -> int -> t
  val blit : t -> int -> int -> t -> int -> int -> int -> int -> unit
  val map : (elt -> elt -> elt) -> 
    t -> int -> int -> t -> int -> int -> int -> int -> unit
end

module MakeIndexed(RI:RAWIMAGE with type elt = int)
    (CON:CONTAINER_INDEXED with type rawimage = RI.t) : 
  IMAGEINDEXED with type t = CON.container
               and  type elt = int
	       and  type mapelt = CON.mapelt
