#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>
#include <sys/stat.h>
#if defined(HAVE_CLAMAV)
# include <clamav.h>
#else
#warning No antivirus library defined, this module will not operate with any virus database
#endif
#include <gtk/gtk.h>
#ifdef HAVE_EDV2
# include <endeavour2.h>
#endif

#include "cfg.h"
#include "win.h"
#include "wincb.h"
#include "windb.h"
#include "core.h"
#include "cfglist.h"
#include "config.h"


/* Utilities */
static gchar *COPY_SHORTEN_STRING(const gchar *s, const gint max);
static const gchar *TIME_DURATION_STRING(
	core_struct *core, const gulong dt, const gulong t
);

#if defined(HAVE_CLAMAV)
static gint WinDBGetPatternStatsFindIterate(
        win_struct *win,
	const gchar *name,
        struct cl_node *clam_pnode,
        guint8 **pattern_rtn,
        gint *pattern_length_rtn,
        gulong *last_updated_time_rtn,
        gulong *discovery_time_rtn
);
#endif
gint WinDBGetPatternStats(
	win_struct *win,
	const gchar *name,
	guint8 **pattern_rtn,
	gint *pattern_length_rtn,
	gulong *last_updated_time_rtn,
	gulong *discovery_time_rtn
);

gint WinDBChanged(win_struct *win);

#if defined(HAVE_CLAMAV)
static void WinDBUpdateIterate(
	core_struct *core, win_struct *win,
	GtkCTree *ctree, GtkCTreeNode *pnode, gchar **strv,
	struct cl_node *clam_pnode,
	const gulong start_time, const gboolean verbose,
	gint *patterns_loaded, gulong *patterns_size_bytes
);
#endif
void WinDBUpdate(
	win_struct *win, const gchar *db_location,
	const gboolean verbose
);
void WinDBQueueUpdate(win_struct *win);
void WinDBClear(win_struct *win);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      Returns a copy of the string that is no longer than max
 *      characters with appropriate shortened notation where
 *      appropriate.
 */
static gchar *COPY_SHORTEN_STRING(const gchar *s, const gint max)
{
	gint len;

	if(s == NULL)
	    return(NULL);

	len = STRLEN(s);
	if((len > max) && (max > 3))
	{
	    /* Need to shorten string */
	    gint i = len - max + 3;

	    return(g_strdup_printf(
		"...%s", (const gchar *)(s + i)
	    ));
	}
	else
	{
	    return(STRDUP(s));
	}
}

#ifndef HAVE_EDV2
static const gchar *EDVSizeStrDelim(gulong v)
{
	static gchar s[80];
	g_snprintf(
	    s, sizeof(s),
	    "%ld", v
	);
	return(s);
}
#endif

/*
 *      Returns a statically allocated string describing the time
 *      lapsed specified by dt.
 */
static const gchar *TIME_DURATION_STRING(
	core_struct *core, const gulong dt, const gulong t
)
{
	gulong ct;
	static gchar buf[80];
			     
	/* Less than one second? */
	if(dt < 1l)
	{
	    g_snprintf(
		buf, sizeof(buf),
		"less than a second ago"
	    );
	}
	/* Less than one minute? */
	else if(dt < (1l * 60l))
	{
	    ct = MAX(dt / 1l, 1l);
	    g_snprintf( 
		buf, sizeof(buf),   
		"%ld second%s ago",
		ct, (ct == 1l) ? "" : "s"
	    );
	}     
	/* Less than one hour? */  
	else if(dt < (60l * 60l))
	{
	    ct = MAX(dt / 60l, 1l);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld minute%s ago",
		ct, (ct == 1l) ? "" : "s"
	    );
	}
	/* Less than one day? */
	else if(dt < (24l * 60l * 60l))
	{
	    ct = MAX(dt / 60l / 60l, 1l);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld hour%s ago",
		ct, (ct == 1l) ? "" : "s"
	    );
	}
	/* Less than one week? */
	else if(dt < (7l * 24l * 60l * 60l))
	{
	    ct = MAX(dt / 60l / 60l / 24l, 1l);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld day%s ago",
		ct, (ct == 1l) ? "" : "s"
	    );
	}
	/* Less than one month (28 days)? */
	else if(dt < (28l * 24l * 60l * 60l))
	{
	    ct = MAX(dt / 60l / 60l / 24l / 7l, 1l);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld week%s ago",
		ct, (ct == 1l) ? "" : "s"
	    );
	}     
	/* Less than 6 months ago? */
	else if(dt < (6l * 28l * 24l * 60l * 60l))
	{
	    ct = MAX(dt / 60l / 60l / 24l / 30l, 1l);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld month%s",
		ct, (ct == 1l) ? "" : "s"
	    );
	}     
	else
	{
#ifdef HAVE_EDV2
	    strncpy((char *)buf, EDVDateString(core->edv_ctx, t), sizeof(buf));
#else
	    gchar *s;
	    time_t tt = (time_t)t;
	    strncpy((char *)buf, ctime(&tt), sizeof(buf));
	    s = (gchar *)strchr((char *)buf, '\n');
	    if(s != NULL)
		*s = '\0';
#endif
	}

	buf[sizeof(buf) - 1] = '\0';

	return(buf);
}


#if defined(HAVE_CLAMAV)
static gint WinDBGetPatternStatsFindIterate(
        win_struct *win,
	const gchar *name,
        struct cl_node *clam_pnode,
        guint8 **pattern_rtn,
        gint *pattern_length_rtn,
        gulong *last_updated_time_rtn,
        gulong *discovery_time_rtn
)
{
#define PROCESS_FIND(_data_,_data_len_,_lut_,_dt_) {	\
 if(pattern_rtn != NULL) {				\
  if(((_data_) != NULL) && ((_data_len_) > 0))		\
   *pattern_rtn = g_memdup((_data_), (_data_len_));	\
 }							\
 if(pattern_length_rtn != NULL)				\
  *pattern_length_rtn = (_data_len_);			\
							\
 if(last_updated_time_rtn != NULL)			\
  *last_updated_time_rtn = (_lut_);			\
 if(discovery_time_rtn != NULL)				\
  *discovery_time_rtn = (_dt_);				\
}

	if(clam_pnode == NULL)
	    return(1);

	/* Iterate through Extended Boyer-Moore list */
	if(clam_pnode->bm_suffix != NULL)
	{
	    gint i;
	    struct cli_bm_patt *pattern;

	    for(i = 0; clam_pnode->bm_suffix[i] != NULL; i++)
	    {
		for(pattern = clam_pnode->bm_suffix[i];
		    pattern != NULL;
		    pattern = pattern->next
		)
		{
		    if(pattern->virname != NULL)
		    {
			if(!strcmp(pattern->virname, name))
			{
			    PROCESS_FIND(
				(const guint8 *)pattern->pattern,
				pattern->length,
				0l,
				0l
			    );
			    return(0);
			}
		    }
		    if(pattern->viralias != NULL)
		    {
			if(!strcmp(pattern->viralias, name))
			{
			    PROCESS_FIND(
				(const guint8 *)pattern->pattern,
				pattern->length,
				0l,
				0l
			    );
			    return(0);
			}
		    }
		}
	    }
	}

	/* Extended Aho-Corasick */
	if(clam_pnode->ac_nodes > 0)
	{
	    gint i;
	    struct cli_ac_node *clam_ac_node;

	    /* Iterate through Extended Aho-Corasick list */
	    for(i = 0; i < clam_pnode->ac_nodes; i++)
	    {
		/* Get this Extended Aho-Corasick node */
		clam_ac_node = clam_pnode->ac_nodetable[i];
		if((clam_ac_node != NULL) ?
		    (clam_ac_node->list != NULL) : FALSE
		)
		{
		    /* Iterate through this list of patterns */
		    struct cli_ac_patt *pattern;
		    for(pattern = clam_ac_node->list;
			pattern != NULL;
			pattern = pattern->next
		    )
		    {
			if(pattern->virname != NULL)
			{
			    if(!strcmp(pattern->virname, name))
			    {
				/* The pattern data is in 16 bits with
				 * the high bit set to 0x00, need to
				 * reduce/convert to 8 bit
				 */
				const gint len = (gint)pattern->length;
				guint8 *p8 = (len > 0) ?
				    (guint8 *)g_malloc(len) : NULL;
				if(p8 != NULL)
				{
				    const guint8 *sp8 = (const guint8 *)pattern->pattern;
				    guint8	*tp8 = p8,
						*tpend8 = tp8 + len;
				    while(tp8 < tpend8)
				    {
					*tp8 = *sp8;
					tp8 += 1;
					sp8 += 2;
				    }
				}
			        PROCESS_FIND(
				    p8,
				    len,
				    0l,
				    0l
				);
				g_free(p8);
				return(0);
			    }
			}
			if(pattern->viralias != NULL)
			{
			    if(!strcmp(pattern->viralias, name))
			    {
				/* The pattern data is in 16 bits with
				 * the high bit set to 0x00, need to
				 * reduce/convert to 8 bit
				 */
				const gint len = (gint)pattern->length;
				guint8 *p8 = (len > 0) ?
				    (guint8 *)g_malloc(len) : NULL;
				if(p8 != NULL)
				{
				    const guint8 *sp8 = (const guint8 *)pattern->pattern;
				    guint8	*tp8 = p8,
						*tpend8 = tp8 + len;
				    while(tp8 < tpend8)
				    {
					*tp8 = *sp8;
					tp8 += 1;
					sp8 += 2;
				    }
				}
			        PROCESS_FIND(
				    p8,
				    len,
				    0l,
				    0l
				);
				g_free(p8);
				return(0);
			    }
			}
		    }
		}
	    }
	}


	/* Iterate through MD5 list */
	if(clam_pnode->md5_hlist != NULL)
	{
	    gint i;
	    struct cli_md5_node *pattern;

	    for(i = 0; clam_pnode->md5_hlist[i] != NULL; i++)
	    {
		for(pattern = clam_pnode->md5_hlist[i];
		    pattern != NULL;
		    pattern = pattern->next
		)
		{
		    if(pattern->virname != NULL)
		    {
			if(!strcmp(pattern->virname, name))
			{
			    PROCESS_FIND(
				(const guint8 *)pattern->md5,
				pattern->size,
				0l,
				0l
			    );
			    return(0);
			}
		    }
		    if(pattern->viralias != NULL)
		    {
			if(!strcmp(pattern->viralias, name))
			{
			    PROCESS_FIND(
				(const guint8 *)pattern->md5,
				pattern->size,
				0l,
				0l
			    );
			    return(0);
			}
		    }
		}
	    }
	}

	/* Iterate through ZIP metadata list */
	if(clam_pnode->zip_mlist != NULL)
	{
	    struct cli_meta_node *pattern;

	    for(pattern = clam_pnode->zip_mlist;
		pattern != NULL;
		pattern = pattern->next
	    )
	    {
		if(pattern->virname != NULL)
		{
		    if(!strcmp(pattern->virname, name))
		    {
			PROCESS_FIND(
			    (const guint8 *)pattern->filename,
			    STRLEN(pattern->filename),
			    0l,
			    0l
			);
			return(0);
		    }
		}
	    }
	}

	/* Iterate through RAR metadata list */
	if(clam_pnode->rar_mlist != NULL)
	{
	    struct cli_meta_node *pattern;

	    for(pattern = clam_pnode->rar_mlist;
		pattern != NULL;
		pattern = pattern->next
	    )
	    {
		if(pattern->virname != NULL)
		{
		    if(!strcmp(pattern->virname, name))
		    {
			PROCESS_FIND(
			    (const guint8 *)pattern->filename,
			    STRLEN(pattern->filename),
			    0l,
			    0l
			);
			return(0);
		    }
		}
	    }
	}

#undef PROCESS_FIND

	return(1);
}
#endif

/*
 *	Gets the pattern statistics.
 *
 *	The name specifies the name of the pattern.
 */
gint WinDBGetPatternStats(
	win_struct *win,
	const gchar *name,
	guint8 **pattern_rtn,
	gint *pattern_length_rtn,
	gulong *last_updated_time_rtn,
	gulong *discovery_time_rtn
)
{
	if(pattern_rtn != NULL)
	    *pattern_rtn = NULL;
	if(pattern_length_rtn != NULL)
	    *pattern_length_rtn = 0;
	if(last_updated_time_rtn != NULL)
	    *last_updated_time_rtn = 0l;
	if(discovery_time_rtn != NULL)
	    *discovery_time_rtn = 0l;

	if((win == NULL) || STRISEMPTY(name))
	    return(-2);

#if defined(HAVE_CLAMAV)
	return(WinDBGetPatternStatsFindIterate(
	    win, name,
	    (struct cl_node *)win->db,
	    pattern_rtn,
	    pattern_length_rtn,
	    last_updated_time_rtn,
	    discovery_time_rtn
	));
#else
	return(-1);
#endif
}


/*
 *	Checks if the virus database that is currently displayed on
 *	the Virus Database list (from the last call to WinDBUpdate())
 *	has changed.
 *
 *	Returns:
 *
 *	1	The database has changed.
 *	0	The database has not changed.
 *	-1	If there was no prior call to WinDBUpdate() or that
 *		the database was not previously loaded or is empty.
 *	-2	Invalid value.
 */
gint WinDBChanged(win_struct *win)
{
	if(win == NULL)
	    return(-2);

#if defined(HAVE_CLAMAV)
	if(win->db_stat != NULL)
	{
	    struct cl_stat *clam_stat = (struct cl_stat *)win->db_stat;
	    if(cl_statchkdir(clam_stat) == 1)
		return(1);
	    else
		return(0);
	}
#endif

	return(-1);
}


#define INSERT_NODE_FOLDER(_label_,_pnode_,_snode_) {	\
 if(ncolumns > 0)					\
  strv[0] = (_label_);					\
 node = gtk_ctree_insert_node(				\
  ctree, (_pnode_), (_snode_),				\
  strv, WIN_LIST_PIXMAP_TEXT_SPACING,			\
  core->folder_closed_pixmap,				\
  core->folder_closed_mask,				\
  core->folder_opened_pixmap,				\
  core->folder_opened_mask,				\
  FALSE, TRUE						\
 );							\
 if(node != NULL) {					\
  win_db_item_struct *d = WIN_DB_ITEM(g_malloc0(	\
   sizeof(win_db_item_struct)				\
  ));							\
  d->type = WIN_DB_ITEM_TYPE_FOLDER;			\
  d->name = STRDUP(_label_);				\
  gtk_ctree_node_set_row_data_full(			\
   ctree, node,						\
   d, WinDBListItemDestroyCB				\
  );							\
 }							\
}

#define INSERT_NODE_ITEM(_label_,_pnode_,_snode_) { \
 if(ncolumns > 0)					\
  strv[0] = (_label_);					\
 node = gtk_ctree_insert_node(				\
  ctree, (_pnode_), (_snode_),				\
  strv, WIN_LIST_PIXMAP_TEXT_SPACING,			\
  core->db_item_pixmap, core->db_item_mask,		\
  NULL, NULL,						\
  TRUE, FALSE						\
 );							\
 if(node != NULL) {					\
  win_db_item_struct *d = WIN_DB_ITEM(g_malloc0(	\
   sizeof(win_db_item_struct)				\
  ));							\
  d->type = WIN_DB_ITEM_TYPE_PATTERN;			\
  d->name = STRDUP(_label_);				\
  gtk_ctree_node_set_row_data_full(			\
   ctree, node,						\
   d, WinDBListItemDestroyCB				\
  );							\
 }							\
}

#if defined(HAVE_CLAMAV)
static void WinDBUpdateIterate(
	core_struct *core, win_struct *win,
	GtkCTree *ctree, GtkCTreeNode *pnode, gchar **strv,
	struct cl_node *clam_pnode,
	const gulong start_time, const gboolean verbose,
	gint *patterns_loaded, gulong *patterns_size_bytes
)
{
	gint i;
	struct cli_ac_node *clam_ac_node;
	GtkCList *clist = GTK_CLIST(ctree);
	const gint ncolumns = clist->columns;

/* Increment *patterns_loaded, thaw/freeze the list, and update
 * the status bar
 */
#define UPDATE_PATTERNS_LOADED_STATUS(_dpatterns_,_dpattern_size_) { \
 /* Count this pattern as loaded */			\
 *patterns_loaded = (*patterns_loaded) + (_dpatterns_);	\
 *patterns_size_bytes = (*patterns_size_bytes) + (_dpattern_size_); \
							\
 /* Update progress every 50 patterns loaded */		\
 if(verbose && (((*patterns_loaded) % 50) == 0)) {	\
  WinStatusProgress(win, -1.0f, FALSE);			\
  WinStatusTime(win, (gulong)time(NULL) - start_time, TRUE); \
 }							\
							\
 /* Update status message every 100 patterns loaded */	\
 if(verbose && (((*patterns_loaded) % 100) == 0)) {	\
  gchar *s2 = STRDUP(EDVSizeStrDelim(			\
    (gulong)(*patterns_loaded)				\
  ));							\
  gchar *s = g_strdup_printf(				\
   "%s patterns loaded...",				\
   s2							\
  );							\
  g_free(s2);						\
  WinStatusMessage(win, s, TRUE);			\
  g_free(s);						\
 }							\
							\
 /* Thaw and refreeze list every 500 patterns loaded */	\
 if(((*patterns_loaded) % 500) == 0) {			\
  gtk_clist_thaw(clist);				\
  gtk_clist_freeze(clist);				\
 }							\
}

	/* Iterate through Extended Boyer-Moore list */
	if((clam_pnode->bm_suffix != NULL) && (win->stop_count == 0))
	{
	    gchar *pattern_label;
	    struct cli_bm_patt *pattern;
	    GtkCTreeNode *node, *pnode2;

	    for(i = 0; clam_pnode->bm_suffix[i] != NULL; i++)
	    {
		if(win->stop_count > 0)
		    break;

		/* Get this list of patterns */
	        pattern = clam_pnode->bm_suffix[i];

		INSERT_NODE_FOLDER(
		    "Extended Boyer-Moore", pnode, NULL
		);
		if(node == NULL)
		    break;

		pnode2 = node;

		/* Iterate through this list of patterns */
		while(pattern != NULL)
		{
		    if(win->stop_count > 0)
			break;

		    if(!STRISEMPTY(pattern->virname) &&
		       !STRISEMPTY(pattern->viralias)
		    )
		    {
			pattern_label = g_strdup_printf(
			    "%s (%s)",
			    pattern->virname,
			    pattern->viralias
			);
		    }     
		    else if(!STRISEMPTY(pattern->virname))
		    {
			pattern_label = STRDUP(pattern->virname);
		    }
		    else if(!STRISEMPTY(pattern->viralias))
		    {
			pattern_label = STRDUP(pattern->viralias);
		    }
		    else
		    {
			pattern = pattern->next;
			continue;
		    }

		    /* Add this pattern to the list */
		    INSERT_NODE_ITEM(
			pattern_label, pnode2, NULL
		    );
		    g_free(pattern_label);

		    UPDATE_PATTERNS_LOADED_STATUS(1, pattern->length);

		    /* Get the next pattern */
		    pattern = pattern->next;
		}
	    }
	}

	/* Extended Aho-Corasick */
	if((clam_pnode->ac_nodes > 0) && (win->stop_count == 0))
	{
	    GtkCTreeNode *node, *pnode2;

	    INSERT_NODE_FOLDER(
		"Extended Aho-Corasick", pnode, NULL
	    );
	    pnode2 = node;

	    /* Iterate through Extended Aho-Corasick list */
	    for(i = 0; i < clam_pnode->ac_nodes; i++)
	    {
		if(win->stop_count > 0)
		    break;

		/* Get this Extended Aho-Corasick node */
		clam_ac_node = clam_pnode->ac_nodetable[i];
		if((clam_ac_node != NULL) ?
		    (clam_ac_node->list != NULL) : FALSE
		)
		{
		    gchar *pattern_label;

		    /* Get this list of patterns */
		    struct cli_ac_patt *pattern = clam_ac_node->list;

		    GtkCTreeNode *pnode3;

		    /* Create a subfolder if this node has multiple
		     * patterns
		     */
		    if(pattern->next != NULL)
		    {
			if(!STRISEMPTY(pattern->virname) &&
			   !STRISEMPTY(pattern->viralias)  
			)
			{
			    pattern_label = g_strdup_printf(
				"%s (%s)",
				pattern->virname,
				pattern->viralias
			    );
			}
			else if(!STRISEMPTY(pattern->virname))
			{
			    pattern_label = STRDUP(pattern->virname);
			}
			else if(!STRISEMPTY(pattern->viralias))
			{
			    pattern_label = STRDUP(pattern->viralias);
			}
			else
			{
			    pattern_label = STRDUP("");
			}
			INSERT_NODE_FOLDER(
			    pattern_label, pnode2, NULL
			);
			g_free(pattern_label);
			pnode3 = node;
		    }
		    else
		    {
			pnode3 = pnode2;
		    }

		    /* Iterate through this list of patterns */
		    while(pattern != NULL)
		    {
			if(win->stop_count > 0)
			    break;

			if(!STRISEMPTY(pattern->virname) && 
			   !STRISEMPTY(pattern->viralias)
			)
			{
			    pattern_label = g_strdup_printf(
				"%s (%s)",
				pattern->virname,
				pattern->viralias
			    );
			}
			else if(!STRISEMPTY(pattern->virname))
			{
			    pattern_label = STRDUP(pattern->virname);
			}
			else if(!STRISEMPTY(pattern->viralias))
			{
			    pattern_label = STRDUP(pattern->viralias);
			}
			else
			{
			    pattern = pattern->next;
			    continue;
			}   

			/* Add this pattern to the list */
			INSERT_NODE_ITEM(
			    pattern_label, pnode3, NULL
			);
			g_free(pattern_label);

			UPDATE_PATTERNS_LOADED_STATUS(1, pattern->length);

			/* Get the next pattern */
			pattern = pattern->next;
		    }
		}
	    }
	}


	/* Iterate through MD5 list */
	if((clam_pnode->md5_hlist != NULL) && (win->stop_count == 0))
	{
	    gchar *pattern_label;
	    struct cli_md5_node *pattern;
	    GtkCTreeNode *node, *pnode2;

	    for(i = 0; clam_pnode->md5_hlist[i] != NULL; i++)
	    {
		if(win->stop_count > 0)
		    break;

		/* Get this list of patterns */
		pattern = clam_pnode->md5_hlist[i];

		INSERT_NODE_FOLDER(
		    "MD5", pnode, NULL
		);
		if(node == NULL)
		    break;

		pnode2 = node;

		/* Iterate through this list of patterns */
		while(pattern != NULL)
		{
		    if(win->stop_count > 0)
			break;

		    if(!STRISEMPTY(pattern->virname) &&
		       !STRISEMPTY(pattern->viralias) 
		    )
		    {
			pattern_label = g_strdup_printf(
			    "%s (%s)",
			    pattern->virname,
			    pattern->viralias
			);
		    }
		    else if(!STRISEMPTY(pattern->virname))
		    {
			pattern_label = STRDUP(pattern->virname);
		    }
		    else if(!STRISEMPTY(pattern->viralias))
		    {
			pattern_label = STRDUP(pattern->viralias);
		    }
		    else
		    {
			pattern = pattern->next;
			continue;
		    }

		    /* Add this pattern to the list */
		    INSERT_NODE_ITEM(
			pattern_label, pnode2, NULL
		    );
		    g_free(pattern_label);

		    UPDATE_PATTERNS_LOADED_STATUS(1, pattern->size);

		    /* Get the next pattern */
		    pattern = pattern->next;
		}
	    }
	}

	/* Iterate through ZIP metadata list */
	if((clam_pnode->zip_mlist != NULL) && (win->stop_count == 0))
	{
	    GtkCTreeNode *node;

	    INSERT_NODE_FOLDER(
		"ZIP metadata", pnode, NULL
	    );
	    if(node != NULL)
	    {
		gchar *pattern_label;
		struct cli_meta_node *pattern;
		GtkCTreeNode *pnode2 = node;

		for(pattern = clam_pnode->zip_mlist;
		    pattern != NULL;
		    pattern = pattern->next
		)
		{
		    if(win->stop_count > 0)
			break;

		    if(!STRISEMPTY(pattern->virname) &&
		       !STRISEMPTY(pattern->filename)
		    )
		    {
			pattern_label = g_strdup_printf(
			    "%s - %s",
			    pattern->virname,
			    pattern->filename
			);
		    }
		    else if(!STRISEMPTY(pattern->virname))
		    {
			pattern_label = STRDUP(pattern->virname);
		    }
		    else if(!STRISEMPTY(pattern->filename))
		    {
			pattern_label = STRDUP(pattern->filename);
		    }
		    else
		    {
			continue;
		    }

		    /* Add this pattern to the list */
		    INSERT_NODE_ITEM(
			pattern_label, pnode2, NULL
		    );
		    g_free(pattern_label);

		    UPDATE_PATTERNS_LOADED_STATUS(1, pattern->size);
		}
	    }
	}

	/* Iterate through RAR metadata list */
	if((clam_pnode->rar_mlist != NULL) && (win->stop_count == 0))
	{
	    GtkCTreeNode *node;

	    INSERT_NODE_FOLDER(
		"RAR metadata", pnode, NULL
	    );
	    if(node != NULL)
	    {
		gchar *pattern_label;
		struct cli_meta_node *pattern;
		GtkCTreeNode *pnode2 = node;

		for(pattern = clam_pnode->rar_mlist;
		    pattern != NULL;
		    pattern = pattern->next
		)
		{
		    if(win->stop_count > 0)
			break;

		    if(!STRISEMPTY(pattern->virname) &&
		       !STRISEMPTY(pattern->filename)
		    )
		    {
			pattern_label = g_strdup_printf(
			    "%s - %s",
			    pattern->virname,
			    pattern->filename
			);
		    }
		    else if(!STRISEMPTY(pattern->virname))
		    {
			pattern_label = STRDUP(pattern->virname);
		    }
		    else if(!STRISEMPTY(pattern->filename))
		    {
			pattern_label = STRDUP(pattern->filename);
		    }
		    else
		    {
			continue;
		    }

		    /* Add this pattern to the list */
		    INSERT_NODE_ITEM(
			pattern_label, pnode2, NULL
		    );
		    g_free(pattern_label);

		    UPDATE_PATTERNS_LOADED_STATUS(1, pattern->size);
		}
	    }
	}

#undef UPDATE_PATTERNS_LOADED_STATUS

#endif
}

/*
 *	Updates the Win's Virus Database Tree with the information
 *	from the specified virus database location.
 */
void WinDBUpdate(
	win_struct *win, const gchar *db_location,
	const gboolean verbose
)
{
#if defined(HAVE_CLAMAV)
	struct stat stat_buf;
	struct cl_node *database = NULL;
	int database_entries = 0;
	gint status, patterns_loaded = 0;
	gulong patterns_total_size_bytes = 0l;
#endif
	gulong		start_time = 0l,
			end_time = 0l;
	GtkWidget *w, *toplevel;
	GtkCList *clist;
	GtkCTree *ctree;
	const cfg_item_struct *cfg_list;
	core_struct *core;

	if(win == NULL)
	    return;

	toplevel = win->toplevel;
	ctree = GTK_CTREE(win->db_ctree);
	clist = GTK_CLIST(ctree);
	core = CORE(win->core);
	cfg_list = core->cfg_list;


	/* Delete the existing virus database and clear the database
	 * tree
	 */
	WinDBClear(win);


	/* Begin updating */

	/* Reset the stop count and update so the stop button can
	 * be used
	 */
	win->stop_count = 0;
	win->paused = FALSE;
	WinSetPassiveBusy(win, TRUE);
	WinUpdateTitle(win);
	WinUpdate(win);

	if(verbose)
	{
	    WinStatusProgress(win, -1.0f, TRUE);
	    WinStatusTime(win, 0l, TRUE);
	}

	/* If the virus database location was not specified then use
	 * the one specified in the configuration
	 */
	if(db_location == NULL)
	    db_location = CFGItemListGetValueS(
		cfg_list, CFG_PARM_DB_LOCATION
	    );
#if defined(HAVE_CLAMAV)
	if(db_location == NULL)
	    db_location = cl_retdbdir();
#endif
	if(db_location == NULL)
	{
	    if(verbose)
	    {
		WinStatusMessage(
		    win,
"No virus database location specified",
		    FALSE
		);
		WinStatusProgress(win, 0.0f, FALSE);
		WinStatusTime(win, 0l, FALSE);
	    }
	    WinSetPassiveBusy(win, FALSE);
	    return;
	}


	/* Update the Virus Database Stats Label (before loading) */
	w = win->db_stats_label;
	if(w != NULL)
	{
	    const gulong	cur_t = (gulong)time(NULL),
				last_updated_t = CFGItemListGetValueL(
				    cfg_list, CFG_PARM_DB_LAST_UPDATE
				);
	    gchar	*db_location_s = COPY_SHORTEN_STRING(
			    db_location, 35
			),
			*last_updated_s = STRDUP((last_updated_t > 0) ?
		TIME_DURATION_STRING(core, cur_t - last_updated_t, last_updated_t) : "Unknown"
			);
	    gchar *msg = g_strdup_printf(
		"%s\n%s\n",
		db_location_s,
		last_updated_s
	    );
	    gtk_label_set_text(GTK_LABEL(w), msg);
	    g_free(msg);
	    g_free(db_location_s);
	    g_free(last_updated_s);
	}

#if defined(HAVE_CLAMAV)
	/* Check if the virus database location exists */
	if(stat((const char *)db_location, &stat_buf))
	{
	    const gint error_code = (gint)errno;
	    if(verbose)
	    {
		gchar *s, *error_msg = STRDUP(g_strerror(error_code));
		if(error_msg != NULL)
		{
		    *error_msg = (gchar)toupper((int)*error_msg);
		    s = g_strdup_printf(
			"%s: %s",
			db_location,
			error_msg
		    );
		    WinStatusMessage(win, s, FALSE);
		    g_free(s);
		    g_free(error_msg);
		}
		WinStatusProgress(win, 0.0f, FALSE);
		WinStatusTime(win, 0l, FALSE);
	    }
	    WinSetPassiveBusy(win, FALSE);
	    return;
	}


	/* Update the Virus Database Stats Label (before loading) */
	w = win->db_stats_label;
	if(w != NULL)
	{
	    const gulong	cur_t = (gulong)time(NULL),
				last_updated_t = CFGItemListGetValueL(
				    cfg_list, CFG_PARM_DB_LAST_UPDATE
				);
	    gchar	*db_location_s = COPY_SHORTEN_STRING(
			    db_location, 35
			),
			*last_updated_s = STRDUP((last_updated_t > 0) ?
		TIME_DURATION_STRING(core, cur_t - last_updated_t, last_updated_t) : "Unknown"
			);
	    gchar *msg = g_strdup_printf(
		"%s\n%s\nLoading...",
		db_location_s,
		last_updated_s
	    );
	    gtk_label_set_text(GTK_LABEL(w), msg);
	    g_free(msg);
	    g_free(db_location_s);
	    g_free(last_updated_s);
	}

	/* Load the virus database and stats */
	if(verbose)
	{
	    gchar *msg = g_strdup_printf(
		"Loading virus database \"%s\"...",
		db_location
	    );
	    WinStatusMessage(win, msg, TRUE);
	    g_free(msg);
	}
	if(win->stop_count == 0)
	{
	    struct cl_stat *clam_stat = (struct cl_stat *)g_malloc0(
		sizeof(struct cl_stat)
	    );
	    win->db_stat = clam_stat;
	    if(S_ISDIR(stat_buf.st_mode))
	    {
		if(clam_stat != NULL)
		{
		    if(cl_statinidir(db_location, clam_stat))
		    {
			g_free(clam_stat);
			clam_stat = NULL;
			win->db_stat = NULL;
		    }
		}
		start_time = (gulong)time(NULL);
		status = cl_loaddbdir(
		    db_location,
		    &database, &database_entries
		);
	    }
	    else
	    {
		if(clam_stat != NULL)
		{
		    gchar *parent = g_dirname(db_location);
		    if(cl_statinidir(parent, clam_stat))
		    {
			g_free(clam_stat);
			clam_stat = NULL;
			win->db_stat = NULL;
		    }
		    g_free(parent);
		}
		start_time = (gulong)time(NULL);
		status = cl_loaddb(
		    db_location,
		    &database, &database_entries
		);
	    }
	}
	else
	{
	    status = 4;
	}

	/* Virus database loaded successfuly with at least one
	 * pattern?
	 */
	if((status == 0) && (database != NULL) && (database_entries > 0))
	{
	    gint i;
	    const gint ncolumns = clist->columns;
	    gchar	*label,
			**strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	    GtkCTreeNode *node;

	    if(verbose)
		WinStatusProgress(win, -1.0f, TRUE);

	    /* Allocate the row cell values */
	    for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	    /* Build the virus database */
	    status = cl_build(database);
	    end_time = (gulong)time(NULL);
	    if(status != 0)
	    {
		if(verbose)
		{
		    gchar *s = g_strconcat(
			"Virus database build error: ",
			cl_strerror(status),
			NULL
		    );
		    WinStatusMessage(win, s, FALSE);
		    g_free(s);
		    WinStatusProgress(win, 0.0f, FALSE);
		    WinStatusTime(win, 0l, FALSE);
		}
		cl_free(database);
		g_free(strv);
		WinSetPassiveBusy(win, FALSE);
		return;
	    }

	    /* Set the successfully loaded and built virus database */
	    win->db = database;


	    /* Begin adding the nodes and patterns to the Virus
	     * Database Tree
	     */
	    if(verbose)
		WinStatusProgress(win, -1.0f, TRUE);
	    gtk_clist_freeze(clist);

	    /* Create the root node */
	    if(S_ISDIR(stat_buf.st_mode))
		label = g_strdup_printf(
		    "ClamAV Virus Database: %s (directory)",
		    g_basename(db_location)
		);
	    else
		label = g_strdup_printf(
		    "ClamAV Virus Database: %s",
		    g_basename(db_location)
		);
	    INSERT_NODE_FOLDER(label, NULL, NULL);
	    g_free(label);
	    if(node == NULL)
	    {
		gtk_clist_thaw(clist);
		g_free(strv);
		if(verbose)
		    WinStatusTime(win, 0l, FALSE);
		WinSetPassiveBusy(win, FALSE);
		return;
	    }
	    win->db_root_node = node;

	    /* Add all child nodes and items to this node */
	    WinDBUpdateIterate(
		core, win, ctree, node, strv,
		database,
		start_time, verbose,
		&patterns_loaded, &patterns_total_size_bytes
	    );

	    gtk_clist_thaw(clist);
	    if(verbose)
		WinStatusProgress(win, -1.0f, TRUE);

	    g_free(strv);
	}
	/* Unable to load the virus database? */
	else if(win->stop_count == 0)
	{
	    gchar *s = g_strconcat(
		"Virus database load error: ",
		cl_strerror(status),
		NULL
	    );
	    WinStatusMessage(win, s, FALSE);
	    g_free(s);

	    /* Delete the virus database */
	    if(database != NULL)
		cl_free(database);
	}
#endif

	/* Update the Virus Database Stats Label (after loading) */
	w = win->db_stats_label;
	if(w != NULL)
	{
	    const gulong	cur_t = (gulong)time(NULL),
				last_updated_t = CFGItemListGetValueL(
				    cfg_list, CFG_PARM_DB_LAST_UPDATE
				),
				duration_t = ((end_time > 0l) && (start_time > 0l)) ?
				    (end_time - start_time) : 0l;
	    gchar *msg, *patterns_total_size_s, *duration_s;
	    gchar	*db_location_s = COPY_SHORTEN_STRING(
			    db_location, 35
			),
			*last_updated_s = STRDUP((last_updated_t > 0) ?
		TIME_DURATION_STRING(core, cur_t - last_updated_t, last_updated_t) : "Unknown"
			),
			*patterns_loaded_s = STRDUP(EDVSizeStrDelim(
		(gulong)patterns_loaded
			));

	    if(patterns_total_size_bytes >= (1024l * 1024l * 1024l))
		patterns_total_size_s = g_strdup_printf(
		    "%.1f gb",
		    (gfloat)patterns_total_size_bytes / 1024.0f / 1024.0f / 1024.0f
		);
	    else if(patterns_total_size_bytes >= (1024l * 1024l))
		patterns_total_size_s = g_strdup_printf(
		    "%.1f mb",
		    (gfloat)patterns_total_size_bytes / 1024.0f / 1024.0f
		);
	    else if(patterns_total_size_bytes >= 1024l)
		patterns_total_size_s = g_strdup_printf(
		    "%.1f kb",
		    (gfloat)patterns_total_size_bytes / 1024.0f
		);
	    else
		patterns_total_size_s = g_strdup_printf(
		    "%ld %s",
		    patterns_total_size_bytes,
		    (patterns_total_size_bytes == 1l) ? "byte" : "bytes"
		);

	    /* Duration, Longer than 1 hour? */
	    if(duration_t >= (60l * 60l))
		duration_s = g_strdup_printf(
		    "%i:%.2i:%.2i",
		    (gint)(duration_t / (60l * 60l)),
		    (gint)((duration_t / 60) % 60l),
		    (gint)(duration_t % 60l)
		);
	    else	/* Less than 1 hour */
		duration_s = g_strdup_printf(
		    "%i:%.2i",
		    (gint)(duration_t / 60l),
		    (gint)(duration_t % 60l)
		);

	    msg = g_strdup_printf(
		"%s\n%s\n%s (%s) %s%s",
		db_location_s,
		last_updated_s,
		patterns_loaded_s,
		patterns_total_size_s,
		duration_s,
		(win->stop_count > 0) ? " (Interrupted)" : " load time"
	    );
	    gtk_label_set_text(GTK_LABEL(w), msg);
	    g_free(msg);
	    g_free(db_location_s);
	    g_free(last_updated_s);
	    g_free(patterns_loaded_s);
	    g_free(patterns_total_size_s);
	    g_free(duration_s);
	}

	/* Print the results */
	if(verbose)
	{
	    gchar *patterns_loaded_s = STRDUP(EDVSizeStrDelim(
		(gulong)patterns_loaded
	    ));
	    gchar *msg = g_strdup_printf(
		"Virus database loaded with %s patterns",
		patterns_loaded_s
	    );

	    /* Raise Win */
	    WinMap(win);

	    WinStatusMessage(win, msg, FALSE);
	    g_free(msg);
	    g_free(patterns_loaded_s);

	    WinStatusProgress(win, 0.0f, FALSE);
	    WinStatusTime(win, 0l, FALSE);
	}

	WinUpdateTitle(win);
	WinUpdate(win);

	WinSetPassiveBusy(win, FALSE);
}

#undef INSERT_NODE_ITEM
#undef INSERT_NODE_FOLDER

/*
 *	Queues a call to WinDBUpdate().
 */
void WinDBQueueUpdate(win_struct *win)
{
	if((win != NULL) ? (win->db_ctree_load_idleid == 0) : FALSE)
	    win->db_ctree_load_idleid = gtk_idle_add_priority(
		G_PRIORITY_LOW,
		WinDBTreeLoadStartIdleCB, win
	    );
}

/*
 *	Deletes all nodes in the Win's Virus Database Tree,
 *	clears the Stats label, and deletes the virus database listing.
 */
void WinDBClear(win_struct *win)
{
	GtkCList *clist;
	core_struct *core;
	GtkCTree *ctree;

	if(win == NULL)
	    return;

	ctree = GTK_CTREE(win->db_ctree);
	clist = GTK_CLIST(ctree);
	core = CORE(win->core);

	/* Begin clearing the database tree */
	gtk_clist_freeze(clist);

	/* Remove the root node from the database tree */
	if(win->db_root_node != NULL)
	{
	    gtk_ctree_remove_node(ctree, win->db_root_node);
	    win->db_root_node = NULL;
	}

	gtk_clist_thaw(clist);


	/* Clear the stats label */
	gtk_label_set_text(GTK_LABEL(win->db_stats_label), "");


#if defined(HAVE_CLAMAV)
	/* Delete the virus database stat and list */
	if(win->db_stat != NULL)
	{
	    cl_statfree((struct cl_stat *)win->db_stat);
	    g_free(win->db_stat);
	    win->db_stat = NULL;
	}
	if(win->db != NULL)
	{
	    cl_free((struct cl_node *)win->db);
	    win->db = NULL;
	}
#endif	/* HAVE_CLAMAV */
}
