#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#ifdef HAVE_EDV2
# include <endeavour2.h>
#endif
#include "../include/string.h"
#include "../include/disk.h"
#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "obj.h"
#include "win.h"    
#include "wincb.h"
#include "windnd.h"
#include "winlist.h"
#include "windb.h"
#include "winopcb.h"
#include "core.h"
#include "cfglist.h"


static const gchar *SEEK_PAST_ARG(const gchar *s);
static const gchar *SEEK_PAST_SPACES(const gchar *s);
static const gchar *GET_ARG(
	const gchar *s, const gchar end_c, 
	gchar **arg_rtn
);
static const gchar *GET_NAME_FROM_PATH(const gchar *path);
static gchar *SHORTEN_STRING(const gchar *s, const gint m);

void WinScanListItemDestroyCB(gpointer data);
void WinResultsListItemDestroyCB(gpointer data);
void WinDBListItemDestroyCB(gpointer data);

gint WinEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

gint WinOPIDEnterEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint WinOPIDLeaveEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void WinToolBarItemCB(
	toolbar_item_struct *item, gint id, gpointer data
);
void WinToolBarItemEnterCB(
	toolbar_item_struct *item, gint id, gpointer data
);
void WinToolBarItemLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data
);

void WinNotebookSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
);

void WinLocationEntryChangedCB(GtkWidget *widget, gpointer data);
gint WinEntryCompletePathKeyCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
);

gint WinListEventCB(GtkWidget *widget, GdkEvent *event, gpointer data);
void WinListResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
);
void WinListClickColumnCB(
	GtkCList *clist, gint column, gpointer data             
);
void WinListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void WinListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

void WinTreeSelectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
void WinTreeUnselectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
);
void WinTreeExpandCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);
void WinTreeCollapseCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
);

static GtkCTreeNode *WinDBFindIterate(
	GtkCTree *ctree,
	const gchar *needle, GtkCTreeNode *node,
	GtkCTreeNode **sel_node
);
void WinDBFindEntryCB(GtkWidget *widget, gpointer data);
gint WinDBTreeLoadStartIdleCB(gpointer data);

gint WinScanMonitorTOCB(gpointer data);
gint WinDBUpdateNetMonitorTOCB(gpointer data);


gint WinPatternDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
void WinPatternDlgCloseCB(GtkWidget *widget, gpointer data);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Seeks past non-blank characters until a blank character or
 *	null character is encountered.
 */
static const gchar *SEEK_PAST_ARG(const gchar *s)
{
	if(s == NULL)
	    return(NULL);

	while(!ISBLANK(*s) && (*s != '\0'))
	    s++;

	return(s);
}

/*
 *	Seeks past blank characters.
 */
static const gchar *SEEK_PAST_SPACES(const gchar *s)
{
	if(s == NULL)
	    return(NULL);

	while(ISBLANK(*s))
	    s++;

	return(s);
}

/*
 *	Gets a dynamically allocated string *arg_rtn containing the
 *	string from s to the next encountered character specified by
 *	end_c.
 *
 *	If end_c is not encountered then a copy of s is returned as
 *	*arg_rtn.
 *
 *	Returns the pointer in s past end_c if end_c was encountered or
 *	NULL if end_c was not encountered.
 */
static const gchar *GET_ARG(
	const gchar *s, const gchar end_c,
	gchar **arg_rtn
)
{
	gint len;
	const gchar *s_end;

	if(arg_rtn != NULL)
	    *arg_rtn = NULL;

	if(s == NULL)
	    return(NULL);

	s_end = strchr(s, end_c);
	len = (s_end != NULL) ? (s_end - s) : STRLEN(s);

	if(arg_rtn != NULL)
	{
	    gchar *arg = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	    if(len > 0)
		memcpy(arg, s, len * sizeof(gchar));
	    arg[len] = '\0';
	    *arg_rtn = arg;
	}

	return((s_end != NULL) ? (s_end + 1) : NULL);
}

/*
 *      Returns the name portion of the path (if any).
 */
static const gchar *GET_NAME_FROM_PATH(const gchar *path)
{
	const gchar *s;

	if(STRISEMPTY(path))
	    return(path);
			 
	s = strrchr(path, G_DIR_SEPARATOR);
	return((s != NULL) ? (s + 1) : path);
}

/*
 *	Returns a copy of the string that is no longer than m
 *	characters with appropriate shortened notation where
 *	appropriate.
 */
static gchar *SHORTEN_STRING(const gchar *s, const gint m)
{
	gint len;

	if(s == NULL)
	    return(NULL);

	len = STRLEN(s);
	if((len > m) && (m > 3))
	{
	    /* Need to shorten string */
	    const gint i = len - m + 3;
	    return(g_strdup_printf(
		"...%s", (const gchar *)(s + i)
	    ));
	}
	else
	{
	    return(STRDUP(s));
	}
}


/*
 *	Scan List item destroy signal callback.
 */
void WinScanListItemDestroyCB(gpointer data)
{
	ObjDelete(OBJ(data));
}

/*
 *	Results List item destroy signal callback.
 */
void WinResultsListItemDestroyCB(gpointer data)
{
	ObjDelete(OBJ(data));
}

/*
 *	Database List item destroy signal callback.
 */
void WinDBListItemDestroyCB(gpointer data)
{
	win_db_item_struct *d = WIN_DB_ITEM(data);
	if(d == NULL)
	    return;

	g_free(d->name);
	g_free(d);
}

/*
 *	Win toplevel GtkWindow event signal callback.
 */
gint WinEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean press;
	guint keyval, state;
	GdkEventKey *key;
	win_struct *win = WIN(data);
	if((widget == NULL) || (event == NULL) || (win == NULL))
	    return(status);


	switch((gint)event->type)
	{
	  case GDK_DELETE:
	    /* If there a scan, database load, or virus database update
	     * in progress then call the stop callback instead of
	     * the close callback
	     */
	    if(WinScanProcessIsRunning(win) ||
	       (win->db_ctree_load_idleid > 0) ||
	       WinDBUpdateNetProcessIsRunning(win)
	    )
		WinStopCB(win->stop_mi, win);
	    else
		WinCloseCB(win->close_mi, win);
	    status = TRUE;
	    break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    press = (event->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    keyval = key->keyval;
	    state = key->state;
#define DO_STOP_KEY_SIGNAL_EMIT {		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?                                       \
   "key_press_event" : "key_release_event"      \
 );						\
}
	    switch(keyval)
	    {
	      case GDK_Escape:
		if(press)
		    WinStopCB(win->stop_mi, win);
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	    }
#undef DO_STOP_KEY_SIGNAL_EMIT
	    break;
	}

	return(status);
}

/*
 *	Win OPID "enter_notify_event" signal callback.
 */
gint WinOPIDEnterEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	win_opid_struct *opid = WIN_OPID(data);
	if(opid == NULL)
	    return(FALSE);

	WinStatusMessage(opid->win, opid->tooltip, FALSE);
	return(TRUE);
}

/*
 *	Win OPID "leave_notify_event" signal callback.
 */
gint WinOPIDLeaveEventCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	win_opid_struct *opid = WIN_OPID(data);
	if(opid == NULL)
	    return(FALSE);

	WinStatusMessage(opid->win, NULL, FALSE);
	return(TRUE);
}


/*
 *	Win Tool Bar Item callback.
 */
void WinToolBarItemCB(
	toolbar_item_struct *item, gint id, gpointer data
)
{
	win_opid_struct *opid = WIN_OPID(data);
	if(opid == NULL)
	    return;

	if(opid->func_cb != NULL)
	    opid->func_cb(
		NULL,		/* GtkWidget */
		opid->win	/* Win */
	    );
}

/*
 *	Win Tool Bar Item enter callback.
 */
void WinToolBarItemEnterCB(
	toolbar_item_struct *item, gint id, gpointer data
)
{
	WinOPIDEnterEventCB(NULL, NULL, data);
}

/*
 *	Win Tool Bar Item leave callback.
 */
void WinToolBarItemLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data
)
{
	WinOPIDLeaveEventCB(NULL, NULL, data);
}


/*
 *	Win notebook "switch_page" signal callback.
 */
void WinNotebookSwitchPageCB(
	GtkNotebook *notebook, GtkNotebookPage *page, guint page_num,
	gpointer data
)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if(win->page_num != (win_page_num)page_num)
	{
	    win->page_num = (win_page_num)page_num;
	    switch(page_num)
	    {
	      case WIN_PAGE_NUM_SCAN:
		break;

	      case WIN_PAGE_NUM_RESULTS:
		break;

	      case WIN_PAGE_NUM_DB:
		WinDBQueueUpdate(win);
		break;
	    }
	    WinUpdateTitle(win);
	    WinUpdate(win);
	}
}


/*
 *	Location GtkEntry "changed" signal callback.
 */
void WinLocationEntryChangedCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if((widget == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	win->freeze_count++;

	WinUpdate(win);

	win->freeze_count--;
}

/*
 *	Entry complete path "key_press_event" or "key_release_event"
 *	signal callback.
 */
gint WinEntryCompletePathKeyCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gboolean press;
	GtkEntry *entry = GTK_ENTRY(widget);
	win_struct *win = WIN(data);
	if((entry == NULL) || (key == NULL) || (win == NULL))
	    return(status);

	press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;

#define SIGNAL_EMIT_STOP	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	switch(key->keyval)
	{
	  case GDK_Tab:
	    if(press)
	    {
		gchar *s = STRDUP(gtk_entry_get_text(entry));
		if(s != NULL)
		{
		    gint status;

		    s = CompletePath(s, &status);
		    gtk_entry_set_text(entry, (s != NULL) ? s : "");
		    gtk_entry_set_position(entry, -1);
		    g_free(s);

		    switch(status)
		    {
		      case COMPLETE_PATH_NONE:
		      case COMPLETE_PATH_AMBIGUOUS:
			gdk_beep();
			break;
		    }
		}
	    }
	    SIGNAL_EMIT_STOP
	    status = TRUE;
	    break;
	}

#undef SIGNAL_EMIT_STOP

	return(status);
}


/*
 *	Win list event signal callback.
 */
gint WinListEventCB(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gint status = FALSE;
	gboolean press;
	guint keyval, state;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	win_struct *win = WIN(data);
	if((widget == NULL) || (event == NULL) || (win == NULL))
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    press = (event->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    keyval = key->keyval;
	    state = key->state;
#define DO_STOP_KEY_SIGNAL_EMIT {		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?                                       \
   "key_press_event" : "key_release_event"      \
 );						\
}
	    switch(keyval)
	    {
	      case GDK_Insert:
		if(press)
		{
		    if(state & GDK_CONTROL_MASK)
			WinCopyCB(win->copy_mi, win);
		    else if(state & GDK_SHIFT_MASK)
			WinPasteCB(win->paste_mi, win);
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	      case GDK_Delete:
		if(press)
		    WinRemoveCB(win->delete_mi, win);
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;
	    }
#undef DO_STOP_KEY_SIGNAL_EMIT
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:
		break;

	      case GDK_BUTTON2:
		break;

	      case GDK_BUTTON3:
		if(widget == win->scan_clist)
		{
		    GtkMenu *menu = GTK_MENU(win->scan_list_menu);
		    gtk_menu_popup(
			menu, NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
 		}
		else if(widget == win->results_clist)
		{
		    GtkMenu *menu = GTK_MENU(win->results_list_menu);
		    gtk_menu_popup(
			menu, NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		}
		else if(widget == win->db_ctree)
		{
		    GtkMenu *menu = GTK_MENU(win->db_list_menu);
		    gtk_menu_popup(
			menu, NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		}
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_2BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:
		if(widget == win->scan_clist)
		{
		    WinStartCB(NULL, win);
		}
		else if(widget == win->results_clist)
		{

		}
		else if(widget == win->db_ctree)
		{
		    WinDBPatternDetailsCB(NULL, win);
		}
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;

	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;

	    break;
	}

	return(status);
}


/*
 *	Win GtkCList "resize_column" signal callback.
 */
void WinListResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
)
{
	win_struct *win = WIN(data);
	if((clist == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(clist)))
	    return;


}

/*
 *	Win GtkCList "click_column" signal callback.
 */
void WinListClickColumnCB(
	GtkCList *clist, gint column, gpointer data             
)
{
	win_struct *win = WIN(data);
	if((clist == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(clist)))
	    return;

	gtk_clist_freeze(clist);

	gtk_clist_set_sort_column(clist, column);
	gtk_clist_set_sort_type(
	    clist,
	    (clist->sort_type == GTK_SORT_ASCENDING) ?
		GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
	);
	gtk_clist_sort(clist);

	gtk_clist_thaw(clist);
}


/*
 *	Win GtkCList "select_row" signal callback.
 */
void WinListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	core_struct *core;
	win_struct *win = WIN(data);
	if((clist == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(clist)))
	    return;

	core = CORE(win->core);
	if(core == NULL)
	    return;

	/* If the row is not visible then scroll to make it visible */
	if(gtk_clist_row_is_visible(clist, row) !=
	    GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,	/* Row, column */
		0.5f, 0.0f	/* Row, column */
	    );

	/* Scan List */
	if(clist == (GtkCList *)win->scan_clist)
	{
	    const obj_struct *obj = OBJ(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj != NULL)
	    {
		gchar *msg;
		const gint nobjs = g_list_length(clist->selection);
 
		/* Update scan settings */
		WinScanSettingsUpdate(win, obj);

		/* Update status bar message */
		if(nobjs > 1)
		    msg = g_strdup_printf(
			"%i scan items selected",
			nobjs
		    );
		else if(!STRISEMPTY(obj->name))
		    msg = g_strdup_printf(
			"Scan item \"%s\" selected",
			obj->name
		    );
		else
		    msg = STRDUP("Scan item (Untitled) selected");
		WinStatusMessage(win, msg, FALSE);
		g_free(msg);
	    }

	    WinListDNDSetIcon(clist, row, column);
	    WinUpdate(win);
	}
	/* Results List */
	else if(clist == (GtkCList *)win->results_clist)
	{
	    const obj_struct *obj = OBJ(
		gtk_clist_get_row_data(clist, row)
	    );
	    if((obj != NULL) ? !STRISEMPTY(obj->path) : FALSE)
	    {
		gchar *msg;
		const gint nobjs = g_list_length(clist->selection);
		const gchar	*path = obj->path,
				*name = GET_NAME_FROM_PATH(path);
		struct stat stat_buf;

		if(nobjs > 1)
		    msg = g_strdup_printf(
			"%i objects selected",
			nobjs
		    );
		else if(lstat(path, &stat_buf))
		    msg = g_strdup_printf(
			"Object \"%s\" selected",
			name
		    );
#ifdef S_ISREG
		else if(S_ISREG(stat_buf.st_mode))
#else
		else if(FALSE)
#endif
		{
#ifdef HAVE_EDV2
		    gchar *size_str = STRDUP(EDVSizeStrDelim(
			(gulong)stat_buf.st_size
		    ));
#else
		    gchar *size_str = g_strdup_printf(
			"%ld",
			(gulong)stat_buf.st_size
		    );
#endif
		    msg = g_strdup_printf(
			"File \"%s\" selected (%s byte%s)", 
			name,
			size_str,
			(stat_buf.st_size == 1l) ? "" : "s"
		    );
		}
#ifdef S_ISDIR
		else if(S_ISDIR(stat_buf.st_mode))
#else
		else if(FALSE)
#endif
		    msg = g_strdup_printf(
			"Directory \"%s\" selected",
			name
		    );
#ifdef S_ISLNK
		else if(S_ISLNK(stat_buf.st_mode))
#else
		else if(FALSE)
#endif
		    msg = g_strdup_printf(
			"Link \"%s\" selected",
			name
		    );
		else
		    msg = g_strdup_printf(
			"Object \"%s\" selected",
			name
		    );

		WinStatusMessage(win, msg, FALSE);
		g_free(msg);
	    }
	    WinListDNDSetIcon(clist, row, column);
	    WinUpdate(win);
	}
}

/*
 *	Win GtkCList "unselect_row" signal callback.
 */
void WinListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	win_struct *win = WIN(data);
	if((clist == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(clist)))
	    return;

	/* Scan List */
	if(clist == (GtkCList *)win->scan_clist)
	{
	    WinStatusMessage(win, NULL, FALSE);
	    WinUpdate(win);
	}
	/* Results List */
	else if(clist == (GtkCList *)win->results_clist)
	{
	    WinStatusMessage(win, NULL, FALSE);
	    WinUpdate(win);
	}
}


/*
 *      Win GtkCTree "tree_select_row" signal callback.
 */
void WinTreeSelectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data
)
{
	win_struct *win = WIN(data);
	if((ctree == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(ctree)))
	    return;

	/* If the node is not visible then scroll to make it visible */
	if(gtk_ctree_node_is_visible(ctree, node)
	   != GTK_VISIBILITY_FULL
	)
	    gtk_ctree_node_moveto(
		ctree,
		node, -1,
		0.5f, 0.0f	/* Row align, column align */
	    );

	/* Virus Database Tree */
	if(ctree == (GtkCTree *)win->db_ctree)
	{
	    gchar *msg;
	    win_db_item_struct *d = WIN_DB_ITEM(
		gtk_ctree_node_get_row_data(ctree, node)
	    );
	    if((d != NULL) ? (d->name != NULL) : FALSE)
	    {
		switch(d->type)
		{
		  case WIN_DB_ITEM_TYPE_PATTERN:
		    msg = g_strdup_printf(
			"Pattern \"%s\" selected",
			d->name
		    );
		    break;
		  case WIN_DB_ITEM_TYPE_FOLDER:
		    msg = g_strdup_printf(
			"Folder \"%s\" selected",
			d->name
		    );
		    break;
		  default:
		    msg = STRDUP("");
		    break;
		}
	    }
	    else
		msg = STRDUP("");
	    WinStatusMessage(win, msg, FALSE);
	    g_free(msg);

	    WinUpdate(win);
	}
}

/*
 *      Win GtkCTree "tree_unselect_row" signal callback.
 */
void WinTreeUnselectRowCB(
	GtkCTree *ctree, GtkCTreeNode *node, gint column,
	gpointer data                                    
)
{
	win_struct *win = WIN(data);
	if((ctree == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(ctree)))
	    return;

	/* Virus Database Tree */
	if(ctree == (GtkCTree *)win->db_ctree)
	{
	    WinUpdate(win);
	}
}
 
/*
 *      Win GtkCTree "tree_expand" signal callback.
 */
void WinTreeExpandCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{
	win_struct *win = WIN(data);
	if((ctree == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(ctree)))
	    return;

}

/*
 *      Win GtkCTree "tree_collapse" signal callback.
 */
void WinTreeCollapseCB(
	GtkCTree *ctree, GtkCTreeNode *node, gpointer data
)
{ 
	win_struct *win = WIN(data);
	if((ctree == NULL) || (win == NULL))
	    return;

	if(win->freeze_count > 0)
	    return;

	if(!GTK_WIDGET_SENSITIVE(GTK_WIDGET(ctree)))
	    return;


}


/*
 *	Searches for a node who's text matches the specified text
 *	in the specified node and the node's child nodes.
 */
static GtkCTreeNode *WinDBFindIterate(
	GtkCTree *ctree,
	const gchar *needle, GtkCTreeNode *node,
	GtkCTreeNode **sel_node
)
{
	gchar *text;
	guint8 spacing;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkCTreeRow *row;

	while(node != NULL)
	{
	    row = GTK_CTREE_ROW(node);
	    if(row == NULL)
		break;

	    /* Past selected node? */
	    if(*sel_node == NULL)
	    {
		/* Get this node's text and see if it matches the
		 * specified text
		 */
		gtk_ctree_node_get_pixtext(
		    ctree, node, ctree->tree_column,
		    &text, &spacing, &pixmap, &mask
		);
		if(text != NULL)
		{
		    if(strcasestr(text, needle) != NULL)
			return(node);
		}
	    }
	    /* Reached the selected node? */
	    else if(*sel_node == node)
	    {
		/* Set the selected node to NULL to indicate that it has
		 * been reached
		 */
		*sel_node = NULL;
	    }

	    /* Does this node have child nodes? */
	    if(row->children != NULL)
	    {
		/* Check all of this node's child nodes */
		GtkCTreeNode *matched_node = WinDBFindIterate(
		    ctree, needle, row->children, sel_node
		);
		if(matched_node != NULL)
		    return(matched_node);
	    }

	    /* Get next sibling */
	    node = row->sibling;
	}

	return(NULL);
}

/*
 *	Win Virus Database Find GtkEntry "activate" signal callback.
 */
void WinDBFindEntryCB(GtkWidget *widget, gpointer data)
{
	gchar *needle;
	GList *glist;
	GtkWidget *toplevel;
	GtkCTreeNode *node, *sel_node;
	GtkCList *clist;
	GtkCTree *ctree;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;

	/* Get copy of search string */
	needle = STRDUP(
	    gtk_entry_get_text(GTK_ENTRY(win->db_find_entry))
	);
	if(needle == NULL)
	    return;

	ctree = GTK_CTREE(win->db_ctree);
	clist = GTK_CLIST(ctree);

	/* Get last selected node or root node */
	glist = clist->selection_end;
	sel_node = (GtkCTreeNode *)((glist != NULL) ? glist->data : NULL);
	if(sel_node != NULL)
	{
	    node = WinDBFindIterate(
		ctree, needle, win->db_root_node, &sel_node
	    );
	    /* If no node was matched then start from the beginning */
	    if(node == NULL)
	    {
		sel_node = NULL;
		node = WinDBFindIterate(
		    ctree, needle, win->db_root_node, &sel_node
		);
	    }
	}
	else
	{
	    /* Search for node starting from the root node */
	    node = WinDBFindIterate(
		ctree, needle, win->db_root_node, &sel_node
	    );
	}

	/* Got match? */
	if(node != NULL)
	{
	    gtk_ctree_select(ctree, node);
	}
	else
	{
	    gchar *msg = g_strdup_printf(
		"The text \"%s\" was not found",
		needle
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Find Results",
		msg,
		NULL,
		CDIALOG_ICON_SEARCH,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	}

	g_free(needle);
}

/*
 *	Win Virus Database Tree load start idle callback.
 */
gint WinDBTreeLoadStartIdleCB(gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return(FALSE);

	/* Update the Virus Database Tree as needed */
	if(win->db == NULL)
	{
	    WinDBUpdate(win, NULL, TRUE);
	}

	win->db_ctree_load_idleid = 0;
	WinUpdate(win);

	return(FALSE);
}

/*
 *	Win scan monitoring timeout callback.
 */
gint WinScanMonitorTOCB(gpointer data)
{
	gint p;
	gchar buf[1024];
	FILE *fp;
	GtkWidget *toplevel;
	GtkCList *clist;
	core_struct *core;
	win_scan_context_struct *scan_ctx;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return(FALSE);

	toplevel = win->toplevel;
	clist = (GtkCList *)win->results_clist;
	core = CORE(win->core);
	scan_ctx = win->scan_context;
	p = scan_ctx->pid;

	/* User clicked on the stop button? */
	if(win->stop_count > 0)
	{
	    /* Stop scanning
	     *
	     * Mark that this timeout is no longer being called
	     */
	    scan_ctx->toid = 0;

	    /* Tell the scanning process to stop scanning, delete
	     * all the tempory output files, and clean up
	     */
	    WinScanProcessStop(win);
	    WinResultsStatusUpdate(win, "Scan Interrupted");

	    WinStatusMessage(
		win,
"Scan interrupted",
		FALSE
	    );

	    return(FALSE);
	}

	/* Read stderr */
	fp = win->stderr_fp;
	if(fp != NULL)
	{
	    /* Anything from stderr will be added as a string to
	     * the virus/problem column of the results list
	     */
	    gint new_row;
	    gchar *s;

	    while(fgets((char *)buf, sizeof(buf), fp) != NULL)
	    {
		buf[sizeof(buf) - 1] = '\0';

		s = (gchar *)strpbrk((char *)buf, "\n\r");
		if(s != NULL)
		    *s = '\0';

		/* Seek past any spaces */
		s = buf;
		while(ISBLANK(*s))
		    s++;

		new_row = WinResultsListAppend(win, NULL, s, FALSE);
		if(new_row > -1)
		{
		    gtk_clist_moveto(clist, new_row, 0, 1.0f, 0.0f);
		}
	    }
	}

	/* Read stdout */
	fp = win->stdout_fp;
	if(fp != NULL)
	{
	    /* Each line from stdout must have its prefix checked and
	     * handled accordingly
	     */
	    gchar *s;

	    while(fgets((char *)buf, sizeof(buf), fp) != NULL)
	    {
		buf[sizeof(buf) - 1] = '\0';

		s = (gchar *)strpbrk((char *)buf, "\n\r");
		if(s != NULL)
		    *s = '\0';

		/* Seek past any spaces */
		s = buf;
		while(ISBLANK(*s))
		    s++;

		/* Scanning file? */
		if(g_strcasepfx(s, "Scanning:") ||
		   g_strcasepfx(s, "Scanning file:")
		)
		{
		    /* Format is:
		     *
		     * "<pfx>: "<path>" <nscanned> <total>"
		     */
		    const gchar *s2 = s;
		    gchar	*pfx = NULL,
				*path = NULL,
				*shortened_path = NULL,
				*msg;
		    gulong nscanned = 0l, total = 0l;

		    /* Get the prefix */
		    if(s2 != NULL)
		    {
			s2 = GET_ARG(s2, ':', &pfx);
		    }

		    /* Get the path */
		    if(s2 != NULL)
		    {
			/* Get the path */
			s2 = strchr(s2, '"');
			if(s2 != NULL)
			    s2++;
			s2 = GET_ARG(s2, '"', &path);
		    }

		    /* Get the number of files scanned */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_SPACES(s2);
			scan_ctx->nscanned = nscanned = (gulong)ATOL(s2);
		    }

		    /* Get the total files */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_ARG(s2);
			s2 = SEEK_PAST_SPACES(s2);
			scan_ctx->total = total = (gulong)ATOL(s2);
		    }

		    WinUpdateTitle(win);

		    /* Update the results status */
		    shortened_path = SHORTEN_STRING(path, 27);
		    msg = g_strdup_printf(
			"%s (%ld/%ld): %s",
			pfx, nscanned + 1l, total, shortened_path
		    );
		    g_free(shortened_path);
		    WinResultsStatusUpdate(win, msg);
		    g_free(msg);

		    /* Update the status bar message */
		    shortened_path = SHORTEN_STRING(path, 45);
		    msg = g_strdup_printf(
			"%s (%ld/%ld): %s",
			pfx, nscanned + 1l, total, shortened_path
		    );
		    g_free(shortened_path);
		    WinStatusMessage(win, msg, FALSE);
		    g_free(msg);

		    /* Update the status bar progress */
		    WinStatusProgress(
			win,
			(scan_ctx->total > 0l) ?
			    ((gfloat)scan_ctx->nscanned / (gfloat)scan_ctx->total) : 0.0f,
			TRUE
		    );

		    g_free(path);
		    g_free(pfx);
		}
		/* Scanned file? */
		else if(g_strcasepfx(s, "Scanned file:"))
		{
		    /* Format is:
		     *
		     * "<pfx>: <nscanned> <blocks_scanned> <total_blocks_scanned>"
		     */
		    const gchar *s2 = (gchar *)strchr((char *)s, ':');

		    /* Get the number of objects scanned */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_ARG(s2);
			s2 = SEEK_PAST_SPACES(s2);
			scan_ctx->nscanned = (gulong)ATOL(s2);
		    }

		    /* Get the data scanned for this object in blocks */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_ARG(s2);
			s2 = SEEK_PAST_SPACES(s2);
			/* Ignore this */
		    }

		    /* Get the total data scanned in blocks */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_ARG(s2);
			s2 = SEEK_PAST_SPACES(s2);
			scan_ctx->blocks_scanned = (gulong)ATOL(s2);
		    }

		    WinUpdateTitle(win);

		    /* Update the status bar progress */
		    WinStatusProgress(
			win,
			(scan_ctx->total > 0l) ?
			    ((gfloat)scan_ctx->nscanned / (gfloat)scan_ctx->total) : 0.0f,
			TRUE
		    );
		}
		/* Scanning directory? */
		else if(g_strcasepfx(s, "Scanning directory:"))
		{
#if 0
/* We ignore the scanning directory because it usually appears very
 * brief and makes subsequent messages difficult to read in the cases
 * of scanning multiple subdirectories
 */
		    /* Format is:
		     *
		     * "<pfx>: "<path>""
		     */
		    const gchar *s2 = s;
		    gchar	*pfx = NULL,
				*path = NULL,
				*shortened_path = NULL,
				*msg;

		    /* Get the prefix */
		    if(s2 != NULL)
		    {
			s2 = GET_ARG(s2, ':', &pfx);
		    }

		    /* Get the path */
		    if(s2 != NULL)
		    {
			/* Get the path */
			s2 = (const gchar *)strchr((const char *)s2, '"');
			if(s2 != NULL)
			    s2++;
			s2 = GET_ARG(s2, '"', &path);
		    }

		    WinUpdateTitle(win);

		    /* Update the results status */
		    shortened_path = SHORTEN_STRING(path, 27);
		    msg = g_strdup_printf(
			"%s: %s",
			pfx, shortened_path
		    );
		    g_free(shortened_path);
		    WinResultsStatusUpdate(win, msg);
		    g_free(msg);

		    /* Update the status bar message */
		    shortened_path = SHORTEN_STRING(path, 45);
		    msg = g_strdup_printf(
			"%s: %s",
			pfx, shortened_path
		    );
		    g_free(shortened_path);
		    WinStatusMessage(win, msg, TRUE);
		    g_free(msg);

		    g_free(path);
		    g_free(pfx);
#endif
		}
		/* Clean */
		else if(g_strcasepfx(s, "Clean:"))
		{
		    /* Format is:
		     *
		     * "<pfx>: "<path>""
		     */
		    gint new_row;
		    const gchar *s2 = (gchar *)strchr((char *)s, ':');
		    gchar	*path = NULL;

		    /* Get the path */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_ARG(s2);
			s2 = SEEK_PAST_SPACES(s2);
			s2 = (const gchar *)strchr((const char *)s2, '"');
			if(s2 != NULL)
			    s2++;
			s2 = GET_ARG(s2, '"', &path);
		    }

		    /* Add this to the results list */
		    new_row = WinResultsListAppend(win, path, "Clean", FALSE);
		    if(new_row > -1)
		    {
			gtk_clist_moveto(clist, new_row, 0, 1.0f, 0.0f);
		    }

		    WinUpdateTitle(win);

		    g_free(path);
		}
		/* Infected */
		else if(g_strcasepfx(s, "Infected:"))
		{
		    /* Format is:
		     *
		     * "<pfx>: "<path>" <virus>"
		     */
		    gint new_row;
		    const gchar *s2 = (gchar *)strchr((char *)s, ':');
		    gchar	*path = NULL,
				*virus = NULL;

		    /* Get the path */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_ARG(s2);
			s2 = SEEK_PAST_SPACES(s2);
			s2 = (const gchar *)strchr((const char *)s2, '"');
			if(s2 != NULL)
			    s2++;
			s2 = GET_ARG(s2, '"', &path);
		    }

		    /* Get the virus name */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_SPACES(s2);
			virus = STRDUP(s2);
		    }

		    /* Count this as an infected file */
		    scan_ctx->ninfected++;

		    /* Add this to the results list */
		    new_row = WinResultsListAppend(win, path, virus, TRUE);
		    if(new_row > -1)
		    {
			gtk_clist_moveto(clist, new_row, 0, 1.0f, 0.0f);
		    }

		    WinUpdateTitle(win);

		    g_free(path);
		    g_free(virus);
		}
		/* Problem */
		else if(g_strcasepfx(s, "Problem:"))
		{
		    /* Format is:
		     *
		     * "<pfx>: "<path>" <problem>"
		     */
		    gint new_row;
		    const gchar *s2 = (gchar *)strchr((char *)s, ':');
		    gchar	*path = NULL,
				*problem = NULL;

		    /* Get the path */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_ARG(s2);
			s2 = SEEK_PAST_SPACES(s2);
			s2 = strchr(s2, '"');
			if(s2 != NULL)
			    s2++;
			s2 = GET_ARG(s2, '"', &path);
		    }

		    /* Get the problem */
		    if(s2 != NULL)
		    {
			s2 = SEEK_PAST_SPACES(s2);
			problem = STRDUP(s2);
		    }

		    scan_ctx->nproblems++;

		    /* Add this to the results list */
		    new_row = WinResultsListAppend(win, path, problem, FALSE);
		    if(new_row > -1)
		    {
			gtk_clist_moveto(clist, new_row, 0, 1.0f, 0.0f);
		    }

		    WinUpdateTitle(win);

		    g_free(path);
		    g_free(problem);
		}
		/* All else print to the status bar's message */
		else
		{
		    /* Print this line to the status bar's message */
		    WinStatusMessage(win, s, TRUE);
		}
	    }
	}

	/* Process has exited? */
	if(!WinScanProcessIsRunning(win))
	{
	    const gulong duration_t = scan_ctx->duration;
	    gchar *duration_s;

	    /* Stop scanning */
	    scan_ctx->toid = 0;
	    WinScanProcessStop(win);

	    /* Format the duration string
	     *
	     * Longer than 1 hour?
	     */
	    if(duration_t >= (60l * 60l))
		duration_s = g_strdup_printf(
		    "%i:%.2i:%.2i",
		    (gint)(duration_t / (60l * 60l)),
		    (gint)((duration_t / 60) % 60l),
		    (gint)(duration_t % 60l)
		);
	    else	/* Less than 1 hour */
		duration_s = g_strdup_printf(
		    "%i:%.2i",
		    (gint)(duration_t / 60l),
		    (gint)(duration_t % 60l)
		);

	    /* Update the results status */
	    if((scan_ctx->ninfected > 0l) || (scan_ctx->nproblems > 0l))
	    {
		gchar *msg = g_strdup_printf(
		    "%ld %s Found   %ld %s Found   Duration: %s",
		    scan_ctx->ninfected,
		    (scan_ctx->ninfected == 1l) ? "Virus" : "Viruses",
		    scan_ctx->nproblems,
		    (scan_ctx->nproblems == 1l) ? "Problem" : "Problems",
		    duration_s
		);
		WinResultsStatusUpdate(win, msg);
		g_free(msg);
	    }
	    else
	    {
		gchar *msg = g_strdup_printf(
		    "Clean,    Duration: %s",
		    duration_s
		);
		WinResultsStatusUpdate(win, msg);
		g_free(msg);
	    }

	    g_free(duration_s);

	    if(scan_ctx->ninfected > 0l)
	    {
#ifdef HAVE_EDV2
		/* Play the "completed" sound */
		EDVPlaySoundCompleted(core->edv_ctx);
#endif
	    }
	    else
	    {
#ifdef HAVE_EDV2
		/* Play "warning" sound */
		EDVPlaySoundWarning(core->edv_ctx);
#endif
	    }

	    /* Update the status message */
	    WinStatusMessage(
		win,
"Scan done",
		FALSE
	    );

	    /* Raise the Win */
	    WinMap(win);

	    /* Display the report */
	    WinResultsReportCB(win->toplevel, win);

	    return(FALSE);
	}

	/* Update the progress only if the total number of files has
	 * not been received yet
	 */
	if((scan_ctx->total == 0l) && !win->paused)
	    WinStatusProgress(win, -1.0f, FALSE);

	/* Update the duration */
	if(scan_ctx->start_time > 0l)
	{
	    const gulong t = (gulong)time(NULL);
	    scan_ctx->duration = (t > scan_ctx->start_time) ?
		(t - scan_ctx->start_time) : 0l;
	    WinStatusTime(win, scan_ctx->duration, FALSE);
	}

	return(TRUE);
}

/*
 *	Win virus database update monitoring timeout callback.
 *
 *	Monitors the virus database update process and prints its
 *	stdout and stderr messages to the status bar.
 */
gint WinDBUpdateNetMonitorTOCB(gpointer data)
{
	gint p;
	gchar buf[1024];
	FILE *fp;
	GtkWidget *toplevel;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return(FALSE);

	p = win->db_update_net_pid;
	toplevel = win->toplevel;
	core = CORE(win->core);

	/* Need to stop? */
	if(win->stop_count > 0)
	{
	    /* Stop update */
	    win->db_update_net_toid = 0;
	    WinDBUpdateNetProcessStop(win);

	    WinStatusMessage(win, "Virus database update interrupted", FALSE); 
	    return(FALSE);
	}

	/* Read stdout */
	fp = win->stdout_fp;
	if(fp != NULL)
	{
	    gchar *s;

	    while(fgets((char *)buf, sizeof(buf), fp) != NULL)
	    {
		buf[sizeof(buf) - 1] = '\0';

		/* Strip newline characters */
		s = (gchar *)strpbrk((char *)buf, "\n\r");
		if(s != NULL)
		    *s = '\0';

		/* Seek past any spaces */
		s = buf;
		while(ISBLANK(*s))
		    s++;

		WinStatusMessage(win, s, TRUE);
	    }
	}

	/* Read stderr */
	fp = win->stderr_fp;
	if(fp != NULL)
	{
	    gint nerrors = 0;
	    gchar *s;

	    while(fgets((char *)buf, sizeof(buf), fp) != NULL)
	    {
		buf[sizeof(buf) - 1] = '\0';

		nerrors++;

		/* Strip newline characters */
		s = (gchar *)strpbrk((char *)buf, "\n\r");
		if(s != NULL)
		    *s = '\0';

		/* Seek past any spaces */
		s = buf;
		while(ISBLANK(*s))
		    s++;

		/* Print the error message to the user */
#ifdef HAVE_EDV2
		EDVPlaySoundWarning(core->edv_ctx);
#endif
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Update Virus Database Error",
		    s,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);

		/* If more than 10 errors have occured then query
		 * user to stop
		 */
		if((nerrors % 10) == 0)
		{
		    gint response;
		    gchar *msg = g_strdup_printf(
"%i errors have occured during the virus database\n\
update.\n\
\n\
Do you want to stop the virus database update?",
			nerrors
		    );
#ifdef HAVE_EDV2
                    EDVPlaySoundWarning(core->edv_ctx);
#endif
		    CDialogSetTransientFor(toplevel);
		    response = CDialogGetResponse(
                        "Update Virus Database Error",
			msg,
			NULL,
                        CDIALOG_ICON_WARNING,
                        CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
                        CDIALOG_BTNFLAG_NO
                    );
		    g_free(msg);
                    CDialogSetTransientFor(NULL);

		    if(response == CDIALOG_RESPONSE_YES)
		    {
			/* Stop update */
			win->db_update_net_toid = 0;
			WinDBUpdateNetProcessStop(win);

			WinStatusMessage(
			    win,
"Virus database update interrupted",
			    FALSE
			);
			return(FALSE);
		    }
		}
	    }
	}


	/* Has the virus database update process exited? */ 
	if(!WinDBUpdateNetProcessIsRunning(win))
	{
	    /* Mark the virus database update time */
	    CFGItemListSetValueL(
		core->cfg_list, CFG_PARM_DB_LAST_UPDATE,
		(gulong)time(NULL), FALSE
	    );

#ifdef HAVE_EDV2
	    /* Play the "completed" sound */
	    EDVPlaySoundCompleted(core->edv_ctx);
#endif

	    /* Stop the virus database update */
	    win->db_update_net_toid = 0;
	    WinDBUpdateNetProcessStop(win);

	    WinStatusMessage(win, "Virus database update done", TRUE);

	    /* Was there a change in the virus database or that the
	     * virus database was not previously displayed or is
	     * empty?
	     */
	    if(WinDBChanged(win) != 0)
	    {
		/* Clear the Virus Database Tree due to virus database update */
		WinDBClear(win);

		/* Currently viewing the Virus Database Page? */
		if(win->page_num == WIN_PAGE_NUM_DB)
		{
		    /* Need to update the Virus Database Tree since it
		     * is currently being viewed
		     *
		     * Note that the Win will be raised after the update
		     */
		    WinDBQueueUpdate(win);
		}
	    }

	    return(FALSE);
	}

	/* Update the progress */
	if(!win->paused)
	    WinStatusProgress(win, -1.0f, FALSE);

	/* Update the time */
	if(win->db_update_net_start_time > 0l)
	{
	    const gulong t = (gulong)time(NULL);
	    WinStatusTime(
		win,
		(t > win->db_update_net_start_time) ?
		    (t - win->db_update_net_start_time) : 0l,
		FALSE
	    );
	}

	return(TRUE);
}


/*
 *	Pattern dialog "delete_event" signal callback.
 */
gint WinPatternDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	win_pattern_dlg_struct *d = WIN_PATTERN_DLG(data);
	if(d == NULL)
	    return(FALSE);

	gtk_widget_hide(d->toplevel);

	return(TRUE);
}

/*
 *	Pattern dialog close signal callback.
 */
void WinPatternDlgCloseCB(GtkWidget *widget, gpointer data)
{
	win_pattern_dlg_struct *d = WIN_PATTERN_DLG(data);
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
}
