/*
Copyright (C) 2000-2001 Jos Roberto B. de A. Monteiro <jrm@autsens.com>
                        and Pedro Zorzenon Neto <pzn@autsens.com>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define DEVICES_CONFIG_FILE_NAME "/etc/avrprog/devices.conf"
#define STRSIZE 256

#include "devices.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

static FILE * config_file_handle=NULL;
static char config_file_line[STRSIZE];

static int get_parameter (int num, char *parameter);
static int read_config_line (char *chip_id, int *signatyre_bytes,
           int *program_memory_size, int *eeprom_memory_size);
static int open_config_file ();
static int close_config_file ();
static int tell_valid_devices ();


int search_chip_by_signature(unsigned char *sigbytes, char *chip_id,
      	     int *program_memory_size, int *eeprom_memory_size) {

    int config_signature_bytes[4]={256,256,256,256};
    int chip_found=0;
    int i;

    open_config_file();

    //search chip
    while (chip_found==0) {
        i=read_config_line(chip_id, config_signature_bytes,
			   program_memory_size, eeprom_memory_size);
	if (i==0) {
	    fprintf(stderr,"Error: config file misspelling. Please verify '%s'.\n",DEVICES_CONFIG_FILE_NAME);
	    abort();
	}
	if (i==-1) {
	    fprintf(stderr,"Error: no device matchs signature bytes in config file.\n");
            tell_valid_devices();
	    abort();
	}
	chip_found=1;
	for(i=0; i<4; i++) {
	    if ((config_signature_bytes[i] > -1) &&
		(config_signature_bytes[i] != (unsigned int) sigbytes[i]))
		chip_found=0;
	}
    }

    // if program_memory_size is -2, means error and show list of available devices
    if (*program_memory_size==-2) {
	fprintf(stderr,"Error: %s\n", chip_id);
	tell_valid_devices();
	abort();
    }

    // if program_memory_size is -1, means error
    if (*program_memory_size==-1) {
	fprintf(stderr,"Error: %s\n", chip_id);
	abort();
    }

    close_config_file();
    return 1;
}

int search_chip_by_name(unsigned char *sigbytes, char *chip_id,
       		int *program_memory_size, int *eeprom_memory_size) {
    int chip_found=0;
    int config_signature_bytes[4]={256,256,256,256};
    char selected_chip[STRSIZE];
    int i;

    strcpy(selected_chip,chip_id);

    open_config_file();

    //search chip
    while (chip_found==0) {
	i=read_config_line(chip_id, config_signature_bytes,
			   program_memory_size, eeprom_memory_size);
	if (i==0) {
	    fprintf(stderr,"Error: config file misspelling. Please verify '%s'.\n",DEVICES_CONFIG_FILE_NAME);
	    abort();
	}
	if (i==-1) {
	    fprintf(stderr,"Error: no device matchs '%s' in config file.\n",selected_chip);
	    tell_valid_devices();
	    abort();
	}
	if (strcmp(selected_chip,chip_id)==0)
	    chip_found=1;
    }

    // if program_memory_size is -2, means error and show list of available devices
    if (*program_memory_size==-2) {
	fprintf(stderr,"Error: %s\n", chip_id);
	tell_valid_devices();
	abort();
    }

    // if program_memory_size is -1, means error
    if (*program_memory_size==-1) {
	fprintf(stderr,"Error: %s\n", chip_id);
	abort();
    }

    fprintf(stderr,"Warning: chip autodetection not done due to '-d' option.\n");

    close_config_file();
    return 1;
}

static int open_config_file () {
    if (config_file_handle!=NULL)
	close_config_file();
    config_file_handle=fopen(DEVICES_CONFIG_FILE_NAME,"r");
    if (config_file_handle==NULL) {
	printf("Error when opening config file '%s' for reading.\n",DEVICES_CONFIG_FILE_NAME);
	abort(); }
    return 1;
}

static int close_config_file () {
    return fclose(config_file_handle);
}

// returns 0 if error
// returns 1 if line read successfully
// returns -1 if eof found
static int read_config_line (char *chip_id, int *signature_bytes,
           int *program_memory_size, int *eeprom_memory_size) {

    char conv[STRSIZE];
    int i,j;

    config_file_line[0]='-'; //something != '+'
    while (config_file_line[0] != '+') {
	if (feof(config_file_handle)) {
	    return -1; //indicates EOF
	}
	fgets(config_file_line, STRSIZE, config_file_handle); //reads a line
    }

    //read chip id (ex. AT90S1200)
    get_parameter(0,chip_id);
    // reads signature bytes
    for (i=0; i<4; i++) {
	get_parameter(i+1,conv);
	for (j=0; j<2; j++) conv[j]=toupper(conv[j]);
	if (strcmp(conv,"XX")==0) {
	    signature_bytes[i]=-1;
	} else {
	    signature_bytes[i]=(unsigned char) strtoul(conv,NULL,16);
	}
    }
    //read size of program memory
    get_parameter(5,conv);
    *program_memory_size=strtoul(conv,NULL,10);
    //read size of eeprom memory
    get_parameter(6,conv);
    *eeprom_memory_size=strtoul(conv,NULL,10);
    return 1;
}

static int get_parameter (int num, char *parameter) {
    int cont, begin, end;

    //get a config file line parameter
    begin=1; cont=0;
    while (cont<num) {
        if (config_file_line[begin]==';') cont++;
	if (config_file_line[begin]==0) cont=1000;
	begin++;
    }

    end=begin;
    while (cont<(num+1)) {
	end++;
	if (config_file_line[end]==';') cont++;
	if (config_file_line[end]==0) cont=1000;
    }

    if (cont==1000) return 0;

    strncpy(parameter,config_file_line+begin,end-begin);
    parameter[end-begin]=0;

    return 1;
}

static int tell_valid_devices() {
    char chip_id[STRSIZE];
    int sigbytes[5];
    int program_memory_size=0;
    int eeprom_memory_size=0;

    chip_id[200]=0;
    fprintf(stderr,"recognized devices are:");
    open_config_file();
    while (read_config_line(chip_id, sigbytes, &program_memory_size,
			    &eeprom_memory_size) == 1) {
	if (program_memory_size>=0)
	    fprintf(stderr," %s",chip_id);
    }
    close_config_file();
    fprintf(stderr,"\nif your AVR device is not supported, edit '%s'.\n",DEVICES_CONFIG_FILE_NAME);
    return 1;

    close_config_file();

    return 1;
}
