#include "Cache.h"
#include "avm_cpuinfo.h"
#include "avm_output.h"
#include "utils.h"

#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/stat.h>

#include <errno.h>
#include <string.h>
#include <stdio.h>

// limit the maximum size of each prefetched stream in bytes
#define STREAM_SIZE_LIMIT 3000000

#ifndef O_LARGEFILE
#define O_LARGEFILE 0
#endif

// uncomment if you do not want to use threads for precaching
//#define NOTHREADS               // do not commit with this define

AVM_BEGIN_NAMESPACE;

#define __MODULE__ "StreamCache"
//static float ttt = 0;
//static float ttt1 = 0;

Cache::Cache(uint_t size)
    :m_uiSize(size), thread(0), m_pPacket(0), m_iFd(-1),
    cache_access(0), cache_right(0), cache_miss(0), m_bQuit(false),
#ifndef NOTHREADS
    m_bThreaded(true) // standard operation mode
#else
    m_bThreaded(false) // for debugging purpose only - never commit
#endif
{
}

Cache::~Cache()
{
    mutex.Lock();
    m_bQuit = true;
    cond.Broadcast();
    mutex.Unlock();
    delete thread;
    clear();
    //printf("CACHE A %d  MIS %d  RIGH %d\n", cache_access, cache_miss, cache_right);
    if (cache_access != 0)
	AVM_WRITE(__MODULE__, "Destroy... (Total accesses %d, hits %.2f%%, misses %.2f%%, errors %.2f%%)\n",
		  cache_access, 100. * double(cache_right - cache_miss) / cache_access,
		  100. * double(cache_miss) / cache_access,
		  100. * double(cache_access - cache_right) / cache_access);
    //printf("TTTT %f  %f\n", ttt, ttt1);
}

int Cache::addStream(uint_t id, const avm::vector<uint32_t>& table)
{
    AVM_WRITE(__MODULE__, 3, "Adding stream, %d chunks\n", table.size());
    mutex.Lock();
    m_streams.push_back(StreamEntry(&table, 0, m_uiSize));
    cond.Broadcast();
    mutex.Unlock();

    return 0;
}

// starts caching thread once we know file descriptor
int Cache::create(int fd)
{
    m_iFd = fd;
    AVM_WRITE(__MODULE__, "Creating cache for file descriptor: %d\n", m_iFd);
    if (m_streams.size() > 0)
    {
	if (m_bThreaded)
	{
	    mutex.Lock();
	    thread = new PthreadTask(0, &startThreadfunc, this);
	    cond.Wait(mutex);
	    mutex.Unlock();
	}
    }
    else
	AVM_WRITE(__MODULE__, "WARNING no stream for caching!\n");

    return 0;
}

// return true if this stream should be waken and read new data
inline bool Cache::isCachable(StreamEntry& stream, uint_t id)
{
    // trick to allow precaching even very large image sizes
    //printf("ISCHACHE f:%d  s:%d  sum:%d   l: %d ll: %d\n", stream.packets.full(), stream.packets.size(), stream.sum, stream.last, stream.table->size());
    return ((stream.sum < STREAM_SIZE_LIMIT
	     // assuming id=0 is video stream
	     // uncompressed video could be really huge!
	     //|| ((id == 0) && stream.packets.size() < 3)
	    )
	    && stream.last < stream.table->size()
	    && !stream.packets.full()
	    && (stream.filling
                || (stream.sum < STREAM_SIZE_LIMIT/2
		    && stream.packets.size() < m_uiSize/2)));
}

//
// currently preffered picking alghorithm
//
// seems to be having good seek strategy
uint_t Cache::pickChunk()
{
    uint_t id = m_uiId;

    do
    {
	StreamEntry& stream = m_streams[id];

	// determine next needed chunk in this stream
	stream.last = (stream.packets.empty()) ?
	    stream.position : stream.packets.back()->position + 1;

	if (isCachable(stream, id))
	    return id;
	// try next stream
	id++;
	if (id >= m_streams.size())
	    id = 0; // wrap around
    }
    while (id != m_uiId);

    return WAIT; // nothing for caching found
}

// caching thread
void* Cache::threadfunc()
{
    mutex.Lock();
    m_uiId = 0;
    int r = 0;
    while (!m_bQuit)
    {
	cond.Broadcast();
        m_uiId = pickChunk();
	// allow waiting tasks to use cache during read or Wait()
        if (m_uiId == WAIT)
	{
	    m_uiId = 0;

	    // one could be trying to send signal to this thread
	    //AVM_WRITE(__MODULE__, 4, "full packets - waiting for read: s#1: %d s#2: %d\n",
	    //          m_streams[0].packets.size(), m_streams[1].packets.size());
	    cond.Wait(mutex);
	    //AVM_WRITE(__MODULE__, 4, "full packets - waiting done\n");
            continue;
	}

	StreamEntry& stream = m_streams[m_uiId];
	if (stream.packets.size() > 0)
	{
	    if (r > 20000)
	    {
		r = 0;
		// don't stress hardrive too much
		//printf("sleep\n");
		cond.Wait(mutex, 0.001);
                continue; // recheck if the same chunk is still needed
	    }
	}

	uint_t coffset = (*stream.table)[stream.last];
	//int64_t sss = longcount();
	if (lseek64(m_iFd, coffset & ~1, SEEK_SET) == -1)
	{
	    AVM_WRITE(__MODULE__, "WARNING offset: %d unreachable! %s\n", coffset, strerror(errno));
            stream.error = stream.last;
	    cond.Wait(mutex, 0.1);
	    continue;
	}
	//ttt1 += to_float(longcount(), sss);
	mutex.Unlock();

	char bfr[8];
        ::read(m_iFd, bfr, 8);
	uint_t ckid = avm_get_le32(bfr);
	uint_t clen = avm_get_le32(bfr + 4);

	// get free buffer
	if (clen > StreamPacket::MAX_PACKET_SIZE)
	{
	    AVM_WRITE(__MODULE__, "WARNING too large chunk %d\n", clen);
	    clen = 10000;
	}
	m_pPacket = new StreamPacket(clen);
	m_pPacket->position = stream.last;
	//printf("************  %d\n", stream.last);
	//AVM_WRITE(__MODULE__, 4, "id: %d   %d   buffered: %d sum: %d - %d\n",
	//	  m_uiId, stream.last, (stream.last - stream.position),
	//	  m_streams[0].sum, m_streams[1].sum);

	// cache might be read while this chunk is being filled
	//int rd = ::read(m_iFd, m_pPacket->memory, 8);

        unsigned int rs = 0;
	while (rs < m_pPacket->size)
	{
	    int rd = ::read(m_iFd, m_pPacket->memory + rs, m_pPacket->size - rs);
	    //printf("READ %d  %d  of %d\n", rd, rs, m_pPacket->size);
	    if (rd <= 0)
	    {
		if (stream.error == stream.OK)
		    AVM_WRITE(__MODULE__, "WARNING offset: %d short read (%d < %d)! %s\n", coffset, rs, m_pPacket->size, (rd < 0) ? strerror(errno) : "");
		break;
	    }
	    rs += rd;
	}
        r += rs;
	mutex.Lock();
	//printf("memch: "); for (int i = 8; i < 20; i++)  printf(" 0x%02x", (uint8_t) *(m + i)); printf("\n");

	// check if we still want same buffer
	//printf("rs  %d  %d\n", rs, m_pPacket->size);
	if (rs != m_pPacket->size)
	{
	    stream.error = stream.last;
	    cond.Broadcast();
	    cond.Wait(mutex);
            m_pPacket->Release();
	    m_pPacket = 0;
            continue;
	}
	stream.error = stream.OK;
	//uint_t sum = 0; for (uint_t i = 0 ; i < m_pPacket->size; i++) sum += ((unsigned char*) m_pPacket->memory)[i]; printf("PACKETSUM %d   pos: %d  size: %d\n", sum, m_pPacket->position, m_pPacket->size);

	stream.sum += rs;
	m_pPacket->size = rs;
	m_pPacket->flags = (coffset & 1) ? AVIIF_KEYFRAME : 0;
	stream.filling = !(stream.sum > STREAM_SIZE_LIMIT);
	stream.packets.push(m_pPacket);
	//AVM_WRITE(__MODULE__, 4,
	//	  "---  id: %d   pos: %d  sum: %d  size: %d filling: %d\n",
	//	  m_uiId, m_pPacket->position, stream.sum, m_pPacket->size, stream.filling);
	m_pPacket = 0;
    }

    mutex.Unlock();
    return 0;
}

// called by stream reader - most of the time this read should
// be satisfied from already precached chunks
StreamPacket* Cache::readPacket(uint_t id, framepos_t position)
{
    //AVM_WRITE(__MODULE__, 4, "Cache: read(id %d, pos %d)\n", id, position);
    StreamPacket* p = 0;
    int rsize;

    cache_access++;
    if (id >= m_streams.size())
	return 0;

    StreamEntry& stream = m_streams[id];
    if (position >= stream.table->size())
	return 0;

    if (!m_bThreaded)
    {
	//int64_t sss = longcount();
	Locker locker(mutex);
	lseek64(m_iFd, (*stream.table)[position] & ~1, SEEK_SET);

	char bfr[8];
        ::read(m_iFd, bfr, 8);
	uint_t ckid = avm_get_le32(bfr);
	uint_t clen = avm_get_le32(bfr + 4);
	if (clen > StreamPacket::MAX_PACKET_SIZE)
	{
	    AVM_WRITE(__MODULE__, "WARNING too large chunk %d\n", clen);
	    clen = 100000;
	}
	p = new StreamPacket(clen);
	if (p->size > 0)
	{
	    rsize = ::read(m_iFd, p->memory, p->size);
	    //printf("%x   %d    %d  %d  m:%x\n", ckid, clen, rsize, p->size, *(int*)(p->memory));
	    if (rsize <= 0)
	    {
		p->Release();
		return 0;
	    }
	}
	p->flags = (*stream.table)[position] & 1 ? AVIIF_KEYFRAME : 0;
	p->position = position;
	//ttt1 += to_float(longcount(), sss);
	return p;
    }

    mutex.Lock();
    rsize = 1;
    //while (stream.actual != position || stream.packets.size() == 0)
    while (!m_bQuit)
    {
	//printf("STREAMPOS:%d  sp:%d  id:%d  ss:%d\n", position, stream.position, id, stream.packets.size());
	if (!stream.packets.empty())
	{
	    p = stream.packets.front();
	    stream.packets.pop();
	    stream.sum -= p->size;
	    //printf("rpos %d   %d\n", r->position, position);
	    if (p->position == position)
	    {
		cache_right++;
                break;
	    }
	    //AVM_WRITE(__MODULE__, 4, "position: 0x%x want: 0x%x\n", r->position, position);
	    // remove this chunk
	    p->Release();
	    p = 0;
            continue;
	}
	//printf("ERROR  e:%d   p:%d   pl:%d\n", stream.error, position, stream.last);
	if (stream.error == position)
            break;
	cache_miss += rsize;
        rsize = 0;
	m_uiId = id;
	stream.position = position;

	//AVM_WRITE(__MODULE__, 4, "--- actual: %d size: %d\n", id, stream.packets.size());
	//int64_t w = longcount();
	cond.Broadcast();
	cond.Wait(mutex);
	//ttt += to_float(longcount(), w);
	//AVM_WRITE(__MODULE__, 4, "--- actual: %d done - size: %d\n", id, stream.packets.size());
    }

    if (p)
    {
	//AVM_WRITE(__MODULE__, 4, "id: %d bsize: %d memory: %p pp: %d\n",
	//	  id, stream.packets.size(), r->memory, r->position);
    }
    if (stream.packets.size() < (CACHE_SIZE / 2))
	cond.Broadcast(); // wakeup only when buffers are getting low...
    mutex.Unlock();

    return p;
}

int Cache::clear()
{
    AVM_WRITE(__MODULE__, 4, "*** CLEAR ***\n");

    mutex.Lock();
    for (unsigned i = 0; i < m_streams.size(); i++)
    {
	StreamEntry& stream = m_streams[i];
	while (stream.packets.size())
	{
	    StreamPacket* r = stream.packets.front();
	    stream.packets.pop();
	    r->Release();
	}
	stream.sum = 0;
	stream.position = 0;
    }
    m_uiId = 0;
    cond.Broadcast();
    mutex.Unlock();

    return 0;
}

double Cache::getSize()
{
    /*
       int status=0;
       for(int i=0; i<m_uiSize; i++)
       if(req_buf[i].st==req::BUFFER_READY)status++;
       return (double)status/m_uiSize;
     */
    return 1.;
}

void* Cache::startThreadfunc(void* arg)
{
    Cache* c = (Cache*)arg;
    c->mutex.Lock();
    c->cond.Broadcast();
    c->mutex.Unlock();
    return c->threadfunc();
}


/*************************************************************/

InputStream::~InputStream()
{
    close();
}

int InputStream::open(const char *pszFile)
{
    m_iFd = ::open(pszFile, O_RDONLY | O_LARGEFILE);
    if (m_iFd < 0)
    {
	AVM_WRITE("InputStream", "Could not open file %s: %s\n", pszFile, strerror(errno));
        return -1;
    }

    m_iPos = ~0;
    m_iBuffered = 0;
    return m_iFd;
}

void InputStream::close()
{
    delete cache;
    cache = 0;
    if (m_iFd >= 0)
	::close(m_iFd);
    m_iFd = -1;
}

int InputStream::async()
{
    if (!cache)
	cache = new Cache();
    return (cache) ? cache->create(m_iFd) : -1;
}

int InputStream::addStream(uint_t id, const avm::vector<uint32_t>& table)
{
    if (!cache)
	cache = new Cache();
    return (cache) ? cache->addStream(id, table) : -1;
}

int64_t InputStream::len() const
{
    struct stat st;
    fstat(m_iFd, &st);
    return st.st_size;
}

int64_t InputStream::seek(int64_t offset)
{
    m_iPos = 0;
    m_iBuffered = 0;
    m_bEof = false;
    return lseek64(m_iFd, offset, SEEK_SET);
}

int64_t InputStream::seekCur(int64_t offset)
{
    //cout << "seekcur " << offset << "   " << m_iPos << endl;
    m_bEof = false;
    if (m_iPos >= m_iBuffered)
	return lseek64(m_iFd, offset, SEEK_CUR);

    if (offset >= 0)
    {
	m_iPos += offset;
	if (m_iPos >= m_iBuffered)
	    return lseek64(m_iFd, m_iPos - m_iBuffered, SEEK_CUR);
    }
    else
    {
	if (m_iPos < -offset)
	{
	    offset += m_iBuffered - m_iPos;
	    m_iBuffered = 0;
	    return lseek64(m_iFd, offset, SEEK_CUR);
	}
	m_iPos += offset;
    }
    return pos();
}

int64_t InputStream::pos() const
{
    int64_t o = lseek64(m_iFd, 0, SEEK_CUR);
    //printf("POS: %lld   %d  %d   %lld\n", o, m_iPos, m_iBuffered, len());
    if (m_iPos < m_iBuffered)
	o -= (m_iBuffered - m_iPos);
    if (o > len())
        o = len();
    //if (o > 733810000)
    //    cout << "pos " << o  << "   " << m_iPos << endl;
    return o;
}

int InputStream::read(void* buffer, uint_t size)
{
    int r = 0;
    if (m_iBuffered > 0)
    {
	//cout << "pos " << m_iPos << endl;
	uint_t copy = m_iBuffered - m_iPos;
	if (size < copy)
	    copy = size;
	memcpy(buffer, bfr + m_iPos, copy);
	m_iPos += copy;
	r = copy;
	size -= copy;
        buffer = (char*) buffer + copy;
	//cout << "READfrompos " << copy << "  " << size << endl;
    }
    if (size > 0)
    {
	int s = ::read(m_iFd, buffer, size);
	if (s <= 0)
	{
	    m_bEof = true;
	    return -1;
	}
	r += s;
    }

    //cout << "rd " << r << "   " << m_iPos << endl;
    return r;
}

uint8_t InputStream::readByte()
{
    if (m_iPos >= m_iBuffered)
    {
	int r = ::read(m_iFd, bfr, sizeof(bfr));
	if (r <= 0)
	{
	    m_bEof = true;
	    return 255;
	}
	m_iBuffered = r;
	m_iPos = 0;
    }
    return bfr[m_iPos++];
#if 0
    uint8_t c;
    ::read(m_iFd, &c, 1);

    return c;
#endif
}

uint32_t InputStream::readDword()
{
    uint32_t i = readByte();
    i |= (readByte() << 8);
    i |= (readByte() << 16);
    i |= (readByte() << 24);
    return i;
}

uint16_t InputStream::readWord()
{
    return readByte() | (readByte() << 8);
}

#undef __MODULE__

AVM_END_NAMESPACE;
