#include "AsxReader.h"
#include "avm_output.h"

#include <ctype.h>
#include <string.h>
#include <stdio.h>

AVM_BEGIN_NAMESPACE;

enum asx_position
{
    begin,
    open_bracket,
    close_bracket,
    in_comment,
    in_quoted_comment,
    in_tagname,
    after_tagname,
    in_endtag,
    before_tagclose,
    in_propname,
    after_propname,
    prop_equal,
    after_propequal,
    between_props,
    in_propcontents,
    in_quoted_propcontents,
    in_quoted_contents,
    in_contents,
};

static inline bool is_whitespace(char c)
{
    return ((c==' ') || (c=='\t') || (c=='\r') || (c=='\n'));
}

bool ASX_Reader::addURL(const char* url)
{
    while (*url && (!isprint(*url) || *url == '"'))
	url++;

    int i = 0;
    while (url[i] && isprint(url[i]) && url[i] != '"')
	i++;
    if (i > 0)
    {
	avm::string vurl = avm::string(url, i);

	if (strncasecmp(vurl.c_str(), "mms://", 6) != 0
	    && strncasecmp(vurl.c_str(), "http://", 7) != 0)
	{
	    char* h = new char[m_Server.size() * 2 + m_Filename.size()];
	    int p = sprintf(h, "http://%s", m_Server.c_str());
	    if (vurl[0] != '/')
	    {
		h[p++] = '/';
		strcpy(h + p, m_Filename.c_str());
		char* q = strchr(h + p, '?'); // strip all after '?'
                if (q) *q = 0;
	    }
	    //printf("_____addUrl>%s<>%s<____\n", h, vurl.c_str());
	    vurl.insert(0, h);
	}

	//printf("URL:%s<\n", vurl.c_str());
	m_Urls.push_back(vurl);
        return true;
    }
    return false;
}

bool ASX_Reader::create(const char* data, uint_t size)
{
    avm::string last_tag;
    avm::string last_prop;
    avm::string last_prop_val;
    int last_start = 0;
    if (size == 0) return false;

    //printf("PARSE >%s< %d\n", data, size);
    asx_position state = begin;
    for (unsigned i=0; i < size; i++)
    {
	char c = data[i];
	if (c < 6) break; // stop processing
	//printf("i %d %c   %d\n", i, c, state);
	bool is_alpha=isalpha(c);
	bool is_space=is_whitespace(c);
	switch(state)
	{
	case begin:
	    if (is_space)
		continue;
	    if (is_alpha)
	    {
                // sometime only single one line redirection appears
		if (!strncasecmp(data + i, "mms://", 6)
                    || !strncasecmp(data + i, "http://", 7))
                    return addURL(data + i);
	    }
	    if (c=='<')
		state = open_bracket;
            continue;
	case open_bracket:
	    if (is_space)
		continue;
	    if (c=='/')
	    {
		state=in_endtag;
		last_start=i+1;
		continue;
	    }
	    if (c=='!')
	    {
		state=in_comment;
		continue;
	    }
	    if (is_alpha)
	    {
		state=in_tagname;
		last_start=i;
		continue;
	    }
	    //cout<<i<<endl;
	    return false;
	case in_comment:
	    if(c=='"')
		state=in_quoted_comment;
	    if(c=='>')
		state=close_bracket;
	    continue;
	case in_quoted_comment:
	    if(c=='"')
		state=in_comment;
	    continue;
	case close_bracket:
	    if(c=='<')
	    {
		state=open_bracket;
		continue;
	    }
	    state=in_contents;
	    continue;
	case in_tagname:
	    if(is_space)
	    {
		last_tag = avm::string(&data[last_start], i-last_start);
		//printf("TAG1: %s\n", last_tag.c_str());
		state=after_tagname;
	    }
	    if((c=='/')||(c=='>'))
	    {
		last_tag = avm::string(&data[last_start], i-last_start);
		//printf("TAG2: %s\n", last_tag.c_str());
		if(c=='/')
	    	    state=before_tagclose;
		else
		    state=close_bracket;
	    }
	    continue;
	case after_tagname:
	    if(is_space) continue;
	    if(c=='/')
	    {
		state=before_tagclose;
		continue;
	    }
	    state=in_propname;
	    last_start=i;
	    continue;
	case in_endtag:
	    if(c=='>')
	    {
		state=close_bracket;
		last_tag = avm::string(&data[last_start], i-last_start);
		AVM_WRITE("ASX reader", 1, " TAG: %s\n", last_tag.c_str());
		if (strcasecmp(last_tag.c_str(), "asx")==0)
		    return true;
	    }
	    continue;
	case before_tagclose:
	    if(is_space)continue;
	    if(c=='>')
	    {
		state=close_bracket;
		continue;
	    }
	    return false;
	case in_propname:
	    if(c=='=')
	    {
		last_prop = avm::string(&data[last_start], i-last_start);
		state=prop_equal;
	    }
	    if(is_space)
	    {
		last_prop = avm::string(&data[last_start], i-last_start);
		state=after_propname;
	    }
	    continue;
	case after_propname:
	    if(c=='=')
		state=prop_equal;
	    continue;
	case prop_equal:
	    if(is_space)
		continue;
	    if(c=='"')
		state=in_quoted_propcontents;
	    else
	        state=in_propcontents;
	    last_start=i;
	    continue;
	case in_propcontents:
	    if(c=='"')
	    {
		state=in_quoted_propcontents;
		continue;
	    }
	    if(is_space || (c=='/') || (c=='>'))
	    {
		last_prop_val = avm::string(&data[last_start], i-last_start);
		AVM_WRITE("ASX reader", 1, "VALUE: %s  t:%s p:%s\n", last_prop_val.c_str(), last_tag.c_str(), last_prop.c_str());
		if (strcasecmp(last_tag.c_str(), "ref") == 0
		    || strcasecmp(last_tag.c_str(), "entryref")== 0
		    || strcasecmp(last_tag.c_str(), "a") == 0
		    || strcasecmp(last_tag.c_str(), "location") == 0
                    || strcasecmp(last_tag.c_str(), "embed") == 0) // part of some javascript
		    if (strcasecmp(last_prop.c_str(), "href") == 0
			|| strcasecmp(last_prop.c_str(), "src") == 0)
			addURL(last_prop_val.c_str());
		switch(c)
		{
		case '/':
		    state=before_tagclose;
		    break;
		case '>':
		    state=close_bracket;
		    break;
		default:
		    state=between_props;
		    break;
	        }
	    }
	    continue;
	case in_quoted_propcontents:
	    if(c=='"')
	    {
		state=in_propcontents;
		continue;
	    }
	    continue;
	case between_props:
	    if(is_space)
		continue;
	    if(c=='/')
	    {
		state=before_tagclose;
		continue;
	    }
	    else if(c=='>')
	    {
	        state=close_bracket;
		continue;
	    }
	    if(is_alpha)
	    {
		last_start=i;
		state=in_propname;
		continue;
	    }
	    return false;
	case in_contents:
	    if(c=='"')
		state=in_quoted_contents;
	    if(c=='<')
		state=open_bracket;
	    continue;
	case in_quoted_contents:
	    if(c=='"')
		state=in_contents;
	    continue;
	default: // after_propequal
	    AVM_WRITE("ASX reader", "FIXME ERROR after_propequal not handled\n");
            break;
	}
    }
    return (m_Urls.size() > 0);
}

bool ASX_Reader::getURLs(avm::vector<avm::string>& urls) const
{
    //for (unsigned i = 0; i < m_Urls.size(); i++) printf("URLs: %s\n", m_Urls[i].c_str());
    urls = m_Urls;
    return true;
}

AVM_END_NAMESPACE;
