/*
 * rbtree.c - red-black tree implementation
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#include <config.h>

#include "rbtree.h"
#include "log.h"	/* needed to do log assertions */

/** Adjustment and utility functions to satisfy red-black tree properties */
static void rbt_rotateleft(rbtree_t* tree, rbnode_t* node);
static void rbt_rotateright(rbtree_t* tree, rbnode_t* node);
static void rbt_reverse_colors(rbnode_t* n1, rbnode_t* n2);
static void rbt_rebalancing(rbtree_t* tree, rbnode_t* node,
					  rbnode_t* parent,
					  rbnode_t* sibling);
static rbnode_t* rbt_recoloring(rbtree_t* tree, rbnode_t* node);
static rbnode_t* rbt_restructuring(rbtree_t* tree, rbnode_t* node);
static int rbt_double_red_violation(rbnode_t* node);

/** Create an empty red-black tree */
rbtree_t*
rbt_create(int (*cmpf)(const void*, const void*))
{
	rbtree_t* tree;
	tree = (rbtree_t*) malloc(sizeof(rbtree_t));
	if (!tree)
		return NULL;

	tree->root = NULL;
	tree->size = 0;
	tree->cmp = cmpf;
	return tree;
}

/** Rotate a child at the right to the left */
static void
rbt_rotateleft(rbtree_t* tree, rbnode_t* node)
{
	rbnode_t* right = node->right;

	/* let the left child of node's right become the new node's right */
	node->right = right->left;
	if (right->left != NULL)
		right->left->parent = node;

	/* let parent of node point to right child */
	right->parent = node->parent;
	if (node->parent == NULL) /* node is root */
		tree->root = right;
	else if (node->parent->left == node)
		node->parent->left = right;
	else
		node->parent->right = right;

	/* put node to the left of node's right */
	right->left = node;
	node->parent = right;
}

/** Rotate a child at the left to the right */
static void
rbt_rotateright(rbtree_t* tree, rbnode_t* node)
{
	rbnode_t* left = node->left;

	/* let the right child of node's left become the new node's left */
	node->left = left->right;
	if (left->right != NULL)
		left->right->parent = node;

	/* let parent of node point to left child */
	left->parent = node->parent;
	if (node->parent == NULL) /* node is root */
		tree->root = left;
	else if (node->parent->left == node)
		node->parent->left = left;
	else
		node->parent->right = left;

	/* put node to the right of node's left */
	left->right = node;
	node->parent = left;
}

/** Retrieve the color of a node */
rbcolor_t
rbt_color(rbnode_t* node)
{
	return (node == NULL ? RB_BLACK : node->color);
}

/** Reverse the colors of two nodes */
static void
rbt_reverse_colors(rbnode_t* n1, rbnode_t* n2)
{
	rbcolor_t tmp = n1->color;
	n1->color = n2->color;
	n2->color = tmp;
}

/* Rebalancing remedies property violations that may occur after deleting a
 * black node
 */
static void
rbt_rebalancing(rbtree_t* tree, rbnode_t* child, rbnode_t* parent, rbnode_t* sibling)
{
	/* case 1: child is root */
	if (child == tree->root)
		return;

	/* parent and sibling must both exist */
	log_assert(parent != NULL && sibling != NULL);

	/* case 2: sibling is red */
	if (rbt_color(sibling) == RB_RED)
	{
		/* 1. reverse colors of parent and sibling
		 * 2. rotate sibling at parent
		 */
		rbt_reverse_colors(parent, sibling);
		if (parent->left == sibling)
		{
			rbt_rotateright(tree, parent);
			sibling = parent->left;
		}
		else {
			rbt_rotateleft(tree, parent);
			sibling = parent->right;
		}
	}

	/* case 3a: nodes in black */
	if (rbt_color(parent) == RB_BLACK &&
		 rbt_color(sibling) == RB_BLACK &&
		 rbt_color(sibling->left) == RB_BLACK &&
		 rbt_color(sibling->right) == RB_BLACK) {

		/* 1. make sibling red to even things up
		 * 2. propagate the problem to the parent
		 */
		sibling->color = RB_RED;
		rbt_rebalancing(tree, parent, parent->parent,
			rbt_sibling(tree, parent));
	}
	/* case 3b: parent red */
	else if (rbt_color(parent) == RB_RED &&
		 rbt_color(sibling) == RB_BLACK &&
		 rbt_color(sibling->left) == RB_BLACK &&
		 rbt_color(sibling->right) == RB_BLACK)
	{

		/* 1. reverse colors of parent and sibling */
		rbt_reverse_colors(parent, sibling);
	}
	/* case 3c: sibling's child is red */
	else
	{
		if (sibling == parent->right &&
		    rbt_color(sibling->left) == RB_RED &&
		    rbt_color(sibling) == RB_BLACK)
		{

			/* 1. reverse colors of sibling and sibling's left
			 * 2. rotate sibling's left at sibling
			 */
			sibling->color = RB_RED;
			sibling->left->color = RB_BLACK;
			rbt_rotateright(tree, sibling);

			sibling = sibling->parent;
		}
		else if (sibling == parent->left &&
			rbt_color(sibling->right) == RB_RED &&
			rbt_color(sibling) == RB_BLACK)
		{

			/* 1. reverse colors of sibling and sibling's right
			 * 2. rotate sibling's right at sibling
			 */
			sibling->color = RB_RED;
			sibling->right->color = RB_BLACK;
			rbt_rotateleft(tree, sibling);

			sibling = sibling->parent;
		}

		sibling->color = parent->color;
		parent->color = RB_BLACK;
		if (sibling == parent->right)
		{
			sibling->right->color = RB_BLACK;
			rbt_rotateleft(tree, parent);
		}
		else
		{
			sibling->left->color = RB_BLACK;
			rbt_rotateright(tree, parent);
		}
	}
}

/* Restructuring remedies a child-parent double red when the parent red node
 * has a black sibling
 */
static rbnode_t*
rbt_restructuring(rbtree_t* tree, rbnode_t* node)
{
	/* node, parent and grandparent must exist  */
	log_assert(node->parent->parent != NULL);

	if (node->parent == node->parent->parent->left)
	{
		/* case 1: parent < child < grandparent */
		if (node == node->parent->right)
		{
			rbt_rotateleft(tree, node->parent);
			rbt_rotateright(tree, node->parent);
		}
		/* case 2: child < parent < grandparent */
		else
		{
			rbt_rotateright(tree, node->parent->parent);
			node = node->parent;
		}
	}
	else
	{
		/* case 3: parent > child > grandparent */
		if (node == node->parent->left)
		{
			rbt_rotateright(tree, node->parent);
			rbt_rotateleft(tree, node->parent);
		}
		/* case 4: child > parent > grandparent */
		else
		{
			rbt_rotateleft(tree, node->parent->parent);
			node = node->parent;
		}
	}

	/* recoloring */
	node->color = RB_BLACK;
	node->left->color = RB_RED;
	node->right->color = RB_RED;

	return node;
}

/* Recoloring remedies a child-parent double red when the parent red node has a
 * red sibling
 */
static rbnode_t*
rbt_recoloring(rbtree_t* tree, rbnode_t* node)
{
	rbnode_t* uncle = rbt_sibling(tree, node->parent);

	/* node, parent and grandparent must exist */
	log_assert(node->parent->parent != NULL);

	/* paint the parent and uncle black */
	node->parent->color = RB_BLACK;
	uncle->color = RB_BLACK;

	/* pain the grandparent red */
	node->parent->parent->color = RB_RED;

	/* continue at grandparent */
	return node->parent->parent;
}

/** Check for double-red violation for this particular node */
static int
rbt_double_red_violation(rbnode_t* node)
{
	if (node == NULL || node->parent == NULL)
		return 0;

	return (node->color == RB_RED && node->parent->color == RB_RED);
}

/** Get the first value in the total order */
rbnode_t*
rbt_first(rbtree_t* tree)
{
	rbnode_t* node;

 	for (node = tree->root; node != NULL && node->left != NULL;)
		node = node->left;
	return node;
}

/** Get the last value in the total order */
rbnode_t*
rbt_last(rbtree_t* tree)
{
	rbnode_t* node;

 	for (node = tree->root; node != NULL && node->right != NULL;)
		node = node->right;
	return node;
}

/** Get predecessor of node */
rbnode_t*
rbt_predecessor(rbnode_t* node)
{
	/* If node has left child, its predecessor is the maximum value in
	 * its left subtree.
	 * If if does not have a left child, its predecessor is its first
	 * left ancestor.
	 */
	rbnode_t* parent;

	if (node->left != NULL)
	{
		node = node->left;
		while (node->right != NULL)
			node = node->right;
	}
	else
	{
		parent = node->parent;
		while (parent != NULL && parent->left == node)
		{
			node = parent;
			parent = parent->parent;
		}
		node = parent;
	}

	return node;
}

/** Get successor of node */
rbnode_t*
rbt_successor(rbnode_t* node)
{
	/* If node has right child, its successor is the minimum value in
	 * its right subtree.
	 * If if does not have a right child, its successor is its first
	 * right ancestor.
	 */

	rbnode_t* parent;

	if (node->right != NULL)
	{
		node = node->right;
		while (node->left != NULL)
			node = node->left;
	}
	else {
		parent = node->parent;
		while (parent != NULL && parent->right == node)
		{
			node = parent;
			parent = parent->parent;
		}
		node = parent;
	}

	return node;
}

/** Create a new node structure */
static rbnode_t*
rbt_create_node(const void* key)
{
	rbnode_t* node = (rbnode_t*) malloc(sizeof(rbnode_t));

	if (node == NULL)
		return NULL;

	node->parent = node->left = node->right = NULL;
	node->key = key;
	node->color = RB_RED;
	return node;
}

/** Insert a node into the red-black tree */
int
rbt_insert(rbtree_t* tree, const void* key)
{
	/* to insert a node in a red-black tree:
		1. initialize node (paint red)
		2. search location to insert the node
		3. insert the node
		4. adjust the tree
	 */
	int cmp = 0;
	rbnode_t* insert = tree->root;
	rbnode_t* parent = NULL;
	rbnode_t* uncle = NULL;
	rbnode_t* node;

	/* 2. search location to insert the node */
	while (insert != NULL)
	{
		/* check for duplicates */
		if ((cmp = tree->cmp(key, insert->key)) == 0)
			return 0;
                parent = insert;

		if (cmp < 0)
			insert = parent->left;
		else
			insert = parent->right;
	}

	/* node needs only to be created when not already in tree */
	node = rbt_create_node(key);
	if (node == NULL)
		return -1;
	insert = node;

	/* 3. insert the node */
	if (parent == NULL)
	{
		node->parent = NULL;
		tree->root = node;
	}
	else
	{
		node->parent = parent;

		if (cmp < 0)
			parent->left = node;
		else
			parent->right = node;

		/* 4. adjust the tree */

		/* while double-red violation */
		while (rbt_double_red_violation(node))
		{
			/* grandparent must exist */
			log_assert(node->parent->parent != NULL);

			uncle = rbt_sibling(tree, node->parent);

			if (uncle == NULL || uncle->color == RB_BLACK)
			{
				/* uncle is colored black */
				node = rbt_restructuring(tree, node);
			}
			else
			{
				/* uncle is colored red */
				node = rbt_recoloring(tree, node);
			}
		}
	}

	/* root can always be made black, without violating properties */
	tree->root->color = RB_BLACK;
	tree->size++;

	return 1;
}

/** Remove a node from the red-black tree */
rbnode_t*
rbt_remove(rbtree_t* tree, const void* key)
{
	rbnode_t* node;
	rbnode_t* swap;
	rbnode_t* child;
	rbnode_t* sibling;

	/* search node */
	swap = node = rbt_search(tree, key);

	if (node == NULL)
		return node;

	if (node->left != NULL)
	{
		/* if node has a left child, swap it with successor */
		swap = rbt_predecessor(node);
	}
	else if (node->right != NULL)
	{
		/* if node only has right child, swap it with successor */
		swap = rbt_successor(node);
	}

	/* swap keys */
	if (node != swap)
	{
		node->key = swap->key;
		swap->key = key;
	}

	/* swap has at most one non-leaf child */
	log_assert(swap->right==NULL || swap->left==NULL);

	sibling = rbt_sibling(tree, swap);

	if (swap->left != NULL)
		child = swap->left;
	else
		child = swap->right;

	if (child != NULL)
		child->parent = swap->parent;

	/* actual removing */
	if (rbt_color(swap) == RB_RED)
	{
		/* remove a red-colored node, in that case properties still
		 * hold.
		 */
		if (swap->parent->left == swap)
			swap->parent->left = child;
		else
			swap->parent->right = child;
	}
	else
	{
		/* remove a black-colored node */
		if (swap == tree->root)
			tree->root = child;
		else if (swap->parent->left == swap)
			swap->parent->left = child;
		else
			swap->parent->right = child;

		if (rbt_color(child) == RB_BLACK)
		{
			/* child is colored black, rebalancing is needed */
			rbt_rebalancing(tree, child, swap->parent,
				sibling);
		}
		else
		{
			/* child is colored red, recoloring is needed */
			child->color = RB_BLACK;
		}
	}

	/* deattach node from tree */
	swap->parent = NULL;
	swap->left = NULL;
	swap->right = NULL;

	tree->size--;
	return swap;
}

/** Search a node in the red-black tree */
rbnode_t*
rbt_search(rbtree_t* tree, const void* key)
{
	int cmp;
	rbnode_t* node = NULL;

	if (tree && tree->root)
		node = tree->root;
	while (node != NULL)
	{
		if ((cmp = tree->cmp(node->key, key)) == 0)
			return node;
		else if (cmp < 0)
			node = node->right;
		else
			node = node->left;
	}

	return NULL;
}

/** Get sibling of a node */
rbnode_t*
rbt_sibling(rbtree_t* tree, rbnode_t* node)
{
	rbnode_t* sibling = NULL;

	/* no sibling exists */
	if (node == NULL || node->parent == NULL)
		return NULL;

	/* node is or node->parent->left or node->parent->right */
	if (node->parent->left != NULL)
	{
		if (tree->cmp(node->key, node->parent->left->key) == 0)
			sibling = node->parent->right;
		else
			sibling = node->parent->left;
	}
	else if (node->parent->right != NULL)
	{
		if (tree->cmp(node->key, node->parent->right->key) == 0)
			sibling = node->parent->left;
		else
			sibling = node->parent->right;
	}
	return sibling;
}

/** Remove a node out of memeory */
void
rbt_freenode(rbnode_t* node, void (*func)(const void*))
{
	if (node == NULL)
		return;

	rbt_freenode(node->left, func);
	rbt_freenode(node->right, func);

	(*func)(node->key);
	free(node);
}

/** Remove tree out of memeory */
void
rbt_free(rbtree_t* tree, void (*func)(const void*))
{
	rbt_freenode(tree->root, func);
	free(tree);
}
