/*!********************************************************************

   Audacity: A Digital Audio Editor

   @file RealtimeEffectPanel.cpp

   @author Vitaly Sverchinsky

**********************************************************************/

#include "RealtimeEffectPanel.h"

#include <wx/app.h>
#include <wx/sizer.h>
#include <wx/statbmp.h>
#include <wx/stattext.h>
#include <wx/menu.h>
#include <wx/wupdlock.h>
#include <wx/hyperlink.h>

#include <wx/dcbuffer.h>

#include "widgets/HelpSystem.h"
#include "Theme.h"
#include "AllThemeResources.h"
#include "AudioIO.h"
#include "Observer.h"
#include "PluginManager.h"
#include "Project.h"
#include "ProjectHistory.h"
#include "ProjectWindow.h"
#include "Track.h"
#include "AColor.h"
#include "WaveTrack.h"
#include "effects/EffectUI.h"
#include "effects/EffectManager.h"
#include "effects/RealtimeEffectList.h"
#include "effects/RealtimeEffectState.h"
#include "effects/RealtimeEffectStateUI.h"
#include "UndoManager.h"
#include "Prefs.h"
#include "BasicUI.h"

#if wxUSE_ACCESSIBILITY
#include "widgets/WindowAccessible.h"
#endif

namespace
{
   template <typename Visitor>
   void VisitRealtimeEffectStateUIs(Track& track, Visitor&& visitor)
   {
      auto& effects = RealtimeEffectList::Get(track);
      effects.Visit(
         [visitor](auto& effectState, bool)
         {
            auto& ui = RealtimeEffectStateUI::Get(effectState);
            visitor(ui);
         });
   }

   void UpdateRealtimeEffectUIData(Track& track)
   {
      VisitRealtimeEffectStateUIs(
         track, [&](auto& ui) { ui.UpdateTrackData(track); });
   }

   void ReopenRealtimeEffectUIData(AudacityProject& project, Track& track)
   {
      VisitRealtimeEffectStateUIs(
         track,
         [&](auto& ui)
         {
            if (ui.IsShown())
            {
               ui.Hide(&project);
               ui.Show(project);
            }
         });
   }
   //fwd
   class RealtimeEffectControl;
   PluginID ShowSelectEffectMenu(wxWindow* parent, RealtimeEffectControl* currentEffectControl = nullptr);

   class DropHintLine : public wxWindow
   {
   public:
      DropHintLine(wxWindow *parent,
                wxWindowID id,
                const wxPoint& pos = wxDefaultPosition,
                const wxSize& size = wxDefaultSize)
                   : wxWindow(parent, id, pos, size, wxNO_BORDER, wxEmptyString)
      {
         wxWindow::SetBackgroundStyle(wxBG_STYLE_PAINT);
         Bind(wxEVT_PAINT, &DropHintLine::OnPaint, this);
      }
      
      bool AcceptsFocus() const override { return false; }

   private:
      void OnPaint(wxPaintEvent&)
      {
         wxBufferedPaintDC dc(this);
         const auto rect = wxRect(GetSize());

         dc.SetPen(*wxTRANSPARENT_PEN);
         dc.SetBrush(GetBackgroundColour());
         dc.DrawRectangle(rect);
      }
   };

   //Event generated by MovableControl when item is picked,
   //dragged or dropped, extends wxCommandEvent interface
   //with "source" and "target" indices which denote the
   //initial and final element positions inside wxSizer (if present)
   class MovableControlEvent final : public wxCommandEvent
   {
      int mSourceIndex{-1};
      int mTargetIndex{-1};
   public:
      MovableControlEvent(wxEventType eventType, int winid = 0)
         : wxCommandEvent(eventType, winid) { }

      void SetSourceIndex(int index) noexcept { mSourceIndex = index; }
      int GetSourceIndex() const noexcept { return mSourceIndex; }

      void SetTargetIndex(int index) noexcept { mTargetIndex = index; }
      int GetTargetIndex() const noexcept { return mTargetIndex; }

      wxEvent* Clone() const override
      {
         return new MovableControlEvent(*this);
      }
   };
   
   /**
    * \brief Changes default arrow navigation to behave more list- or table-like.
    * Instead of searching focusable items among children first, list navigation
    * searches for siblings when arrow key is pressed. Tab behaviour stays same.
    * Requires wxWANT_CHARS style flag to be set
    */
   template<class WindowBase>
   class ListNavigationEnabled : public wxNavigationEnabled<WindowBase>
   {
   public:
      ListNavigationEnabled()
      {
         WindowBase::Bind(wxEVT_NAVIGATION_KEY, &ListNavigationEnabled::OnNavigationKeyEvent, this);
         WindowBase::Bind(wxEVT_KEY_DOWN, &ListNavigationEnabled::OnKeyDown, this);
         WindowBase::Bind(wxEVT_CHAR_HOOK, &ListNavigationEnabled::OnCharHook, this);
      }

   private:
      void SetFocus() override
      {
         //Prevent attempt to search for a focusable child
         WindowBase::SetFocus();
      }

      void OnCharHook(wxKeyEvent& evt)
      {
         //We want to restore focus to list item once arrow navigation is used
         //on the child item, for this we need a char hook since key/navigation
         //events are sent directly to the focused item
         const auto keyCode = evt.GetKeyCode();
         if((keyCode == WXK_DOWN || keyCode == WXK_UP) &&
            !WindowBase::HasFocus() &&
            WindowBase::IsDescendant(WindowBase::FindFocus()))
         {
            wxWindow::SetFocusFromKbd();
         }
         else
            evt.Skip();
      }
      
      void OnKeyDown(wxKeyEvent& evt)
      {
         const auto keyCode = evt.GetKeyCode();
         if(keyCode == WXK_TAB)
            WindowBase::NavigateIn(wxNavigationKeyEvent::FromTab | (evt.ShiftDown() ? wxNavigationKeyEvent::IsBackward : wxNavigationKeyEvent::IsForward));
         else if(keyCode == WXK_DOWN)
            WindowBase::Navigate(wxNavigationKeyEvent::IsForward);
         else if(keyCode == WXK_UP)
            WindowBase::Navigate(wxNavigationKeyEvent::IsBackward);
         else
            evt.Skip();
      }

      void OnNavigationKeyEvent(wxNavigationKeyEvent& evt)
      {
         if(evt.GetEventObject() == WindowBase::GetParent() && !evt.IsFromTab())
            WindowBase::SetFocusFromKbd();
         else if(evt.GetEventObject() == this && evt.GetCurrentFocus() == this && evt.IsFromTab())
         {
            //NavigateIn
            wxPropagationDisabler disableProp(evt);
            const auto isForward = evt.GetDirection();
            const auto& children = WindowBase::GetChildren();
            auto node = isForward ? children.GetFirst() : children.GetLast();
            while(node)
            {
               auto child = node->GetData();
               if(child->CanAcceptFocusFromKeyboard())
               {
                  if(!child->GetEventHandler()->ProcessEvent(evt))
                  {
                     child->SetFocusFromKbd();
                  }
                  evt.Skip(false);
                  return;
               }
               node = isForward ? node->GetNext() : node->GetPrevious();
            }
         }
         else
            evt.Skip();
      }
      
      bool Destroy() override
      {
         if(WindowBase::IsDescendant(wxWindow::FindFocus()))
         {
            auto next = WindowBase::GetNextSibling();
            if(next != nullptr && next->AcceptsFocus())
               next->SetFocus();
            else
            {
               auto prev = WindowBase::GetPrevSibling();
               if(prev != nullptr && prev->AcceptsFocus())
                  prev->SetFocus();
            }
         }
         return wxNavigationEnabled<WindowBase>::Destroy();
      }
   
   };

   //Alias for ListNavigationEnabled<wxWindow> which provides wxWidgets-style ctor
   class ListNavigationPanel : public ListNavigationEnabled<wxWindow>
   {
   public:
      ListNavigationPanel() = default;

      ListNavigationPanel(wxWindow* parent,
                   wxWindowID id,
                   const wxPoint& pos = wxDefaultPosition,
                   const wxSize& size = wxDefaultSize,
                   const wxString& name = wxPanelNameStr)
      {
         Create(parent, id, pos, size, name);
      }

      void Create(wxWindow* parent,
                   wxWindowID id,
                   const wxPoint& pos = wxDefaultPosition,
                   const wxSize& size = wxDefaultSize,
                   const wxString& name = wxPanelNameStr)
      {
         SetBackgroundStyle(wxBG_STYLE_PAINT);
         ListNavigationEnabled<wxWindow>::Create(parent, id, pos, size, wxNO_BORDER | wxWANTS_CHARS, name);
         Bind(wxEVT_PAINT, &ListNavigationPanel::OnPaint, this);
         Bind(wxEVT_SET_FOCUS, &ListNavigationPanel::OnChangeFocus, this);
         Bind(wxEVT_KILL_FOCUS, &ListNavigationPanel::OnChangeFocus, this);
      }

      void OnChangeFocus(wxFocusEvent& evt)
      {
         Refresh(false);
      }
      
      void OnPaint(wxPaintEvent& evt)
      {
         wxBufferedPaintDC dc(this);

         dc.SetPen(*wxTRANSPARENT_PEN);
         dc.SetBrush(GetBackgroundColour());
         dc.Clear();

         if(HasFocus())
            AColor::DrawFocus(dc, GetClientRect().Deflate(3, 3));
      }
   };

   class HyperLinkCtrlWrapper : public ListNavigationEnabled<wxHyperlinkCtrl>
   {
   public:
      HyperLinkCtrlWrapper(wxWindow *parent,
                           wxWindowID id,
                           const wxString& label,
                           const wxString& url,
                           const wxPoint& pos = wxDefaultPosition,
                           const wxSize& size = wxDefaultSize,
                           long style = wxHL_DEFAULT_STYLE,
                           const wxString& name = wxHyperlinkCtrlNameStr)
      {
         Create(parent, id, label, url, pos, size, style, name);
      }
      
      void Create(wxWindow *parent,
                  wxWindowID id,
                  const wxString& label,
                  const wxString& url,
                  const wxPoint& pos = wxDefaultPosition,
                  const wxSize& size = wxDefaultSize,
                  long style = wxHL_DEFAULT_STYLE,
                  const wxString& name = wxHyperlinkCtrlNameStr)
      {
         ListNavigationEnabled<wxHyperlinkCtrl>::Create(parent, id, label, url, pos, size, style, name);
         Bind(wxEVT_PAINT, &HyperLinkCtrlWrapper::OnPaint, this);
      }
              
      void OnPaint(wxPaintEvent& evt)
      {
         wxPaintDC dc(this);
         dc.SetFont(GetFont());
         dc.SetTextForeground(GetForegroundColour());
         dc.SetTextBackground(GetBackgroundColour());

         auto labelRect = GetLabelRect();
         
         dc.DrawText(GetLabel(), labelRect.GetTopLeft());
         if (HasFocus())
            AColor::DrawFocus(dc, labelRect);
      }
   };

   wxDEFINE_EVENT(EVT_MOVABLE_CONTROL_DRAG_STARTED, MovableControlEvent);
   wxDEFINE_EVENT(EVT_MOVABLE_CONTROL_DRAG_POSITION, MovableControlEvent);
   wxDEFINE_EVENT(EVT_MOVABLE_CONTROL_DRAG_FINISHED, MovableControlEvent);

   //Base class for the controls that can be moved with drag-and-drop
   //action. Currently implementation is far from being generic and
   //can work only in pair with wxBoxSizer with wxVERTICAL layout.
   class MovableControl : public wxWindow
   {
      bool mDragging { false };
      wxPoint mInitialPosition;

      int mTargetIndex { -1 };
      int mSourceIndex { -1 };
   public:

      MovableControl() = default;

      MovableControl(wxWindow* parent,
                   wxWindowID id,
                   const wxPoint& pos = wxDefaultPosition,
                   const wxSize& size = wxDefaultSize,
                   long style = 0,
                   const wxString& name = wxPanelNameStr)
      {
         Create(parent, id, pos, size, style, name);
      }

      void Create(wxWindow* parent,
                   wxWindowID id,
                   const wxPoint& pos = wxDefaultPosition,
                   const wxSize& size = wxDefaultSize,
                   long style = 0,
                   const wxString& name = wxPanelNameStr)
      {
         wxWindow::Create(parent, id, pos, size, style, name);
         Bind(wxEVT_LEFT_DOWN, &MovableControl::OnMouseDown, this);
         Bind(wxEVT_LEFT_UP, &MovableControl::OnMouseUp, this);
         Bind(wxEVT_MOTION, &MovableControl::OnMove, this);
         Bind(wxEVT_KEY_DOWN, &MovableControl::OnKeyDown, this);
         Bind(wxEVT_MOUSE_CAPTURE_LOST, &MovableControl::OnMouseCaptureLost, this);
      }

      void ProcessDragEvent(wxWindow* target, wxEventType eventType)
      {
         MovableControlEvent event(eventType);
         event.SetSourceIndex(mSourceIndex);
         event.SetTargetIndex(mTargetIndex);
         event.SetEventObject(this);
         target->GetEventHandler()->ProcessEvent(event);
      }

      int FindIndexInParent() const
      {
         auto parent = GetParent();
         if(!parent)
            return -1;

         if(auto sizer = parent->GetSizer())
         {
            for(size_t i = 0, count = sizer->GetItemCount(); i < count; ++i)
            {
               if(sizer->GetItem(i)->GetWindow() == this)
                  return static_cast<int>(i);
            }
         }
         return -1;
      }

   private:

      void OnKeyDown(wxKeyEvent& evt)
      {
         const auto keyCode = evt.GetKeyCode();
         if(evt.AltDown() && (keyCode == WXK_DOWN || keyCode == WXK_UP))
         {
#ifdef __WXOSX__
            {//don't allow auto-repeats
               static long lastEventTimestamp = 0;
               if(lastEventTimestamp == evt.GetTimestamp())
                  return;//don't skip
               lastEventTimestamp = evt.GetTimestamp();
            }
#endif
            const auto sourceIndex = FindIndexInParent();
            if(sourceIndex == -1)
            {
               evt.Skip();
               return;
            }
            
            const auto targetIndex = std::clamp(
               keyCode == WXK_DOWN ? sourceIndex + 1 : sourceIndex - 1,
               0,
               static_cast<int>(GetParent()->GetSizer()->GetItemCount()) - 1
            );
            if(sourceIndex != targetIndex)
            {
               mSourceIndex = sourceIndex;
               mTargetIndex = targetIndex;
               ProcessDragEvent(GetParent(), EVT_MOVABLE_CONTROL_DRAG_FINISHED);
            }
         }
         else
            evt.Skip();
      }
      
      void OnMouseCaptureLost(wxMouseCaptureLostEvent& event)
      {
         if(mDragging)
            DragFinished();
      }

      void DragFinished()
      {
         if(auto parent = GetParent())
         {
            wxWindowUpdateLocker freeze(this);
            ProcessDragEvent(parent, EVT_MOVABLE_CONTROL_DRAG_FINISHED);
         }
         mDragging = false;
      }

      void OnMouseDown(wxMouseEvent& evt)
      {
         if(mDragging)
         {
            DragFinished();
            return;
         }

         mSourceIndex = mTargetIndex = FindIndexInParent();
         if(mSourceIndex != -1)
         {
            CaptureMouse();
            ProcessDragEvent(GetParent(), EVT_MOVABLE_CONTROL_DRAG_STARTED);

            mInitialPosition = evt.GetPosition();
            mDragging=true;
         }
      }

      void OnMouseUp(wxMouseEvent& evt)
      {
         if(!mDragging)
            return;

         ReleaseMouse();

         DragFinished();
      }

      void OnMove(wxMouseEvent& evt)
      {
         if(!mDragging)
            return;

         auto parent = GetParent();
         if(!parent)
            return;

         wxPoint newPosition = wxGetMousePosition() - mInitialPosition;
         Move(GetParent()->ScreenToClient(newPosition));

         if(auto boxSizer = dynamic_cast<wxBoxSizer*>(parent->GetSizer()))
         {
            if(boxSizer->GetOrientation() == wxVERTICAL)
            {
               auto targetIndex = mSourceIndex;

               //assuming that items are ordered from top to bottom (i > j <=> y(i) > y(j))
               //compare wxSizerItem position with the current MovableControl position!
               if(GetPosition().y < boxSizer->GetItem(mSourceIndex)->GetPosition().y)
               {
                  //moving up
                  for(int i = 0; i < mSourceIndex; ++i)
                  {
                     const auto item = boxSizer->GetItem(i);

                     if(GetRect().GetTop() <= item->GetPosition().y + item->GetSize().y / 2)
                     {
                        targetIndex = i;
                        break;
                     }
                  }
               }
               else
               {
                  //moving down
                  for(int i = static_cast<int>(boxSizer->GetItemCount()) - 1; i > mSourceIndex; --i)
                  {
                     const auto item = boxSizer->GetItem(i);
                     if(GetRect().GetBottom() >= item->GetPosition().y + item->GetSize().y / 2)
                     {
                        targetIndex = i;
                        break;
                     }
                  }
               }

               if(targetIndex != mTargetIndex)
               {
                  mTargetIndex = targetIndex;
                  ProcessDragEvent(parent, EVT_MOVABLE_CONTROL_DRAG_POSITION);
               }
            }
         }
      }
   };

#if wxUSE_ACCESSIBILITY
   class RealtimeEffectControlAx : public wxAccessible
   {
   public:
      RealtimeEffectControlAx(wxWindow* win = nullptr) : wxAccessible(win) { }

      wxAccStatus GetName(int childId, wxString* name) override
      {
         if(childId != wxACC_SELF)
            return wxACC_NOT_IMPLEMENTED;
         
         if(auto movable = wxDynamicCast(GetWindow(), MovableControl))
            //i18n-hint: argument - position of the effect in the effect stack
            *name = wxString::Format(_("Effect %d"), movable->FindIndexInParent() + 1);
         return wxACC_OK;
      }

      wxAccStatus GetChildCount(int* childCount) override
      {
         const auto window = GetWindow();
         *childCount = window->GetChildren().size();
         return wxACC_OK;
      }

      wxAccStatus GetChild(int childId, wxAccessible** child) override
      {
         if(childId == wxACC_SELF)
            *child = this;
         else
         {
            const auto window = GetWindow();
            const auto& children = window->GetChildren();
            const auto childIndex = childId - 1;
            if(childIndex < children.size())
               *child = children[childIndex]->GetAccessible();
            else
               *child = nullptr;
         }
         return wxACC_OK;
      }

      wxAccStatus GetRole(int childId, wxAccRole* role) override
      {
         if(childId != wxACC_SELF)
            return wxACC_NOT_IMPLEMENTED;

         *role = wxROLE_SYSTEM_PANE;
         return wxACC_OK;
      }

      wxAccStatus GetState(int childId, long* state) override
      {
         if(childId != wxACC_SELF)
            return wxACC_NOT_IMPLEMENTED;

         const auto window = GetWindow();
         if(!window->IsEnabled())
            *state = wxACC_STATE_SYSTEM_UNAVAILABLE;
         else
         {
            *state = wxACC_STATE_SYSTEM_FOCUSABLE;
            if(window->HasFocus())
               *state |= wxACC_STATE_SYSTEM_FOCUSED;
         }
         return wxACC_OK;
      }
   };
#endif

   //UI control that represents individual effect from the effect list
   class RealtimeEffectControl : public ListNavigationEnabled<MovableControl>
   {
      wxWeakRef<AudacityProject> mProject;
      std::shared_ptr<Track> mTrack;
      std::shared_ptr<RealtimeEffectState> mEffectState;
      std::shared_ptr<EffectSettingsAccess> mSettingsAccess;

      ThemedAButtonWrapper<AButton>* mChangeButton{nullptr};
      AButton* mEnableButton{nullptr};
      ThemedAButtonWrapper<AButton>* mOptionsButton{};

      Observer::Subscription mSubscription;

   public:
      RealtimeEffectControl() = default;

      RealtimeEffectControl(wxWindow* parent,
                   wxWindowID winid,
                   const wxPoint& pos = wxDefaultPosition,
                   const wxSize& size = wxDefaultSize)
      {
         Create(parent, winid, pos, size);
      }

      void Create(wxWindow* parent,
                   wxWindowID winid,
                   const wxPoint& pos = wxDefaultPosition,
                   const wxSize& size = wxDefaultSize)
      {
         //Prevents flickering and paint order issues
         MovableControl::SetBackgroundStyle(wxBG_STYLE_PAINT);
         MovableControl::Create(parent, winid, pos, size, wxNO_BORDER | wxWANTS_CHARS);

         Bind(wxEVT_PAINT, &RealtimeEffectControl::OnPaint, this);
         Bind(wxEVT_SET_FOCUS, &RealtimeEffectControl::OnFocusChange, this);
         Bind(wxEVT_KILL_FOCUS, &RealtimeEffectControl::OnFocusChange, this);

         auto sizer = std::make_unique<wxBoxSizer>(wxHORIZONTAL);

         //On/off button
         auto enableButton = safenew ThemedAButtonWrapper<AButton>(this);
         enableButton->SetTranslatableLabel(XO("Power"));
         enableButton->SetImageIndices(0, bmpEffectOff, bmpEffectOff, bmpEffectOn, bmpEffectOn, bmpEffectOff);
         enableButton->SetButtonToggles(true);
         enableButton->SetBackgroundColorIndex(clrEffectListItemBackground);
         mEnableButton = enableButton;

         enableButton->Bind(wxEVT_BUTTON, [this](wxCommandEvent&) {

            mEffectState->SetActive(mEnableButton->IsDown());
         });

         //Central button with effect name, show settings
         const auto optionsButton = safenew ThemedAButtonWrapper<AButton>(this, wxID_ANY);
         optionsButton->SetImageIndices(0,
            bmpHButtonNormal,
            bmpHButtonHover,
            bmpHButtonDown,
            bmpHButtonHover,
            bmpHButtonDisabled);
         optionsButton->SetBackgroundColorIndex(clrEffectListItemBackground);
         optionsButton->SetForegroundColorIndex(clrTrackPanelText);
         optionsButton->SetButtonType(AButton::TextButton);
         optionsButton->Bind(wxEVT_BUTTON, &RealtimeEffectControl::OnOptionsClicked, this);

         //Remove/replace effect
         auto changeButton = safenew ThemedAButtonWrapper<AButton>(this);
         changeButton->SetImageIndices(0, bmpMoreNormal, bmpMoreHover, bmpMoreDown, bmpMoreHover, bmpMoreDisabled);
         changeButton->SetBackgroundColorIndex(clrEffectListItemBackground);
         changeButton->SetTranslatableLabel(XO("Replace effect"));
         changeButton->Bind(wxEVT_BUTTON, &RealtimeEffectControl::OnChangeButtonClicked, this);
         
         auto dragArea = safenew wxStaticBitmap(this, wxID_ANY, theTheme.Bitmap(bmpDragArea));
         dragArea->Disable();
         sizer->Add(dragArea, 0, wxLEFT | wxCENTER, 5);
         sizer->Add(enableButton, 0, wxLEFT | wxCENTER, 5);
         sizer->Add(optionsButton, 1, wxLEFT | wxCENTER, 5);
         sizer->Add(changeButton, 0, wxLEFT | wxRIGHT | wxCENTER, 5);
         mChangeButton = changeButton;
         mOptionsButton = optionsButton;

         auto vSizer = std::make_unique<wxBoxSizer>(wxVERTICAL);
         vSizer->Add(sizer.release(), 0, wxUP | wxDOWN | wxEXPAND, 10);

         SetSizer(vSizer.release());

#if wxUSE_ACCESSIBILITY
         SetAccessible(safenew RealtimeEffectControlAx(this));
#endif
      }

      static const PluginDescriptor *GetPlugin(const PluginID &ID) {
         auto desc = PluginManager::Get().GetPlugin(ID);
         return desc;
      }

      //! @pre `mEffectState != nullptr`
      TranslatableString GetEffectName() const
      {
         const auto &ID = mEffectState->GetID();
         const auto desc = GetPlugin(ID);
         return desc
            ? desc->GetSymbol().Msgid()
            : XO("%s (missing)")
               .Format(PluginManager::GetEffectNameFromID(ID).GET());
      }

      void SetEffect(AudacityProject& project,
         const std::shared_ptr<Track>& track,
         const std::shared_ptr<RealtimeEffectState> &pState)
      {
         mProject = &project;
         mTrack = track;
         mEffectState = pState;

         mSubscription = mEffectState->Subscribe([this](RealtimeEffectStateChange state) {
            state == RealtimeEffectStateChange::EffectOn 
               ? mEnableButton->PushDown() 
               : mEnableButton->PopUp();

            if (mProject)
               ProjectHistory::Get(*mProject).ModifyState(false);
         });

         TranslatableString label;
         if (pState) {
            label = GetEffectName();
            mSettingsAccess = pState->GetAccess();
         }
         else
            mSettingsAccess.reset();
         if (mEnableButton)
            mSettingsAccess && mSettingsAccess->Get().extra.GetActive()
               ? mEnableButton->PushDown()
               : mEnableButton->PopUp();
         if (mOptionsButton)
         {
            mOptionsButton->SetTranslatableLabel(label);
            mOptionsButton->SetEnabled(pState && GetPlugin(pState->GetID()));
         }
      }

      void RemoveFromList()
      {
         if(mProject == nullptr || mEffectState == nullptr)
            return;

         auto& ui = RealtimeEffectStateUI::Get(*mEffectState);
         // Don't need autosave for the effect that is being removed
         ui.Hide();

         auto effectName = GetEffectName();
         //After AudioIO::RemoveState call this will be destroyed
         auto project = mProject.get();
         auto trackName = mTrack->GetName();

         AudioIO::Get()->RemoveState(*project, &*mTrack, mEffectState);
         ProjectHistory::Get(*project).PushState(
            /*! i18n-hint: undo history record
             first parameter - realtime effect name
             second parameter - track name
             */
            XO("Removed %s from %s").Format(effectName, trackName),
            /*! i18n-hint: undo history record
             first parameter - realtime effect name */
            XO("Remove %s").Format(effectName)
         );
      }

      void OnOptionsClicked(wxCommandEvent& event)
      {
         if(mProject == nullptr || mEffectState == nullptr)
            return;//not initialized

         const auto ID = mEffectState->GetID();
         const auto effectPlugin = EffectManager::Get().GetEffect(ID);

         if(effectPlugin == nullptr)
         {
            ///TODO: effect is not available
            return;
         }

         auto& effectStateUI = RealtimeEffectStateUI::Get(*mEffectState);

         effectStateUI.UpdateTrackData(*mTrack);
         effectStateUI.Toggle( *mProject );
      }

      void OnChangeButtonClicked(wxCommandEvent& event)
      {
         if(!mTrack || mProject == nullptr)
            return;
         if(mEffectState == nullptr)
            return;//not initialized

         const auto effectID = ShowSelectEffectMenu(mChangeButton, this);
         if(effectID.empty())
            return;

         auto &em = RealtimeEffectManager::Get(*mProject);
         auto oIndex = em.FindState(&*mTrack, mEffectState);
         if (!oIndex)
            return;

         auto oldName = GetEffectName();
         auto &project = *mProject;
         auto trackName = mTrack->GetName();
         if (auto state = AudioIO::Get()
            ->ReplaceState(project, &*mTrack, *oIndex, effectID)
         ){
            // Message subscription took care of updating the button text
            // and destroyed `this`!
            auto effect = state->GetEffect();
            assert(effect); // postcondition of ReplaceState
            ProjectHistory::Get(project).PushState(
               /*i18n-hint: undo history,
                first and second parameters - realtime effect names
                */
               XO("Replaced %s with %s")
                  .Format(oldName, effect->GetName()),
               /*! i18n-hint: undo history record
                first parameter - realtime effect name */
               XO("Replace %s").Format(oldName));
         }
      }

      void OnPaint(wxPaintEvent&)
      {
         wxBufferedPaintDC dc(this);
         const auto rect = wxRect(GetSize());

         dc.SetPen(*wxTRANSPARENT_PEN);
         dc.SetBrush(GetBackgroundColour());
         dc.DrawRectangle(rect);

         dc.SetPen(theTheme.Colour(clrEffectListItemBorder));
         dc.SetBrush(theTheme.Colour(clrEffectListItemBorder));
         dc.DrawLine(rect.GetBottomLeft(), rect.GetBottomRight());

         if(HasFocus())
            AColor::DrawFocus(dc, GetClientRect().Deflate(3, 3));
      }

      void OnFocusChange(wxFocusEvent& evt)
      {
         Refresh(false);
         evt.Skip();
      }
   };

   static wxString GetSafeVendor(const PluginDescriptor& descriptor)
   {
      if (descriptor.GetVendor().empty())
         return XO("Unknown").Translation();

      return descriptor.GetVendor();
   }

   PluginID ShowSelectEffectMenu(wxWindow* parent, RealtimeEffectControl* currentEffectControl)
   {
      wxMenu menu;

      if(currentEffectControl != nullptr)
      {
         //no need to handle language change since menu creates it's own event loop
         menu.Append(wxID_REMOVE, _("No Effect"));
         menu.AppendSeparator();
      }

      auto& pluginManager = PluginManager::Get();

      std::vector<const PluginDescriptor*> effects;
      int selectedEffectIndex = -1;

      auto compareEffects = [](const PluginDescriptor* a, const PluginDescriptor* b)
      {
         const auto vendorA = GetSafeVendor(*a);
         const auto vendorB = GetSafeVendor(*b);

         return vendorA < vendorB ||
                (vendorA == vendorB &&
                 a->GetSymbol().Translation() < b->GetSymbol().Translation());
      };

      for(auto& effect : pluginManager.EffectsOfType(EffectTypeProcess))
      {
         if(!effect.IsEffectRealtime() || !effect.IsEnabled())
            continue;
         effects.push_back(&effect);
      }

      std::sort(effects.begin(), effects.end(), compareEffects);

      wxString currentSubMenuName;
      std::unique_ptr<wxMenu> currentSubMenu;

      auto submenuEventHandler = [&](wxCommandEvent& event)
      {
         selectedEffectIndex = event.GetId() - wxID_HIGHEST;
      };

      for(int i = 0, count = effects.size(); i < count; ++i)
      {
         auto& effect = *effects[i];

         const wxString vendor = GetSafeVendor(effect);

         if(currentSubMenuName != vendor)
         {
            if(currentSubMenu)
            {
               currentSubMenu->Bind(wxEVT_MENU, submenuEventHandler);
               menu.AppendSubMenu(currentSubMenu.release(), currentSubMenuName);
            }
            currentSubMenuName = vendor;
            currentSubMenu = std::make_unique<wxMenu>();
         }

         const auto ID = wxID_HIGHEST + i;
         currentSubMenu->Append(ID, effect.GetSymbol().Translation());
      }
      if(currentSubMenu)
      {
         currentSubMenu->Bind(wxEVT_MENU, submenuEventHandler);
         menu.AppendSubMenu(currentSubMenu.release(), currentSubMenuName);
         menu.AppendSeparator();
      }
      menu.Append(wxID_MORE, _("Get more effects..."));

      menu.Bind(wxEVT_MENU, [&](wxCommandEvent& event)
      {
         if(event.GetId() == wxID_REMOVE)
            currentEffectControl->RemoveFromList();
         else if(event.GetId() == wxID_MORE)
            OpenInDefaultBrowser("https://plugins.audacityteam.org/");
      });

      if(parent->PopupMenu(&menu, parent->GetClientRect().GetLeftBottom()) && selectedEffectIndex != -1)
         return effects[selectedEffectIndex]->GetID();

      return {};
   }
}

class RealtimeEffectListWindow : public wxScrolledWindow
{
   wxWeakRef<AudacityProject> mProject;
   std::shared_ptr<Track> mTrack;
   AButton* mAddEffect{nullptr};
   wxStaticText* mAddEffectHint{nullptr};
   wxWindow* mAddEffectTutorialLink{nullptr};
   wxWindow* mEffectListContainer{nullptr};

   Observer::Subscription mEffectListItemMovedSubscription;

public:
   RealtimeEffectListWindow(wxWindow *parent,
                     wxWindowID winid = wxID_ANY,
                     const wxPoint& pos = wxDefaultPosition,
                     const wxSize& size = wxDefaultSize,
                     long style = wxScrolledWindowStyle,
                     const wxString& name = wxPanelNameStr)
      : wxScrolledWindow(parent, winid, pos, size, style, name)
   {
      Bind(wxEVT_SIZE, &RealtimeEffectListWindow::OnSizeChanged, this);
#ifdef __WXMSW__
      //Fixes flickering on redraw
      wxScrolledWindow::SetDoubleBuffered(true);
#endif
      auto rootSizer = std::make_unique<wxBoxSizer>(wxVERTICAL);

      auto effectListContainer = safenew ThemedWindowWrapper<wxPanel>(this, wxID_ANY);
      effectListContainer->SetBackgroundColorIndex(clrMedium);
      effectListContainer->SetSizer(safenew wxBoxSizer(wxVERTICAL));
      effectListContainer->SetDoubleBuffered(true);
      effectListContainer->Hide();
      mEffectListContainer = effectListContainer;

      auto addEffect = safenew ThemedAButtonWrapper<AButton>(this, wxID_ANY);
      addEffect->SetImageIndices(0,
            bmpHButtonNormal,
            bmpHButtonHover,
            bmpHButtonDown,
            bmpHButtonHover,
            bmpHButtonDisabled);
      addEffect->SetTranslatableLabel(XO("Add effect"));
      addEffect->SetButtonType(AButton::TextButton);
      addEffect->SetBackgroundColorIndex(clrMedium);
      addEffect->SetForegroundColorIndex(clrTrackPanelText);
      addEffect->Bind(wxEVT_BUTTON, &RealtimeEffectListWindow::OnAddEffectClicked, this);
      mAddEffect = addEffect;

      auto addEffectHint = safenew ThemedWindowWrapper<wxStaticText>(this, wxID_ANY, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxST_NO_AUTORESIZE);
      //Workaround: text is set in the OnSizeChange
      addEffectHint->SetForegroundColorIndex(clrTrackPanelText);
      mAddEffectHint = addEffectHint;

      auto addEffectTutorialLink = safenew ThemedWindowWrapper<wxHyperlinkCtrl>(
         this, wxID_ANY, _("Watch video"),
         "https://www.audacityteam.org/realtime-video", wxDefaultPosition,
         wxDefaultSize, wxHL_ALIGN_LEFT | wxHL_CONTEXTMENU);
      
      //i18n-hint: Hyperlink to the effects stack panel tutorial video
      addEffectTutorialLink->SetTranslatableLabel(XO("Watch video"));
#if wxUSE_ACCESSIBILITY
      safenew WindowAccessible(addEffectTutorialLink);
#endif

      addEffectTutorialLink->Bind(
         wxEVT_HYPERLINK, [](wxHyperlinkEvent& event)
         { BasicUI::OpenInDefaultBrowser(event.GetURL()); });

      mAddEffectTutorialLink = addEffectTutorialLink;

      //indicates the insertion position of the item
      auto dropHintLine = safenew ThemedWindowWrapper<DropHintLine>(effectListContainer, wxID_ANY);
      dropHintLine->SetBackgroundColorIndex(clrDropHintHighlight);
      dropHintLine->Hide();

      rootSizer->Add(mEffectListContainer, 0, wxEXPAND | wxBOTTOM, 10);
      rootSizer->Add(addEffect, 0, wxLEFT | wxRIGHT | wxBOTTOM | wxEXPAND, 20);
      rootSizer->Add(addEffectHint, 0, wxLEFT | wxRIGHT | wxBOTTOM | wxEXPAND, 20);
      rootSizer->Add(addEffectTutorialLink, 0, wxLEFT | wxRIGHT | wxEXPAND, 20);

      SetSizer(rootSizer.release());
      SetMinSize({});

      Bind(EVT_MOVABLE_CONTROL_DRAG_STARTED, [dropHintLine](const MovableControlEvent& event)
      {
         if(auto window = dynamic_cast<wxWindow*>(event.GetEventObject()))
            window->Raise();
      });
      Bind(EVT_MOVABLE_CONTROL_DRAG_POSITION, [this, dropHintLine](const MovableControlEvent& event)
      {
         constexpr auto DropHintLineHeight { 3 };//px

         auto sizer = mEffectListContainer->GetSizer();
         assert(sizer != nullptr);

         if(event.GetSourceIndex() == event.GetTargetIndex())
         {
            //do not display hint line if position didn't change
            dropHintLine->Hide();
            return;
         }

         if(!dropHintLine->IsShown())
         {
            dropHintLine->Show();
            dropHintLine->Raise();
            if(auto window = dynamic_cast<wxWindow*>(event.GetEventObject()))
               window->Raise();
         }

         auto item = sizer->GetItem(event.GetTargetIndex());
         dropHintLine->SetSize(item->GetSize().x, DropHintLineHeight);

         if(event.GetTargetIndex() > event.GetSourceIndex())
            dropHintLine->SetPosition(item->GetRect().GetBottomLeft() - wxPoint(0, DropHintLineHeight));
         else
            dropHintLine->SetPosition(item->GetRect().GetTopLeft());
      });
      Bind(EVT_MOVABLE_CONTROL_DRAG_FINISHED, [this, dropHintLine](const MovableControlEvent& event)
      {
         dropHintLine->Hide();

         if(mProject == nullptr)
            return;

         auto& effectList = RealtimeEffectList::Get(*mTrack);
         const auto from = event.GetSourceIndex();
         const auto to = event.GetTargetIndex();
         if(from != to)
         {
            auto effectName =
               effectList.GetStateAt(from)->GetEffect()->GetName();
            bool up = (to < from);
            effectList.MoveEffect(from, to);
            ProjectHistory::Get(*mProject).PushState(
               (up
                  /*! i18n-hint: undo history record
                   first parameter - realtime effect name
                   second parameter - track name
                   */
                  ? XO("Moved %s up in %s")
                  /*! i18n-hint: undo history record
                   first parameter - realtime effect name
                   second parameter - track name
                   */
                  : XO("Moved %s down in %s"))
                  .Format(effectName, mTrack->GetName()),
               XO("Change effect order"), UndoPush::CONSOLIDATE);
         }
         else
         {
            wxWindowUpdateLocker freeze(this);
            Layout();
         }
      });
      SetScrollRate(0, 20);
   }

   void OnSizeChanged(wxSizeEvent& event)
   {
      if(auto sizerItem = GetSizer()->GetItem(mAddEffectHint))
      {
         //We need to wrap the text whenever panel width changes and adjust widget height
         //so that text is fully visible, but there is no height-for-width layout algorithm
         //in wxWidgets yet, so for now we just do it manually

         //Restore original text, because 'Wrap' will replace it with wrapped one
         mAddEffectHint->SetLabel(_("Realtime effects are non-destructive and can be changed at any time."));
         mAddEffectHint->Wrap(GetClientSize().x - sizerItem->GetBorder() * 2);
         mAddEffectHint->InvalidateBestSize();
      }
      event.Skip();
   }

   void OnEffectListItemChange(const RealtimeEffectListMessage& msg)
   {
      auto sizer = mEffectListContainer->GetSizer();
      const auto insertItem = [this, &msg](){
         auto& effects = RealtimeEffectList::Get(*mTrack);
         InsertEffectRow(msg.srcIndex, effects.GetStateAt(msg.srcIndex));
         mAddEffectHint->Hide();
         mAddEffectTutorialLink->Hide();
      };
      const auto removeItem = [&](){
         auto& ui = RealtimeEffectStateUI::Get(*msg.affectedState);
         // Don't need to auto-save changed settings of effect that is deleted
         // Undo history push will do it anyway
         ui.Hide();
         
         auto window = sizer->GetItem(msg.srcIndex)->GetWindow();
         sizer->Remove(msg.srcIndex);
         wxTheApp->CallAfter([ref = wxWeakRef { window }] {
            if(ref) ref->Destroy();
         });
         
         if(sizer->IsEmpty())
         {
            if(mEffectListContainer->IsDescendant(FindFocus()))
               mAddEffect->SetFocus();
            
            mEffectListContainer->Hide();
            mAddEffectHint->Show();
            mAddEffectTutorialLink->Show();
         }
      };

      wxWindowUpdateLocker freeze(this);
      if(msg.type == RealtimeEffectListMessage::Type::Move)
      {
         const auto sizer = mEffectListContainer->GetSizer();

         const auto movedItem = sizer->GetItem(msg.srcIndex);

         const auto proportion = movedItem->GetProportion();
         const auto flag = movedItem->GetFlag();
         const auto border = movedItem->GetBorder();
         const auto window = movedItem->GetWindow();

         if(msg.srcIndex < msg.dstIndex)
            window->MoveAfterInTabOrder(sizer->GetItem(msg.dstIndex)->GetWindow());
         else
            window->MoveBeforeInTabOrder(sizer->GetItem(msg.dstIndex)->GetWindow());
         
         sizer->Remove(msg.srcIndex);
         sizer->Insert(msg.dstIndex, window, proportion, flag, border);
      }
      else if(msg.type == RealtimeEffectListMessage::Type::Insert)
      {
         insertItem();
      }
      else if(msg.type == RealtimeEffectListMessage::Type::WillReplace)
      {
         removeItem();
      }
      else if(msg.type == RealtimeEffectListMessage::Type::DidReplace)
      {
         insertItem();
      }
      else if(msg.type == RealtimeEffectListMessage::Type::Remove)
      {
         removeItem();
      }
      SendSizeEventToParent();
   }

   void ResetTrack()
   {
      mEffectListItemMovedSubscription.Reset();

      mTrack.reset();
      mProject = nullptr;
      ReloadEffectsList();
   }

   void SetTrack(AudacityProject& project, const std::shared_ptr<Track>& track)
   {
      if (mTrack == track)
         return;

      mEffectListItemMovedSubscription.Reset();

      mTrack = track;
      mProject = &project;
      ReloadEffectsList();

      if (track)
      {
         auto& effects = RealtimeEffectList::Get(*mTrack);
         mEffectListItemMovedSubscription = effects.Subscribe(
            *this, &RealtimeEffectListWindow::OnEffectListItemChange);

         UpdateRealtimeEffectUIData(*track);
      }
   }

   void EnableEffects(bool enable)
   {
      if (mTrack)
         RealtimeEffectList::Get(*mTrack).SetActive(enable);
   }

   void ReloadEffectsList()
   {
      wxWindowUpdateLocker freeze(this);
      
      const auto hadFocus = mEffectListContainer->IsDescendant(FindFocus());
      //delete items that were added to the sizer
      mEffectListContainer->Hide();
      mEffectListContainer->GetSizer()->Clear(true);

      
      if(!mTrack || RealtimeEffectList::Get(*mTrack).GetStatesCount() == 0)
         mEffectListContainer->Hide();
      
      auto isEmpty{true};
      if(mTrack)
      {
         auto& effects = RealtimeEffectList::Get(*mTrack);
         isEmpty = effects.GetStatesCount() == 0;
         for(size_t i = 0, count = effects.GetStatesCount(); i < count; ++i)
            InsertEffectRow(i, effects.GetStateAt(i));
      }
      mAddEffect->SetEnabled(!!mTrack);
      //Workaround for GTK: Underlying GTK widget does not update
      //its size when wxWindow size is set to zero
      mEffectListContainer->Show(!isEmpty);
      mAddEffectHint->Show(isEmpty);
      mAddEffectTutorialLink->Show(isEmpty);
      
      SendSizeEventToParent();
   }

   void OnAddEffectClicked(const wxCommandEvent& event)
   {
      if(!mTrack || mProject == nullptr)
         return;

      const auto effectID = ShowSelectEffectMenu(dynamic_cast<wxWindow*>(event.GetEventObject()));
      if(effectID.empty())
         return;

      if(auto state = AudioIO::Get()->AddState(*mProject, &*mTrack, effectID))
      {
         auto effect = state->GetEffect();
         assert(effect); // postcondition of AddState
         const auto effectName = effect->GetName();
         ProjectHistory::Get(*mProject).PushState(
            /*! i18n-hint: undo history record
             first parameter - realtime effect name
             second parameter - track name
             */
            XO("Added %s to %s").Format(effectName, mTrack->GetName()),
            //i18n-hint: undo history record
            XO("Add %s").Format(effectName));
      }
   }

   void InsertEffectRow(size_t index,
      const std::shared_ptr<RealtimeEffectState> &pState)
   {
      if(mProject == nullptr)
         return;

      // See comment in ReloadEffectsList
      if(!mEffectListContainer->IsShown())
         mEffectListContainer->Show();

      auto row = safenew ThemedWindowWrapper<RealtimeEffectControl>(mEffectListContainer, wxID_ANY);
      row->SetBackgroundColorIndex(clrEffectListItemBackground);
      row->SetEffect(*mProject, mTrack, pState);
      mEffectListContainer->GetSizer()->Insert(index, row, 0, wxEXPAND);
   }
};


struct RealtimeEffectPanel::PrefsListenerHelper : PrefsListener
{
   AudacityProject& mProject;

   explicit PrefsListenerHelper(AudacityProject& project)
       : mProject { project }
   {}

   void UpdatePrefs() override
   {
      auto& trackList = TrackList::Get(mProject);
      for (auto waveTrack : trackList.Any<WaveTrack>())
         ReopenRealtimeEffectUIData(mProject, *waveTrack);
   }
};

RealtimeEffectPanel::RealtimeEffectPanel(
   AudacityProject& project, wxWindow* parent, wxWindowID id, const wxPoint& pos,
   const wxSize& size,
   long style, const wxString& name)
      : wxPanel(parent, id, pos, size, style, name)
      , mProject(project)
      , mPrefsListenerHelper(std::make_unique<PrefsListenerHelper>(project))
{
   auto vSizer = std::make_unique<wxBoxSizer>(wxVERTICAL);

   auto header = safenew ThemedWindowWrapper<ListNavigationPanel>(this, wxID_ANY);
   header->SetMinClientSize({254, -1});
#if wxUSE_ACCESSIBILITY
   safenew WindowAccessible(header);
#endif
   header->SetBackgroundColorIndex(clrMedium);
   {
      auto hSizer = std::make_unique<wxBoxSizer>(wxHORIZONTAL);
      auto toggleEffects = safenew ThemedAButtonWrapper<AButton>(header);
      toggleEffects->SetImageIndices(0, bmpEffectOff, bmpEffectOff, bmpEffectOn, bmpEffectOn, bmpEffectOff);
      toggleEffects->SetButtonToggles(true);
      toggleEffects->SetTranslatableLabel(XO("Power"));
      toggleEffects->SetBackgroundColorIndex(clrMedium);
      mToggleEffects = toggleEffects;

      toggleEffects->Bind(wxEVT_BUTTON, [this](wxCommandEvent&) {
         if (mEffectList)
         {
            mEffectList->EnableEffects(mToggleEffects->IsDown());
         
            ProjectHistory::Get(mProject).ModifyState(false);
         }
      });

      hSizer->Add(toggleEffects, 0, wxSTRETCH_NOT | wxALIGN_CENTER | wxLEFT, 5);
      {
         auto vSizer = std::make_unique<wxBoxSizer>(wxVERTICAL);

         auto headerText = safenew ThemedWindowWrapper<wxStaticText>(header, wxID_ANY, wxEmptyString);
         headerText->SetFont(wxFont(wxFontInfo().Bold()));
         headerText->SetTranslatableLabel(XO("Realtime Effects"));
         headerText->SetForegroundColorIndex(clrTrackPanelText);

         auto trackTitle = safenew ThemedWindowWrapper<wxStaticText>(header, wxID_ANY, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxST_ELLIPSIZE_END);
         trackTitle->SetForegroundColorIndex(clrTrackPanelText);
         mTrackTitle = trackTitle;

         vSizer->Add(headerText);
         vSizer->Add(trackTitle);

         hSizer->Add(vSizer.release(), 1, wxEXPAND | wxALL, 10);
      }
      auto close = safenew ThemedAButtonWrapper<AButton>(header);
      close->SetTranslatableLabel(XO("Close"));
      close->SetImageIndices(0, bmpCloseNormal, bmpCloseHover, bmpCloseDown, bmpCloseHover, bmpCloseDisabled);
      close->SetBackgroundColorIndex(clrMedium);

      close->Bind(wxEVT_BUTTON, [this](wxCommandEvent&) { Close(); });

      hSizer->Add(close, 0, wxSTRETCH_NOT | wxALIGN_CENTER | wxRIGHT, 5);

      header->SetSizer(hSizer.release());
   }
   vSizer->Add(header, 0, wxEXPAND);

   auto effectList = safenew ThemedWindowWrapper<RealtimeEffectListWindow>(this, wxID_ANY);
   effectList->SetBackgroundColorIndex(clrMedium);
   vSizer->Add(effectList, 1, wxEXPAND);

   mHeader = header;
   mEffectList = effectList;

   SetSizerAndFit(vSizer.release());

   Bind(wxEVT_CHAR_HOOK, &RealtimeEffectPanel::OnCharHook, this);
   mTrackListChanged = TrackList::Get(mProject).Subscribe([this](const TrackListEvent& evt) {
         auto track = evt.mpTrack.lock();
         auto waveTrack = dynamic_cast<WaveTrack*>(track.get());

         if (waveTrack == nullptr)
            return;
         
         switch (evt.mType)
         {
         case TrackListEvent::TRACK_DATA_CHANGE:
            if (mCurrentTrack.lock() == track)
               mTrackTitle->SetLabel(track->GetName());
            UpdateRealtimeEffectUIData(*waveTrack);
            break;
         case TrackListEvent::DELETION:
            if (evt.mExtra == 0)
               mPotentiallyRemovedTracks.push_back(track);
            break;
         case TrackListEvent::ADDITION:
            // Addition can be fired as a part of "replace" event.
            // Calling UpdateRealtimeEffectUIData is mostly no-op,
            // it will just create a new State and Access for it.
            UpdateRealtimeEffectUIData(*waveTrack);
            break;
         default:
            break;
         }
   });

   mUndoSubscription = UndoManager::Get(mProject).Subscribe(
      [this](UndoRedoMessage message)
      {
         if (
            message.type == UndoRedoMessage::Type::Purge ||
            message.type == UndoRedoMessage::Type::BeginPurge ||
            message.type == UndoRedoMessage::Type::EndPurge)
            return;

         auto& trackList = TrackList::Get(mProject);

         // Realtime effect UI is only updated on Undo or Redo
         auto waveTracks = trackList.Any<WaveTrack>();
         
         if (
            message.type == UndoRedoMessage::Type::UndoOrRedo ||
            message.type == UndoRedoMessage::Type::Reset)
         {
            for (auto waveTrack : waveTracks)
               UpdateRealtimeEffectUIData(*waveTrack);
         }

         // But mPotentiallyRemovedTracks processing happens as fast as possible.
         // This event is fired right after the track is deleted, so we do not
         // hold the strong reference to the track much longer than need.
         if (mPotentiallyRemovedTracks.empty())
            return;

         // Collect RealtimeEffectUIs that are currently shown
         // for the potentially removed tracks
         std::vector<RealtimeEffectStateUI*> shownUIs;
         
         for (auto track : mPotentiallyRemovedTracks)
         {
            // By construction, track cannot be null
            assert(track != nullptr);

            VisitRealtimeEffectStateUIs(
               *track,
               [&shownUIs](auto& ui)
               {
                  if (ui.IsShown())
                     shownUIs.push_back(&ui);
               });
         }

         // For every UI shown - check if the corresponding state
         // is reachable from the current track list.
         for (auto effectUI : shownUIs)
         {
            bool reachable = false;
            
            for (auto track : waveTracks)
            {               
               VisitRealtimeEffectStateUIs(
                  *track,
                  [effectUI, &reachable](auto& ui)
                  {
                     if (effectUI == &ui)
                        reachable = true;
                  });

               if (reachable)
                  break;
            }

            if (!reachable)
               // Don't need to autosave for an unreachable state
               effectUI->Hide();
         }

         mPotentiallyRemovedTracks.clear();
      });
}

RealtimeEffectPanel::~RealtimeEffectPanel()
{
}

void RealtimeEffectPanel::SetTrack(const std::shared_ptr<Track>& track)
{
   //Avoid creation-on-demand of a useless, empty list in case the track is of non-wave type.
   if(track && dynamic_cast<WaveTrack*>(&*track) != nullptr)
   {
      mTrackTitle->SetLabel(track->GetName());
      mToggleEffects->Enable();
      track && RealtimeEffectList::Get(*track).IsActive()
         ? mToggleEffects->PushDown()
         : mToggleEffects->PopUp();
      mEffectList->SetTrack(mProject, track);

      mCurrentTrack = track;
      //i18n-hint: argument - track name
      mHeader->SetName(wxString::Format(_("Realtime effects for %s"), track->GetName()));
   }
   else
      ResetTrack();
}

void RealtimeEffectPanel::ResetTrack()
{
   mTrackTitle->SetLabel(wxEmptyString);
   mToggleEffects->Disable();
   mEffectList->ResetTrack();
   mCurrentTrack.reset();
   mHeader->SetName(wxEmptyString);
}

void RealtimeEffectPanel::SetFocus()
{
   mHeader->SetFocus();
}

void RealtimeEffectPanel::OnCharHook(wxKeyEvent& evt)
{
   if(evt.GetKeyCode() == WXK_ESCAPE && IsShown() && IsDescendant(FindFocus()))
      Close();
   else
      evt.Skip();
}
