/****************************************************************
**
** Attal : Lords of Doom
**
** serverInterface.cpp
** interface for the server
**
** Version : $Id: serverInterface.cpp,v 1.16 2004/10/10 11:46:37 audoux Exp $
**
** Author(s) : Pascal Audoux - Carlo
**
** Date : 01/11/2000
**
** Licence :    
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/


#include "serverInterface.h"

#include <stdlib.h>
#include <time.h>
// include files for QT
#include <qapplication.h>
#include <qfiledialog.h>
#include <qhbuttongroup.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qstatusbar.h>
#include <qvbuttongroup.h>
// application specific includes
#include "conf.h"
#include "libCommon/dataTheme.h"
#include "libCommon/log.h"

#include "libClient/gui.h"

#include "displayScenarii.h"

extern DataTheme DataTheme;
extern QString DATA_PATH;
extern QString SCENAR_PATH;
extern QString CAMPAIGN_PATH;
extern QString THEME;
extern QString PORT;
extern bool FAST;

#define CLIENT

/** add comments here */
ServerInterface::ServerInterface()
	:QMainWindow()
{
	setCaption( tr( "Attal - Lords of Doom (Server)" ) );
	initActions();
	initMenuBar();
	initStatusBar();
	if( ! DataTheme.init() ) {
		/// XXX: we could manage this better (later :) )
		QMessageBox::critical( this, tr( "Can't load theme" ), tr( "Theme " ) + THEME + tr( " has not been loaded successfully" ) );
	}

	_widget = new ServerWidget( this );
	setCentralWidget( _widget );

	connect( _widget, SIGNAL( sig_stop() ), SLOT( slot_stop() ) );
	connect( _widget, SIGNAL( sig_load( QString ) ), SLOT( slot_load( QString ) ) );
	connect( _widget, SIGNAL( sig_save() ), SLOT( slot_save() ) );

	_config = 0;

	if( !init() ) {
		logDD( "quit" );
		//XXX ugly hack, but qApp->quit() seem not work in constructor
		exit(0);
	}

	setMinimumSize( 350, 200 );
	
	srand( time( NULL ) );
}

ServerInterface::~ServerInterface()
{
}

void ServerInterface::closeEvent(QCloseEvent* ce)
{
	_engine->setFinished(true);
	ce->accept();
}

bool ServerInterface::init()
{
	bool ret = false;
	_config = new ConfigConnection( this );
	_config->setHost( "localhost" );
	_config->setPort( PORT.toInt() );
	if(FAST){
		_config->accept();
	} else { 
		_config->exec();
	}
	if( _config->result() == QDialog::Accepted ) {
		_server = new AttalServer( _config->getPort() );
		if( _server->ok() ) {
			ret = true;
			_engine = new Engine( _server );
			_server->setEngine( _engine );
			connect( _engine, SIGNAL( sig_newPlayer( AttalPlayerSocket * ) ), _widget, SLOT( slot_newPlayer( AttalPlayerSocket * ) ) );
			connect( _server, SIGNAL( sig_endConnection( QString ) ), _widget, SLOT( slot_endConnection( QString ) ) );
		} else {
			delete _server;
			_server = 0;
			if( QMessageBox::warning( this,
						tr( "Server error" ),
						tr( "Could not listen for sockets." ),
						tr( "Try again" ),
						tr( "Quit" ),
						0, 0, 1 )  == 0 ) {
				ret = init();
			} else {
				ret = false;
			}
		}
	} 

	if (_config){
		delete _config;
	}
	_config = 0;
	return ret;
}

void ServerInterface::initStatusBar()
{
	statusBar()->message( tr( "Status Bar" ), 0 );
}

void ServerInterface::initActions()
{
	_actions.resize( NB_ACTIONS );
	
	QAction * action;
	QSignalMapper * sigmap = new QSignalMapper( this );

	action = new QAction( tr( "Load scenario" ), tr( "Load scenario" ), this );
	_actions.insert( ACTION_LOADSCENARIO, action );
	sigmap->setMapping( action, ACTION_LOADSCENARIO );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
	
	action = new QAction( tr( "Load campaign" ), tr( "Load campaign" ), this );
	_actions.insert( ACTION_LOADCAMPAIGN, action );
	sigmap->setMapping( action, ACTION_LOADCAMPAIGN );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
	
	action = new QAction( tr( "Load game" ), tr( "Load game" ), this );
	_actions.insert( ACTION_LOADGAME, action );
	sigmap->setMapping( action, ACTION_LOADGAME );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
	
	action = new QAction( tr( "Save game" ), tr( "Save game" ), this );
	_actions.insert( ACTION_SAVE, action );
	sigmap->setMapping( action, ACTION_SAVE );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
	
	action = new QAction( tr( "End game" ), tr( "End game" ), this );
	_actions.insert( ACTION_END, action );
	sigmap->setMapping( action, ACTION_END );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
		
	action = new QAction( tr( "Quit" ), tr( "&Quit" ), this );
	_actions.insert( ACTION_QUIT, action );
	sigmap->setMapping( action, ACTION_QUIT );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
		
	action = new QAction( tr( "Fill with AI" ), tr( "Fill with AI" ), this );
	_actions.insert( ACTION_FILLAI, action );
	sigmap->setMapping( action, ACTION_FILLAI );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
	
	action = new QAction( tr( "Add AI player" ), tr( "Add AI player" ), this );
	_actions.insert( ACTION_ADDAI, action );
	sigmap->setMapping( action, ACTION_ADDAI );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );
		
	connect( sigmap, SIGNAL( mapped( int ) ), SLOT( slot_action( int ) ) );
}

void ServerInterface::initMenuBar()
{
	QPopupMenu * menuFile = new QPopupMenu();
	_actions[ ACTION_LOADSCENARIO ]->addTo( menuFile );
	_actions[ ACTION_LOADCAMPAIGN ]->addTo( menuFile );
	menuFile->insertSeparator();
	_actions[ ACTION_LOADGAME ]->addTo( menuFile );
	_actions[ ACTION_SAVE ]->addTo( menuFile );
	_actions[ ACTION_END ]->addTo( menuFile );
	_actions[ ACTION_QUIT ]->addTo( menuFile );

	QPopupMenu * menuGame = new QPopupMenu();
	_actions[ ACTION_FILLAI ]->addTo( menuGame );
	_actions[ ACTION_ADDAI ]->addTo( menuGame );

	menuBar()->insertItem( tr( "&File" ), menuFile );
	menuBar()->insertItem( tr( "&Game" ), menuGame );
}

void ServerInterface::slot_action( int num )
{
	switch( num ) {
	case ACTION_LOADSCENARIO:{
		DisplayScenariiDialog * scen = new DisplayScenariiDialog( this );
		if( scen->exec() ) {
			slot_load( scen->getFileName() );
		}
		} break;
	case ACTION_LOADCAMPAIGN:{
		QString filename;
		filename = QFileDialog::getOpenFileName( CAMPAIGN_PATH, "*.cmp", this );
		loadCampaign( filename );
		} break;
	case ACTION_LOADGAME:{
		QString filename;
		filename = QFileDialog::getOpenFileName( "", "*.scn *.gam", this );
		slot_load( filename );
		} break;
	case ACTION_SAVE:
		slot_save();
		break;
	case ACTION_END:
		slot_stop();
		break;
	case ACTION_QUIT:
		_engine->setFinished(true);
		qApp->quit();
		break;
	case ACTION_FILLAI:
		fillWithAI();
		break;
	case ACTION_ADDAI:
		addAI();
		break;
	}
}

void ServerInterface::fillWithAI()
{
	QString filename = _widget->getFilename();
	if( _engine->loadGame(filename, true ) == false ) {
		int play = _engine->getNumFillPlayers();
		if( play > 0 ) {
			for( int i = 0; i < play; i++ ) {
				addAI();	
			}	
		}
	}
}

void ServerInterface::addAI()
{
	int pid; 

	if( ( pid = fork() ) < 0 ) {
		return;
	}
	
	if( ! pid ) {
		execl( "./attal-ai", "attal-ai", "-fast", NULL );
	}
}

void ServerInterface::slot_status( QString text )
{
	statusBar()->message( text, 0 );
}

void ServerInterface::slot_stop()
{
	_engine->endGame();
	_widget->setGameLoaded( false );
}

void ServerInterface::loadCampaign( const QString & fileName )
{
	if( ( _server->getNbSocket() == 1 ) && ( ! fileName.isNull() ) ) {
		if( _engine->loadCampaign( fileName ) ) {
			_engine->startCampaign();
			_widget->setGameLoaded( true );
		}
	}
}

void ServerInterface::slot_load( QString filename )
{
	if( ( _server->getNbSocket() > 0 ) && ( !filename.isNull() ) ) {
		if( _engine->loadGame( filename  , false) ) {
			_widget->setGameLoaded( true );
			_engine->startGame();
		}
	}
}

void ServerInterface::slot_save()
{
	QString filename;
	filename = QFileDialog::getSaveFileName( "", "*.gam", this );
	if (!filename.isNull()) {
		_engine->saveGame( filename );
	}
}

//
// ----- ChooseFileRadioButton -----
//

ChooseFileRadioButton::ChooseFileRadioButton( QWidget * parent, const char * name )
	:QRadioButton( parent, name )
{
	QHBoxLayout * layout = new QHBoxLayout( this );
	layout->addSpacing( 20 );
	
	_edit = new QLineEdit( this );
	layout->addWidget( _edit, 1 );
	layout->addSpacing( 10 );
	
	_choose = new QPushButton( this );
	_choose->setText( tr( "Choose" ) );
	FIXEDSIZE( _choose );
	layout->addWidget( _choose );
	
	layout->activate();
	
	connect( _choose, SIGNAL( clicked() ), SLOT( slot_choose() ) );
	connect( this, SIGNAL( toggled( bool ) ), SLOT( slot_toggle( bool ) ) );
	setFixedHeight( _choose->size().height() + 4 );
}

void ChooseFileRadioButton::slot_choose()
{
	QString filename = QFileDialog::getOpenFileName( "", "*.scn *.gam", this ); 
	if( ! filename.isNull() ) {
		_edit->setText( filename );
	}
}

void ChooseFileRadioButton::slot_toggle( bool st )
{
	_choose->setEnabled( st );
	_edit->setEnabled( st );
}

//
// ----- ServerWidget -----
//

ServerWidget::ServerWidget( QWidget * parent, const char * name )
	: QWidget( parent, name )
{
	_group = new QVButtonGroup( this );

	QRadioButton * radio1 = new QRadioButton( _group );
	radio1->setText( "Demo 1 player" );
	radio1->setChecked( true );

	QRadioButton * radio2 = new QRadioButton( _group );
	radio2->setText( "Demo 2 players" );

	_radio3 = new ChooseFileRadioButton( _group );
	_radio3->slot_toggle( false );

	_playerList = new QListView( this );
	_playerList->addColumn( tr( "Name" ) );
	_playerList->addColumn( tr( "Address" ) );
	_playerList->setMinimumHeight( 50 );

	_groupBottom = new QHButtonGroup( this );

	QPushButton * butStart = new QPushButton( _groupBottom );
	butStart->setText( tr( "Start" ) );
	FIXEDSIZE( butStart );

	QPushButton * butStop = new QPushButton( _groupBottom );
	butStop->setText( tr( "Save" ) );
	FIXEDSIZE( butStop );

	QPushButton * butSave = new QPushButton( _groupBottom );
	butSave->setText( tr( "End" ) );
	FIXEDSIZE( butSave );

	_groupBottom->find( 0 )->setEnabled( true );
	_groupBottom->find( 1 )->setEnabled( false );
	_groupBottom->find( 2 )->setEnabled( false );

	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->addWidget( _group );
	layout->addSpacing( 5 );
	layout->addWidget( _playerList, 1 );
	layout->addSpacing( 5 );
	layout->addWidget( _groupBottom );
	layout->activate();

	connect( _groupBottom, SIGNAL( clicked( int ) ), SLOT( slot_choice( int ) ) );
}

void ServerWidget::slot_newPlayer( AttalPlayerSocket * player )
{
	_playerList->insertItem( new QListViewItem( _playerList, player->getPlayer()->getName(), player->address().toString() )  );
}

void ServerWidget::slot_endConnection( QString name )
{
	bool found = false;
	QListViewItem * item = _playerList->firstChild();

	do {
		if( item ) {
			if( item->text( 0 ) == name ) {
				found = true;
				_playerList->takeItem( item );
				delete item;
			} else {
				item = item->nextSibling();
			}
		} else {
			found = true;
		}
	} while( !found );
}

void ServerWidget::setGameLoaded( bool b )
{
	_loaded = b;
	if( _loaded ) {
		_groupBottom->find( 0 )->setEnabled( false );
		_groupBottom->find( 1 )->setEnabled( true );
		_groupBottom->find( 2 )->setEnabled( true );
		_group->setEnabled( false );
	} else {
		_groupBottom->find( 0 )->setEnabled( true );
		_groupBottom->find( 1 )->setEnabled( false );
		_groupBottom->find( 2 )->setEnabled( false );
		_group->setEnabled( true );
	}
}

QString ServerWidget::getFilename()
{
	QString filename;
	switch( _group->id( _group->selected() ) ) {
		case 0:
			filename = SCENAR_PATH + "demo_1player.scn";
			break;
		case 1:
			filename = SCENAR_PATH + "demo_2players.scn";
			break;
		case 2:
			filename = _radio3->getText();
			break;
	}
	return filename;

}

void ServerWidget::slot_choice( int choice )
{
	switch( choice ) {
	case 0: {
		QString filename;
		switch( _group->id( _group->selected() ) ) {
		case 0:
			filename = SCENAR_PATH + "demo_1player.scn";
			break;
		case 1:
			filename = SCENAR_PATH + "demo_2players.scn";
			break;
		case 2:
			filename = _radio3->getText();
			break;
		}
		emit sig_load( filename );

		break;
	}
	case 1:
		emit sig_save();
		break;
	case 2:
		emit sig_stop();
		break;
	}
}

//
// ----- ConfigConnection -----
//

ConfigConnection::ConfigConnection( QWidget * parent, const char * name )
	: QDialog( parent, name, true )
{
	setCaption( tr( "Start server" ) );
	QVBoxLayout * layout = new QVBoxLayout( this );

	QHBoxLayout * layH1 = new QHBoxLayout();
	layH1->addSpacing( 5 );
	QLabel * labHost = new QLabel( tr( "Host : " ), this );
	FIXEDSIZE( labHost );
	layH1->addWidget( labHost );
	layH1->addSpacing( 5 );
	_host = new QLineEdit( this );
	_host->setFixedSize( 160, 20 );
	_host->setEnabled(false);
	layH1->addWidget( _host );
	layH1->addStretch( 1 );
	layout->addLayout( layH1, 1 );

	QHBoxLayout * layH2 = new QHBoxLayout();
	layH2->addSpacing( 5 );
	QLabel * labPort = new QLabel( tr( "Port : " ), this );
	FIXEDSIZE( labPort );
	layH2->addWidget( labPort );
	layH2->addSpacing( 5 );
	_port = new QLineEdit( this );
	_port->setFixedSize( 80, 20 );
	layH2->addWidget( _port );
	layH2->addStretch( 1 );
	layout->addLayout( layH2, 1 );

	
	QHBoxLayout * layH3 = new QHBoxLayout();
	QPushButton * pbOk = new QPushButton( this );
	pbOk->setText( tr( "Start" ) );
	FIXEDSIZE( pbOk );
	layH3->addStretch( 1 );
	layH3->addWidget( pbOk );
	layH3->addStretch( 1 );
	QPushButton * pbCan = new QPushButton( this );
	pbCan->setText( tr( "Quit" ) );
	FIXEDSIZE( pbCan );
	layH3->addWidget( pbCan );
	layH3->addStretch( 1 );
	layout->addLayout( layH3, 2 );

	layout->activate();

	connect( pbOk, SIGNAL( clicked() ), SLOT( accept() ) );
	connect( pbCan, SIGNAL( clicked() ), SLOT( reject() ) );

	setFixedSize( 250, 150 );
}
