/****************************************************************
**
** Attal : Lords of Doom
**
** miniMap.cpp
** draw and manage the mini-map
**
** Version : $Id: miniMap.cpp,v 1.13 2005/12/28 10:06:05 audoux Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo - Forest Darling
**
** Date : 05/12/2000
**
** Licence :    
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "miniMap.h"

 
// generic include files
// include files for QT
#include <QBrush>
#include <QMouseEvent>
#include <QPainter>
#include <QPaintEvent>
#include <QPixmap>
#include <QResizeEvent>

// application specific includes
#include "libCommon/log.h"
#include "libCommon/dataTheme.h"
#include "libClient/graphicalCell.h"

extern DataTheme DataTheme;

QPen MiniMap::_dashPen = QPen( Qt::white, 1, Qt::DashLine );
int MiniMap::_defaultHeight = 200;
int MiniMap::_defaultWidth = 200;

MiniMap::MiniMap( Map * map, QWidget * parent, const char * name )
	: QWidget( parent, name )
{
	setAttribute( Qt::WA_PaintOutsidePaintEvent, true );
	_map = map;
	_gmapw = _map->width();
	_gmaph = _map->height();
	connect( _map, SIGNAL( resized() ), SLOT( slot_graphicalMapResized() ) );

	setFixedSize( _defaultHeight, _defaultWidth );
	setBackgroundColor( Qt::black );

	_qp = new QPixmap( _defaultHeight, _defaultWidth );
	_height = _defaultHeight;
	_width = _defaultWidth;
	_ntiles = (int)( DataTheme.tiles.count() );
	_viewrect = new QRect( 0, 0, 0, 0 );
}

void MiniMap::resizeMap( uint width, uint height )
{
	_height = height;
	_width = width;
}

void MiniMap::redrawMap( Map * map )
{
	int w = map->getWidth();
	int h = map->getHeight();

	if( /*isVisible() && */( w > 0 ) && ( h > 0 ) ) {
		_qp->fill(Qt::black);
		_map = map;
	  	_sizeH = _width/w;
	  	_sizeV = _height/h;

		//QPainter paint( _qp );
		QBrush brush;
		for( int i = 0; i < h; i++ ) {
			for( int j = 0; j < w; j++ ) {
				// XXX: to improve (color in theme ?)
				GenericCell * cell = map->at( i, j );
				if( cell->getType() < _ntiles) {
					redrawCellFast( cell );
				} else {
					logEE( "Each cell should have a type" );
				}
			}
		}

		QPainter paint( this );
		paint.drawPixmap( 0, 0, * _qp );
		
		paint.setPen( _dashPen );
		paint.drawRect( * _viewrect );
	}
}


void MiniMap::redrawCell( GenericCell * cell )
{
	uint w = _map->getWidth();
	uint h = _map->getHeight();
	uint row = cell->getRow();
	uint col = cell->getCol();
	int type = cell->getType();

	if( /*isVisible() && */ ( w > 0 ) && ( h > 0 ) ) {
	  
		int sizeH = _width/w;
	  int sizeV = _height/h;

		QPainter paintQp( _qp );
		QBrush brush;

		// XXX: to improve (color in theme ?)
		if(  type < _ntiles ) {
			if( type == 0 ) {
				paintQp.fillRect( col * sizeH, row * sizeV, sizeH, sizeV, Qt::black );
			} else  {
				QColor c = DataTheme.tiles.at( type )->getColor();
				paintQp.fillRect( col * sizeH, row * sizeV, sizeH, sizeV, c );
			}
		} else {
			logEE( "Each cell should have a type" );
		}
		
		QPainter paint( this );
		paint.drawPixmap( 0, 0, * _qp );
		
		paint.setPen( _dashPen );
		paint.drawRect( * _viewrect );
	}
}

void MiniMap::redrawCellFast( GenericCell * cell )
{
	uint row = cell->getRow();
	uint col = cell->getCol();
	int type = cell->getType();

	QPainter paint( _qp );
	QBrush brush;

	// XXX: to improve (color in theme ?)
	if( type < _ntiles ) {
		if( type == 0 ) {
			paint.fillRect( col * _sizeH, row * _sizeV, _sizeH, _sizeV, Qt::black );
		} else  {
			QColor c = DataTheme.tiles.at( type )->getColor();
			paint.fillRect( col * _sizeH, row * _sizeV, _sizeH, _sizeV, c );
		}
		
		QPainter paint2( this );
		paint2.setPen( _dashPen );
		paint2.drawRect( * _viewrect );
	} else {
		logEE( "Each cell should have a type" );
	}
}

void MiniMap::paintEvent( QPaintEvent * /* event */ )
{
	//QWidget::paintEvent( event );
	
	QPainter paint( this );
	paint.drawPixmap( 0, 0, * _qp );

	paint.setPen( _dashPen );
	paint.drawRect(*_viewrect);
}

void MiniMap::resizeEvent( QResizeEvent * event )
{
	QWidget::resizeEvent( event );
	_qp->resize( event->size() );
	_qp->fill( Qt::black );
	
	QPainter paint( this );
	paint.drawPixmap( 0, 0, * _qp );
	
	paint.setPen( _dashPen );
	paint.drawRect( * _viewrect );
}

void MiniMap::mousePressEvent( QMouseEvent * event )
{
	//QWidget::mousePressEvent( event );
	translatePoint( event->x(), event->y() );
}

void MiniMap::translatePoint( int x, int y )
{
	uint modx, mody;
	
	if( x < 0 ) {
		x = 0;
	} else if( x >= width() ) {
		x = width() - 1;
	}
	
	if( y < 0 ) {
		y = 0;
	} else if( y >= height() ) {
		y = height() - 1;
	}
	
	modx = x * _gmapw / width();
	mody = y * _gmaph / height();
	emit sig_mouseReleasedMinimap( modx, mody );
}

void MiniMap::mouseMoveEvent( QMouseEvent * event )
{
	translatePoint( event->x(), event->y() );
}

void MiniMap::slot_graphicalMapResized()
{
	_gmapw = _map->width();
	_gmaph = _map->height();
	
	repaintViewRect();
}

void MiniMap::repaintViewRect()
{
	if( _gmapw && _gmaph ) {
		_viewrect->setRect(_viewx * width() / _gmapw, _viewy * height() / _gmaph, _vieww * width() / _gmapw, _viewh * height() / _gmaph);
		repaint();
	}
}

void MiniMap::slot_mapviewChanged( int x, int y, int w, int h )
{
	if ( w && h ) {
		_vieww = w;
		_viewh = h;
	}
	
	_viewx = x;
	_viewy = y;

	repaintViewRect();
}

void MiniMap::slot_mapviewScrolled( int x, int y )
{
	slot_mapviewChanged( x, y, 0, 0);
}
