/****************************************************************
**
** Attal : Lords of Doom
**
** displayBase.cpp
** inside the base
**
** Version : $Id: displayBase.cpp,v 1.43 2006/03/11 14:59:36 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo
**
** Date : 03/09/2000
**
** Licence :    
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "displayBase.h"

 
// generic include files
// include files for QT
#include <QApplication>
#include <QDesktopWidget>
#include <QLabel>
#include <QLayout>
#include <QMessageBox>
#include <QPixmap>
#include <QPushButton>
#include <QSignalMapper>

// application specific include files
#include "conf.h"
#include "libCommon/attalSettings.h"
#include "libCommon/attalSocket.h"
#include "libCommon/dataTheme.h"
#include "libCommon/genericBuilding.h"
#include "libCommon/genericLord.h"
#include "libCommon/log.h"

#include "libClient/attalButton.h"
#include "libClient/displayCreature.h"
#include "libClient/game.h"
#include "libClient/gui.h"
#include "libClient/imageTheme.h"
#include "libClient/insideAction.h"
#include "libClient/insideBase.h"
#include "libClient/market.h"
#include "libClient/tavern.h"
#include "libClient/unitExchange.h"


extern QString DATA_PATH;
extern QString IMAGE_PATH;
extern ImageTheme ImageTheme;
extern DataTheme DataTheme;

DisplayBase::DisplayBase( QWidget * parent, const char * name )
	: QWidget( parent, name )
{
	_player = 0;
	_base = 0;
	_socket = 0;
	_game = 0;
	_allBuildings = 0;
	_tavern = 0;
	
	_inside = new InsideBase( this );	
	
	_panel = new BaseRightPanel( this );
	FIXEDSIZE( _panel );

	_view = new InsideBaseView( _inside, this );
	
	_layVerticalView = 0;
	_layout = new QHBoxLayout( this );	

	updateDispositionMode();	
	
	connect( _view, SIGNAL( sig_building( GenericInsideBuilding * ) ), SLOT( slot_building( GenericInsideBuilding * ) ) );
	connect( _panel, SIGNAL( sig_quit() ), SIGNAL( sig_quit() ) );
	//connect( _baseButtons, SIGNAL( sig_exchange() ), _baseLords, SLOT( slot_exchange() ) );
	//connect( _baseLords, SIGNAL( sig_exchange( bool ) ), _baseButtons, SLOT( slot_exchange( bool ) ) );
}

DisplayBase::~DisplayBase()
{
}

void DisplayBase::updateDispositionMode()
{
	AttalSettings::DispositionMode mode = AttalSettings::getInstance()->getDispositionMode();
	
	_layout->removeWidget( _panel );
	if( _layVerticalView ) {
		_layout->removeItem( _layVerticalView );
		delete _layVerticalView;
	}
	_layVerticalView = new QVBoxLayout();

	switch( mode ) {
		case AttalSettings::DM_VERYCOMPACT:
			_view->setMinimumSize( 0, 0 );
			_view->setMaximumSize( 960, 720 );
			_view->setVScrollBarMode( Q3ScrollView::Auto );
			_view->setHScrollBarMode( Q3ScrollView::Auto );
			_layVerticalView->addWidget( _view, 1 );
			//layV1->addStretch( 1 );
		break;
		case AttalSettings::DM_COMPACT:
			_view->setMinimumSize( 0, 0 );
			_view->setMaximumSize( 960, 720 );
			_view->setVScrollBarMode( Q3ScrollView::Auto );
			_view->setHScrollBarMode( Q3ScrollView::Auto );
			_layVerticalView->addWidget( _view, 1 );
			//layV1->addStretch( 1 );
		break;
		case AttalSettings::DM_FULL:
			_view->setFixedSize( 960, 720 );
			_view->setVScrollBarMode( Q3ScrollView::AlwaysOff );
			_view->setHScrollBarMode( Q3ScrollView::AlwaysOff );
			_layVerticalView->addWidget( _view );
			_layVerticalView->addStretch( 1 );
		break;
	}

	_layout->addLayout( _layVerticalView, 1 );
	_layout->addWidget( _panel );
	_layout->activate();
}

void DisplayBase::handleSocket()
{
	if( _tavern ) {
		_tavern->handleSocket();
	}
		
	_game->handleSocket();
	_socket->reReadData();
	
	switch( _socket->getCla1() ) {
	case SO_MSG:
		break;
	case SO_GAME:
		break;
	case SO_TURN:
		break;
	case SO_MODIF:
		if( ( _socket->getCla2() == C_MOD_BASE ) &&
		    ( _socket->getCla3() == C_BASE_BUILDING ) )	{
			int row = _socket->readInt();
			int col = _socket->readInt();
			uchar level = _socket->readChar();
			bool create = (bool)_socket->readChar();
			if( ( _base->getCell()->getRow() == row ) &&
			    ( _base->getCell()->getCol() == col ) ) {
				if( create ) {
					GenericInsideBuilding * building =  _base->getBuildingByType( level );
					_inside->addBuilding( building );
				} else {
					_inside->delBuilding( level );
				}
			}
			if( _allBuildings ) {
				_allBuildings->updateView();
			}
		} else if( ( _socket->getCla2() == C_MOD_BASE ) && ( _socket->getCla3() == C_BASE_UNIT ) ) {
			reinit();
			emit sig_castle();
		}
		if( ( _socket->getCla2() == C_MOD_PLAYER ) && ( _socket->getCla3() == C_PLAY_RESS ) ) {
			emit sig_resource();
			reinit();
			/// XXX:_baseSummary->reinit();
		} else if( ( _socket->getCla2() == C_MOD_PLAYER ) && ( _socket->getCla3() == C_PLAY_POPUL ) ) {
			reinit();
		}
		break;
	case SO_QR:
		break;
	case SO_MVT:
		break;
	case SO_TECHNIC:
		break;
	case SO_EXCH:
		break;
	case SO_CONNECT:
		break;
	case SO_FIGHT:
		break;
	default:
		logEE( "Unknown socket_class" );
	}
}

void DisplayBase::slot_building( GenericInsideBuilding * building )
{
	if( _base ) {
		InsideBuildingModel * model = DataTheme.bases.at( _base->getRace() )->getBuildingModel( building->getLevel() );
		if( model->getAction() ) {
			switch( model->getAction()->getType() ) {
			case INSIDE_NONE:
				break;
			case INSIDE_VILLAGE:
				actionAllBuildings();
				break;
			case INSIDE_CASTLE:
				actionAllCreatures();
				break;
			case INSIDE_CREA:
				actionSomeCreatures( building );
				break;
			case INSIDE_MARKET:
				actionMarket();
				break;
			case INSIDE_TAVERN:
				actionTavern();
				break;
			default:
				logEE( "Unknown action %d", model->getAction()->getType() );
				break;
			}
		}
	}
}

void DisplayBase::actionAllBuildings()
{
	_allBuildings = new InsideActionAllBuildings( _base );
	if( _allBuildings ) {
		_allBuildings->setSocket( _socket );
		_allBuildings->setPlayer( _player );
		_allBuildings->exec();
		delete _allBuildings;
		_allBuildings = 0;
	}
	reinit();
}

void DisplayBase::actionAllCreatures()
{
	DisplayCreatureBase creatures( _base, this );
	creatures.initSocket( _socket );
	creatures.init( _player );
	creatures.exec();
	reinit();
}

void DisplayBase::actionSomeCreatures( GenericInsideBuilding * building )
{
	if( _base ) {
		InsideAction * action = DataTheme.bases.at( _base->getRace() )->getBuildingModel( building->getLevel() )->getAction();
		if( action ) {
			//InsideActionSomeCreature dialog( action->getParam( 0 ), action->getParam( 1 ) );
			BuyCreature dialog( this );
			dialog.initSocket( _socket );
			int race = action->getParam( 0 );
			int level = action->getParam( 1 );
			dialog.init( _player, _base, DataTheme.creatures.at( race, level ) );
			if( dialog.exec() ) {
				// try to buy some creature
			}
		}
	}
	reinit();
}

void DisplayBase::actionMarket()
{
	Market mark( this );

	mark.initPlayer( _player );
	mark.setPrices( _player->getPriceMarket());
	mark.initSocket( _socket );
	mark.exec();
	reinit();
}

void DisplayBase::actionTavern()
{
	_tavern = new Tavern( this );
	_tavern->initSocket( _socket );
	_tavern->initPlayer( _player );
	_tavern->start( _base );
	reinit();
}


void DisplayBase::setPlayer( Player * player )
{
	_player = player;
	_panel->setPlayer( player );
}

void DisplayBase::setBase( GenericBase * base )
{
	_base = base;
	_panel->setBase( base );
	_inside->setBase( base );
}

void DisplayBase::reinit()
{
	/// XXX:_baseSummary->reinit();
	_panel->reinit();
}

void DisplayBase::reinitBase()
{
	_panel->reinit();
	_inside->reinit();
	updateDispositionMode();
}

void DisplayBase::setSocket( AttalSocket * socket )
{
	_socket = socket;
	_panel->setSocket( socket );
}


//
// ----- BaseRightPanel -----
//

BaseRightPanel::BaseRightPanel( QWidget * parent, const char * name )
: QWidget( parent, name )
{
	_player = 0;
	_socket = 0;
	_base = 0;
	_displayLords = 0;

	_pbLord = new AttalButton( this );
	_pbLord->setFixedSize( 60, 60 );
	_popul = new QLabel( this );
	_popul->setAlignment( Qt::AlignHCenter );
	_totPopul = new QLabel( this );
	_totPopul->setAlignment( Qt::AlignHCenter );
	
	_lordLabel = new QLabel( this );
	_lordLabel->setText( "\n\n" );
	_lordLabel->setAlignment( Qt::AlignHCenter | Qt::WordBreak );
	_lordLabel->setMinimumWidth( 64 );
	_lordLabel->setMinimumHeight( _lordLabel->sizeHint().height() );
	
	_pbInfo = new QPushButton( this );
	_pbInfo->setText( "Info" );
	FIXEDSIZE( _pbInfo );
	_ress = new RessourceWin(this,tr(""),RS_VERT);
	
	_bases = new BaseButtons( this );
	
	/// XXX: use pixmap instead
	AttalButton * pbQuit = new AttalButton( this, AttalButton::BT_OK );
	FIXEDSIZE( pbQuit );

	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->setMargin( 5 );
	layout->setSpacing( 10 );
	layout->addWidget( _pbLord );
	layout->addWidget( _lordLabel );
	layout->addSpacing( 10 );
	layout->addWidget( _pbInfo );
	layout->addWidget( _ress );
	layout->addWidget( _popul, 0, Qt::AlignHCenter );
	layout->addWidget( _totPopul, 0, Qt::AlignHCenter );
	layout->addStretch( 1 );
	layout->addWidget( _bases );
	layout->addWidget( pbQuit );
	layout->activate();
	
	connect( pbQuit, SIGNAL( clicked() ), SIGNAL( sig_quit() ) );
	connect( _pbLord, SIGNAL( clicked() ), SLOT( slot_lords() ) );
	connect( _pbInfo, SIGNAL( clicked() ), SLOT( slot_info() ) );
}

void BaseRightPanel::slot_lords()
{
	_displayLords = new DisplayBaseLord( this );
	/// XXX: could be gathered
	_displayLords->init( _player, _base, _socket );
	_displayLords->exec();
	reinit();
	delete _displayLords;
	_displayLords = 0;
}
	
void BaseRightPanel::slot_info()
{
	DisplayBaseInfo displayInfo( this );
	displayInfo.init( _player, _base );
	displayInfo.exec();
}

void BaseRightPanel::setPlayer( Player * player )
{
	/// XXX: set lord pixmap
	_player = player;
	_ress->setPlayer(player);
}

void BaseRightPanel::setSocket( AttalSocket * socket )
{
	_socket = socket;
}

void BaseRightPanel::setBase( GenericBase * base )
{
	_base = base;
	reinit();
}

void BaseRightPanel::reinit()
{
	GenericLord * lord = 0;
	QString text;

	 text = "\nNo lord\n";
	if( _base->getGarrisonLord() ) {
		lord = _base->getGarrisonLord();
		text.sprintf( "Lord\n%s\n(Guarrison)", lord->getName().toLatin1().constData() );
	} else if( _base->getVisitorLord() ) {
		lord = _base->getVisitorLord();
		text.sprintf( "Lord\n%s\n(Visitor)", lord->getName().toLatin1().constData() );
	}
	if( lord ) {
		_pbLord->setPixmap( * _player->getLordPixmapById( lord->getId() ) );
	} else {
		/// XXX: should be optimized in imageTheme
		_pbLord->setPixmap( QPixmap( IMAGE_PATH + "/lords/noLord.png" ) );
	}
	
	_lordLabel->setText( text );
	if( _displayLords ) {
		_displayLords->reinit();
	}
	_ress->reinit();
	_popul->setText( QString(" Base Pop \n%1 ").arg( _base->getPopulation() ) );
	FIXEDSIZE( _popul );
	_totPopul->setText( QString(" Total Pop \n%1 ").arg( _player->getPopulation() ) );
}

//
// ----- BaseSummary -----
//

BaseSummary::BaseSummary( QWidget * parent, const char * name )
	: QWidget( parent, name )
{
	_base = 0;
	int i;
	_popul = new QLabel( this );

	QVBoxLayout * layout = new QVBoxLayout( this );
	
	QHBoxLayout * lay1 = new QHBoxLayout();
	_ress = new RessourceWin(this,tr(""),RS_GRID);
	lay1->addWidget( _ress );
	
	layout->addLayout( lay1 );
	layout->addStretch( 2 );
	layout->addWidget( _popul );
	
	QHBoxLayout * lay2 = new QHBoxLayout();
	lay2->addStretch( 1 );
	for( i = 0; i < 4; i++ ) {
		_unitIco[i] = new UnitSummary( this );
		lay2->addWidget( _unitIco[i] );
		lay2->addStretch( 1 );
	}
	
	layout->addLayout( lay2 );
	
	QHBoxLayout * lay3 = new QHBoxLayout();
	lay3->addStretch( 1 );
	for( i = 0; i < 4; i++ ) {
		_unitIco[4+i] = new UnitSummary( this );
		lay3->addWidget( _unitIco[4+i] );
		lay3->addStretch( 1 );
	}	
	
	layout->addLayout( lay3 );
	
	layout->addStretch( 1 );
	
	layout->activate();
	setFixedHeight( 240 );
}

void BaseSummary::setPlayer( Player * player )
{
	_player = player;

	_ress->setPlayer(player);
	for( int i = 0; i < 8; i++ ) {
		_unitIco[i]->setPlayer( player );
	}
}

void BaseSummary::reinit()
{
	_ress->reinit();
	_popul->setText(QString("  Population %1 ").arg(_base->getPopulation(),10,10));
}

void BaseSummary::setBase( GenericBase * base )
{
	_base = base;
	reinit();
}

//
// ----- DisplayBaseInfo  -----
//

DisplayBaseInfo::DisplayBaseInfo( QWidget * parent, const char * /* name */)
: QDialog( parent, Qt::Dialog )
{
	_widget = new BaseSummary( this );
	
	AttalButton * pbOk = new AttalButton( this, AttalButton::BT_OK );
	
	QHBoxLayout * layH1 = new QHBoxLayout();
	layH1->addStretch( 1 );
	layH1->addWidget( pbOk );
	layH1->addStretch( 1 );
	
	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->setSpacing( 5 );
	layout->setMargin( 5 );
	layout->addWidget( _widget );
	layout->addLayout( layH1 );
	layout->activate();	
	
	connect( pbOk, SIGNAL( clicked() ), SLOT( accept() ) );
}

void DisplayBaseInfo::init( Player * player, GenericBase * base )
{
	_widget->setBase( base );
	_widget->setPlayer( player );
}

void DisplayBaseInfo::reinit()
{
	_widget->reinit();
}

//
// ----- BaseLords -----
//

BaseLords::BaseLords( QWidget * parent, const char * name )
	: QWidget( parent, name )
{
	_base = 0;
	_isExchange = false;
	_socket = 0;
	
	setFixedHeight( 240 );	
	
	QVBoxLayout * layout = new QVBoxLayout( this );
	
	_garrison = new BaseTroop( true, this );
	layout->addWidget( _garrison, 1 );
	
	_visitor = new BaseTroop( false, this );
	layout->addWidget( _visitor, 1 );
	
	layout->activate();

	connect( _garrison, SIGNAL( sig_photo() ), SLOT( slot_photoGarrison() ) );
	connect( _garrison, SIGNAL( sig_unit( int ) ), SLOT( slot_unitGarrison( int ) ) );
	connect( _visitor, SIGNAL( sig_photo() ), SLOT( slot_photoVisitor() ) );
	connect( _visitor, SIGNAL( sig_unit( int ) ), SLOT( slot_unitVisitor( int ) ) );

	_garrisonSelected = false;
	_visitorSelected = false;
	_unitGarrison = -1;
	_unitVisitor = -1;
}


void BaseLords::setPlayer( Player * player )
{
	_player = player;

	_garrison->setPlayer( player );
	_visitor->setPlayer( player );
}

void BaseLords::setBase( GenericBase * base )
{
	_base = base;

	_garrison->setBase( base );
	_visitor->setBase( base );

	_garrisonSelected = false;
	_visitorSelected = false;
}

void BaseLords::reinit()
{
	_garrison->reinit();
	_visitor->reinit();
}

void BaseLords::slot_exchange()
{
	_isExchange = !_isExchange;
}

void BaseLords::slot_photoGarrison()
{
	unselectUnits();

	if( _garrisonSelected ) {
		/// XXX: show info lord
		_garrisonSelected = false;
	} else {
		if( _visitorSelected ) {
			exchangeLords();
		} else if( _base->isGarrisonLord() ) {
			_garrisonSelected = true;
		}
	}
}

void BaseLords::slot_photoVisitor()
{
	unselectUnits();
	
	if( _visitorSelected ) {
		/// XXX: show info lord
		_visitorSelected = false;
	} else {
		if( _garrisonSelected ) {
			exchangeLords();
		} else if( _base->isVisitorLord() ) {
			_visitorSelected = true;
		}
	}	
}

void BaseLords::exchangeLords()
{	
	_garrisonSelected = false;
	_visitorSelected = false;
	
	_base->exchangeLords();

	GenericLord * visitor = _base->getVisitorLord();
	_visitor->setLord( visitor );
	if( visitor ) {
		_socket->sendLordGarrison( visitor, false );
	}

	GenericLord * garrison = _base->getGarrisonLord();
	_garrison->setLord( garrison );
	if( garrison ) {
		_socket->sendLordGarrison( garrison, true );
	}
}

void BaseLords::exchangeUnitsGV()
{
	GenericLord * garrison = _base->getGarrisonLord();
	GenericLord * visitor = _base->getVisitorLord();

	if(visitor){
		if(garrison){
			_socket->sendExchangeUnit( garrison, _unitGarrison, visitor, _unitVisitor );
		} else {
			_socket->sendExchangeBaseUnit( _base, _unitGarrison, visitor, _unitVisitor );
		}
	}
	_garrison->reinit();
	_visitor->reinit();
}
	
void BaseLords::exchangeUnitsVG()
{
	GenericLord * garrison = _base->getGarrisonLord();
	GenericLord * visitor = _base->getVisitorLord();
	if(garrison){
		_socket->sendExchangeUnit( garrison, _unitGarrison, visitor, _unitVisitor );
	} else {
		_socket->sendExchangeBaseUnit( _base, _unitGarrison, visitor, _unitVisitor );
	}

	_garrison->reinit();
	_visitor->reinit();
}


void BaseLords::exchangeUnitsGG( int u1, int u2 )
{
	if( u1 == u2 ) {
		return;
	}
	
	GenericLord * garrison = _base->getGarrisonLord();
	if(garrison){
		_socket->sendExchangeUnit( garrison, u1, garrison, u2 );
	} else {
		_socket->sendExchangeBaseUnit( _base, u1, 0, u2 );
	}
	_garrison->reinit();
}

void BaseLords::exchangeUnitsVV( int u1, int u2 )
{
	if( u1 == u2 ) {
		return;
	}

	GenericLord * visitor = _base->getVisitorLord();
	_socket->sendExchangeUnit( visitor, u1, visitor, u2 );
	
	_visitor->reinit();
}

void BaseLords::slot_unitGarrison( int num )
{
	unselectPhoto();
	
	if( _isExchange ) {
		if( _unitGarrison != -1 ) {
		} else {
		}
	} else {
		if( ( _unitGarrison != -1 ) || ( _unitVisitor != -1 ) ) {
			if( _unitGarrison != -1 ) {
				exchangeUnitsGG( _unitGarrison, num );
			} else {
				_unitGarrison = num;
				exchangeUnitsVG();
			}
			unselectUnits();
		} else {
			if( _base->getGarrisonUnit( num ) != 0 ) {
				_unitGarrison = num;
				_garrison->selectUnit( _unitGarrison );
				emit sig_exchange( true );
			}
		}
	}
}

void BaseLords::slot_unitVisitor( int num )
{
	unselectPhoto();
	
	if( _isExchange ) {
		
	} else {
		if( ( _unitGarrison != -1 ) || ( _unitVisitor != -1 ) ) {
			if( _unitVisitor != -1 ) {
				exchangeUnitsVV( _unitVisitor, num );
			} else {
				_unitVisitor = num;
				exchangeUnitsGV();
			}
			unselectUnits();
		} else {
			if( _base->getVisitorUnit( num ) != 0 ) {
				_unitVisitor = num;
				_visitor->selectUnit( _unitVisitor );
				emit sig_exchange( true );
			}
		}
	}
}

void BaseLords::unselectUnits()
{
	_unitGarrison = -1;
	_garrison->unselectUnits();
	_unitVisitor = -1;
	_visitor->unselectUnits();
	emit sig_exchange( false );
}

void BaseLords::unselectPhoto()
{
	_garrisonSelected = false;
	_garrison->unselectPhoto();
	_visitorSelected = false;
	_visitor->unselectPhoto();
}

//
// ----- DisplayBaseLord -----
//

DisplayBaseLord::DisplayBaseLord( QWidget * parent, const char * /* name */)
: QDialog( parent, Qt::Dialog )
{
	_widget = new BaseLords( this );
	
	AttalButton * pbOk = new AttalButton( this, AttalButton::BT_OK );
	
	QHBoxLayout * layH1 = new QHBoxLayout();
	layH1->addStretch( 1 );
	layH1->addWidget( pbOk );
	layH1->addStretch( 1 );
	
	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->addWidget( _widget, 1 );
	layout->addLayout( layH1 );
	layout->activate();
	
	connect( pbOk, SIGNAL( clicked() ), SLOT( accept() ) );
}

void DisplayBaseLord::init( Player * player, GenericBase * base, AttalSocket * socket )
{
	_widget->setPlayer( player );
	_widget->setSocket( socket );
	_widget->setBase( base );
}

void DisplayBaseLord::reinit()
{
 _widget->reinit();
}

//
// ----- BaseButtons -----
//

BaseButtons::BaseButtons( QWidget * parent, const char * name )
	: QWidget( parent, name )
{
	QVBoxLayout * layout = new QVBoxLayout( this );
	
	_pbExch = new QPushButton( this );
	_pbExch->setFixedSize( 50, 40 );
	layout->addWidget( _pbExch );
	
	layout->addStretch( 1 );
		
	layout->activate();
	setFixedHeight( 240 );	
	
	_pbExch->setEnabled( false );
	
	connect( _pbExch, SIGNAL( clicked() ), SIGNAL( sig_exchange() ) );
}

void BaseButtons::slot_exchange( bool st )
{
	if( st ) {
		_pbExch->setEnabled( true );
	} else {
		_pbExch->setEnabled( false );
	}
}

//
// ----- UnitSummary -----
//

UnitSummary::UnitSummary( QWidget * parent, const char * name )
	: QWidget( parent, name )
{
	_ico = new QLabel( this );
	_ico->setFixedSize( 50, 40 );
	_nb = new QLabel( this );
	_nb->setFixedSize( 50, 20 );
	_nb->setAlignment( Qt::AlignCenter );
	_nb->move( 0, 40 );

	setFixedSize( 50, 60 );
}

void UnitSummary::setCreature( int /*id*/ )
{
	/// XXX: TODO
	logEE( "not yet implemented" );
}

void UnitSummary::setNumber( int num )
{
	if( num == 0 ) {
		_nb->setText( "" );
	} else {
		_nb->setText( "+" + QString::number( num ) );
	}
}

//
// ----- BaseTroop -----
//

BaseTroop::BaseTroop( bool isGarrison, QWidget * parent, const char * name )
	: QWidget( parent, name )
{
	_isGarrison = isGarrison;
	_lord = 0;

	QHBoxLayout * layout = new QHBoxLayout( this );
	QVBoxLayout * layV2 = new QVBoxLayout();

	_photo = new Icon( this );
	_name = new QLabel( this );
	_name->setFixedSize( 60, 20 );
	layV2->addWidget( _photo );
	layV2->addWidget( _name );
	layout->addLayout( layV2 );
	layout->addStretch( 1 );
	QVBoxLayout * layV[ MAX_UNIT ];

	QSignalMapper * sigmap = new QSignalMapper( this );
	for( int i = 0; i < MAX_UNIT; i++ ) {
		layV[i] = new QVBoxLayout();
		_units[i] = new Icon( this );
		_numUnit[i] = new QLabel( this );
		_numUnit[i]->setFixedSize( 60, 20 );
		_numUnit[i]->setAlignment( Qt::AlignCenter );
		layV[i]->addWidget( _units[i] );
		layV[i]->addWidget( _numUnit[i] );
		layout->addLayout( layV[i] );
		sigmap->setMapping( _units[i], i );
		connect( _units[i], SIGNAL( sig_clicked() ), sigmap, SLOT( map() ) );
	}
	layout->addStretch( 1 );

	layout->activate();

	connect( _photo, SIGNAL( sig_clicked() ), SIGNAL( sig_photo() ) );
	connect( sigmap, SIGNAL( mapped( int ) ), SIGNAL( sig_unit( int ) ) );
}

void BaseTroop::clear()
{
	_lord = 0;
	_photo->clear();
	_name->clear();

	for( int i = 0; i < MAX_UNIT; i++ ) {
		_units[i]->clear();
		_numUnit[i]->clear();
	}
}

void BaseTroop::setLord( GenericLord * lord )
{
	_lord = lord;
	reinit();
}

void BaseTroop::setBase( GenericBase * base )
{
	_base = base;
	if( _isGarrison ) {
		_lord = base->getGarrisonLord();
	} else {
		_lord = base->getVisitorLord();
	}
	reinit();
}

void BaseTroop::reinit()
{
	QString text;

	if( _lord == 0 ) {
		clear();
		if( _isGarrison ) {
			_name->setText( _base->getName());
			for( int i = 0; i < MAX_UNIT; i++ ) {
				if( _base->getUnit( i ) ) {
					_units[i]->setPixmap( QPixmap( ImageTheme.getPhotoCreature( _base->getUnit( i ) ) ) );
					_numUnit[i]->setText( QString::number( _base->getUnit( i )->getNumber() ) );
					text = _base->getUnit(i)->showDescription();
					_units[i]->setPopupText( text  );
				} else {
					_numUnit[i]->clear();
				}
			}
		}
		return;
	}
	
	_photo->setPixmap( *_player->getLordPixmapById( _lord->getId() ) );
	_name->setText( _lord->getName());
	
	for( int i = 0; i < MAX_UNIT; i++ ) {
		if( _lord->getUnit( i ) ) {
			_units[i]->setPixmap( QPixmap( ImageTheme.getPhotoCreature( _lord->getUnit( i ) ) ) );
			_numUnit[i]->setText( QString::number( _lord->getUnit( i )->getNumber() ) );
			text = _lord->getUnit(i)->showDescription();
			_units[i]->setPopupText( text  );
		} else {
			_units[i]->clear();
			_numUnit[i]->clear();
		}
	}	
}

