/****************************************************************
**
** Attal : Lords of Doom
**
** clientInterface.cpp
** Manages the whole game
**
** Version : $Id: clientInterface.cpp,v 1.82 2006/08/09 11:36:00 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo
**
** Date : 17/08/2000
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/


#include "clientInterface.h"

// generic include files
// include files for QT
#include <QAction>
#include <QApplication>
#include <QCloseEvent>

#include <QFileDialog>
#include <QLayout>
#include <QLabel>
#include <QMenuBar>
#include <QMessageBox>
#include <QMenu>
#include <QProcess>
#include <QPushButton>
#include <QSignalMapper>
#include <QStatusBar>
#include <QTimer>

#include <Q3Canvas>

// application specific includes
#include "conf.h"
#include "client/clientWidgets.h"
#include "libCommon/attalSettings.h"
#include "libCommon/attalSocket.h"
#include "libCommon/log.h"
#include "libCommon/skill.h"
#include "libCommon/unit.h"
#include "libCommon/dataTheme.h"

#include "libClient/aboutDialog.h"
#include "libClient/attalStyle.h"
#include "libClient/building.h"
#include "libClient/chatWidget.h"
#include "libClient/displayLord.h"
#include "libClient/displayBase.h"
#include "libClient/gainLevel.h"
#include "libClient/game.h"
#include "libClient/gui.h"
#include "libClient/imageTheme.h"
#include "libClient/optionsDialog.h"

#include "libFight/fight.h"
#include "libFight/fightResult.h"

#include "libServer/attalServer.h"
#include "libServer/engine.h"
#include "libServer/campaign.h"

#include "libAi/analyst.h"

extern QString DATA_PATH;
extern QString IMAGE_PATH;
extern QString THEME;
extern QString SCENARIO_PATH;
extern QString CAMPAIGN_PATH;
extern QString SAVE_PATH;
extern QString PORT;


extern DataTheme DataTheme;
extern ImageTheme ImageTheme;




//
// ----- ClientInterface -----
//

ClientInterface::ClientInterface()
{
	_socket = 0;
	_fight = 0;
	_level = 0;
	_base = 0;
	_config = 0;
	_state = SG_MAP;
	_mini = true;
	_full = true;
	_inLoad = false;
	_help = 0;
	_proc = 0;
	_server = 0;
	_engine = 0;
	_actScen = 0;
	_nbScen = 1;
	_inLoad = false;
	_winner = true;
	_ready = true;

	qApp->setStyle( new AttalStyle( DATA_PATH + "style.dat" ) );

	initActions();
	initMenuBar();
	initStatusBar();
	_centralWindow = new QStackedWidget(this);
	setCentralWidget( _centralWindow );
	loadMenu();

	if( DataTheme.init() && ImageTheme.init() ) {
		_game = new Game( _centralWindow );
		_centralWindow->addWidget( _game );
		_centralWindow->setCurrentWidget( _game );

		connect( _game, SIGNAL( sig_statusMsg( const QString & ) ), SLOT( slot_status( const QString & ) ) );
		connect( _game, SIGNAL( sig_base( GenericBase * ) ), SLOT( slot_base( GenericBase * ) ) );
		connect( _game, SIGNAL( sig_fight( GenericLord *, CLASS_FIGHTER ) ), SLOT( slot_fight( GenericLord *, CLASS_FIGHTER ) ) );
		connect( _game, SIGNAL( sig_endGame() ),this, SLOT( slot_endGame() ) );
		connect( _game, SIGNAL( sig_exchange() ), SLOT( slot_exchange() ) );
		connect( _game, SIGNAL( sig_quit() ), SLOT( slot_quit() ) );
		connect( _game, SIGNAL( sig_options() ), SLOT( slot_options() ) );
	 
		_startDialog = new StartGameDialog(this);
 
		connect( _startDialog, SIGNAL( sig_newScen()), this, SLOT (slot_newScen()));
		connect( _startDialog, SIGNAL( sig_loadScen()), this, SLOT (slot_loadScen()));
		connect( _startDialog, SIGNAL( sig_newCamp()), this, SLOT (slot_newCamp()));
		connect( _startDialog, SIGNAL( sig_loadCamp()), this, SLOT (slot_loadCamp()));
		connect( _startDialog, SIGNAL( sig_dialogClosed()), this, SLOT (slot_dialogClosed()));
	} else {
		/// XXX: we could manage this better (later :) )
		QMessageBox::critical( this, tr( "Can't load theme" ), tr( "Theme " ) + THEME + tr( " has not been loaded successfully" ) );
	}
	//XXX here insert main menu
	if(_msg) {
		delete _msg;
		_msg = 0;
	}
}

ClientInterface::~ClientInterface()
{
	if( _socket ) {
		actionDisconnect( QUIT_GAME );
	}
	if( _config ) {
		delete _config;
	}
	
	delete _game;
	
	if( _fight ) {
		delete _fight;
	}
	if( _base ) {
		delete _base;
	}
	//free before ImageTheme, after DataTheme (clear of ImageTheme depend on DataTheme data)
	ImageTheme.clear();
	DataTheme.clear();
}

void ClientInterface::changeEvent ( QEvent * e )
{
	//used to replace setCaption
	switch( e->type() ) {
		case QEvent::WindowTitleChange:
			setWindowTitle( tr( "Attal - Lords of Doom" ) );
		break;
		default:
			QWidget::changeEvent ( e );
		break;
	}
}

void ClientInterface::closeEvent( QCloseEvent * event )
{
	if( actionQuit() ) {
		event->accept();
	} else {
		event->ignore();
	}
}

QAction * ClientInterface::addAction( const QString & label, const QKeySequence & shortcut, MENU_ACTIONS id, QSignalMapper * sigmap )
{
	QAction * action;

	action = new QAction( label, this );
	action->setShortcut( shortcut );
	_actions.insert( id, action );
	sigmap->setMapping( action, id );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );

	return action;
}

void ClientInterface::initActions()
{
	_actions.resize( NB_ACTIONS );
	
	QAction * action;
	QSignalMapper * sigmap = new QSignalMapper( this );

	addAction( tr( "&Connect to server" ), QKeySequence( tr( "CTRL+C" ) ), ACTION_CONNECT, sigmap );
	addAction( tr( "&Start game" ), QKeySequence( tr( "CTRL+N" ) ), ACTION_START, sigmap );
	action = addAction( tr( "&End game" ), QKeySequence( tr( "CTRL+E" ) ), ACTION_END, sigmap );
	action->setDisabled( true );
	action = addAction( tr( "&Save game" ), QKeySequence( tr( "CTRL+W" ) ), ACTION_SAVE, sigmap );
	action->setDisabled( true );
	addAction( tr( "Fast &Server" ), QKeySequence( tr( "CTRL+S" ) ), ACTION_FASTCONNECT, sigmap );
	action = addAction( tr( "&Disconnect" ), QKeySequence( tr( "CTRL+D" ) ), ACTION_DISCONNECT, sigmap );
	action->setDisabled( true );
	addAction( tr( "&Mini map" ), QKeySequence( tr( "CTRL+M" ) ), ACTION_MINIMAP, sigmap );
	addAction( tr( "&Fullscreen" ), QKeySequence( tr( "CTRL+F" ) ), ACTION_FULL, sigmap );
	addAction( tr( "Next &Lord" ), QKeySequence( tr( "CTRL+L" ) ), ACTION_NEXTLORD, sigmap );
	addAction( tr( "Next &Base" ), QKeySequence( tr( "CTRL+B" ) ), ACTION_NEXTBASE, sigmap );
	addAction( tr( "&Help" ), QKeySequence( tr( "Key_F1" ) ), ACTION_HELP, sigmap );
	addAction( tr( "&About" ), QKeySequence( tr( "CTRL+A" ) ), ACTION_ABOUT, sigmap );
	addAction( tr( "&Quit" ), QKeySequence( tr( "CTRL+Q" ) ), ACTION_QUIT, sigmap );
	action = addAction( tr( "Free size" ), QKeySequence( "" ), ACTION_FREESIZE, sigmap );
	action->setOn( true );
	addAction( tr( "800x600" ), QKeySequence( "" ), ACTION_800, sigmap );
	addAction( tr( "1024x768" ), QKeySequence( "" ), ACTION_1024, sigmap );
	addAction( tr( "1280x1024" ), QKeySequence( "" ), ACTION_1280, sigmap );
	/// XXX: we have to use QActionGroup for switch between size of screen

	addAction( tr( "Test Style" ), QKeySequence( "CTRL+T" ), ACTION_TESTSTYLE, sigmap );

	connect( sigmap, SIGNAL( mapped( int ) ), SLOT( slot_action( int ) ) );
}

void ClientInterface::initMenuBar()
{
	QMenu * menuFile = menuBar()->addMenu( tr( "&File" ) );
	QMenu * menuGame = menuBar()->addMenu( tr( "&Game" ) );
	QMenu * menuScreen = menuBar()->addMenu( tr( "&Screen" ) );
	QMenu * menuHelp = menuBar()->addMenu( tr( "&Help" ) );

	menuFile->addAction( _actions[ ACTION_START ] );
	menuFile->addAction( _actions[ ACTION_END ] );
	menuFile->addAction( _actions[ ACTION_SAVE ] );
	menuFile->addAction( _actions[ ACTION_CONNECT ] );
	menuFile->addAction( _actions[ ACTION_FASTCONNECT ] );
	menuFile->addAction( _actions[ ACTION_DISCONNECT ] );
	menuFile->addAction( _actions[ ACTION_QUIT ] );

	menuGame->addAction( _actions[ ACTION_MINIMAP ] );
	menuGame->addAction( _actions[ ACTION_FULL ] );
	
	menuScreen->addAction( _actions[ ACTION_FREESIZE ] );
	menuScreen->addAction( _actions[ ACTION_800 ] );
	menuScreen->addAction( _actions[ ACTION_1024 ] );
	menuScreen->addAction( _actions[ ACTION_1280 ] );
#ifdef TEST
	menuScreen->addAction( _actions[ ACTION_TESTSTYLE ] );
#endif
	
	menuHelp->addAction( _actions[ ACTION_HELP ] );
	menuHelp->addAction( _actions[ ACTION_ABOUT ] );

}

void ClientInterface::slot_action( int num )
{
	switch( num ) {
	case ACTION_START:
		actionStart();
		break;
	case ACTION_END:
		if( _engine) {
			_actScen = 0;
			actionDisconnect(QUIT_GAME);
		}
		break;
	case ACTION_SAVE:
		actionSave();
		break;
	case ACTION_CONNECT:
		actionConnect(false);
		break;
	case ACTION_FASTCONNECT:
		actionFastConnect();
		break;
	case ACTION_DISCONNECT:
		actionDisconnect(END_GAME);
		break;
	case ACTION_MINIMAP:
		actionMinimap();
		break;
	case ACTION_FULL:
		actionFullScreen();
		break;
	case ACTION_NEXTLORD:
		actionNextLord();
		break;
	case ACTION_NEXTBASE:
		actionNextBase();
		break;
	case ACTION_HELP:
		actionHelp();
		break;
	case ACTION_ABOUT:
		actionAbout();
		break;
	case ACTION_QUIT:
		actionQuit();
		break;
		
	case ACTION_FREESIZE:
		actionResize( SIZEMODE_FREE );
		break;
	case ACTION_800:
		actionResize( SIZEMODE_800 );
		break;
	case ACTION_1024:
		actionResize( SIZEMODE_1024 );
		break;
	case ACTION_1280:
		actionResize( SIZEMODE_1280 );
		break;
	case ACTION_TESTSTYLE:
		actionTestStyle();
		break;
	}
}

void ClientInterface::slot_exchange()
{
	if( _base ) {
		_base->reinit();
	}
}

void ClientInterface::slot_result( bool result )
{
	_winner = result;
}

void ClientInterface::endEngine()
{
	killAI();
	_actScen = 0;
	_nbScen = 1;
	_winner = true;

	if(_server){
		delete _server;
		_server = 0;
	}
	if(_engine){
		delete _engine;
		_engine = 0;
	}
	
	_actions[ ACTION_START ]->setEnabled( true );
	_actions[ ACTION_END ]->setEnabled( false );
	_actions[ ACTION_SAVE ]->setEnabled( false );

}

void ClientInterface::slot_endGame()
{
	if( _actScen == 0 || ( _actScen == _nbScen) ) {
		endEngine();
	}

	//static int count = 0;
	_state = SG_MAP;
	_game->show();
	
	if( _fight ) {
		_fight->hide();
		delete _fight;
		_fight = 0;
	}
	if( _base ) {
		_base->hide();
		delete _base;
		_base = 0;
	}
}

void ClientInterface::slot_dialogClosed()
{
	actionDisconnect( QUIT_GAME );
	endEngine();
}

void ClientInterface::actionFastConnect()
{

	_proc = new QProcess(this);
	if( ! _proc ) {
		actionQuit();
	}
	QStringList arglist;
	
	arglist.append( "--fast" );

#ifdef WIN32
	_proc->start( "attal-server.exe", arglist );
#else
	QFile file("./attal-server");

	if (file.exists()) {
		_proc->start( "./attal-server", arglist );
	} else {
		_proc->start( "attal-server", arglist );
	}
#endif
	while( !_proc->waitForStarted()) {}
	QTimer::singleShot( 2000, this, SLOT( slot_clientConnect() ) );
}

void ClientInterface::slot_clientConnect()
{
	actionConnect( true );
}

void ClientInterface::actionConnect(bool fast)
{
	if( !_socket ) {
		if( _config == 0 ) {
			_config = new ConfigConnection( this );
		}
		_config->setHost( "localhost" );
		_config->setPort( PORT.toInt() );
		_config->setPlayerName( _game->getGamePlayer()->getName() );
		if(fast){
			_config->accept();
		} else { 
			_config->exec();
		}
		if( _config->result() == QDialog::Accepted ) {
			_socket = new AttalSocket();
			_game->setSocket( _socket );
			_game->setPlayerName( _config->getPlayerName() );
			connect( _socket, SIGNAL( readyRead() ), SLOT( slot_readSocket() ) );
			connect( _socket, SIGNAL( hostFound() ), SLOT( slot_hostfound() ) );
			connect( _socket, SIGNAL( connectionClosed() ), SLOT( slot_connectionClosed() ) );
			connect( _socket, SIGNAL( error( QAbstractSocket::SocketError ) ), SLOT( slot_error( QAbstractSocket::SocketError ) ) );
			_socket->connectToHost( _config->getHost(), _config->getPort() );
			_actions[ ACTION_START ]->setEnabled( false );
			_actions[ ACTION_END ]->setEnabled( false );
			_actions[ ACTION_SAVE ]->setEnabled( false );
			_actions[ ACTION_CONNECT ]->setEnabled( false );
			_actions[ ACTION_DISCONNECT ]->setEnabled( true );
			_actions[ ACTION_FASTCONNECT ]->setEnabled(false);
		}
	} else {
		QMessageBox::critical( this, tr( "Can't connect" ),
			tr( "You're already connected to a server. Please disconnect first." ) );
	}
}

void ClientInterface::slot_hostfound()
{
	if(_game && _game->getChat()){	
		_game->getChat()->newMessage(QString ("Host found , connecting... "));
	}	
}

void ClientInterface::slot_error( QAbstractSocket::SocketError error )
{
	logEE("Cannot connect to server");
	switch( error ) {
		case QAbstractSocket::ConnectionRefusedError: 
			{
			logEE( "Connection Refused" );
			if( _game && _game->getChat() ) {	
				_game->getChat()->newMessage( QString( "Connection Refused" ) );
			}	
			actionDisconnect(QUIT_GAME);
			return;
			}
			break;
		case QAbstractSocket::HostNotFoundError:
			logEE( "Host not found" );
			if( _game && _game->getChat() ) {	
				_game->getChat()->newMessage( QString( "Host not found" ) );
			}	
			break;
		case QAbstractSocket::RemoteHostClosedError:  {
			logEE("The remote host closed the connection.");	
			}
			break;
		case QAbstractSocket::SocketAccessError:   logEE("The socket operation failed because the application lacked the required privileges.");	break;
		case QAbstractSocket::SocketResourceError:   logEE("The local system ran out of resources (e.g., too many sockets).");	break;
		case QAbstractSocket::SocketTimeoutError:    logEE("The socket operation timed out.");	break;
		case QAbstractSocket::DatagramTooLargeError:    logEE("The datagram was larger than the operating system's limit (which can be as low as 8192 bytes).");	break;
		case QAbstractSocket::NetworkError:    logEE("An error occurred with the network (e.g., the network cable was accidentally plugged out).");	break;
		case QAbstractSocket::AddressInUseError:    logEE("The address specified to QUdpSocket::bind() is already in use and was set to be exclusive.");	break;
		case QAbstractSocket::SocketAddressNotAvailableError:    logEE("The address specified to QUdpSocket::bind() does not belong to the host.");	break;
		case QAbstractSocket::UnsupportedSocketOperationError:   logEE("The requested socket operation is not supported by the local operating system (e.g., lack of IPv6 support).");	break;
		default: {
			logEE( "Other socket error occured" );
			actionDisconnect(QUIT_GAME);
			if( _game && _game->getChat() ) {	
				_game->getChat()->newMessage( QString( "Other socket error occured" ) );
			}	
			return;
						 }
			break;
	}

	actionDisconnect(QUIT_GAME_SOCKET_CLOSED);
}

void ClientInterface::slot_connectionClosed()
{
	logEE( "Server disconnect" );
	if(_game && _game->getChat()){	
		_game->getChat()->newMessage(QString ("Server Disconnect "));
	}	
	killServer();
}

void ClientInterface::killServer()
{
	if(_proc) {
		_proc->close();
		//QTimer::singleShot( 5000, _proc, SLOT( kill() ) );
		_proc->terminate();
		_proc->waitForFinished();
	}
}

void ClientInterface::actionSave()
{
	if(_engine) {
		QString filen;
		filen = QFileDialog::getSaveFileName( this ,"", SAVE_PATH , "*.gam" );

		if(!filen.contains(".gam")){
			filen.append(".gam");
		}
		if (!filen.isNull()) {
			_engine->saveGame( filen );
		}

		if(_engine->getCampaign()) {
			QString filenamecmp = filen;
			QString filename;

			filenamecmp.remove(".gam");
			filenamecmp.append(".cms");	
			//filenamecmp = QFileDialog::getSaveFileName( this, "",CAMPAIGN_PATH ,  "*.cms" );

#ifdef WIN32
			filename = filen.section("\\",-1,-1);
#else
			filename = filen.section("/",-1,-1);
#endif

			QFile f( filenamecmp );

			if (! f.open(QIODevice::WriteOnly) ) {
				logEE( "Could not open file %s for writing\n", filename.toLatin1().constData() );
				return;
			}
			Campaign  * campaign = new Campaign;

			QTextStream ts( &f );

			uint nbScen = _engine->getCampaign()->getScenarioNumber();

			campaign->setCurrentScenario(_actScen);

			for( uint i = 0; i < nbScen; i++ ) {
				if(i == _actScen){
					campaign->addScenario( filename );
				} else {
					campaign->addScenario( _engine->getCampaign()->getScenario(i) );
				}
			}

			campaign->setTheme( _engine->getCampaign()->getTheme() );
			campaign->setName( _engine->getCampaign()->getName() );
			campaign->setDescription( _engine->getCampaign()->getDescription() );
			campaign->save( & ts );

			f.close();
		}
	}
}

void ClientInterface::actionStart()
{
	_server = new AttalServer( PORT.toInt() );

	if( _server->isListening() ) {

		_engine = new Engine( _server );
		connect( _engine, SIGNAL( sig_newPlayer( AttalPlayerSocket * ) ), this, SLOT( slot_ready() ) );
		connect( _engine, SIGNAL( sig_result(bool) ), this, SLOT( slot_result(bool)) );
		//connect( _engine, SIGNAL( sig_endGame() ), this, SLOT( slot_endGame()) );

	} else {
		delete _server;
		_server = 0;
	}

	actionConnect( true );
		
	_startDialog->show();

}

void ClientInterface::slot_newScen()
{
	DisplayScenariiDialog * scen = new DisplayScenariiDialog( this );
	if( scen->exec() ) {
		_startDialog->hide();
		load( scen->getFileName() );
	} 
}

void ClientInterface::slot_newCamp()
{
	QString filename;
	filename = QFileDialog::getOpenFileName( this, tr( "Load campaign" ), CAMPAIGN_PATH, "*.cmp" );
	if ( !filename.isEmpty() ) {
		_startDialog->hide();
		loadCampaign( filename );
	}
}

void ClientInterface::slot_loadScen()
{
	QString filename;
	filename = QFileDialog::getOpenFileName( this, tr( "Load game" ), SAVE_PATH, "*.scn *.gam" );
	if ( !filename.isEmpty() ) {
		_startDialog->hide();
		load( filename );
	}
}

void ClientInterface::slot_loadCamp()
{
	QString filename;
	filename = QFileDialog::getOpenFileName( this, tr( "Load campaign" ), SAVE_PATH, "*.cms" );
	if ( !filename.isEmpty() ) {
		_startDialog->hide();
		loadCampaign( filename );
	}
}

void ClientInterface::load(QString filename)
{
	if( ( _server->getNbSocket() > 0 ) && ( !filename.isNull() ) ) {
		/* semaphore to avoid process 2 signal at same time */
		if(!_inLoad) {
			_inLoad = true;
			fillWithAI(filename);
			if( _engine->loadGame( filename  , false) ) {
				_actions[ ACTION_START ]->setEnabled( false );
				_actions[ ACTION_END ]->setEnabled( true );
				_actions[ ACTION_SAVE ]->setEnabled( true );
				_actions[ ACTION_DISCONNECT ]->setEnabled( false );
				_engine->startGame();
			}
			_inLoad = false;
		}
	}

}

void ClientInterface::loadCampaign(QString fileName)
{
	QString currentPath;
		
	if( ( _server->getNbSocket() == 1 ) && ( ! fileName.isNull() ) ) {
		if( _engine->loadCampaign( fileName ) ) {
			if( _engine->getCampaign() ) {
				QString filename;
				_nbScen = _engine->getCampaign()->getScenarioNumber();
				uint i;
				for( i = _engine->getCampaign()->getCurrentScenario() ; i < _nbScen; i++ ) {
					currentPath = CAMPAIGN_PATH;
					_actScen = i;
					killAI();
					filename = _engine->getCampaign()->getScenario( i );
					if(filename.contains(".gam")){
						currentPath = SAVE_PATH;
					}

					if(_winner) {
						fillWithAI( currentPath + filename );
//restart:
						if( _engine->loadGame( currentPath + filename , false) ) {
							//keep these instruction in right order (otherwise, various bugs)
							_winner = false;
							_actions[ ACTION_START ]->setEnabled( false );
							_actions[ ACTION_END ]->setEnabled( true );
							_actions[ ACTION_SAVE ]->setEnabled( true );
							_actions[ ACTION_DISCONNECT ]->setEnabled( false );
							_engine->startGame();
						} else {
							//QMessageBox msb( tr( "Problem" ), tr( "Do you want to continue campaign (control right number of AI)?" ), QMessageBox::Warning, QMessageBox::Yes | 			QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, 0, this );
							//if( msb.exec() == QMessageBox::Yes){
							//	goto restart;
							//}
						}
					} else {
						_winner = true;
						break;
					}
				}
				_actScen = 0;
				_nbScen = 1;
				_engine->cancelCampaign();
			}
		}
	}

}

void ClientInterface::addInternalAI()
{
	QStringList arglist;
	Analyst * ai = new Analyst();
	AttalSocket * socket = new AttalSocket;
	socket->connectToHost( "localhost", PORT.toInt() );
	ai->aiConnect( socket );
	_aiList.append( ai );
}

bool ClientInterface::fillWithAI(QString filename)
{
	//logDD("filename %s",filename.toLatin1().constData());

	if ( _engine->loadGame(filename, true ) == false ) {
		int nplay = _engine->getNumFillPlayers();
		//logDD("nplay %d",nplay);
		if( nplay > 0 ) {
			//QTimer::singleShot( (nplay+2)* 1000, this, SLOT( slot_ready() ) );
			for( int i = 0; i < nplay; i++ ) {
				_ready = false;
				//logDD("ai %d",i);
				addInternalAI();	
				while( !_ready){
					qApp->processEvents();
				}
			}	
			return true;
		}
	}
	return false;
}

void ClientInterface::slot_ready()
{
	_ready =  true;
}

bool ClientInterface::killAI()
{
	int count;
	Analyst * ai;
		
	count = _aiList.count();
	if( _aiList.count() > 0 ) {
		for( int i = 0; i < count; i++ ) {
				ai = _aiList.takeFirst();
				delete ai->getSocket();
				delete ai;
		}
		return true;	
	}

	return false;
}

void ClientInterface::actionDisconnect(int choice)
{

	if(choice==END_GAME){
		QMessageBox msb( tr( "Match" ), tr( "Do you want abandon the match ?" ), QMessageBox::Warning, QMessageBox::Yes | 			QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, 0, this );
		if( msb.exec() == QMessageBox::Yes){
			choice = QUIT_GAME;
		} else {
			return;
		}
	}

	updateScreen();
	_game->setSocket( 0 );
	if( _socket ) {
		if(choice!=QUIT_GAME_SOCKET_CLOSED) {
			_socket->close();
			delete _socket;
		}
		_socket = 0;
	}
	_game->endGame();
	if( !_socket ) {
		_actions[ ACTION_START ]->setEnabled( true );
		_actions[ ACTION_END ]->setEnabled( false );
		_actions[ ACTION_SAVE ]->setEnabled( false );
		_actions[ ACTION_CONNECT ]->setEnabled( true );
		_actions[ ACTION_DISCONNECT ]->setEnabled( false );
		_actions[ ACTION_FASTCONNECT ]->setEnabled( true );
	}

	_game->getChat()->newMessage(QString ("Disconnect from server"));
}

void ClientInterface::actionMinimap()
{
	if( _game ) {
		_mini = ! _mini;
		_game->displayMiniMap( _mini );
		_actions[ ACTION_MINIMAP ]->setOn( _mini );
		_actions[ ACTION_FULL ]->setEnabled( _mini );
	}
}

void ClientInterface::actionFullScreen()
{
	if( _game ) {
		_full = ! _full;
		_game->displayFullScreen( _full );
		_actions[ ACTION_FULL ]->setOn( _full );
		_actions[ ACTION_MINIMAP ]->setEnabled( _full );
	}
}

void ClientInterface::actionNextLord()
{
	if( _game ) {
		_game->nextLord();
	}
}

void ClientInterface::actionNextBase()
{
	if( _game ) {
		_game->nextBase();
	}
}

void ClientInterface::actionHelp()
{
	if( ! _help ) {
		_help = new DisplayHelp( this );
	}
	_help->resize( 800, 600 );
	_help->show();
}

void ClientInterface::actionAbout()
{
	AboutDialog dialog;
	dialog.exec();
}


bool ClientInterface::actionQuit()
{
	bool ret = true;
	QMessageBox msb( tr("Are you sure ?"), tr("Do you really want to quit?"), QMessageBox::Warning, QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, 0, this );
	if ( msb.exec() == QMessageBox::Yes ) {
		if( _socket ) {
			_game->setSocket( 0 );
			_socket->close();
			delete _socket;
			_socket = 0;
		}
		_game->endGame();
		AttalSettings::getInstance()->save();
		qApp->exit(0);
	} else {
		ret = false;
	}	

	return ret;
}

void ClientInterface::actionResize( SizeMode mode )
{
	_actions[ ACTION_FREESIZE ]->setOn( false );
	_actions[ ACTION_800 ]->setOn( false );
	_actions[ ACTION_1024 ]->setOn( false );
	_actions[ ACTION_1280 ]->setOn( false );
	
	switch( mode ) {
	case SIZEMODE_FREE:
		_actions[ ACTION_FREESIZE ]->setOn( true );
		setMinimumSize( 0, 0 );
		setMaximumSize( 10000, 10000 );
		AttalSettings::getInstance()->setDispositionMode( AttalSettings::DM_FULL );
		_game->updateDispositionMode();
		break;
	case SIZEMODE_800:
		_actions[ ACTION_800 ]->setOn( true );
		setFixedSize( 800, 600 );
		AttalSettings::getInstance()->setDispositionMode( AttalSettings::DM_VERYCOMPACT );
		_game->updateDispositionMode();
		break;
	case SIZEMODE_1024:
		_actions[ ACTION_1024 ]->setOn( true );
		setFixedSize( 1024, 768 );
		AttalSettings::getInstance()->setDispositionMode( AttalSettings::DM_COMPACT );
		_game->updateDispositionMode();
		break;
	case SIZEMODE_1280:
		_actions[ ACTION_1280 ]->setOn( true );
		setFixedSize( 1280, 1024);
		AttalSettings::getInstance()->setDispositionMode( AttalSettings::DM_FULL );
		_game->updateDispositionMode();
		break;
	}
}

void ClientInterface::actionTestStyle()
{
#ifdef TEST
	TestAttalStyle test;
	test.exec();
#endif
}

void ClientInterface::slot_status( const QString & text )
{
	statusBar()->showMessage( text, 0 );
}

void ClientInterface::initStatusBar()
{
	statusBar()->showMessage( tr( "Status Bar" ), 0 );
}

void ClientInterface::loadMenu()
{
	//XXX : don't display text, but a good start
	_msg = new GameMessage();
	_msg->setCaption( tr("Loading Data...") );
	_msg->addText( tr("Loading Data...") );
	_msg->resize( 300, 70 );
	_msg->setModal(false);
	_msg->show();
}

void ClientInterface::slot_base( GenericBase * base )
{
	_state = SG_BASE;
	if( !_base ) {
		_base = new DisplayBase( _centralWindow );
		_base->setPlayer( _game->getGamePlayer() );
		_base->setSocket( _socket );
		_base->setGame( _game );
		_base->show();
		_centralWindow->addWidget( _base );
		connect( _base, SIGNAL( sig_quit() ), SLOT( slot_map() ) );
	} else {
		_base->reinitBase();
		_base->show();
	}
	menuBar()->hide();
	statusBar()->hide();
	_base->setBase( base );
	_game->hide();
	_centralWindow->setCurrentWidget( _base );
	ImageTheme.playMusicBase( base->getRace() );
}

void ClientInterface::slot_map()
{
	if(_game) {
		_game->updateWindows();
	}
	updateScreen();
	ImageTheme.playMusicMap();
}

void ClientInterface::updateScreen()
{
	switch( _state ) {
	case SG_MAP:
		return;
		break;
	case SG_FIGHT:
		_fight->hide();
		break;
	case SG_BASE:
		_base->hide();
		break;
	default:
		logEE( "Should not happen" );
		break;
	}
	_state = SG_MAP;
	menuBar()->show();
	statusBar()->show();
	if( _game->getChat()) {
		_game->getChat()->clear();
	}
	_game->show();
	_centralWindow->setCurrentWidget( _game );
	ImageTheme.playMusicMap();
}

void ClientInterface::slot_fight( GenericLord * lord, CLASS_FIGHTER cla )
{
	if ( _fight == 0 ) {
		_fight = new Fight( _centralWindow );
		_fight->setGame( _game );
		_fight->show();
		_centralWindow->addWidget( _fight );
		connect( _fight, SIGNAL( sig_quit() ), SLOT( slot_map() ) );
	} else {
		_fight->reinit();
		_fight->show();
	}
	_state = SG_FIGHT;
	_game->hide();
	menuBar()->hide();
	statusBar()->show();
	_fight->setSocket( _socket );
	_fight->setLord( lord, cla );
	_centralWindow->setCurrentWidget( _fight );
	ImageTheme.playMusicFight();
}


void ClientInterface::slot_readSocket()
{
	_socket->readData();
	switch( _state ) {
	case SG_MAP:
		_game->handleSocket();
		break;
	case SG_FIGHT:
		_fight->handleSocket();
		break;
	case SG_BASE:
		_base->handleSocket();
		break;
	}

	if( _socket->bytesAvailable() > 0 ) {
		slot_readSocket();
	}
}


void ClientInterface::slot_options()
{
	OptionsDialog options( this );
	
	if( options.exec() ) {
		if( options.hasChanged() ) {
			if( _game == _centralWindow->currentWidget() ) {
				_game->updateOptions();
			} else if( _base == _centralWindow->currentWidget() ) {
				/// XXX: we can't change options from base for the moment
			} else {
				/// XXX: we can't change disposition for fight mode for the moment
			}
		}
	}
}

