/*
** ATOP - System & Process Monitor 
** 
** The program 'atop' offers the possibility to view the activity of
** the system on system-level as well as process-level.
** 
** This source-file contains functions to read the process-administration
** of every running process from kernel-space and extract the required
** activity-counters.
** ================================================================
** Author:      Gerlof Langeveld - AT Computing, Nijmegen, Holland
** E-mail:      gerlof@ATComputing.nl
** Date:        November 1996
** LINUX-port:  June 2000
** 
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public License as published by the
** Free Software Foundation; either version 2, or (at your option) any
** later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU General Public License for more details.
**
** $Log: photoproc.c,v $
** Revision 1.17  2003/07/07 09:26:59  gerlof
** Cleanup code (-Wall proof).
**
** Revision 1.16  2003/06/30 11:30:43  gerlof
** Enlarge counters to 'long long'.
**
** Revision 1.15  2003/02/06 12:09:23  gerlof
** Exchange tab-character in command-line by space.
**
** Revision 1.14  2003/01/24 14:19:39  gerlof
** Exchange newline byte in command-line by space.
**
** Revision 1.13  2003/01/17 14:21:41  root
** Change-directory to /proc to optimize opening /proc-files
** via relative path-names i.s.o. absolute path-names.
**
** Revision 1.12  2003/01/17 07:31:29  gerlof
** Store the full command-line for every process.
**
** Revision 1.11  2003/01/06 13:03:09  gerlof
** Improved command-name parsing (command-names containing a close-bracket
** were not parsed correctly).
**
** Revision 1.10  2002/10/03 11:12:39  gerlof
** Modify (effective) uid/gid to real uid/gid.
**
** Revision 1.9  2002/07/24 11:13:31  gerlof
** Changed to ease porting to other UNIX-platforms.
**
** Revision 1.8  2002/07/08 09:27:45  gerlof
** Avoid buffer overflow during sprintf by using snprintf.
**
** Revision 1.7  2002/01/22 13:39:53  gerlof
** Support for number of cpu's.
**
** Revision 1.6  2001/11/22 08:33:43  gerlof
** Add priority per process.
**
** Revision 1.5  2001/11/13 08:26:15  gerlof
** Small bug-fixes.
**
** Revision 1.4  2001/11/07 09:18:43  gerlof
** Use /proc instead of /dev/kmem for process-level statistics.
**
** Revision 1.3  2001/10/04 13:57:34  gerlof
** Explicit include of sched.h (i.s.o. linux/sched.h via linux/mm.h).
**
** Revision 1.2  2001/10/04 08:47:26  gerlof
** Improved verification of kernel-symbol addresses
**
** Revision 1.1  2001/10/02 10:43:29  gerlof
** Initial revision
**
*/

static const char rcsid[] = "$Id: photoproc.c,v 1.17 2003/07/07 09:26:59 gerlof Exp $";

#include <sys/types.h>
#include <sys/param.h>
#include <dirent.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include "atop.h"
#include "photoproc.h"

#define	SCANSTAT 	"%c %*d %*d %*d %*d %*d "		\
			"%*d %lld %*d %lld %*d %lld %lld %*d "	\
			"%*d %d %*d %*d %*d %lld %lld %lld "	\
			"%*d %lld %lld %*d %*d %*d %*d %*d " 	\
			"%*d %*d %*d %lld %*d %*d %d "	 	\
			"%c %lld %lld %lld %lld %lld %lld" /* ATOP-extension */

char	kernpatch;

int
photoproc(struct pstat *proclist, int maxproc)
{
	static int	firstcall = 1;
	static time_t	bootepoch;

	register struct pstat	*curproc;

	FILE		*fp;
	DIR		*dirp;
	struct dirent	*entp;
	int		nr, i;

	char		command[64], state, extstats, origdir[1024];
	char		line[1024], *cmdhead, *cmdtail;
	int		pid, ruid, rgid, prio, curcpu, pval=0;
	count_t		utime, stime, starttime;
	count_t		minflt, majflt, size, rss, nswap,
			startcode, endcode,
			diskrio=0, diskwio=0,
			tcpsnd=0,  tcprcv=0, udpsnd=0, udprcv=0;

	/*
	** one-time initialization stuff
	*/
	if (firstcall)
	{
		/*
		** check if this kernel is patched
		*/
		if ( (fp = fopen("/proc/1/stat", "r")) )
		{
			extstats = 'X';

			fscanf(fp, "%d (%[^)]) " SCANSTAT,
				&pid,       command, 
				&state,     &minflt,  &majflt,
				&utime,     &stime,   &prio,
				&starttime, &size,    &rss,
				&startcode, &endcode, &nswap,  &curcpu,
				&extstats,  &diskrio, &diskwio,
				&tcpsnd,    &tcprcv, 
				&udpsnd,    &udprcv);

			if (extstats == 'A')
				kernpatch = 1;

			fclose(fp);
		}

		bootepoch = getboot();
		firstcall = 0;
	}

	/*
	** read all subdirectory-names below the /proc directory
	*/
	getcwd(origdir, sizeof origdir);
	chdir("/proc");
	dirp = opendir(".");

	while ( (entp = readdir(dirp)) && pval < maxproc )
	{
		/*
		** skip non-numerical names
		*/
		if (!isdigit(entp->d_name[0]))
			continue;

		/*
		** change to the process' subdirectory
		*/
		if ( chdir(entp->d_name) != 0 )
			continue;

		/*
		** open file "stat" and obtain required info
		*/
		if ( (fp = fopen("stat", "r")) == NULL)
		{
			chdir("..");
			continue;
		}

		(void) fgets(line, sizeof line, fp);

		sscanf(line, "%d", &pid);		/* fetch pid */

		cmdhead = strchr (line, '(');		/* fetch commandname */
		cmdtail = strrchr(line, ')');
		if ( (nr = cmdtail-cmdhead-1) > sizeof command)
			nr = sizeof command;
		memcpy(command, cmdhead+1, nr);
		memset(&command[nr], 0, sizeof command - nr);

		nr = sscanf(cmdtail+2, SCANSTAT,
			&state,    &minflt,    &majflt,
			&utime,    &stime,     &prio,    &starttime, &size,
			&rss,      &startcode, &endcode, &nswap,     &curcpu,
			&extstats, &diskrio,   &diskwio,
			&tcpsnd,   &tcprcv,    &udpsnd,  &udprcv);

		fclose(fp);

		if (nr < 11)		/* parsing succeeded ? */
		{
			chdir("..");
			continue;
		}

		/*
		** open file "status" and obtain required info
		*/
		if ( (fp = fopen("status", "r")) == NULL)
		{
			chdir("..");
			continue;
		}

		while (fgets(line, sizeof line, fp))
		{
			if (memcmp(line, "Uid:", 4)==0)
			{
				sscanf(line, "Uid: %d", &ruid);
				continue;
			}

			if (memcmp(line, "Gid:", 4)==0)
			{
				sscanf(line, "Gid: %d", &rgid);
				break;
			}
		}

		fclose(fp);

		/*
		** store required info in process-structure
		*/
		curproc	= proclist+pval;

		curproc->gen.pid   = pid;
		curproc->gen.ruid  = ruid;
		curproc->gen.rgid  = rgid;
		curproc->gen.state = state;

		strncpy(curproc->gen.name, command, PNAMLEN);
		curproc->gen.name[PNAMLEN] = 0;

		curproc->gen.excode = 0;
		curproc->gen.btime  = starttime/HZ+bootepoch;
		curproc->cpu.utime  = utime;
		curproc->cpu.stime  = stime;
		curproc->cpu.pri    = prio;
		curproc->cpu.curcpu = curcpu;
		curproc->mem.iosw   = nswap;
		curproc->mem.minflt = minflt;
		curproc->mem.majflt = majflt;
		curproc->mem.vmem   = size / 1024;
		curproc->mem.rmem   = rss  * (getpagesz()/1024);
		curproc->mem.vgrow  = 0;	/* calculated later */
		curproc->mem.rgrow  = 0;	/* calculated later */
		curproc->mem.shtext = (endcode-startcode)/1024;

		curproc->dsk.ior    = diskrio;
		curproc->dsk.iow    = diskwio;
		curproc->net.tcpsnd = tcpsnd;
		curproc->net.tcprcv = tcprcv;
		curproc->net.udpsnd = udpsnd;
		curproc->net.udprcv = udprcv;

		/*
		** store the full command line;
		** the command-line may contain:
		**    - null-bytes as a separator between the arguments
		**    - newlines (e.g. arguments for awk or sed)
		**    - tabs (e.g. arguments for awk or sed)
		** these special bytes will be converted to spaces
		*/
		memset(curproc->gen.cmdline, 0, CMDLEN+1);

		if ( (fp = fopen("cmdline", "r")) != NULL)
		{
			register char *p = curproc->gen.cmdline;

			nr = fread(p, 1, CMDLEN, fp);
			fclose(fp);

			if (nr >= 0)	/* anything read ? */
			{
				for (i=0; i < nr-1; i++, p++)
					switch (*p)
					{
					   case '\0':
					   case '\n':
					   case '\t':
						*p = ' ';
					}
			}
		}

		pval++;

		chdir("..");
	}

	closedir(dirp);

	chdir(origdir);

	return(pval);
}
