# -*- mode: python; coding: utf-8 -*-

from __future__ import with_statement

import os
import logging
import sys
import time
import inspect
import random
import optparse
from itertools import chain
import fnmatch

from iniparse import INIConfig

from pyarco.Pattern import Singleton
from pyarco.Thread import ThreadPool
from pyarco.UI import high, cout, cout_config
from pyarco.Conio import *

import atheist
import atheist.const
from atheist.utils import *
import atheist.plugins
from atheist.gvar import Log



class ConfigFacade(object):
    '''forwards missing attributes and key-items to the holder objects'''
    def __init__(self, holder=None):
        self.attr_holders = []
        self.key_holders = []
        if holder:
            self.attr_holders.append(holder)

    def __getattr__(self, name):
        for holder in self.attr_holders:
            try:
                return getattr(holder, name)
            except AttributeError:
                pass

        raise AttributeError(name)

    def __getitem__(self, name):
        for holder in self.key_holders:
            try:
                return holder[name]
            except KeyError:
                pass

        raise KeyError(name)

    def get(self, name, value=None):
        "dict-like get method"
        try:
            return self[name]
        except KeyError:
            return value


def public(fn):
    fn.public = True
    return fn


class AInternal:
    '''Namespace for atheist API'''

    def __init__(self, mng):
        self.mng = mng

        self._temp_n = 0
        self._temp_lock = threading.Lock()

        self.symbols = [x[1] for x in inspect.getmembers(
                self, inspect.ismethod) if hasattr(x[1], 'public')]

        pluginpath = self.mng.cfg.pluginpath + \
            [os.path.join(os.path.dirname(atheist.plugins.__file__))]

        self.plugins = []
        for dname in pluginpath:
            self.plugins.extend(atheist.plugins.Loader(dname))

        self.symbols.extend(self.plugins)
        Log.debug("Registered plugins: %s" % sorted([x.__name__ for x in self.plugins]))

        # add built-ins classes to the scope
        for name,symbol in inspect.getmembers(atheist, inspect.isclass):
            if issubclass(symbol, atheist.Public):
                self.symbols.append(symbol)

        self.scope = {}
        for s in self.symbols:
            self.scope[s.__name__] = s

    def get_Task_classes(self):
        return [x for x in self.symbols if inspect.isclass(x) and issubclass(x, atheist.Task)]

    @public
    def get_task(self, _id):
        for t in chain(self.mng.ts):
            if t.tid == _id: return t
        Log.error("There is no task with id '%s'" % _id)
        return None

    @public
    def load(self, fname):
        _globals = self.mng.exec_env.copy()
        globals_before_keys = _globals.keys()
        atheist.exec_file(fname, _globals)

        # FIXME: better imp.new_module?
        retval = types.ModuleType(fname)
        for key,val in _globals.items():
            if key not in globals_before_keys:
                setattr(retval, key, val)

        return retval

    @public
    def temp_name(self, prefix=''):
        with self._temp_lock:
            self._temp_n += 1
            return '/tmp/atheist/%s_%s_%s' % (prefix, os.getpid(), self._temp_n)


class Manager(object):
    def __init__(self, argv):
        self.aborted = False
        self.abort_observers = []

        self._task_index = 0
        self._taskcases = []

        self.ntests = 0
        self.ntasks = 0
        self.ok = 0
        self.fail = 0
        self.ts = []

        self.RunnerClass = atheist.Runner
        self.reporters = []

        self.parser = OptParser(usage=USAGE, version=atheist.const.VERSION)
        opt_cfg, args = self.parser.parse_args(argv)
        self.cfg = ConfigFacade(opt_cfg)

        self._check_option_conflicts_and_ilegals()

        if self.cfg.gen_template:
            print atheist.file_template()
            sys.exit(0)

        self.cfg.loglevel = logging.WARNING
        if self.cfg.verbosity == 1:
            self.cfg.loglevel = logging.INFO
        elif self.cfg.verbosity >= 2:
            self.cfg.loglevel = logging.DEBUG
        if self.cfg.quiet:
            self.cfg.loglevel = logging.ERROR

        Log.setLevel(self.cfg.loglevel)


        self.cfg.timetag = ''
        if self.cfg.use_timetag:
            self.cfg.timetag = '%(asctime)s '
            formatter = atheist.log.XXLoggingFormatter(\
                self.cfg.timetag + '[%(levelinitial)s] %(message)s',
                atheist.log.DATEFORMAT)
            Log.handlers[0].setFormatter(formatter) # change default formatter

        if self.cfg.log:
            filelog = logging.FileHandler(self.cfg.log)
#            filelog.setFormatter(formatter)
            filelog.setLevel(logging.DEBUG)
            Log.addHandler(filelog)

        Log.debug('Log level is %s' % logging.getLevelName(Log.level))

        if not self.cfg.quiet:
            self.reporters.append(atheist.ConsoleReporter(self))


        # make a new parser to get plugin options
        self.parser = OptParser(usage=USAGE, version=atheist.const.VERSION)
        self.parser.ignore_unknown = False

        self.api = AInternal(self)
        for klass in self.api.get_Task_classes():
            klass.set_mng(self)

        group = optparse.OptionGroup(self.parser, "Plugin options")
        for plugin in self.api.plugins:
            plugin.add_options(group)
        self.parser.add_option_group(group)

        opt_cfg, self.args = self.parser.parse_args(argv)
        self.cfg.attr_holders.append(opt_cfg)

        if os.path.exists(ATHEIST_CFG):
            self.cfg.attr_holders.append(INIConfig(file(ATHEIST_CFG)))

        for plugin in self.api.plugins:
            plugin.config(self)

        self.exec_env = self.api.scope
        self.exec_env['args'] = self.cfg.task_args.split(',')

        compath.root = os.path.abspath(self.cfg.basedir)

        self.cfg.exclude = EXCLUDE + self.cfg.ignore.split(',')
        try:
            self.cfg.exclude += [self.cfg.ui.ignore]
        except AttributeError:
            pass

        Log.debug("excluding %s" % self.cfg.exclude)

        cout_config(not self.cfg.plain)


    def _check_option_conflicts_and_ilegals(self):
        if self.cfg.stdout and self.cfg.stdout_on_fail:
            self.parser.error("-o/--stdout and -f/--stdout-on-fail are incompatible options")

        if self.cfg.verbosity and self.cfg.quiet:
            self.parser.error("-q/--quiet and -v/--verbose are incompatible options")

        if self.cfg.use_timetag and self.cfg.verbosity == 0 or self.cfg.quiet:
            self.parser.error("-t/--timetag requires some verbosity")

        if self.cfg.workers < 0:
            self.parser.error("-w/--workers can not be negative.")

        if self.cfg.verbosity:
            self.cfg.report_detail = 10


    def abort(self):
        self.aborted = True
        for ob in self.abort_observers: ob()


    def reload(self):
        self._task_index = 0
        self._taskcases = []

        for inline in self.cfg.inline:
            Log.debug("inline script: %s" % inline)
            fname = os.path.join(ATHEIST_TMP, 'inline_%s.test' % os.getpid())
            fd = open(fname, 'w')
            fd.write(inline)
            fd.close()
            try:
                self.add(atheist.TaskCase(fname, self))
            finally:
                Log.debug("removing %s" % fname)
                os.remove(fname)

        for i in sorted([os.path.abspath(x) for x in self.args]):
            if os.path.isdir(i):
                self.process_directory(i)

            elif os.path.isfile(i):
                if i in [SETUP, TEARDOWN]: continue
                self.add(atheist.TaskCase(i, self))

            else:
                try:
                    file(i).close()  # suspicious file, checking
                except IOError, e:
                    Log.critical(e)
                    sys.exit(1)


    def next(self):
        self._task_index += 1
        return self._task_index

    def add(self, taskcase):
        self._taskcases.append(taskcase)

    def process_directory(self, dname):
        def excluded(fname):
            return any(fnmatch.fnmatch(fname, pattern) for pattern in self.cfg.exclude)

        for root, dirs, files in os.walk(dname):
            dirs.sort()
            Log.info("Entering directory '%s'" % compath(root))
            for f in sorted([x for x in files if x.endswith(ATHEIST_EXT)]):
                if excluded(f):
                    continue
                self.add(atheist.TaskCase(os.path.join(root, f), self))

            for d in self.cfg.exclude:
                if d in dirs: dirs.remove(d)


    def itertasks(self):
        return chain(*[x.tasks for x in self._taskcases])

    def itercases(self):
        return iter(self._taskcases)

    def __len__(self):
        return sum([len(s.tasks) for s in self._taskcases])

    def run(self, ob):
        def count(*args):
            self.done += 1

        self.tini = time.time()
        self.done = 0  # completed tasks

        cases = self._taskcases[:]

        if len(cases) == 1:
            cases[0].run(ob)

        else:
            nworkers = self.cfg.workers
            if nworkers == 0:
                nworkers = min(100, 1+len(cases)/2)
                Log.info("Creating %s workers" % nworkers)

            pool = ThreadPool(nworkers)

            if self.cfg.random is not None:
                if self.cfg.random == 0:
                    random.seed()  # random seed
                else:
                    random.seed(self.cfg.random)

                random.shuffle(cases)

            for tc in cases:
                pool.add(tc.run, [ob], callback=count)

            #pool.join()
            while not self.aborted:
                if self.done == len(cases): break
                time.sleep(0.5)

            pool.join(True, True)

        self._calculate()


    def _calculate(self):
        self.ntasks = len([x for x in self.itertasks()])
        tests = [x for x in self.itertasks() if x.check]
        self.ntests = len(tests)
        self.ok    = len([x for x in tests if x.result == OK])
        self.fail = self.ntests - self.ok

    def str_stats(self):
        delta = time.time() - self.tini
        if delta > 60:
            t = "%s.%s" % (time.strftime("%H:%M:%S", time.gmtime(delta)),
                           ("%.2f" % (delta % 1.0))[2:])
        else:
            t = "%.2fs" % delta

        if self.ok == self.ntests:
            return cout(BOLD, LIGHT_GREEN_BG, "  ALL OK!!  ", NORM, " - %s - %s - %s " % (
                t, count(self.ntests, 'test'), count(self.ntasks, 'task')))

        return "%s - %s - %s/%s" % \
            (cout(BOLD, LIGHT_RED_BG, "    FAIL    ", NORM),
            t, high(self.ok),
            count(self.ntests, 'test'))

    def ALL(self):
        return self.ok == self.ntests



class OptParser(optparse.OptionParser):
    def __init__(self, *args, **kargs):
        self.ignore_unknown = True
        optparse.OptionParser.__init__(self, *args, **kargs)

        self.add_option('-a', '--task-args', dest='task_args', metavar='ARGS',
                          default='',
                          help='colon-separated options for the tasks')

        self.add_option('-b', '--base-dir', dest='basedir', metavar='PATH',
                          default=os.getcwd(),
                          help='set working directory')

        self.add_option('-C', '--clean-only', dest='clean_only',
                          action='store_true',
                          help="execute nothing, only remove generated files")

        self.add_option('-d', '--describe', dest='describe',
                          action='store_true',
                          help='execute nothing, only describe tasks')

        self.add_option('-e', '--stderr', dest='stderr',
                          action='store_true',
                          help='print task process stderr')

        self.add_option('-f', '--stdout-on-fail', dest='stdout_on_fail',
                          action='store_true',
                          help='print task output but only if it fail')

        self.add_option('-g', '--gen-template', dest='gen_template',
                          action='store_true',
                          help='generate a taskcase file template with default values')

# FIXME: test this
        self.add_option('-i', '--report-detail', metavar='LEVEL', default=1, type=int,
                          help='Report verbosity (0:nothing, [1:case], 2:task, 3:composite, 4:condition)')

        self.add_option('-j', '--skip-hooks', dest='skip_hooks',
                          action='store_true',
                          help='skip _setup and _teardown files')

        self.add_option('-k', '--keep-going', dest='keep_going',
                          action='store_true', default=False,
                          help='continue even with failed tasks')

        self.add_option('-l', '--list', dest='list_only',
                          action='store_true',
                          help='list tasks but do not execute them')

        self.add_option('-o', '--stdout', dest='stdout',
                          action='store_true',
                          help='print task stdout')

        # FIXME: probar esto
        self.add_option('-p', '--plugin-dir', dest='pluginpath', metavar='PATH',
                          action='append', default=[],
                          help='a directory containing plugins')

        self.add_option('-q', '--quiet', dest='quiet',
                          action='store_true',
                          help='do not show result summary nor warnings, only totals')

        self.add_option('-r', '--random', dest='random', type='int', default=None,
                          help='shuffle taskcases using the specified seed (0:random seed)')

        self.add_option('-s', '--script', dest='inline',
                          action='append', default=[],
                          help='specifies command line script')

        self.add_option('-t', '--time-tag', dest='use_timetag',
                          action='store_true',
                          help='include time info in the logs')

        self.add_option('-v', '--verbose', dest='verbosity',
                          action='count', default=0,
                          help='incresse verbosity')

        self.add_option('-w', '--workers', default=1, type=int,
                          help='number of simultaneous tasks (deafult:1) 0:auto-select.')

        self.add_option('--cols', dest='screen_width', default='80', metavar='WIDTH',
                          type=int, help="terminal width (in chars)")

        self.add_option('--dirty', dest='clean',
                          action='store_false', default=True,
                          help="do not remove generated files after task execution")

        self.add_option('--disable-bar', action='store_true',
                          help="Don't show progress bar")

        # FIXME: a test for this
        a= self.add_option('--ignore', metavar='PATTERN', dest='ignore', default='',
                        help="test files to ignore (glob patterns) separated with semicolon")

        self.add_option('--log', default=None,
                          help="Log to specified filename")

        self.add_option('--plain', default=False, action='store_true',
                          help="Avoid color codes in console output")


# YAGNI
#        self.add_option('-x', '--extension', dest='ext',
#                          default='.test',
#                          help='file extension for the task files')



    def _process_args(self, largs, rargs, values):
        "Override OptParse._process_args to give ability to ignore unknown args"
        while rargs:
            arg = rargs[0]
            if arg == "--":
                del rargs[0]
                return
            elif arg[0:2] == "--":
                try:
                    self._process_long_opt(rargs, values)
                except optparse.BadOptionError, e:
                    if not self.ignore_unknown:
                        raise

            elif arg[:1] == "-" and len(arg) > 1:
                try:
                    self._process_short_opts(rargs, values)
                except optparse.BadOptionError, e:
                    if not self.ignore_unknown:
                        raise

            elif self.allow_interspersed_args:
                largs.append(arg)
                del rargs[0]
            else:
                return
