/*
 * atanks - obliterate each other with oversize weapons
 * Copyright (C) 2003  Thomas Hudson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * */


#include "environment.h"
#include "globaldata.h"
#include "virtobj.h"
#include "missile.h"
#include "tank.h"
#include "files.h"

ENVIRONMENT::ENVIRONMENT (GLOBALDATA *global)
{
	gravity = 0.150;
	viscosity = 0.5;
	landSlideType = LANDSLIDE_GRAVITY;
        landSlideDelay = MAX_GRAVITY_DELAY;
	windstrength = 8;
	windvariation = 1;
	techLevel = 5;
	landType = LANDTYPE_RANDOM;
	meteors = 0;
	lightning = 0;
	fog = 0;
        wallType = 0.0;
        current_wallType = 0;

	done = new char[global->screenWidth];
	fp = new int[global->screenWidth];
	h = new int[global->screenWidth];
	surface = new int[global->screenWidth];
	dropTo = new int[global->screenWidth];
	velocity = new double[global->screenWidth];
	dropIncr = new double[global->screenWidth];
	height = new double[global->screenWidth];
	
	db = create_bitmap (global->screenWidth, global->screenHeight);
	if (!db)
		cout << "Failed to create db bitmap: " << allegro_error;
	terrain = create_bitmap (global->screenWidth, global->screenHeight);
	if (!terrain)
		cout << "Failed to create terrain bitmap: " << allegro_error;
	sky = create_bitmap (global->screenWidth, global->screenHeight - MENU);
	if (!sky)
		cout << "Failed to create sky bitmap: " << allegro_error;

	_global = global;

	initialise ();
}

// Following are config file definitions
#define	GFILE_LATEST_VERSION	4

// stop using wind strength as an int. Use the new double version
#define GFILE_WINDSTRENGTH	&windstrength, sizeof (int), 1, file
#define GFILE_NEW_WINDSTRENGTH  &windstrength, sizeof (double), 1, file
 
#define	GFILE_WINDVARIATION	&windvariation, sizeof (double), 1, file
#define	GFILE_VISCOSITY		&viscosity, sizeof (double), 1, file
#define	GFILE_GRAVITY		&gravity, sizeof (double), 1, file
#define	GFILE_TECHLEVEL		&techLevel, sizeof (double), 1, file
#define	GFILE_METEORS		&meteors, sizeof (double), 1, file
#define	GFILE_LIGHTNING		&lightning, sizeof (double), 1, file
#define	GFILE_FOG		&fog, sizeof (double), 1, file
#define	GFILE_LANDTYPE		&landType, sizeof (double), 1, file
#define	GFILE_LANDSLIDETYPE	&landSlideType, sizeof (double), 1, file
#define GFILE_WALLTYPE          &wallType, sizeof(double), 1, file

#define	GFILE_VERSION_1(a)	a (GFILE_WINDSTRENGTH);\
				a (GFILE_WINDVARIATION);\
				a (GFILE_VISCOSITY);\
				a (GFILE_GRAVITY);\
				a (GFILE_TECHLEVEL);\
				a (GFILE_METEORS);\
				a (GFILE_FOG);\
				a (GFILE_LANDTYPE);\
				a (GFILE_LANDSLIDETYPE);

#define	GFILE_VERSION_2(a)	a (GFILE_WINDSTRENGTH);\
				a (GFILE_WINDVARIATION);\
				a (GFILE_VISCOSITY);\
				a (GFILE_GRAVITY);\
				a (GFILE_TECHLEVEL);\
				a (GFILE_METEORS);\
				a (GFILE_LIGHTNING);\
				a (GFILE_FOG);\
				a (GFILE_LANDTYPE);\
				a (GFILE_LANDSLIDETYPE);

#define GFILE_VERSION_3(a)      a (GFILE_WINDSTRENGTH);\
                                a (GFILE_WINDVARIATION);\
                                a (GFILE_VISCOSITY);\
                                a (GFILE_GRAVITY);\
                                a (GFILE_TECHLEVEL);\
                                a (GFILE_METEORS);\
                                a (GFILE_LIGHTNING);\
                                a (GFILE_FOG);\
                                a (GFILE_LANDTYPE);\
                                a (GFILE_LANDSLIDETYPE);\
                                a (GFILE_WALLTYPE);

#define GFILE_VERSION_4(a)      a (GFILE_NEW_WINDSTRENGTH);\
                                a (GFILE_WINDVARIATION);\
                                a (GFILE_VISCOSITY);\
                                a (GFILE_GRAVITY);\
                                a (GFILE_TECHLEVEL);\
                                a (GFILE_METEORS);\
                                a (GFILE_LIGHTNING);\
                                a (GFILE_FOG);\
                                a (GFILE_LANDTYPE);\
                                a (GFILE_LANDSLIDETYPE);\
                                a (GFILE_WALLTYPE);


#define	GFILE_WRITE_LATEST	GFILE_VERSION_4(fwrite)
#define	GFILE_READ_VERSION_1	GFILE_VERSION_1(fread)
#define	GFILE_READ_VERSION_2	GFILE_VERSION_2(fread)
#define GFILE_READ_VERSION_3    GFILE_VERSION_3(fread)
#define GFILE_READ_VERSION_4    GFILE_VERSION_4(fread)


/*
This function saves the environment settings to a text
file. Each line has the format
name=value\n

The function returns TRUE on success and FALSE on failure.
-- Jesse
*/
int ENVIRONMENT::saveToFile_Text (FILE *file)
{
   if (! file) return FALSE;

   fprintf(file, "*ENV*\n");

   fprintf(file, "WINDVARIATION=%lf\n", windvariation);
   fprintf(file, "VISCOSITY=%lf\n", viscosity);
   fprintf(file, "GRAVITY=%lf\n", gravity);
   fprintf(file, "TECHLEVEL=%lf\n", techLevel);
   fprintf(file, "METEORS=%lf\n", meteors);
   fprintf(file, "LIGHTNING=%lf\n", lightning);
   fprintf(file, "FOG=%lf\n", fog);
   fprintf(file, "LANDTYPE=%lf\n", landType);
   fprintf(file, "LANDSLIDETYPE=%lf\n", landSlideType);
   fprintf(file, "WALLTYPE=%lf\n", wallType);
   fprintf(file, "LANDSLIDEDELAY=%lf\n", landSlideDelay);

   fprintf(file, "***\n");      
   return TRUE;
}



int ENVIRONMENT::saveToFile (FILE *file)
{
	int version = GFILE_LATEST_VERSION;
	if (!file)
		return (-1);

	fwrite (&version, sizeof (int), 1, file);

	GFILE_WRITE_LATEST

	return (ferror (file));
}



/*
This function loads environment settings from a text
file. The function returns TRUE on success and FALSE if
any erors are encountered.
-- Jesse
*/
int ENVIRONMENT::loadFromFile_Text (FILE *file)
{
    char line[MAX_CONFIG_LINE];
    int equal_position, line_length;
    char field[MAX_CONFIG_LINE], value[MAX_CONFIG_LINE];
    char *result = NULL;
    bool done = false;

    // read until we hit line "*ENV*" or "***" or EOF
    do
    {
      result = fgets(line, MAX_CONFIG_LINE, file);
      if (! result)     // eof
         return FALSE;
      if (! strncmp(line, "***", 3) )     // end of record
         return FALSE;
    } while ( strncmp(line, "*ENV*", 5) );     // read until we hit new record

      while ( (result) && (!done) )
      {
       // read a line
       memset(line, '\0', MAX_CONFIG_LINE);
       result = fgets(line, MAX_CONFIG_LINE, file);
       if (result)
       {
          // if we hit end of the record, stop
          if (! strncmp(line, "***", 3) ) return TRUE;
          // find equal sign
          line_length = strlen(line);
          // strip newline character
          if ( line[line_length - 1] == '\n')
          {
               line[line_length - 1] = '\0';
               line_length--;
          }
          equal_position = 1;
          while ( ( equal_position < line_length) && (line[equal_position] != '=') )
              equal_position++;
          // make sure we have valid equal sign

          if ( equal_position <= line_length )
          {
              // seperate field from value
             memset(field, '\0', MAX_CONFIG_LINE);
             memset(value, '\0', MAX_CONFIG_LINE);
             strncpy(field, line, equal_position);
             strcpy(value, & (line[equal_position + 1]));
             // check for fields and values
             if (! strcasecmp(field, "windvariation") )
                sscanf(value, "%lf", &windvariation);
             else if (! strcasecmp(field, "viscosity") )
                sscanf(value, "%lf", &viscosity);
             else if (! strcasecmp(field, "gravity") )
                sscanf(value, "%lf", &gravity);
             else if (! strcasecmp(field, "techlevel"))
                sscanf(value, "%lf", &techLevel);
             else if (! strcasecmp(field, "meteors"))
                sscanf(value, "%lf", &meteors);
             else if (! strcasecmp(field, "lightning") )
                sscanf(value, "%lf", &lightning);
             else if (! strcasecmp(field, "fog") )
                sscanf(value, "%lf", &fog);
             else if (! strcasecmp(field, "landtype"))
                sscanf(value, "%lf", &landType);
             else if (! strcasecmp(field, "landslidetype"))
                sscanf(value, "%lf", &landSlideType);
             else if (! strcasecmp(field, "walltype"))
                sscanf(value, "%lf", &wallType);
             else if (! strcasecmp(field, "landslidedelay"))
                sscanf(value, "%lf", &landSlideDelay);
          }    // end of found field=value line

      }     // end of read a line properly
     }     // end of while not done

   return TRUE;
}


int ENVIRONMENT::loadFromFile (FILE *file)
{
	int version;
	if (!file)
		return (-1);

	fread (&version, sizeof (int), 1, file);

	switch (version) {
		case 1:
			GFILE_READ_VERSION_1;
			break;
		case 2:
			GFILE_READ_VERSION_2;
			break;
                case 3:
                        GFILE_READ_VERSION_3;
                        break;
                case 4:
                        GFILE_READ_VERSION_4;
                        break;
		default:
			break;
	}

	return (ferror (file));
}

void ENVIRONMENT::initialise ()
{
	for (int count = 0; count < _global->screenWidth; count++) {
		h[count] = 0;
		fp[count] = 0;
		done[count] = 1;
		dropTo[count] = _global->screenHeight - 1;
		surface[count] = 0;
	}
	for (int count = 0; count < MAX_OBJECTS; count++)
		objects[count] = NULL;
	for (int count = 0; count < MAXPLAYERS * 3; count++)
		order[count] = NULL;

	clear_to_color (sky, PINK);
	clear_to_color (db, WHITE);
	clear_to_color (terrain, PINK);

	oldFogX = 0;
	oldFogY = 0;

	combineUpdates = TRUE;
}

int ENVIRONMENT::isItemAvailable (int itemNum)
{
	if (itemNum < WEAPONS) {
		if ((weapon[itemNum].warhead) ||
			(weapon[itemNum].techLevel > techLevel))
			return (FALSE);
	} else if (item[itemNum - WEAPONS].techLevel > techLevel) {
		return (FALSE);
	}
	return (TRUE);
}

void ENVIRONMENT::generateAvailableItemsList ()
{
	int slot = 0;
	for (int itemNum = 0; itemNum < THINGS; itemNum++)
	{
		if (!isItemAvailable (itemNum))
			continue;
		availableItems[slot] = itemNum;
		slot++;
	}
	numAvailable = slot;
}

int ENVIRONMENT::addObject (VIRTUAL_OBJECT *object)
{
	int objCount = 0;

	while ((objects[objCount] != NULL) && (objCount < MAX_OBJECTS))
		objCount++;
	if (objCount < MAX_OBJECTS)
		objects[objCount] = object;

	return (objCount);
}

int ENVIRONMENT::removeObject (VIRTUAL_OBJECT *object)
{
	int objCount = object->getIndex ();

	if (objCount < MAX_OBJECTS) {
		objects[objCount] = NULL;
		if (object->getClass () == TANK_CLASS)
			for (int ordCount = 0; ordCount < _global->maxNumTanks; ordCount++)
				if (order[ordCount] == object)
					order[ordCount] = NULL;
	} else {
		return (objCount);
	}

	return (0);
}

VIRTUAL_OBJECT *ENVIRONMENT::getNextOfClass (int classNum, int *objCount)
{
	for (;*objCount < MAX_OBJECTS; (*objCount)++) {
		if ((objects[*objCount] != NULL) &&
			((classNum == ANY_CLASS) || (classNum == objects[*objCount]->getClass ())))
			break;
	}

	if (*objCount < MAX_OBJECTS)
		return (objects[*objCount]);
	else
		return (NULL);
}

void ENVIRONMENT::newRound ()
{
	for (int objCount = 0; objCount < MAX_OBJECTS; objCount++) {
		if (objects[objCount]) {
			if ((!objects[objCount]->isSubClass (TANK_CLASS)) &&
			(!objects[objCount]->isSubClass (FLOATTEXT_CLASS))) {
				delete objects[objCount];
				objects[objCount] = NULL;
			}
		}
	}

        // set wall type
        if (wallType == WALL_RANDOM)
           current_wallType = rand() % 3;
        else
           current_wallType = (int) wallType;

        // time_to_fall = (rand() % MAX_GRAVITY_DELAY) + 1;
        time_to_fall = (rand() & (int)landSlideDelay) + 1;
}

int ENVIRONMENT::ingamemenu ()
{
	int button[INGAMEBUTTONS] = { -35, -10, 15 }, pressed, updatew[INGAMEBUTTONS];
	char buttext[INGAMEBUTTONS][10] = { "Return", "Main Menu", "Quit" };
	int z, zz;

	pressed = -1;
	for (z = 0; z < INGAMEBUTTONS; z++)
		updatew[z] = 0;
	if (! _global->os_mouse) show_mouse (NULL);
	make_update (_global->halfWidth - 100, _global->halfHeight - 50, 200, 100);
	rectfill (db, _global->halfWidth - 100, _global->halfHeight - 50, _global->halfWidth + 99, _global->halfHeight + 49, makecol (128, 128, 128));
	rect (db, _global->halfWidth - 100, _global->halfHeight - 50, _global->halfWidth + 99, _global->halfHeight + 49, BLACK);
	while (1) {
        LINUX_SLEEP;
		if (keypressed ()) {
			k = readkey ();
			if (k >> 8 == KEY_ESC)
				return (0);
		}
		
		if (mouse_b & 1) {
			zz = 0;
			for (z = 0; z < INGAMEBUTTONS; z++) {
				if (mouse_x >= _global->halfWidth - 75 && mouse_x < _global->halfWidth + 75 && mouse_y >= button[z] + _global->halfHeight
				    && mouse_y < button[z] + 20 + _global->halfHeight) {
					zz = 1;
					if (pressed > -1)
						updatew[pressed] = 1;
					pressed = z;
					updatew[z] = 1;
				}
			}
			if (!zz) {
				if (pressed > -1)
					updatew[pressed] = 1;
				pressed = -1;
			}
		}
		if (pressed > -1 && !mouse_b & 1)
			return (pressed);
		for (z = 0; z < INGAMEBUTTONS; z++) {
			if (updatew[z]) {
				updatew[z] = 0;
				make_update (_global->halfWidth - 75, _global->halfHeight + button[z], 150, 20);
			}
		}
		make_update (mouse_x, mouse_y, ((BITMAP *) (_global->gfxData.M[0].dat))->w, ((BITMAP *) (_global->gfxData.M[0].dat))->h);
		make_update (lx, ly, ((BITMAP *) (_global->gfxData.M[0].dat))->w, ((BITMAP *) (_global->gfxData.M[0].dat))->h);
		lx = mouse_x;
		ly = mouse_y;
		if (! _global->os_mouse)  show_mouse (NULL);
		for (z = 0; z < INGAMEBUTTONS; z++) {
			draw_sprite (db, (BITMAP *) _global->gfxData.M[(pressed == z) ? 8 : 7].dat, _global->halfWidth - 75, _global->halfHeight + button[z]);
			textout_centre_ex (db, font, buttext[z], _global->halfWidth, _global->halfHeight + button[z] + 6, WHITE, -1);
		}
		if (! _global->os_mouse) show_mouse (db);
		do_updates ();
	}
}

void ENVIRONMENT::do_updates ()
{
	int count;

	if (fog && _global->currTank) {
		int x, y;
		x = (int)_global->currTank->x;
		y = (int)_global->currTank->y;
		if (oldFogX != x || oldFogY != y) {
			rectfill (screen, oldFogX - 100, oldFogY - 100, oldFogX + 100, oldFogY + 100, makecol (128, 128, 128));
			oldFogX = x;
			oldFogY = y;
		}
		blit (db, screen, x - 100, y - 100, x - 100, y - 100, 200, 200);
		blit (db, screen, 0, 0, 0, 0, _global->screenWidth, MENU);
	} else {
		if (_global->updateCount == MAXUPDATES)
			_global->updateCount = 1;
		for (count = 0; count < _global->updateCount && count < MAXUPDATES; count++) {
			blit (db, screen, _global->updates[count].x, _global->updates[count].y, _global->updates[count].x, _global->updates[count].y, _global->updates[count].w, _global->updates[count].h);
			// Debug rectangle
			//rect (screen, _global->updates[count].x, _global->updates[count].y, _global->updates[count].x + _global->updates[count].w, _global->updates[count].y + _global->updates[count].h, WHITE);
		}
	}
	_global->lastUpdatesCount = _global->updateCount;
	memcpy (_global->lastUpdates, _global->updates, sizeof (BOX) * _global->updateCount);
	_global->updateCount = 0;
}

void ENVIRONMENT::replaceCanvas ()
{
	int count;

	if (_global->lastUpdatesCount == MAXUPDATES)
		_global->lastUpdatesCount = 1;
	for (count = 0; count < _global->lastUpdatesCount && count < MAXUPDATES; count++) {
		blit (sky, db, _global->lastUpdates[count].x, _global->lastUpdates[count].y - MENU, _global->lastUpdates[count].x, _global->lastUpdates[count].y, _global->lastUpdates[count].w, _global->lastUpdates[count].h);
		masked_blit (terrain, db, _global->lastUpdates[count].x, _global->lastUpdates[count].y, _global->lastUpdates[count].x, _global->lastUpdates[count].y, _global->lastUpdates[count].w, _global->lastUpdates[count].h);
	}
	_global->lastUpdatesCount = 0;
}

void ENVIRONMENT::make_update (int x, int y, int w, int h)
{
	int combined = 0;

	if (combineUpdates && _global->updateCount && _global->updateCount < MAXUPDATES)
	{
		BOX prev, next;
		prev.x = _global->updates[_global->updateCount - 1].x;
		prev.y = _global->updates[_global->updateCount - 1].y;
		prev.w = _global->updates[_global->updateCount - 1].x + _global->updates[_global->updateCount - 1].w;
		prev.h = _global->updates[_global->updateCount - 1].y + _global->updates[_global->updateCount - 1].h;

		next.x = x;
		next.y = y;
		next.w = x + w;
		next.h = y + h;

		if (((next.w > prev.x - 3) && (prev.w > next.x - 3)) &&
		    ((next.h > prev.y - 3) && (prev.h > next.y - 3)))
		{
			next.x = (next.x < prev.x)?next.x:prev.x;
			next.y = (next.y < prev.y)?next.y:prev.y;
			next.w = (next.w > prev.w)?next.w:prev.w;
			next.h = (next.h > prev.h)?next.h:prev.h;
			_global->updates[_global->updateCount - 1].x = next.x;
			_global->updates[_global->updateCount - 1].y = next.y;
			_global->updates[_global->updateCount - 1].w = next.w - next.x;
			_global->updates[_global->updateCount - 1].h = next.h - next.y;
			combined = 1;
		}
	}
	if (!combined)
	{
		_global->updates[_global->updateCount].x = x;
		_global->updates[_global->updateCount].y = y;
		_global->updates[_global->updateCount].w = w;
		_global->updates[_global->updateCount].h = h;
		if (_global->updateCount < MAXUPDATES)
			_global->updateCount++;
	}
	if (_global->updateCount == MAXUPDATES)
	{
		_global->updates[0].x = x;
		_global->updates[0].y = y;
		_global->updates[0].w = _global->screenWidth;
		_global->updates[0].h = _global->screenHeight;
	}
	if (!_global->stopwindow) {
		if (x < _global->window.x)
			_global->window.x = x;
		if (y < _global->window.y)
			_global->window.y = y;
		if (x + w > _global->window.w)
			_global->window.w = (x + w) - 1;
		if (y + h > _global->window.h)
			_global->window.h = (y + h) - 1;
		if (_global->window.x < 0)
			_global->window.x = 0;
		if (_global->window.y < MENU)
			_global->window.y = MENU;
		if (_global->window.w > (_global->screenWidth-1))
			_global->window.w = (_global->screenWidth-1);
		if (_global->window.h > (_global->screenHeight-1))
			_global->window.h = (_global->screenHeight-1);
	}
}

void ENVIRONMENT::make_bgupdate (int x, int y, int w, int h)
{
	int combined = 0;
	if (combineUpdates && _global->lastUpdatesCount && _global->lastUpdatesCount < MAXUPDATES)
	{
		BOX prev, next;
		prev.x = _global->lastUpdates[_global->lastUpdatesCount - 1].x;
		prev.y = _global->lastUpdates[_global->lastUpdatesCount - 1].y;
		prev.w = _global->lastUpdates[_global->lastUpdatesCount - 1].x + _global->lastUpdates[_global->lastUpdatesCount - 1].w;
		prev.h = _global->lastUpdates[_global->lastUpdatesCount - 1].y + _global->lastUpdates[_global->lastUpdatesCount - 1].h;

		next.x = x;
		next.y = y;
		next.w = x + w;
		next.h = y + h;

		if (((next.w > prev.x - 3) && (prev.w > next.x - 3)) &&
		    ((next.h > prev.y - 3) && (prev.h > next.y - 3)))
		{
			next.x = (next.x < prev.x)?next.x:prev.x;
			next.y = (next.y < prev.y)?next.y:prev.y;
			next.w = (next.w > prev.w)?next.w:prev.w;
			next.h = (next.h > prev.h)?next.h:prev.h;
			_global->lastUpdates[_global->lastUpdatesCount - 1].x = next.x;
			_global->lastUpdates[_global->lastUpdatesCount - 1].y = next.y;
			_global->lastUpdates[_global->lastUpdatesCount - 1].w = next.w - next.x;
			_global->lastUpdates[_global->lastUpdatesCount - 1].h = next.h - next.y;
			combined = 1;
		}
	}
	if (!combined)
	{
		_global->lastUpdates[_global->lastUpdatesCount].x = x;
		_global->lastUpdates[_global->lastUpdatesCount].y = y;
		_global->lastUpdates[_global->lastUpdatesCount].w = w;
		_global->lastUpdates[_global->lastUpdatesCount].h = h;
		if (_global->lastUpdatesCount < MAXUPDATES)
			_global->lastUpdatesCount++;
	}
	if (_global->lastUpdatesCount == MAXUPDATES)
	{
		_global->lastUpdates[0].x = x;
		_global->lastUpdates[0].y = y;
		_global->lastUpdates[0].w = _global->screenWidth;
		_global->lastUpdates[0].h = _global->screenHeight;
	}
}
