# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""usage: as_run --serv [options] user@mach:/nom_profil.export

old syntax : as_serv as_exec [options] user@mach:/nom_profil.export

Export file must be the last argument.

Object :
    - retreive the export file from the client
    - read export and ...
    - ... call as_run on the right node, into a xterm or not,
     now or differed...
"""

from asrun.common.i18n import _
from asrun.mystring     import print3
from asrun.calcul       import ASTER_CALCUL, parse_submission_result
from asrun.profil       import ASTER_PROFIL
from asrun.common_func  import get_tmpname
from asrun.common.sysutils import local_user, local_full_host


def SetParser(run):
    """Configure the command-line parser, add options name to store to the list,
    set actions informations.
    run : ASTER_RUN object which manages the execution
    """
    acts_descr = {
        'serv' : {
            'method' : Serv,
            'syntax' : 'user@mach:/nom_profil.export',
            'help'   : _(u'start an execution (calling as_run in a separate process)')
        },
        'call_server' : {
            'method' : Start,
            'syntax' : 'user@mach:/nom_profil.export',
            'help'   : _(u'call a server to start an execution (calling as_run --serv on the server)')
        },
    }
    run.SetActions(
            actions_descr = acts_descr,
            actions_order = ['serv', 'call_server'],
            group_options=False,
            options_descr = {},
    )


def Serv(run, *args):
    """Start an execution, another as_run, in a separate process (but on the same server/cluster).
    """
    # ----- check argument
    if len(args) != 1:
        run.parser.error(_(u"'--%s' requires one argument") % run.current_action)

    service = ASTER_CALCUL(run, filename=args[0])
    iret, bid = service.start()
    # prinf job info
    print3("JOBID=%s QUEUE=%s" % (service.jobid, service.queue))

    run.Sortie(iret)


def Start(run, *args):
    """Call a server to start a calculation : call --serv option on the
    local or remote server.
    This option is intended to be called on the client machine (directly
    by the gui for example).
    """
    # ----- check argument
    if len(args) != 1:
        run.parser.error(_(u"'--%s' requires one argument") % run.current_action)

    fprof = get_tmpname(run, run['tmp_user'], basename='profil_astk')
    iret = run.Copy(fprof, args[0], niverr='<F>_PROFILE_COPY')
    prof = ASTER_PROFIL(fprof, run)
    filename = "%s:%s" % (local_full_host, fprof)
    if local_user != '':
        filename = "%s@%s" % (local_user, filename)
    cmd = run.get_as_run_cmd()
    cmd.append("--serv")
    cmd.append(filename)
    run.DBG("call --serv on %s@%s, filename=%s" %(prof['username'], prof['serveur'], filename))
    iret, output = run.Shell(" ".join(cmd), user=prof['username'][0], mach=prof['serveur'][0])

    jobid, queue = parse_submission_result(output)
    print3("JOBID=%s QUEUE=%s" % (jobid, queue))
    run.Sortie(iret)

