# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
Functions usefully for an asrun client.
"""

import os
import re
from shutil import copyfile

from asrun.installation import confdir
from asrun.rcfile       import read_rcfile, parse_config, decode_config_line, time_to_seconds

PREFERENCES  = 'prefs'
SERVER_CONF = 'config_serveurs'

_INFO = 'astkrc_version'
_RC_VERSION = "1.1"


class CLIENT(object):
   """
   Functions usefully for an asrun client.
   """
   def __init__(self, rcdir, run=None):
      """
      Initialization
      """
      self._rcdir = rcdir
      self.run    = run
      # cache for user preferences and servers configuration
      self._pref  = None
      self._serv  = None
      self._serv_rcversion = None

   def rcfile(self, filename, rcdir=None):
      """
      Return resource filename
      """
      rcdir = rcdir or self._rcdir
      return os.path.join(rcdir, filename)
   
   def init_user_resource(self, filename):
      """
      Initialize 'rcdir'/'filename' if it's not exist.
      """
      if not os.path.exists(self.rcfile(filename)):
         copyfile(self.rcfile(filename, os.path.join(confdir, 'astkrc')), self.rcfile(filename))

   def get_user_preferences(self, force=False):
      """
      Return user preferences.
      """
      if force or not self._pref:
         self.init_user_resource(PREFERENCES)
         self._pref = {}
         read_rcfile(self.rcfile(PREFERENCES), self._pref)
      return self._pref

   def init_server_config(self, force=False, refresh=False):
      """
      Return servers configuration from 'rcdir'/config_serveurs
      """
      if force or not self._serv:
         self.init_user_resource(SERVER_CONF)
         self._serv = {}
         read_rcfile(self.rcfile(SERVER_CONF), self._serv, mcfact_key='serveur', mcsimp=['vers', 'noeud'])
         if self._serv.get(_INFO) is not None:
            try:
               self._serv_rcversion = '%.1f' % self._serv[_INFO]
            except:
               pass
            del self._serv[_INFO]
      if refresh:
         self._refresh_server_config()

   def get_server_rcinfo(self):
      """
      Return informations about the servers resource file.
      """
      return self._serv_rcversion

   def get_server_list(self):
      """
      Return the list of available servers.
      """
      self.init_server_config()
      return self._serv.keys()

   def get_server_config(self, server):
      """
      Return the configuration of 'server'
      """
      self.init_server_config()
      return self._serv.get(server, {})

   def save_server_config(self):
      """
      Write 'rcdir'/config_serveurs with current values.
      """
      content = ["""# AUTOMATICALLY GENERATED - DO NOT EDIT !
astkrc_version : %s
#""" % _RC_VERSION,]
      
      sorted_servers = [(self._serv[serv]['__id__'], serv) for serv in self._serv.keys()]
      sorted_servers.sort()
      for iocc, server in sorted_servers:
         lkeys = self._serv[server].keys()
         lkeys.sort()
         content.append('serveur : %s' % server)
         for key in lkeys:
            if key.startswith('__'):
               continue
            value = self._serv[server][key]
            if value == "":
               value = "_VIDE"
            if key in ('vers', 'noeud'):
               for val in value.split():
                  content.append('%s : %s' % (key, val))
            else:
               content.append('%s : %s' % (key, value))
         content.append("#")
      open(self.rcfile(SERVER_CONF), 'w').write(os.linesep.join(content))

   def _refresh_server_config(self):
      """
      Refresh configuration of each server presents in self._serv.
      """
      assert self.run is not None, "ASTER_RUN object necessary to call _refresh_server_config"
      for server in self.get_server_list():
         cfg = self._serv.get(server)
         if cfg.get('etat') != "on":
            continue
         cmd = '%s --info' % os.path.join(cfg['rep_serv'], 'bin', 'as_run')
         iret, output = self.run.Shell(cmd, mach=cfg['nom_complet'], user=cfg['login'])
         self._serv[server].update(self._parse_info(output))

   def _parse_info(self, content):
      """
      Parse information write by as_run --info
      """
      info = {}
      mat = re.search("@PARAM@(.*)@FINPARAM@", content, re.DOTALL)
      if mat is not None:
         for line in mat.group(1).splitlines():
            try:
               key, val = decode_config_line(line)
               info[key] = val
            except:
               pass
      mat = re.search("@VERSIONS@(.*)@FINVERSIONS@", content, re.DOTALL)
      if mat is not None:
         lvers = []
         for line in mat.group(1).splitlines():
            try:
               key, val = decode_config_line(line)
               lvers.append(val)
            except:
               pass
         info['vers'] = ' '.join(lvers)
      mat = re.search("@NOEUDS@(.*)@FINNOEUDS@", content, re.DOTALL)
      if mat is not None:
         lnode = []
         for line in mat.group(1).splitlines():
            try:
               key, val = decode_config_line(line)
               lnode.append(val)
            except:
               pass
         info['noeud'] = ' '.join(lnode)
      return info


if __name__ == "__main__":
   client = CLIENT(os.path.join(os.environ['HOME'], '.astkrc'))
   prefs = client.get_user_preferences()
   client.init_server_config() #force=True, refresh=True)
   servers = client.get_server_list()
   for serv in servers:
      cfg = client.get_server_config(serv)
      print serv, cfg

