/** \file libassogiate/mime-type.hh */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef LIBASSOGIATE__MIME_TYPE_HH
#define LIBASSOGIATE__MIME_TYPE_HH

#include <map>
#include <stdexcept>
#include <glibmm/refptr.h>
#include <glibmm/ustring.h>
#include <libxml++/exceptions/parse_error.h>
#include <libxml++/nodes/node.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

using Glib::RefPtr;
using Glib::ustring;

/******************************************************************************/
enum Location {
/******************************************************************************/

	NO_LOCATION = 0,
	SYSTEM_STANDARD = 1 << 0,
	SYSTEM_OVERRIDE = 1 << 1,
	USER_STANDARD = 1 << 2,
	USER_OVERRIDE = 1 << 3

}; /* enum Location */

inline Location
operator|(Location lhs, Location rhs)
{	return Location(unsigned(lhs) | unsigned(rhs)); }

inline Location
operator&(Location lhs, Location rhs)
{	return Location(unsigned(lhs) & unsigned(rhs)); }

inline Location
operator^(Location lhs, Location rhs)
{	return Location(unsigned(lhs) ^ unsigned(rhs)); }

inline Location
operator~(Location flags)
{	return Location(~unsigned(flags)); }

inline Location&
operator|=(Location& lhs, Location rhs)
{	return (lhs = Location(unsigned(lhs) | unsigned(rhs))); }

inline Location&
operator&=(Location& lhs, Location rhs)
{	return (lhs = Location(unsigned(lhs) & unsigned(rhs))); }

inline Location&
operator^=(Location& lhs, Location rhs)
{	return (lhs = Location(unsigned(lhs) ^ unsigned(rhs))); }

/******************************************************************************/
struct Message : public std::map<ustring, ustring> {
/******************************************************************************/

	ustring get_untranslated() const throw();
	ustring get_translated() const throw();
	
	void set_from_user(const ustring& value) throw();

}; /* struct Message */

class MagicMatch; /* defined below */

/******************************************************************************/
struct MatchSet : public xmlpp::NonCopyable {
/******************************************************************************/

	virtual ~MatchSet();

	std::list<MagicMatch*> submatches;

}; /* struct MatchSet */

/******************************************************************************/
struct MagicMatch : public MatchSet {
/******************************************************************************/

	enum Type {
		STRING,
		HOST16, HOST32,
		BIG16, BIG32,
		LITTLE16, LITTLE32,
		BYTE
	};
	
	Type type;
	int first_offset;
	int last_offset;
	ustring value, mask;

}; /* struct MagicMatch */

/******************************************************************************/
struct Magic : public MatchSet {
/******************************************************************************/

	int priority;

}; /* struct Magic */

/******************************************************************************/
struct XMLRoot {
/******************************************************************************/

	bool operator==(const XMLRoot& other) const throw();

	ustring namespace_uri;
	ustring local_name;

}; /* struct XMLRoot */

/******************************************************************************/
class MimeType : public xmlpp::NonCopyable {
/******************************************************************************/

public:

	static std::pair<ustring, ustring> split_name(const ustring& full_name)
		throw(std::invalid_argument);

	MimeType(const ustring& type, const ustring& subtype) throw();
	virtual ~MimeType();
	
	ustring m_type, m_subtype;
	ustring get_full_name() const throw();
	ustring get_icon_name() const throw();

	Message m_description;
	
	std::list<ustring> m_aliases, m_superclasses;

	std::list<ustring> m_glob_patterns;
	std::list<Magic*> m_magics;
	std::list<XMLRoot> m_xml_roots;
	
	Location get_locations() const throw();
	bool is_in_location(Location location) const throw();
	MimeType* limit_locations(Location locations) const throw();
	
	void accumulate(const xmlpp::Node& node, Location source)
		throw(xmlpp::exception);

	void output(xmlpp::Element& node) const throw();

private:

	static MagicMatch* parse_match(const xmlpp::Element& node)
		throw(xmlpp::exception);

	static void output_match(xmlpp::Element& parent, const MagicMatch& match)
		throw();
	
	typedef std::multimap<Location, const xmlpp::Node*> LocationNodesMap;
	LocationNodesMap m_location_nodes;

}; /* class MimeType */

typedef std::multimap<ustring, std::pair<xmlpp::Node*, Location> > MimeNodeMap;

/******************************************************************************/
class NodeMapExtender {
/******************************************************************************/

public:

	virtual ~NodeMapExtender();

	virtual void extend_node_map(MimeNodeMap& node_map) throw() = 0;

}; /* class NodeMapExtender */

} /* namespace assoGiate */

#endif /* LIBASSOGIATE__MIME_TYPE_HH */
