// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/CCInstantiation.h"
#include "Puma/CStructure.h"
#include "Puma/CClassInstance.h"
#include "Puma/CUnionInstance.h"
#include "Puma/CFctInstance.h"
#include "Puma/CNamespaceInfo.h"
#include "Puma/CTemplateInfo.h"
#include "Puma/CTemplateParamInfo.h"
#include "Puma/CTree.h"
#include "Puma/CCOverloading.h"
#include "Puma/CConstant.h"
#include "Puma/CCConversions.h"
#include "Puma/CUnit.h"
#include "Puma/CCParser.h"
#include "Puma/TokenStream.h"
#include "Puma/UnitManager.h"
#include "Puma/CSemDatabase.h"
#include "Puma/PreMacroManager.h"
#include "Puma/CTranslationUnit.h"
#include "Puma/PreprocessorParser.h"
#include "Puma/CFileInfo.h"
#include "Puma/CSourceInfo.h"
#include "Puma/CCConversions.h"
#include "Puma/TemplateInstanceUnit.h"

#include <stdio.h>      /* sprintf() */
#include <stdlib.h>     /* strtol() */
#include <sstream>      /* ostringstream */
using namespace std;

namespace Puma {


/*DEBUG*/int TRACE_INSTANCE_CODE = 0;
/*DEBUG*/int TRACE_PARSE_INSTANCE = 0;


#define SEM_ERROR(loc__,mesg__) \
  {if (report) {\
    err << sev_error << loc__->token ()->location () \
        << mesg__ << endMessage;}}


CCInstantiation::CCInstantiation (ErrorSink &e, bool rep) : err (e) {
  instance      = 0;
  last_instance = 0;
  report        = rep;
  trans_unit    = 0;
  current_scope = 0;

  base_candidate = new InstantiationCandidate;
  candidates[0] = base_candidate;
  base_candidate->initialize (rep ? &e : (ErrorSink*)0);
}


CCInstantiation::~CCInstantiation () {
  for (long i = 0; i < candidates.length (); i++) {
    InstantiationCandidate *candidate = candidates.lookup (i);
    if (candidate == base_candidate)
      base_candidate = 0;
    delete candidate;
  }
  if (base_candidate)
    delete base_candidate;
}


// start instantiation process
CObjectInfo *CCInstantiation::instantiate (CTree *node,
 CObjectInfo *binfo, bool real_inst, CStructure *cs, bool inst_immediately) {
  // don't call twice
  if (instance || ! binfo || ! node)
    return instance;

  First ().initialize (node, binfo);
  current_scope = cs;

  if (! ObjectInfo ()->SemDB ()->Project ()->config ().Option ("--real-instances"))
    real_inst = false;

  // §14.8.2 deduce template arguments
  if (! First ().deduceArguments (real_inst))
    return instance;

  // choose the appropriate template specialization (if any)
  if (! chooseSpecialization ()) {
    return instance;
  }

  // instantiate template if not already done before
  if (! alreadyInstantiated (real_inst)) {
    // create an instance of the template
    createPseudoInstance ();

    if (instance) {
      // do not delete the deduced arguments
      First ().forgetDeducedArgs ();
      if (base_candidate != &First ())
        base_candidate->forgetDeducedArgs ();

      instance->TemplateInstance ()->canInstantiate (real_inst);

      if (real_inst && inst_immediately) {
        instance = instance->TemplateInstance ()->instantiate (cs) ? instance : 0;
      }
    }
  }

  return instance;
}


// §14.5.4.1 matching of template partial specializations
bool CCInstantiation::chooseSpecialization () {
  bool no_default;
  unsigned entries;
  CObjectInfo *info;
  CTemplateInfo *tinfo;
  CT_TemplateName *name;
  CT_TemplateArgList *args;

  // only class templates can be partially specialized
  if (ObjectInfo ()->FunctionInfo ())
    return true;

  // iterate specializations
  for (unsigned i = 0; i < TemplateInfo ()->Specializations (); i++) {
    tinfo = TemplateInfo ()->Specialization (i);
    if (! tinfo->ObjectInfo ()) // internal error
      continue;
    info = tinfo->ObjectInfo ()->DefObject ();
    name = tinfo->SpecializationName ();
    if (! name) // internal error
      continue;
    args = name->Arguments ();
    if (! args) // internal error
      continue;

    // create new instantiation candidate
    InstantiationCandidate *cand = new InstantiationCandidate;
    cand->initialize (PointOfInstantiation (), info, tinfo, report ? &err : (ErrorSink*)0);

    // add the template arguments of the partial specialization
    entries = args->Entries ();
    for (unsigned j = 0; j < entries; j++)
      cand->addArgument (args->Entry (j));
    // add default arguments of the base template if not overwritten
    no_default = false;
    for (unsigned j = entries; j < TemplateInfo ()->Parameters (); j++) {
      // no default argument? this is an error
      if (! TemplateInfo ()->Parameter (j)->DefaultArgument ()) {
        no_default = true;
        cand->reset ();
        delete cand;
        break;
      }
      cand->addArgument (TemplateInfo ()->Parameter (j)->DefaultArgument ());
    }
    if (no_default)
      continue;
    
    // §14.5.4.1.2 try to match the partial specialization 
    // against the actual template argument list
    if (! cand->match ()) {
      // does not match, not a candidate for instantiation
      cand->reset ();
      delete cand;
    } else {
      candidates[candidates.length ()] = cand;
    }
  }
  
  // if exactly one matching specialization is found, the 
  // instantiation is generated from that specialization
  if (candidates.length () == 2) {
    // remove the base template as candidate
    candidates.remove (0); 
  // if more than one specialization is matching, the
  // best specialization is chosen using the partial 
  // ordering rules (§14.5.4.2 and §14.5.5.2)
  } else if (candidates.length () > 2) {
    return chooseBestSpecialization ();
  }
  
  return true;
}


bool CCInstantiation::chooseBestSpecialization () {
  // run a tournament to choose the best specialization 
  // comparing two specializations using the partial 
  // ordering rules (§14.5.4.2 and §14.5.5.2)
  InstantiationCandidate *challenger, *champion;
  unsigned num;

  num = candidates.length ();
  champion = candidates.lookup (num-1);
  for (unsigned i = num-1; i > 1; i--) {
    // current challenger
    challenger = candidates.lookup (i-1); 
    
    // let the champion face the challenger 
    switch (champion->compare (*challenger)) {
      // champion has won the round 
      case 1: 
        // elliminate challenger
        delete candidates.lookup (i-1); 
        candidates.remove (i-1); 
        break;
      // challenger has won the round
      case -1:
        // elliminate old champion
        delete candidates.lookup (i); 
        candidates.remove (i); 
        // challenger is the new champion
        champion = challenger; 
        break;
      // challenger and champion are equal
      default:
        // case 0: ambiguous up till now
        champion = challenger; 
        break;
    }
  }
  
  // verify that the champion is better than all other remaining 
  // candidates (the champion did not yet faced)
  for (unsigned i = candidates.length (); i > 1; i--) {
    challenger = candidates.lookup (i-1); 
    if (challenger == champion)
      continue;
      
    if (champion->compare (*challenger) == 1) {
      delete candidates.lookup (i-1); 
      candidates.remove (i-1);
    }
  }
  
  // if exactly one specialization left over, the 
  // instantiation is generated from that specialization
  if (candidates.length () == 2) {
    // remove the base template as candidate
    candidates.remove (0); 
    return true;
  } 

  // more than one specialization left over,
  // so the instantiation is ambiguous 
  if (report) {
    std::ostringstream name;
    name << ObjectInfo ()->Name ();
    base_candidate->printArgumentList (name);
    err << sev_error << getPoiToken()->location ()
        << "Instantiation of `" << name.str ().c_str ()
        << "' is ambiguous" << endMessage;
    for (int i = 1; i < candidates.length (); i++) {
      InstantiationCandidate *cand = candidates[i];
      if (cand->ObjectInfo ()->Tree () && cand->ObjectInfo ()->Tree ()->token ())
        err << cand->ObjectInfo ()->Tree ()->token ()->location ();
      if (i == 1)
        err << "candidates are: ";
      else
        err << "                ";
      err << name.str ().c_str () << endMessage;
    }
  }

  return false;
}


bool CCInstantiation::alreadyInstantiated (bool real_inst) {
  last_instance = 0;
  for (unsigned i = 0; i < TemplateInfo ()->Instances (); i++) {
    CObjectInfo *ti = TemplateInfo ()->Instance (i);
    bool cont = false;

    if ((TemplateInfo ()->isBaseTemplate () ==
         ti->TemplateInstance ()->Template ()->isBaseTemplate ()) &&
        (DeducedArgs () == ti->TemplateInstance ()->DeducedArgs ())) {
      for (unsigned j = 0; j < ti->TemplateInstance ()->DeducedArgs (); j++)
        if (*DeducedArg (j) != *ti->TemplateInstance ()->DeducedArg (j))
          cont = true;
    } else
      cont = true;

    if (! cont) {
      instance = ti;
      if ((ti->FunctionInfo () && ti->FunctionInfo ()->isFctDef ()) ||
          (ti->Record () && ti->Record ()->isDefined ())) {
        last_instance = 0;
        return true;
      } else
        last_instance = ti;
    }
  }

  instance = 0;
  if (last_instance) {
    if (real_inst &&
        ((ObjectInfo ()->FunctionInfo () && ObjectInfo ()->FunctionInfo ()->isFctDef ()) ||
         (ObjectInfo ()->Record () && ObjectInfo ()->Record ()->isDefined ())))
      return false;
    instance = last_instance;
    last_instance = 0;
    return true;
  }
  return false;
}


void CCInstantiation::createPseudoInstance () {
  CTemplateInstance *ti = 0;
  if (ObjectInfo ()->FunctionInfo ()) {
    instance = makeScope ()->newFunction (true);
    ti = instance->TemplateInstance ();
  } else if (ObjectInfo ()->UnionInfo ()) {
    instance = makeScope ()->newUnion (true);
    ti = instance->TemplateInstance ();
  } else if (ObjectInfo ()->ClassInfo ()) {
    instance = makeScope ()->newClass (true);
    ti = instance->TemplateInstance ();
  } else if (ObjectInfo ()->TemplateParamInfo ()) {
    instance = ObjectInfo ()->TemplateParamInfo ()->TemplateTemplate ()->newTemplateParam (false);
    instance->TemplateParamInfo ()->TemplateInstance (new CTemplateInstance);
    instance->TemplateParamInfo ()->isTypeParam (ObjectInfo ()->TemplateParamInfo ()->isTypeParam ());
    instance->TemplateParamInfo ()->ValueType (ObjectInfo ()->TemplateParamInfo ()->ValueType ()->Duplicate ());
    instance->TemplateParamInfo ()->TemplateTemplate (ObjectInfo ()->TemplateParamInfo ()->TemplateTemplate ());
    ObjectInfo ()->NextObject (instance);
    ti = instance->TemplateParamInfo ()->TemplateInstance ();
  } else {
    return; // unsupported object type
  }
  instance->Name (ObjectInfo ()->Name ());  
  instance->SemDB (ObjectInfo ()->SemDB ());
  instance->SourceInfo ()->FileInfo (ObjectInfo ()->SourceInfo ()->FileInfo ());
  instance->SourceInfo ()->StartToken (PointOfInstantiation ()->token_node ());
  instance->TypeInfo (ObjectInfo ()->TypeInfo ()->Duplicate ());
  ti->PointOfInstantiation (PointOfInstantiation (), current_scope);
  ti->Template (TemplateInfo ());
  ti->Object (instance);
  ti->isPseudoInstance (true);
  if (instance->FunctionInfo ())
    instance->TypeInfo ()->TypeFunction ()->FunctionInfo (instance->FunctionInfo ());
  else if (instance->TemplateParamInfo ())
    instance->TypeInfo ()->TypeTemplateParam ()->TemplateParamInfo (instance->TemplateParamInfo ());
  else
    instance->TypeInfo ()->TypeRecord ()->Record (instance->Record ());
  for (unsigned i = 0; i < DeducedArgs (); i++)
    ti->addDeducedArg (DeducedArg (i));
  for (unsigned i = 0; i < base_candidate->DeducedArgs (); i++)
    ti->addInstantiationArg (base_candidate->DeducedArg (i));
  TemplateInfo ()->addInstance (instance);
}


void CCInstantiation::setupFromPseudoInstance (CTemplateInstance* instinfo, CStructure* scope) {
  instance = instinfo->Object ();
  current_scope = (scope && scope == instance) ? scope->Parent ()->Structure () : scope;
  CObjectInfo* defobj = instinfo->Template ()->ObjectInfo ()->DefObject ();
  base_candidate->initialize (instinfo->PointOfInstantiation (), defobj, defobj->Template ());
  for (unsigned i = 0; i < instinfo->DeducedArgs (); i++)
    base_candidate->addDeducedArg (instinfo->DeducedArg (i));
}


bool CCInstantiation::instantiate (CTemplateInstance* instinfo, CStructure* instscope) {
  CTranslationUnit *tu;
  CStructure *scope;
  CProject *project;
  bool fct_inst;
  CUnit *unit;

  // setup from the pseudo instance information to
  // reset the correct instantiation information
  setupFromPseudoInstance (instinfo, instscope);

  // check maximal instantiation depth
  if (maxInstDepthReached ()) {
    First ().forgetDeducedArgs ();
    removeInstance ();
    return false;
  }

  // increase instantiation depth
  TemplateInfo ()->increaseDepth ();

  scope = instance->TemplateParamInfo () ? makeScope () : instance->Scope ()->Structure ();
  fct_inst = (ObjectInfo ()->FunctionInfo ());
  project = ObjectInfo ()->SemDB ()->Project ();
  unit = new TemplateInstanceUnit (err, getPoiToken()->unit(), instinfo);
  unit->scanner ().configure (project->config ());
  unit->name (scope->Name ());

  // create code for the template instance
  fillUnit (*unit);

  // setup new parser for the instantiated code
  trans_unit = tu = new CTranslationUnit (*unit, *project);
  CCParser p;
  p.configure (project->config ());
  if (TRACE_PARSE_INSTANCE) {
#ifdef __PUMA_TRACING__
    p.trace(std::cout);
#endif
  }

  // setup the preprocessor
  TokenStream stream;
  stream.push (unit); 
  project->unitManager ().init ();
  PreprocessorParser cpp (&project->err (), 
    &project->unitManager (), &tu->local_units ());
  cpp.macroManager ()->init (unit->name ());
  cpp.stream (&stream);
  cpp.configure (project->config (), false); // do not process --include option

  // initialize semantic analyzer
  p.semantic ().init (*ObjectInfo ()->SemDB (),
    *ObjectInfo ()->SourceInfo ()->FileInfo ()->Primary (), 
    scope, fct_inst, ! fct_inst, this);   
  ((ErrorCollector&)p.builder ().err ()).index (0);
  p.semantic ().error_sink (p.builder ().err ());

  // start parsing 
  TokenProvider provider (cpp);
  tu->tree (p.syntax ().run (provider)); 
  tu->cpp_tree (cpp.syntaxTree ());

  // decrease instantiation depth
  TemplateInfo ()->decreaseDepth ();

  // report errors and clean up
  if (((ErrorCollector&)p.builder ().err ()).severity () > sev_warning) {
    removeInstance ();
    if (report) {
      err << getPoiToken()->location ()
          << "In instantiation of `" << scope->Name ().c_str ()+1 
          << "':" << endMessage;
      p.builder ().errors (err);
    }
    //delete trans_unit;
    //trans_unit = 0;
    //delete unit;
  // no errors detected
  } else {
    if (instance) {
      //instance->TemplateInstance ()->canDelete ();
    } else {
      //delete trans_unit;
      //trans_unit = 0;
      //delete unit;
    }
  }

  First ().forgetDeducedArgs ();
  return instance;
}


bool CCInstantiation::maxInstDepthReached () {
  // ISO says not more than 17
  unsigned max_depth = 17;
  // maximum set by user?
  const ConfOption *opt = TemplateInfo ()->SemDB ()->Project ()->config ().Option ("--template-depth");
  if (opt && opt->Arguments () > 0) {
    max_depth = strtol (opt->Argument (0), NULL, 10);
  }
  if (TemplateInfo ()->Depth () > max_depth) {
    CTemplateInstance* topinst = 0;
    Unit* unit = getPoiToken()->unit ();
    while (unit->isTemplateInstance ()) {
      TemplateInstanceUnit* tiunit = (TemplateInstanceUnit*)unit;
      topinst = tiunit->TemplateInstance ();
      unit = tiunit->ContainingUnit ();
    }
    if (topinst && topinst->Object()) {
      err << getPoiToken(topinst)->location ()
          << "In instantiation of `" << *topinst->Object()->TypeInfo()
          << "':" << endMessage;
    }
    err << sev_error << getPoiToken()->location ()
        << "maximum instantiation depth ("
        << max_depth << ") reached" << endMessage;
    return true;
  }
  return false;
}


Token* CCInstantiation::getPoiToken(CTemplateInstance* ti) {
  Token* tok = 0;
  if (ti) {
    CTree* poi = ti->PointOfInstantiation();
    do {
      tok = poi->token();
    } while (! tok && (poi = poi->Parent()));
  } 
  else {
    tok = First().getPointOfInstantiationToken();
  }
  return tok;
}


void CCInstantiation::insertInstance (CObjectInfo *info) {
  if (info && info->TemplateInstance ()) {
    instance = info;
    if (last_instance && last_instance != instance)
      instance->NextObject (last_instance);
    instance->TemplateInstance ()->TranslationUnit (trans_unit);
  }
}


void CCInstantiation::removeInstance () {
  instance = 0;
}


CStructure *CCInstantiation::makeScope () {
  CStructure *scope;

  // create namespace name
  std::ostringstream sname;
  sname << "%" << ObjectInfo ()->Name ();
  base_candidate->printArgumentList (sname);

  scope = TemplateInfo ()->Parent ()->newNamespace ();
  scope->NamespaceInfo ()->aroundInstantiation (true);
  scope->Name (sname.str ().c_str ());
  scope->TypeInfo (&CTYPE_UNDEFINED);
  scope->SemDB (ObjectInfo ()->SemDB ());
  scope->SourceInfo ()->FileInfo (ObjectInfo ()->SourceInfo ()->FileInfo ());
  scope->SourceInfo ()->StartToken (PointOfInstantiation ()->token_node ());
  return scope;
}


void CCInstantiation::listParameters (CUnit &unit, CTemplateInfo *ti) {
  CTemplateParamInfo *tp;
  CT_TypeParamDecl *tpd;
  CT_NonTypeParamDecl *ntpd;
  const char* name;

  for (unsigned i = 0; i < ti->Parameters (); ++i) {
    if (i > 0) {
      unit << ",";
    }
    tp = ti->Parameter (i);
    char anonymous[100];
    name = tp->Name ();
    if (*tp->Name () == '%') {
      sprintf(anonymous, "__puma_%s", name+1);
      name = anonymous;
    }
    if (tp->isTypeParam ()) {
      tpd = (CT_TypeParamDecl*)tp->Tree ();
      copyTree (unit, tpd, findPrivateName (tpd->Name ()), name);
    } else {
      ntpd = (CT_NonTypeParamDecl*)tp->Tree ();
      copyTree (unit, ntpd, findPrivateName (ntpd->Declarator ()), name);
    }
  }
}


void CCInstantiation::fillUnit (CUnit &unit) {
  CTemplateParamInfo *param, *tp;
  DeducedArgument *arg;
  CTree *default_arg;
  CTemplateInfo *ti;

  for (unsigned i = 0; i < DeducedArgs (); i++) {
    arg = DeducedArg (i);
    param = arg->TemplateParam ();
    if (! param)
      continue;

    // get the name of the parameter, generate one
    // if the parameter is anonymous
    char anonymous[100];
    const char* name = param->Name ();
    if (*name == '%') {
      sprintf(anonymous, "__puma_%s", name+1);
      name = anonymous;
    }

    // deduced template-template argument
    if (param->isTemplate ()) {
      ti = param->TemplateTemplate ();
      unit << "template< ";
      listParameters(unit, param->TemplateTemplate ());
      unit << " > struct " << name;
      unit << " {\n  typedef ";
      if (arg->isDefaultArg () && arg->TemplateArg ()) {
        copyTree (unit, arg->TemplateArg (), arg->TemplateArg ()->end_token ());
      } else if (arg->Type ()) {
        arg->Type ()->TypeText (unit, 0, true);
      }
      unit << "<";
      for (unsigned j = 0; j < ti->Parameters (); ++j) {
        if (j > 0) {
          unit << ",";
        }
        tp = ti->Parameter (j);
        if (*tp->Name () != '%') {
          unit << tp->Name ();
        } else {
          unit << "__puma_" << tp->Name ()+1;
        }
      }
      unit << "> __puma_redirect;\n};\n" << endu;
    // deduced type argument
    } else if (param->isTypeParam ()) {
      // default template argument
      if (arg->isDefaultArg () && arg->TemplateArg ()) {
        unit << "typedef ";
        default_arg = arg->TemplateArg ();
        copyTree (unit, default_arg, findPrivateName (default_arg), name);
        unit << ";\n" << endu;
      } else if (arg->Type ()) {
        unit << "typedef ";
        arg->Type ()->TypeText (unit, name, true);
        unit << ";\n" << endu;
      }
    // deduced non-type argument
    } else {
      unit << "static ";
      CT_NonTypeParamDecl* decl = (CT_NonTypeParamDecl*)param->Tree ();
      copyTree (unit, decl->DeclSpecs (), 0, name);
      unit << " ";
      copyTree (unit, decl->Declarator (), findName (decl->Declarator ()), name,
        param->ValueType () && ! param->ValueType ()->isConst ());
      // default template argument
      if (arg->isDefaultArg () && arg->TemplateArg ()) {
        unit << " = ";
        copyTree (unit, arg->TemplateArg (), arg->TemplateArg ()->end_token ());
      } else if (arg->Value ()) {
        unit << " = " << *(arg->Value ());
      }
      unit << ";\n" << endu;
    }
  }

  // check for a template parameter in the class name -> desturbs
  Array<CTree*> desturbing;
  if (ObjectInfo ()->Tree ()->NodeName () == CT_ClassDef::NodeId ()) {
    CT_ClassDef *clsdef = (CT_ClassDef*)ObjectInfo ()->Tree ();
    if (clsdef->Name ()->NodeName () == CT_TemplateName::NodeId ())
      desturbing.append (((CT_TemplateName*)clsdef->Name ())->Arguments ());
    CT_MembList *members = clsdef->Members ();
    if (members) { // normally true, but in case of parse errors (?) ...
      for (int i = 0; i < members->Entries (); i++) {
        if (members->Entry (i)->NodeName () == CT_FctDef::NodeId ()) {
          CT_FctDef *fctdef = (CT_FctDef*)members->Entry (i);
          CT_SimpleName *nm =((CT_Declarator*)fctdef->Declarator ())->Name ();
          if (nm->NodeName () == CT_TemplateName::NodeId ())
            desturbing.append (((CT_TemplateName*)nm)->Arguments ());
        }
      }
    }
  }

  // copy the object declaration without the template header
  if (TemplateInfo ()->Tree ()) {
    CTree* decl = TemplateInfo ()->Tree ()->Declaration ();
    copyTree (unit, decl, decl->end_token(), &desturbing);
  }

  unit << "\n" << endu;

  if (TRACE_INSTANCE_CODE) {
    cout << endl;
    if (PointOfInstantiation ()) {
      cout << getPoiToken()->location () << ": ";
    }
    cout << unit.name ();
    if (instance && instance->Scope () && instance->Scope ()->Scope ()) {
      cout << ": instantiated in scope " << instance->Scope ()->Scope ()->QualName (true);
    }
    cout << endl;
    if (TemplateInfo ()->Tree () && TemplateInfo ()->Tree ()->token () && TemplateInfo ()->ObjectInfo ()) {
      cout << TemplateInfo ()->Tree ()->token ()->location () << ": instantiation of template " 
           << TemplateInfo ()->ObjectInfo ()->QualName (true) << endl;
    }
    cout << "-------------------" << endl;
    unit.print (cout);
    cout << "-------------------\n" << endl;
  }
}


void CCInstantiation::copyTree (CUnit &unit, CTree *tree, CTree *sn, const char *name, bool make_const) {
  // replace name with name of the current template parameter
  if (tree == sn) {
    unit << " ";
    if (make_const) 
      unit << "const ";
    if (sn->NodeName () == CT_PrivateName::NodeId ())
      unit << name << " ";
  // replace template template parameter by its deduced value
  } else if (tree->NodeName () == CT_SimpleName::NodeId ()) {
    replaceName (unit, (CT_SimpleName*)tree, 0);
    return;
  // copy the rest
  } else if (tree->NodeName () == CT_Token::NodeId ()) {
    Token* t = tree->token ();
    if (t) {
      unit << t->text ();
      Token* next = t->unit()->next(t);
      if (next && (next->is_whitespace() || next->is_comment()))
        unit << (next->is_whitespace() ? next->text() : " ");
      else if (! next)
        unit << " ";
    }
  }

  for (unsigned i = 0; i < (unsigned)tree->Sons (); i++) {
    copyTree (unit, tree->Son (i), sn, name, make_const);
  }

  if (tree->NodeName () == CT_TemplateName::NodeId ()) {
    extendTemplateName (unit, (CT_TemplateName*)tree);
  }
}


void CCInstantiation::copyTree (CUnit &unit, CTree *tree, Token* end_token, Array<CTree*> *desturbing) {
  // first check if this subtree is disturbing and should
  // be omitted in the generated unit
  if (desturbing)
    for (int i = 0; i < desturbing->length (); i++)
      if (desturbing->lookup (i) == tree)
        return; // prune here

  // replace template template parameter by its deduced value
  if (tree->NodeName () == CT_SimpleName::NodeId ()) {
    replaceName (unit, (CT_SimpleName*)tree, end_token);
    return;
  // copy the rest
  } else if (tree->NodeName () == CT_Token::NodeId ()) {
    Token* t = tree->token ();
    if (t) {
      unit << t->text ();
      if (t != end_token) {
        Token* next = t->unit()->next(t);
        if (next && (next->is_whitespace() || next->is_comment()))
          unit << (next->is_whitespace() ? next->text() : " ");
        else if (! next)
          unit << " ";
      }
    }
  }

  CTree* body = 0;
  if (tree->NodeName () == CT_FctDef::NodeId ()) {
    body = ((CT_FctDef*)tree)->Body ();
  }

  for (unsigned i = 0; i < (unsigned)tree->Sons (); i++) {
    CTree* son = tree->Son (i);
    if (body && body == son) {
      if (body->NodeName () == CT_Error::NodeId ()) {
        // if function body is error node => replace by ';'
        unit << ";\n";
      } else {
        // don't instantiate function body, instantiated on demand
        unit << "{}\n";
      }
      continue;
    }
    copyTree (unit, tree->Son (i), end_token, desturbing);
  }

  if (tree->NodeName () == CT_TemplateName::NodeId ()) {
    extendTemplateName (unit, (CT_TemplateName*)tree);
  }
}


void CCInstantiation::extendTemplateName (CUnit &unit, CT_TemplateName *name) {
  CObjectInfo *info = name->TemplateName ()->Object ();
  if (info) {
    // is template template parameter?
    CTemplateParamInfo *pinfo = info->TemplateParamInfo ();
    if (pinfo && pinfo->isTemplate ()) {
      // which template parameter is it?
      int pos = First ().getPosition (pinfo);
      if (pos != -1) {
        // extend the template name
        unit << "::__puma_redirect ";
      }
    }
  }
}


void CCInstantiation::replaceName (CUnit &unit, CT_SimpleName *name, Token *end_token) {
  bool replaced = false;
  
  CObjectInfo *info = name->Object ();
  if (info) {
    // is template template parameter?
    CTemplateParamInfo *pinfo = info->TemplateParamInfo ();
    if (pinfo && pinfo->isTemplate ()) {
      // which template parameter is it?
      int pos = First ().getPosition (pinfo);
      if (pos != -1 && name->Parent ()->NodeName () != CT_TemplateName::NodeId ()) {
        // get deduced argument for this parameter and insert it here
        DeducedArgument *darg = DeducedArg (pos);
        if (darg->TemplateArg ()) {
          copyTree (unit, darg->TemplateArg (), end_token);
          replaced = true;
        } else {
          replaced = darg->Type() || darg->Value();
          unit << *darg << " ";
        }
      }
    }
  }
  // not replaced, so copy it as it is
  if (! replaced) {
    for (unsigned i = 0; i < (unsigned)name->Sons (); i++) {
      copyTree (unit, name->Son (i), end_token);
    }
  }
}


CT_SimpleName *CCInstantiation::findPrivateName (CTree *node) const {
  const char *id = node->NodeName ();
  if (id == CT_PrivateName::NodeId ())
    return (CT_SimpleName*)node;
  else if (id == CT_NamedType::NodeId ())
    return findPrivateName (((CT_NamedType*)node)->Declarator ());
  else if (id == CT_FctDeclarator::NodeId ())
    return findPrivateName (((CT_FctDeclarator*)node)->Declarator ());
  else if (id == CT_ArrayDeclarator::NodeId ())
    return findPrivateName (((CT_ArrayDeclarator*)node)->Declarator ());
  else if (id == CT_PtrDeclarator::NodeId ())
    return findPrivateName (((CT_PtrDeclarator*)node)->Declarator ());
  else if (id == CT_MembPtrDeclarator::NodeId ())
    return findPrivateName (((CT_MembPtrDeclarator*)node)->Declarator ());
  else if (id == CT_BracedDeclarator::NodeId ())
    return findPrivateName (((CT_BracedDeclarator*)node)->Declarator ());
  else if (id == CT_BitFieldDeclarator::NodeId ())
    return findPrivateName (((CT_BitFieldDeclarator*)node)->Declarator ());
  else if (id == CT_RefDeclarator::NodeId ())
    return findPrivateName (((CT_RefDeclarator*)node)->Declarator ());
  else if (id == CT_InitDeclarator::NodeId ())
    return findPrivateName (((CT_InitDeclarator*)node)->Declarator ());
  return (CT_SimpleName*)0;
}


CT_SimpleName *CCInstantiation::findName (CTree *node) const {
  const char *id = node->NodeName ();
  if (node->IsSimpleName ())
    return (CT_SimpleName*)node;
  else if (id == CT_NamedType::NodeId ())
    return findName (((CT_NamedType*)node)->Declarator ());
  else if (id == CT_FctDeclarator::NodeId ())
    return findName (((CT_FctDeclarator*)node)->Declarator ());
  else if (id == CT_ArrayDeclarator::NodeId ())
    return findName (((CT_ArrayDeclarator*)node)->Declarator ());
  else if (id == CT_PtrDeclarator::NodeId ())
    return findName (((CT_PtrDeclarator*)node)->Declarator ());
  else if (id == CT_MembPtrDeclarator::NodeId ())
    return findName (((CT_MembPtrDeclarator*)node)->Declarator ());
  else if (id == CT_BracedDeclarator::NodeId ())
    return findName (((CT_BracedDeclarator*)node)->Declarator ());
  else if (id == CT_BitFieldDeclarator::NodeId ())
    return findName (((CT_BitFieldDeclarator*)node)->Declarator ());
  else if (id == CT_RefDeclarator::NodeId ())
    return findName (((CT_RefDeclarator*)node)->Declarator ());
  else if (id == CT_InitDeclarator::NodeId ())
    return findName (((CT_InitDeclarator*)node)->Declarator ());
  return (CT_SimpleName*)0;
}


} // namespace Puma
