/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	VCD document and view class
 *
 *	by Tony Sideris	(11:57AM Dec 05, 2001)
 *================================================*/
#include "arson.h"

#include <qabstractlayout.h>
#include <qtoolbutton.h>
#include <qfileinfo.h>
#include <qlineedit.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qframe.h>
#include <qcombobox.h>

#include <kfiledialog.h>
#include <kstddirs.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "tempfile.h"
#include "vcddoc.h"
#include "listwnd.h"
#include "process.h"
#include "mainwnd.h"

#define ARSONVCD_TEMPBASE		"vcd"

enum {
	VCDTYPE_VCD1 = 0,
	VCDTYPE_VCD2,
	VCDTYPE_SVCD,
	VCDTYPE_HQVCD,
	_VCDTYPE_MAX,
};

struct {
	QString desc;			//	Description
	const char *flag;		//	Command line switch to vcdxgen

}	g_vcd_types[_VCDTYPE_MAX] = {

	{ I18N_NOOP("VCD 1.1"), "vcd11" },
	{ I18N_NOOP("VCD 2.0 (default)"), "vcd2" },
	{ I18N_NOOP("SVCD"), "svcd" },
	{ I18N_NOOP("HQVCD"), "hqvcd" },	//	WTF is that ?!?
};

/*========================================================*/

class vcdProgressDlg : public ArsonImgCreateProgressDlg
{
public:
	vcdProgressDlg (ArsonVcdDoc *pDoc);

private:
	virtual ArsonProcessMgr *createProcessMgr (void);
	virtual void processOpts (ArsonProcessOpts &opts);

	ArsonVcdDoc *m_pDoc;
	QComboBox *m_pVcdType;
};

void vcdProgressDlg::processOpts (ArsonProcessOpts &opts)
{
	ArsonImgCreateProgressDlg::processOpts(opts);

	opts.addString("type", g_vcd_types[m_pVcdType->currentItem()].flag);
}

ArsonProcessMgr *vcdProgressDlg::createProcessMgr (void)
{
	return m_pDoc->createCdWriter(ui());
}

/*========================================================*/

vcdProgressDlg::vcdProgressDlg (ArsonVcdDoc *pDoc)
	: ArsonImgCreateProgressDlg(ArsonFrame::getFrame(), "vcddlg"), m_pDoc(pDoc)
{
	QLabel *pv = new QLabel(i18n("VCD Type:"), ctrlParent());

	pv->setSizePolicy(QSizePolicy(QSizePolicy::Maximum, QSizePolicy::Fixed));
	m_pVcdType = new ArsonProgressCombobox(ctrlParent(), "vcdtype");

	for (int index = 0; index < _VCDTYPE_MAX; ++index)
		m_pVcdType->insertItem(i18n(g_vcd_types[index].desc));

	m_pVcdType->setCurrentItem(VCDTYPE_VCD2);

	layoutRow() << pv << m_pVcdType;
	showLabelEdit();
}

/*========================================================*/
/*	VCD document class implementation
 *========================================================*/

ArsonVcdDoc::ArsonVcdDoc (QWidget *parent, const char *name)
	: ArsonFileListDoc(parent, name)
{
	//	Nothing...
}

/*========================================================*/

QString ArsonVcdDoc::propDocType (void) const
{
	return "vcd";
}

/*========================================================*/
/*	Factory functions
 *========================================================*/

ArsonProcessMgr *ArsonVcdDoc::createCdWriter (ArsonProcessUI *pUI)
{
	return new ArsonVcdWriter(pUI, this);
}

void ArsonVcdDoc::buildFileFilter (ArsonFileFilter &filter)
{
	filter.addFilter(i18n("*.mpg *.mpeg *.mpe|MPEG Files"));
}

ArsonProgressDlg *ArsonVcdDoc::createProgressDlg (void)
{
	return new vcdProgressDlg(this);
}

/*========================================================*/

void ArsonVcdDoc::addItem (ArsonListItem *ptr, ArsonLvPos *pp)
{
	KMessageBox::information(ArsonFrame::getFrame(),
		i18n("arson will create VCD image from the given MPEG files, but these files must already be in VCD format. arson performs no conversion on MPEG files. Continue?"),
		i18n("Warning"), "vcd-warning");

	ArsonFileListDoc::addItem(ptr, pp);
}

/*========================================================*/

QString ArsonVcdDoc::progressString (int length) const
{
	return QString(i18n("%1 total in VCD (of %2)"))
		.arg(arsonByteSize(length))
		.arg(arsonByteSize(getMaxProgress()));
}

/*========================================================*/

namespace arson {
	class vcdFileItem : public ArsonFileListFileItem
	{
	public:
		vcdFileItem (const KURL &url) : ArsonFileListFileItem(url) { }

		virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd)
		{
			static QPixmap pm;

			if (pm.isNull())
				pm = loadIcon("video");

			pi->setPixmap(0, pm);

			ArsonFileListFileItem::refresh(pi, pd);
		}
	};
};

ArsonFileListItem *ArsonVcdDoc::createFileItem (const KURL &url) const
{
	return new arson::vcdFileItem(url);
}

/*========================================================*/

void ArsonVcdDoc::create (void)
{
	static ArsonListHeader vcd_headers[] = {
		ArsonListHeader(i18n("Filename"), 85),
		ArsonListHeader(i18n("Size"), 15),
	};

	ArsonFileListDoc::create();

	listWnd()->setListHeaders(vcd_headers,
		ARRSIZE(vcd_headers));
}

/*========================================================*/
/*	VCD process manager class impl
 *========================================================*/

ArsonVcdWriter::ArsonVcdWriter (ArsonProcessUI *pUI, ArsonVcdDoc *pDoc)
	: ArsonCueWriter(pUI), m_pDoc(pDoc),
	m_bConverted(false), m_filebase(QString::null)
{
	//	Nothing...
}

ArsonVcdWriter::~ArsonVcdWriter (void)
{
	QFileInfo fi (QFile::encodeName(m_filebase + ".toc"));
	QDir dir = fi.dir();

	//	Remove all the temporary files created by vcdxbuild
	if (dir.exists())// && opts().getBool("deltmp"))
	{
		const QString filter = fi.baseName() + "*";
		ArsonFileCleanup() << dir.entryList(filter);
	}
#ifdef ARSONDBG
	else { Trace("ERROR: dir doesn't exist! (%s)\n", (const char *) m_filebase); }
#endif	//	ARSONDBG
}

/*========================================================*/

void ArsonVcdWriter::begin (const ArsonProcessOpts &opts)
{
	m_filebase = ArsonTempFile::tempFileName(ARSONVCD_TEMPBASE, "bin");
		//opts.getString("imgfile");
	
	if (m_filebase != QString::null)
	{
		ArsonCueWriter::begin(opts);

		//	Build a toc/bin file using the specified mpeg files
		try {
			ArsonVcdxbuildProcess *ptr = new ArsonVcdxbuildProcess(this, m_filebase);
			ArsonFileListItem *pItem;

			//	Add each mpeg file to the new process
			for (ArsonListDoc::ItemIterator it(m_pDoc); it; ++it)
				if (ArsonFileListFileItem *pi = (ArsonFileListFileItem *) it.item())
					ptr->addFile(pi->local());

			setProcess(ptr);
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
#ifdef ARSONDBG
	else { Trace("ERROR: FILEBASE not set!\n"); }
#endif	//	ARSONDBG
}

/*========================================================*/

void ArsonVcdWriter::taskComplete (ArsonProcess *ptr)
{
	ArsonCueWriter::taskComplete(ptr);

	if (!m_bConverted)// && !opts().getBool("noburn"))
	{
		//	Burn the new toc/bin to CD.
		m_bConverted = true;

		setCueFile(m_filebase + ".toc");
		writeCd();
	}
}

/*========================================================*/
