/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __KONFIG_H__
#define __KONFIG_H__

#include "arson.h"

#include <qvaluelist.h>
#include <qmap.h>
#include <qstringlist.h>

#include "mainwnd.h"
#include "device.h"
#include "lookup.h"

/*==================================*/
/*	DEFINES
 *==================================*/

#define ACONFIG		(g_CONFIG)
#define ACFGKEY(k)	(#k)

#if STDC_HEADERS
# include <string.h>
#else
# if !HAVE_STRCHR
#  define strchr index
#  define strrchr rindex
# endif
char *strchr (), *strrchr ();
# if !HAVE_MEMCPY
#  define memcpy(d, s, n) bcopy ((s), (d), (n))
#  define memmove(d, s, n) bcopy ((s), (d), (n))
# endif
#endif

enum {
	PROGGRP_AUDIOCD = 0,
	PROGGRP_DATACD,
	PROGGRP_CDCOPY,
	PROGGRP_MP3ENC,
	PROGGRP_DEVSCAN,
	PROGGRP_RIPPER,
	PROGSEQ_BLANK,
	_PROGGRP_MAX
};

enum {
	AUDIOSEQ_CDRDAO = 0,
	AUDIOSEQ_CDRECORD,
	_AUDIOSEQ_MAX
};

enum {
	DATASEQ_CDRECORD = 0,
	DATASEQ_CDRDAO,
	_DATASEQ_MAX
};

enum {
	COPYSEQ_CDRDAO = 0,
	COPYSEQ_CDRECORD,
	_COPYSEQ_MAX
};

enum {
	MP3ENCSEQ_BLADEENC = 0,
	MP3ENCSEQ_LAME,
	_MP3ENCSEQ_MAX
};

enum {
	DEVSEQ_CDRECORD = 0,
	DEVSEQ_CDRDAO,
	_DEVSEQ_MAX
};

enum {
	RIPPERSEQ_CDDA2WAV = 0,
	RIPPERSEQ_CDPARANOIA,
	_RIPPERSEQ_MAX
};

enum {
	BLANKSEQ_CDRECORD = 0,
	BLANKSEQ_CDRDAO,
	_BLANKSEQ_MAX
};

/*========================================================*/

class ArsonProgramSeqs
{
public:
	ArsonProgramSeqs (void);
	void save (KConfig *pk);
	void load (KConfig *pk);

	void setSeq (int grp, int seq) { m_seq[grp] = seq; }
	int seq (int grp) const { return m_seq[grp]; }

private:
	int m_seq[_PROGGRP_MAX];
};

/*========================================================*/

class ArsonConfigBase
{
public:
	ArsonConfigBase (const char *name)
		: m_name(name) { }

	virtual bool load (KConfig *pk);
	virtual void save (KConfig *pk, bool finalize = true);

protected:
	const char *m_name;
};

class ArsonFeatureConfig : public ArsonConfigBase
{
public:
	ArsonFeatureConfig (const char *name, uint flags = 0)
		: ArsonConfigBase(name), m_flags(flags) { }

	virtual bool load (KConfig *pk);
	virtual void save (KConfig *pk, bool finalize = true);
	
	bool is (uint mask) const { return (m_flags & mask) != 0; }
	void set (uint mask) { m_flags |= mask; }
	void clear (uint mask) { m_flags &= ~mask; }
	void toggle (uint mask) { m_flags ^= mask; }
	void fromBool (uint mask, bool val) { if (val) set(mask); else clear(mask); }
	uint flags (void) const { return m_flags; }
	void setFlags (uint flags) { m_flags = flags; }
	
private:
	const char *m_name;
	uint m_flags;
};

/*========================================================*/

class QComboBox;

class ArsonConfig
{
public:
	ArsonConfig (void);

	static const char *programHelpSwitch (int index);
	static QString programName (int index);
	static QString programDesc (int index);

	bool load (void);
	void save (bool finalize = true);

	void autoDetectPrograms (const QString &xtra = QString::null);
	void showMissingPrograms (void);

	int speed (void) const { return m_nSpeed; }
	int cdlenMin (void) const { return m_nCdLenMin; }
	int cdlenMB (void) const { return m_nCdLenMB; }
	int nice (void) const { return m_niceLevel; }
	int startDoc (void) const { return m_startDoc; }
	bool is (uint mask) const { return (m_flags & mask) != 0; }
	void set (uint mask) { m_flags |= mask; }
	void clear (uint mask) { m_flags &= ~mask; }
	void toggle (uint mask) { m_flags ^= mask; }
	void fromBool (uint mask, bool val) { if (val) set(mask); else clear(mask); }
	const char *device (void) const { return m_strDevice; }
	const QRect &geometry (void) const { return m_geometry; }
	const QString &extraProgramPaths (void) const { return m_extraPaths; }
	const QString &temporaryDirectory (void) const { return m_tempDirectory; }
	const QString &driver (void) const { return m_strDriver; }
	const QString &srcDriver (void) const { return m_strSrcDriver; }
	const ArsonDeviceList &devices (void) const { return m_devs; }
	ArsonDeviceList &devices (void) { return m_devs; }
	const ArsonProgramSeqs &seq (void) const { return m_seq; }
	ArsonProgramSeqs &seq (void) { return m_seq; }

	void setSpeed (int speed) { if (speed > 0) m_nSpeed = speed; }
	void setCdLenMin (int length) { if (length > 0) m_nCdLenMin = length; }
	void setCdLenMB (int length) { if (length > 0) m_nCdLenMB = length; }
	void setDevice (const char *sz) { if (sz) m_strDevice = sz; }
	void setNice (int level) { if (InArray(level, 21)) m_niceLevel = level; }
	void setStartDoc (int doc) { if (InArray(doc,_DOCUMENT_MAX)) m_startDoc = doc; }
	void setGeometry (const QRect &rect) { m_geometry = rect; }
	void setExtraProgramPaths (const QString &paths) { m_extraPaths = paths; }
	void setTemporaryDirectory (const QString &dir) { m_tempDirectory = dir; }
	void setDriver (const QString &drv) { m_strDriver = drv; }
	void setSrcDriver (const QString &drv) { m_strSrcDriver = drv; }

	class Program;
	class RipperCfg;

	const RipperCfg &ripper (void) const { return m_ripper; }
	RipperCfg &ripper (void) { return m_ripper; }

	const Program *program (int index) const;
	Program *program (int index);

	enum {
		flagResv1 = 0x0001,
		flagResv4 = 0x0002,
		flagResv5 = 0x0004,
		flagDndTip = 0x0008,
		flagResv2 = 0x0010,
		flagRenice = 0x0020,
		flagBurnProof = 0x0040,
		flagResv6 = 0x0080,
		flagResv3 = 0x0100,
		flagShowCmdLine = 0x0200,
		flagResv7 = 0x0400,
		flagMd5Verify = 0x0800,
		flagMd5Reset = 0x1000,
	};

	enum {
		PROGRAM_UNKNOWN = -1,
		PROGRAM_CDRECORD,
		PROGRAM_CDRDAO,
		PROGRAM_MKISOFS,
		PROGRAM_MPG123,
#ifdef OGG
		PROGRAM_OGG123,
#endif
#ifdef FLAC
		PROGRAM_FLAC,
#endif
		PROGRAM_CDDA2WAV,
		PROGRAM_CDPARANOIA,
		PROGRAM_BLADEENC,
		PROGRAM_LAME,
		PROGRAM_OGGENC,
		PROGRAM_READCD,
		PROGRAM_SHORTEN,
		PROGRAM_SHNLEN,
		PROGRAM_MD5SUM,
		PROGRAM_SOX,
		PROGRAM_NORMALIZE,
		PROGRAM_VCDXBUILD,
		PROGRAM_VCDXGEN,
		PROGRAM_ID3V2,
		PROGRAM_CDRINIT,
		PROGRAM_CDRFIN,
		_PROGRAMS_MAX
	};

	enum {
		privateInitCdr = 0x0001,
	};

	struct Program {
		const char *path (void) const { return m_program; }
		const char *params (void) const { return m_params; }
		bool valid (void) const { return arsonIsExecutable(m_program); }

		QString m_program;
		QString m_params;
	};

	class RipperCfg : public ArsonFeatureConfig
	{
	public:
		RipperCfg (const ArsonConfig *cfg);

		virtual bool load (KConfig *pk);
		virtual void save (KConfig *pk, bool finalize = true);

		QStringList recentCdiFormats (void) const;

		int format (void) const { return m_outputFormat; }
		const QString &outdir (void) const { return m_strOutDir; }
		const QString &cdiFormat (void) const { return m_cdiFormat; }
		const QString &email (void) const { return m_email; }
		const QString &srcdev (void) const { return m_srcdev; }
		const QString &quality (void) const { return m_quality; }
		const QString &defaultComment (void) const { return m_defaultComment; }
		const ArsonLookupOrder &lookups (void) const { return m_lookups; }
		void setFormat (int fmt) { m_outputFormat = fmt; }
		void setOutdir (const QString &dir) { m_strOutDir = dir; }
		void setEmail (const QString &email) { m_email = email; }
		void setLookups (const ArsonLookupOrder &lp) { m_lookups = lp; }
		void setSrcdev (const char *dev) { m_srcdev = dev; }
		void setQuality (const QString &qual) { m_quality = qual; }
		void setDefaultComment (const QString &cmt) { m_defaultComment = cmt; }
		void setCdiFormat (const QString &fmt);

		enum {
			ripperResv1 = 0x01,
			ripperRecv2 = 0x02,
			ripperResv3 = 0x04,
			ripperEmailPrompt = 0x08,
			ripperCdLookup = 0x10,
			ripperOverCdtext = 0x20,
		};

	private:
		const ArsonConfig *m_cfg;	//	The owner
		QString m_strOutDir;		//	Default output directory
		int m_outputFormat;			//	Default output format
		QString m_cdiFormat;		//	How to format filenames
		QString m_srcdev;			//	The source device (ioctl or scsi)
		QString m_email;			//	User email address
		QString m_quality;			//	The quality preset
		QString m_defaultComment;	//	The default track comment
		ArsonLookupOrder m_lookups;	//	CD Lookups
	};

private:
	QString detectProgram (const char *name, const QString &xtra = QString::null);
	bool validDeviceScannersExist (void) const;

	bool cdrInit (bool init);

	Program m_programs[_PROGRAMS_MAX];	//	External programs
	QString m_strDevice;				//	Selected SCSI device
	uint m_flags;						//	Various boolean options
	uint m_private;						//	Various booleans flags that dont persist
	int m_nSpeed;						//	Burn speed
	int m_nCdLenMin;					//	Max CD length in minutes
	int m_nCdLenMB;						//	Max CD length in MB
	QRect m_geometry;					//	Main frame geometry
	QString m_extraPaths;				//	Extra program paths
	QString m_tempDirectory;			//	User-specified temporary directory or NULL of none
	int m_niceLevel;					//	CD-writer priority
	int m_startDoc;						//	Initial document
	QString m_strDriver;				//	Writer cdrdao driver (--driver)
	QString m_strSrcDriver;				//	Source cdrdao driver (--source-driver)
	RipperCfg m_ripper;					//	Ripper configuration
	ArsonProgramSeqs m_seq;				//	Program sequences (what program to use)
	ArsonDeviceList m_devs;					//	CDROM and CDR devices
};

/*========================================================*/

class ArsonProgramHelp
{
public:
	ArsonProgramHelp (int program = ArsonConfig::PROGRAM_UNKNOWN);

	void show (QWidget *parent = NULL) const;
	int setProgram (int program);

	const QString &helpMsg (void) const { return m_help; }
	bool valid (void) const { return !m_help.isEmpty(); }

private:
	QString m_help;
	int m_program;
};

/*==================================*/
/*	GLOBALS
 *==================================*/

extern ArsonConfig g_CONFIG;

/*========================================================*/
#endif	/*	__KONFIG_H__	*/
