/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	The device page.
 *
 *	by Tony Sideris	(11:26PM Apr 28, 2002)
 *================================================*/
#include "arson.h"

#include <qpushbutton.h>
#include <qfileinfo.h>
#include <qdir.h>

#include <klineeditdlg.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kstddirs.h>
#include <klocale.h>

#include "devicepage.h"
#include "listwnd.h"
#include "konfig.h"

/*==========================================================*/
/*	Class Implementation
 *==========================================================*/

ArsonDevicePage::ArsonDevicePage (ArsonConfig &config, ArsonConfigDlg *pd, QWidget *parent)
	: ArsonDevicePageBase(parent), ArsonConfigPage(config), m_devs(config.devices())
{
	static ArsonListHeader hdrs[] = {
		ArsonListHeader(i18n("Device ID"), 25),
		ArsonListHeader(i18n("Device Path"), 25),
		ArsonListHeader(i18n("Description"), 50),
	};

	m_icon = KGlobal::iconLoader()->loadIcon(
		"cdrom_unmount", KIcon::Small);

	m_pIoctlHdr
		= m_pIoctlNone
		= m_pScsiHdr
		= m_pScsiNone
		= NULL;

	drvs->setSelectionMode(QListView::Single);
	drvs->setSorting(-1);
	drvs->setListHeaders(hdrs, ARRSIZE(hdrs));

	m_pScsiHdr = new QListViewItem(drvs, i18n("SCSI Devices"));
	m_pScsiHdr->setPixmap(0, m_icon);
	m_pIoctlHdr = new QListViewItem(drvs, i18n("IOCTL Devices"));
	m_pIoctlHdr->setPixmap(0, m_icon);

	m_pIoctlHdr->setOpen(true);
	m_pScsiHdr->setOpen(true);
	
	QObject::connect(drvs, SIGNAL(selectionChanged(QListViewItem*)),
		this, SLOT(item_selected(QListViewItem*)));

	fillIoctl();
	fillScsi();
}

/*==========================================================*/

bool ArsonDevicePage::inBranch (QListViewItem *item) const
{
	QListViewItem *psel = drvs->selectedItem();
	return (psel && psel->parent() == item);
}

QCString ArsonDevicePage::selectedKey (void) const
{
	if (QListViewItem *pi = drvs->selectedItem())
	{
		QListViewItem *parent = pi->parent();

		if (parent == m_pScsiHdr || parent == m_pIoctlHdr)
			return pi->text((parent == m_pScsiHdr) ? 0 : 1).latin1();
	}

	return QCString();
}

/*==========================================================*/

void ArsonDevicePage::rescan_clicked (void)
{
	m_devs.scanbus();
	fillScsi();

	emit devsChanged(m_devs);
}

/*========================================================*/

QString ArsonDevicePage::getDevPath (const QString &old)
{
	QDir dir ("/dev/");

	if (old != QString::null)
		dir = QFileInfo(old).dir();
	
	return KFileDialog::getOpenFileName(
		dir.exists() ? dir.absPath() : QString::null,
		QString::null, this,
		i18n("Select device path (ie: /dev/cdrom, /dev/sg0, etc)"));
}

/*==========================================================*/

void ArsonDevicePage::modify_clicked (void)
{
	const QCString sel = selectedKey();
	QListViewItem *pi = drvs->selectedItem();

	if (!sel.isEmpty() && pi)
	{
		const QString inp (getDevPath(pi->text(1)));

		if (inp == QString::null)
			return;

		if (ArsonDevice *pd = m_devs.device(sel))
		{
			if (!pd->scsiDevice())
			{
				QListViewItem *pi;

				m_devs.delIoctlDev(sel);
				m_devs.addIoctlDev(inp);
			}
			else
				pd->setDev(inp);

			pi->setText(1, inp);

			emit devsChanged(m_devs);
		}
	}
}

/*==========================================================*/

void ArsonDevicePage::add_ioctl (void)
{
	const QString inp (getDevPath());

	if (inp != QString::null && m_devs.addIoctlDev(inp))
	{
//		new QListViewItem(m_pIoctlHdr, i18n("n/a"), inp);
		fillIoctl();
		emit devsChanged(m_devs);
	}
}

void ArsonDevicePage::del_ioctl (void)
{
	if (inBranch(m_pIoctlHdr) && m_devs.delIoctlDev(selectedKey()))
	{
//		delete drvs->selectedItem();
		fillIoctl();
		emit devsChanged(m_devs);
	}
}

/*==========================================================*/

namespace arson {
	void clear_branch (QListViewItem *item)
	{
		QListViewItem *ptr;
		Assert(item != NULL);

		while ((ptr = item->firstChild()))
			delete ptr;
	}
};

void ArsonDevicePage::fillIoctl (void)
{
	const QStringList keys (m_devs.keys());
	QStringList::ConstIterator it, end;
	int count = 0;

	delete m_pIoctlNone;
	m_pIoctlNone = NULL;
	arson::clear_branch(m_pIoctlHdr);

	for (it = keys.begin(), end = keys.end(); it != end; ++it)
	{
		if (const ArsonDevice *pd = m_devs.device(*it))
		{
			if (pd->scsiDevice())
				continue;

			QListViewItem *pi = new QListViewItem(
				m_pIoctlHdr,
				i18n("n/a"), pd->dev(), pd->description());

			++count;
		}
	}

	if (!count)
		m_pIoctlNone = new QListViewItem(m_pIoctlHdr, i18n("<None>"));
}

void ArsonDevicePage::fillScsi (void)
{
	const QStringList keys (m_devs.keys());
	QStringList::ConstIterator it, end;
	int count = 0;

	delete m_pScsiNone;
	m_pScsiNone = NULL;
	arson::clear_branch(m_pScsiHdr);
	
	for (it = keys.fromLast(), end = keys.end(); it != end; --it)
	{
		if (const ArsonScsiDevice *pd = (const ArsonScsiDevice *) m_devs.device(*it))
		{
			if (!pd->scsiDevice())
				continue;

			QListViewItem *pi = new QListViewItem(m_pScsiHdr,
				pd->id(), pd->dev(), pd->description());

			++count;
		}
	}

	if (!count)
		m_pScsiNone = new QListViewItem(m_pScsiHdr, i18n("<None>"));
}

/*==========================================================*/

void ArsonDevicePage::item_selected (QListViewItem *item)
{
	/*	Will only be true if none of the header or
	 *	placeholder items have been selected, in
	 *	other words, a valid entry, not a support
	 *	item.
	 */
	const bool en =
		(item != m_pIoctlNone &&
		item != m_pIoctlHdr &&
		item != m_pScsiNone &&
		item != m_pScsiHdr);

	modify->setEnabled(en);

	del->setEnabled(m_pIoctlHdr &&
		inBranch(m_pIoctlHdr) &&
		item != m_pIoctlNone);

	drvs->clearSelection();
	item->setSelected(en);
}

/*==========================================================*/

void ArsonDevicePage::on_accept (void)
{
	cfg().devices() = m_devs;
}

/*========================================================*/
