/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Data CD document/view implementation
 *
 *	by Tony Sideris	(05:41AM Aug 02, 2001)
 *================================================*/
#include "arson.h"

#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qpopupmenu.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qheader.h>
#include <qxml.h>

#include <klineeditdlg.h>
#include <klocale.h>
#include <kaction.h>

#include "xmlwriter.h"
#include "datadoc.h"
#include "listwnd.h"
#include "mainwnd.h"
#include "process.h"
#include "isofs.h"
#include "tools.h"

#define ARSON_DEFAULT_DATA_LABEL		i18n("NewCD")

/*========================================================*/
/*	Data document class implementation
 *========================================================*/

ArsonDataDoc::ArsonDataDoc (QWidget *parent, const char *name)
	: ArsonFileListDoc(parent, name)
{
	setRenameColumn(0);
}

/*========================================================*/

QString ArsonDataDoc::propDocType (void) const
{
	return "data";
}

/*========================================================*/

namespace arson {
	bool recursiveSymLink (const QFileInfo &fi, const QDir &cwd)
	{
		if (fi.isSymLink())
		{
			const QDir root (QDir::rootDirPath());
			const QDir sym (fi.readLink());
			QDir copy (cwd.canonicalPath());

			Trace("Symlink checking (%s) in %s:\n",
				sym.absPath().latin1(),
				cwd.absPath().latin1());

			while (copy != root)
			{
				Trace("  checking %s\n", copy.absPath().latin1());

				if (sym.absPath() == copy.absPath())
					return true;

				copy.cdUp();
			}
		}

		return false;
	}
};

void ArsonDataDoc::addDirectory (const QString &dirName, ArsonListInserter &pos)
{
	const QDir dir (dirName);
	ArsonLvPos lvp (pos.getParent(), pos.getAfter());
	const QStringList sl (dir.entryList(/*QDir::DefaultFilter, QDir::DirsFirst*/));

	//	Add current directory (and cd to it)
	addDirItem(dirName, &lvp);
	pos.setParent(lvp.result);
	
	//	Recurse through the new directory
	for (QStringList::ConstIterator it = sl.begin(), end = sl.end();
		 it != end; ++it)
	{
		if ((*it) != "." && (*it) != "..")
		{
			QFileInfo fi (dir.filePath(*it));

			if (fi.isDir())
			{
				if (arson::recursiveSymLink(fi, dir))
				{
					arsonErrorMsg(
						i18n("Recursive symlink detected (%1), skipping...")
						.arg(fi.absFilePath()));
					
					continue;
				}

				addDirectory(fi.absFilePath(), pos);
			}
			else
			{
				ArsonLvPos lvp (pos.getParent(), pos.getAfter());

				if (addFileItem(KURL(QString("file:") + fi.absFilePath()), &lvp))
					pos.setAfter(lvp.result);
			}
		}
	}

	//	cd ..
	pos.clearParent();
}

/*========================================================*/

ArsonDocWidget::Status *ArsonDataDoc::initStatus (QStatusBar *psb)
{
	if (Status *ps = ArsonFileListDoc::initStatus(psb))
	{
/*
		QToolButton *pb = new QToolButton(this);

		QToolTip::add(pb, i18n("Calculate Actual ISO Size"));
		pb->setIconSet(QIconSet(ArsonFileListItem::loadIcon("1rightarrow"), QIconSet::Small));
		
		ps->addPane("isosize", i18n("Actual ISO Size: 0"));
		ps->addPane("isobtn", pb);

		QObject::connect(pb, SIGNAL(clicked()),
			this, SLOT(slotCalcIsoSize()));
*/
		
		return ps;
	}

	return NULL;
}

/*========================================================*/

void ArsonDataDoc::slotCalcIsoSize (void)
{
/*
	QStringList sl;
	graftPointReader rdr (sl);

	if (visitAll(rdr))
	{
		ArsonIsoSizeProcess proc (QString::null, &flags, sl);

		if (proc.execute())
		{
		}
	}
*/
}

/*========================================================*/
/*	Visit each node calling onDataItem for files
 *========================================================*/

ArsonDataContentsVisiter::ArsonDataContentsVisiter (void)
	: m_path("/")
{
	//	Nothing...
}

bool ArsonDataContentsVisiter::beginBranch (QListViewItem *pi, ArsonListItem *pl)
{
	if (ArsonFileTreeVisiter::beginBranch(pi, pl))
	{
		m_path.append(pi->text(0) + "/");
		return true;
	}
/*
	if (!pi->firstChild())
	{
		arsonErrorMsg(	//	mkisofs will die if given one...
			i18n("Empty directories are not allowed (%1).")
			.arg(path()));
	}
*/
	return false;
}

void ArsonDataContentsVisiter::endBranch (void)
{
	const int len = m_path.length();

	if (len > 1)
	{
		const int pos = m_path.findRev("/", m_path.length() - 2);
		Assert(pos != -1);

		m_path = m_path.left(pos + 1);
	}

	Trace("Path: %s\n", m_path.latin1());
	
	ArsonFileTreeVisiter::endBranch();
}

bool ArsonDataContentsVisiter::visitFile (QListViewItem *pi, ArsonFileListFileItem *pf)
{
	return onDataItem(pf->local(), path() + pi->text(0));
}

QString ArsonDataContentsVisiter::path (void) const
{
/*
	STRSTACK::ConstIterator it, end;
	QString result ("/");

	for (it = m_dirs.begin(), end = m_dirs.end(); it != end; ++it)
		result.append((*it) + "/");

	return result;
*/
	return m_path;
}

/*========================================================*/

void ArsonDataDoc::deleteContents (void)
{	
	ArsonFileListDoc::deleteContents();

	m_pRootItem->setText(0, ARSON_DEFAULT_DATA_LABEL);
}

/*========================================================*/

void ArsonDataDoc::connectTo (ArsonActionConnector &ac)
{
	ac.connect("list_folder_new", SLOT(slotNewFolder()));
	
	ArsonFileListDoc::connectTo(ac);

	ac.disconnect("list_down");
	ac.disconnect("list_up");
	ac.disconnect("list_shuffle");
}

/*========================================================*/

void ArsonDataDoc::slotNewFolder (void)
{
	bool ok;
	ArsonLvPos pos (this);
	const QString name = KLineEditDlg::getText(
		i18n("Name for new folder:"),
		i18n("New Folder"), &ok, this);

	if (ok && !name.isEmpty())
		addDirItem(name, &pos);
}

/*========================================================*/

bool ArsonDataDoc::onStartElement (const QString &name,
	const QXmlAttributes &attr, ArsonListInserter &pos)
{
	if (name == QString("arson-") + propDocType() && m_pRootItem)
	{
		const QString label = attr.value("label");

		if (!label.isEmpty())
			m_pRootItem->setText(0, label);
	}

	return ArsonFileListDoc::onStartElement(name, attr, pos);
}

void ArsonDataDoc::editRootElement (ArsonXmlWriter &xml)
{
	ArsonFileListDoc::editRootElement(xml);
	
	if (m_pRootItem)
		xml.addAttribute("label", m_pRootItem->text(0));
}

/*========================================================*/

QString ArsonDataDoc::isoLabel (void) const
{
	return m_pRootItem ? m_pRootItem->text(0) : QString::null;
}

QListViewItem *ArsonDataDoc::rootItem (void) const
{
	return m_pRootItem;
}

/*========================================================*/

namespace arson {
	class dataFileItem : public ArsonFileListFileItem
	{
	public:
		dataFileItem (const KURL &url) : ArsonFileListFileItem(url) { }

		virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd)
		{
			static QPixmap pm;

			if (pm.isNull())
				pm = loadIcon("mime_empty");

			pi->setPixmap(0, pm);
			pi->setText(0, display());
			pi->setText(1, KURL::decode_string(persist()));
			pi->setText(2, arsonByteSize(length()));
		}
	};

	class dataDirItem : public ArsonFileListDirItem
	{
	public:
		dataDirItem (const QString &dn) : ArsonFileListDirItem(dn) { }

		virtual void refresh (QListViewItem *pi, ArsonDocWidget *pd)
		{
			static QPixmap pm;

			if (pm.isNull())
				pm = loadIcon("folder");

			pi->setPixmap(0, pm);
			ArsonFileListDirItem::refresh(pi, pd);
			pi->setText(2, arsonByteSize(length()));
		}

		virtual uint length (void) const
		{
			return 2048;
		}
	};
};

ArsonFileListItem *ArsonDataDoc::createFileItem (const KURL &url) const
{
	ArsonFileListFileItem *pi = new arson::dataFileItem(url);
	ArsonCdImageFile img (pi->local());

	if (img.imgFormat() != ArsonCdImageFile::Unknown &&
		img.imgFormat() != ArsonCdImageFile::Dir)
	{
		const int res = KMessageBox::warningYesNo(kapp->mainWidget(), //KMessageBox::WarningYesNo,
			i18n("The file you are adding to this layout (%1) looks like an image file. In the future, to burn an image file, use the \"Create CD from Image\" command (in the File menu). Do you want to add this file to the current layout, or burn the image to CD?").arg(url.path()),
			i18n("Adding Image File"),
			i18n("&Add to Layout"),
			i18n("&Burn the Image..."),
			"add-image"
			);

		if (res == KMessageBox::No)
		{
			delete pi;
			
			img.write();
			return NULL;
		}
	}

	return pi;
}

ArsonFileListItem *ArsonDataDoc::createDirItem (const QString &dn) const
{
	return new arson::dataDirItem(dn);
}

/*========================================================*/

ArsonListWnd *ArsonDataDoc::createListWnd (void)
{
	static ArsonListHeader hdrs[] = {
		ArsonListHeader(i18n("Filename"), 40),
		ArsonListHeader(i18n("Source"), 45),
		ArsonListHeader(i18n("Size"), 15),
	};

	ArsonListWnd *ptr = ArsonFileListDoc::createListWnd();
	ptr->setListHeaders(hdrs, ARRSIZE(hdrs));
	ptr->setRootIsDecorated(true);

	ptr->setSortable(false);	//	arson sorting
	ptr->setSorting(0);			//	std sorting

	ptr->header()->setClickEnabled(true);
	ptr->setShowSortIndicator(true);

	m_pRootItem = new QListViewItem(ptr, ARSON_DEFAULT_DATA_LABEL);
	m_pRootItem->setPixmap(0,
		ArsonFileListItem::loadIcon("cdwriter_unmount"));
	m_pRootItem->setExpandable(true);
	m_pRootItem->setOpen(true);

	ptr->setItemsRenameable(true);
	ptr->setRenameable(0, true);
	ptr->setRenameable(1, false);
	ptr->setRenameable(2, false);

	return ptr;
}

/*========================================================*/
/*	The ISO process manager
 *========================================================*/

class isoDataMgr : public ArsonIsoMgr
{
public:
	isoDataMgr (ArsonProcessUI *pUI, const QStringList &gp)
		: ArsonIsoMgr(pUI, gp) { }

private:
	virtual ArsonMkisoProcess *mkisofsProcess (const ArsonIsoFlags *pFlags,
		const QString &outfile, const QStringList &what)
	{
		if (ArsonMkisoProcess *ptr = ArsonIsoMgr::mkisofsProcess(pFlags, outfile, what))
		{
			(*ptr) << "-graft-points";
			return ptr;
		}

		return NULL;
	}

	virtual ArsonIsoSizeProcess *sizeProcess (const ArsonIsoFlags *pFlags,
		const QString &outfile, const QStringList &what)
	{
		if (ArsonIsoSizeProcess *ptr = ArsonIsoMgr::sizeProcess(pFlags, outfile, what))
		{
			(*ptr) << "-graft-points";
			return ptr;
		}

		return NULL;
	}
};

/*========================================================*/
/*	Fill up the graft points
 *========================================================*/

class graftPointReader : public ArsonDataContentsVisiter
{
public:
	graftPointReader (QStringList &sl)
		: ArsonDataContentsVisiter(), m_sl(sl) { }

	void repl (QString &in, const char *str, const char *with)
	{
		int index = in.find(str);
		const int wlen = qstrlen(with);
		const int slen = qstrlen(str);

		while (index != -1)
		{
			in.replace(index, slen, with);
			index = in.find(str, index + wlen);
		}
	}

	bool onDataItem (const QString &src, const QString &dst)
	{
		QString s (src);
		QString d (dst);

		repl(s, "\\", "\\\\");
		repl(s, "=", "\\=");
		repl(d, "\\", "\\\\");
		repl(d, "=", "\\=");

		m_sl.append(d + "=" + s);
		return true;
	}

private:
	QStringList &m_sl;
};

/*========================================================*/
/*	The progress dialog for data CDs
 *========================================================*/

class dataCdDlg : public ArsonIsoProgressDlg
{
public:
	dataCdDlg (ArsonDataDoc *pd, QWidget *parent)
		: ArsonIsoProgressDlg(parent, "isodatacd")
	{
		graftPointReader rdr (m_graftPoints);

		pd->visitAll(rdr);
		setAutoIsoLabel(false);
		setIsoLabel(pd->isoLabel());
	}

private:
	virtual ArsonIsoMgr *createIsoMgr (void)
	{
		return new isoDataMgr(ui(), m_graftPoints);
	}

	QStringList m_graftPoints;
};

/*========================================================*/

ArsonProgressDlg *ArsonDataDoc::createProgressDlg (void)
{
	return new dataCdDlg(this, this);
}

/*========================================================*/
